;-*-SCHEME-*-

;;;
;;; StoK: Scheme to KL1 translator
;;;   Copyright (C) 1997 Yoshihiro Oyama
;;;     <oyama@is.s.u-tokyo.ac.jp>
;;;       December 19, 1997
;;;

;;
;; R4RS library
;;
;; We implement some R4RS primitives by a combination of
;; 'kernel' primitives in StoK.
;;

(module r4rs)

;; quote cannot be used as a procedure like
;; (map quote '(1 2 3))
;; (define (quote x) (%quote x))

;; They also cannot be used as a procedure like
;; (cons lambda 123)
; lambda, if, set!, global-ref, cond

;; They can be used as a procedure
(define (and x y) (%and x y))
(define (or x y) (%or x y))

;; They also cannot be used as a procedure like
;; (cons let 123)
; let, let*, begin

;; They can be used as a procedure
(define (not x) (%not x))
(define (boolean? x) (%boolean? x))
(define (pair? x) (%pair? x))
(define (car x) (%car x))
(define (cdr x) (%cdr x))
(define (cadr x) (%cadr x))
(define (caddr x) (%caddr x))

(define (null? x) (%null? x))
;; list can be used as a procedure in R4RS like
;; (apply list '(1 2 3)), but we dont currently support that.
; list
 
;(define (length lis)
;  (if (null? lis)
;      0
;      (+ 1 (length (cdr lis)))))
(define (length x) (%length x))

;(define (append lis1 lis2)
;  (if (null? lis1)
;      lis2
;      (cons
;       (car lis1)
;       (append (cdr lis1) lis2))))
(define (append x y) (%append x y))

(define (reverse lis1)
  (let loop ((lis lis1)
	     (result '()))
    (if (null? lis)
	result
	(loop (cdr lis) (cons (car lis) result)))))

(define (member x lis)
  (if (null? lis)
      #f
      (if (equal? x (car lis))
	  #t
	  (member x (cdr lis)))))

(define (assoc x lis)
  (if (null? lis)
      #f
      (if (equal? x (caar lis))
	  (car lis)
	  (member x (cdr lis)))))

(define (symbol? x) (%symbol? x))
(define (symbol->string x) (%symbol->string x))
(define (string->symbol x) (%string->symbol x))

(define (number? x) (%number? x))
(define (= x y) (%= x y))
(define (< x y) (%< x y))
(define (> x y) (%> x y))
(define (<= x y) (%<= x y))
(define (>= x y) (%>= x y))

(define (+ x y) (%+ x y))
(define (- x y) (%- x y))
(define (* x y) (%* x y))
(define (quotient x y) (%quotient x y))
(define (remainder x y) (%remainder x y))

(define (integer->string n1)
  (let loop ((n n1)
	     (prefix-list '()))
    (if (< n 10)
	(list->string
	 (cons (integer->chared-decimal n) prefix-list))
	(loop
	 (quotient n 10)
	 (cons (integer->chared-decimal (remainder n 10)) prefix-list)))))
(define (integer->chared-decimal n)
  (integer->char (+ n (char->integer #\0))))


(define (string->integer str)
  (let ((len (string-length str)))
    (let loop ((i 0)
	       (result 0))
      (if (>= i len)
	  result
	  (let ((c (string-ref str i)))
	    (if (or (char<? c #\0) (char<? #\9 c))
		(error 'string->integer "error in string->integer" str)
		(loop (+ i 1) (+ (* result 10) (chared-decimal->integer c)))))))))
(define (chared-decimal->integer c)
  (- (char->integer c) (char->integer #\0)))

(define (char? x) (%char? x))
(define (char<? x y) (%char<? x y))
(define (char>? x y) (%char>? x y))
(define (char<=? x y) (%char<=? x y))
(define (char>=? x y) (%char>=? x y))

(define (char->integer x) (%char->integer x))
(define (integer->char x) (%integer->char x))

(define (char-upcase c)
  (if (and (char<=? #\a c) (char<=? c #\z))
      (let ((c-int     (char->integer c))
	    (low-a-int (char->integer #\a))
	    (up-a-int  (char->integer #\A)))
	(integer->char (+ (- c-int low-a-int) up-a-int)))
      c))

(define (char-downcase c)
  (if (and (char<=? #\A c) (char<=? c #\Z))
      (let ((c-int     (char->integer c))
	    (low-a-int (char->integer #\a))
	    (up-a-int  (char->integer #\A)))
	(integer->char (+ (- c-int up-a-int) low-a-int)))
      c))

(define (string? x) (%string? x))
(define (string-length x) (%string-length x))
(define (string-ref str i) (%string-ref str i))
(define (string-set str i o) (%string-set str i o))
(define (string-append x y) (%string-append x y))

(define (string->list str)
  (let ((len (string-length str)))
    (let loop ((i 0))
      (if (= i len)
	  '()
	  (cons
	   (string-ref str i)
	   (loop (+ i 1)))))))

(define (list->string lis)
  (let ((len (length lis)))
    (let loop ((i 0)
	       (l lis)
	       (str (make-string len)))
      (if (= i len)
	  str
	  (loop (+ i 1) (cdr l) (string-set str i (car l)))))))

(define (procedure? x) (%procedure? x))

(define (map f lis)
  (if (null? lis)
      '()
      (cons (f (car lis)) (map f (cdr lis)))))

(define (current-input-port) (%current-input-port))
(define (current-output-port) (%current-output-port))
(define (open-input-file x) (%open-input-file x))
(define (open-output-file x) (%open-output-file x))
(define (close-input-port x) (%close-input-port x))
(define (close-output-port x) (%close-output-port x))
(define (stok-read-char x) (%stok-read-char x))
(define (eof-object? x) (%eof-object? x))

(define (stok-display obj port)
  (cond
   ((boolean? obj)
    (if obj
	(stok-display-string-without-quote "#t" port)
	(stok-display-string-without-quote "#f" port)))
   ((symbol? obj) (stok-display-string-without-quote (symbol->string obj) port))
   ((string? obj) (stok-display-string-without-quote obj port))
   ((integer? obj) (stok-display-string-without-quote (integer->string obj) port))
   ((char? obj)   (stok-display-string-without-quote (list->string (list obj)) port))
   ((pair? obj) 
    (let ((newport1 (stok-display-string-without-quote "(" port))
	  (newport2 (stok-display (car obj) newport1))
	  (newport3 (stok-display-string-without-quote " . " newport2))
	  (newport4 (stok-display (cdr obj) newport3)))
      (stok-display-string-without-quote ")" newport4)))
   ((procedure? obj) (stok-display-string-without-quote "#<procedure>" port))))
    
(define (stok-newline port) (%stok-newline port))


(define (stok-write-char x port) (%stok-write-char x port))
(define (exit) (%exit))

(define (error sym str obj) (%error sym str obj))

;;;
;;;
;;; Scheme Lexer
;;;
;;; modes
;;;  blank mode
;;;  object mode
;;;
;;; consider comment line finishing with EOF
;;; consider (a b(c)), #\a(a b c)
;;;

(define (stok-read port)
  (let ((nestlevel 0))
    (let* ((obj-and-newport (stok-read-blank port nestlevel))
	   (obj     (stok-read-get-obj  obj-and-newport))
	   (newport (stok-read-get-port obj-and-newport)))
      obj-and-newport)))

(define (stok-read-combine-obj-and-newport obj newport) (cons obj newport))
(define (stok-read-get-obj obj-and-newport)  (car obj-and-newport))
(define (stok-read-get-port obj-and-newport) (cdr obj-and-newport))

(define (stok-read-blank-char? c)
  (let ((n (char->integer c)))
    (or
     (= n 9)
     (= n 10)
     (= n 13)
     (= n 32)
)))


;;;
;;; returns a structure composed of
;;; (1) a read object and
;;; (2) a port with a new position.
;;;
(define (stok-read-blank port nestlevel)
  (let* ((c-and-newport (stok-read-char port))
	 (c       (stok-read-get-obj     c-and-newport))
	 (newport (stok-read-get-port c-and-newport)))
    (cond
     ((eof-object? c) c-and-newport) ; return the object as it is
     ((stok-read-blank-char? c) (stok-read-blank newport nestlevel))
     ((char=? c #\()
      (let* ((obj-and-new-newport (stok-read-blank newport (+ nestlevel 1)))
	     (obj         (stok-read-get-obj     obj-and-new-newport))
	     (new-newport (stok-read-get-port obj-and-new-newport)))
	(if (= nestlevel 0)
	    obj-and-new-newport
	    (let ((nextobj-and-new-new-newport
		   (stok-read-blank new-newport nestlevel)))
	      (stok-read-combine-obj-and-newport
	       (cons obj (stok-read-get-obj nextobj-and-new-new-newport))
	       (stok-read-get-port nextobj-and-new-new-newport))))))
     ((char=? c #\))
      (if (= nestlevel 0)
	  (error 'stok-read-blank-toplevel "~a~%" c)
	  (stok-read-combine-obj-and-newport '() newport)))
     ((char=? c #\")
      (let* ((str-and-new-newport (stok-read-string newport))
	     (str         (stok-read-get-obj     str-and-new-newport))
	     (new-newport (stok-read-get-port str-and-new-newport)))
	(if (= nestlevel 0)
	    (stok-read-combine-obj-and-newport str new-newport)
	    (let ((nextobj-and-new-new-newport
		   (stok-read-blank new-newport nestlevel)))
	      (stok-read-combine-obj-and-newport
	       (cons str (stok-read-get-obj nextobj-and-new-new-newport))
	       (stok-read-get-port nextobj-and-new-new-newport))))))
     ((char=? c #\;)
      (let ((new-newport (stok-read-comment-until-line-end newport)))
	(stok-read-blank new-newport nestlevel)))
     ((char=? c #\#)
      (let* ((ch-and-new-newport (stok-read-character newport))
	     (ch (stok-read-get-obj ch-and-new-newport))
	     (new-newport (stok-read-get-port ch-and-new-newport)))
	(if (= nestlevel 0)
	    ch-and-new-newport
	    (let ((nextobj-and-new-new-newport
		   (stok-read-blank new-newport nestlevel)))
	      (stok-read-combine-obj-and-newport
	       (cons ch (stok-read-get-obj nextobj-and-new-new-newport))
	       (stok-read-get-port nextobj-and-new-new-newport))))))
     (else
      (stok-read-object newport nestlevel (list c))))))


(define (stok-read-object port nestlevel uncompleted-object-list)
  (let* ((c-and-newport (stok-read-char port))
	 (c       (stok-read-get-obj     c-and-newport))
	 (newport (stok-read-get-port c-and-newport)))
    (cond
     ((stok-read-blank-char? c)
      (let ((got-object (stok-read-compose-object uncompleted-object-list)))
	(if (= nestlevel 0)
	    (stok-read-combine-obj-and-newport got-object newport)
	    (let ((nextobj-and-new-newport (stok-read-blank newport nestlevel)))
	      (stok-read-combine-obj-and-newport
	       (cons got-object (stok-read-get-obj nextobj-and-new-newport))
	       (stok-read-get-port nextobj-and-new-newport))))))
     ((char=? c #\()
      (let ((listobj-and-new-newport (stok-read-blank newport (+ nestlevel 1))))
	(if (and
	     (pair? uncompleted-object-list)
	     (null? (cdr uncompleted-object-list))
	     (char=? (car uncompleted-object-list) #\'))
	    ;;; quoted list
	    (if (= nestlevel 0)
		(stok-read-combine-obj-and-newport
		 (list 'quote (stok-read-get-obj listobj-and-new-newport))
		 (stok-read-get-port listobj-and-new-newport))
		(let* ((new-newport (stok-read-get-port listobj-and-new-newport))
		       (nextobj-and-new-new-newport
			(stok-read-blank new-newport nestlevel)))
		  (stok-read-combine-obj-and-newport
		   (cons (list 'quote (stok-read-get-obj listobj-and-new-newport))
			 (stok-read-get-obj nextobj-and-new-new-newport))
		   (stok-read-get-port nextobj-and-new-new-newport))))
	    ;;; normal list
	    (if (= nestlevel 0)
		(stok-read-combine-obj-and-newport
		 (stok-read-get-obj listobj-and-new-newport)
		 (stok-read-get-port listobj-and-new-newport))
		(let* ((new-newport (stok-read-get-port listobj-and-new-newport))
		       (nextobj-and-new-new-newport
			(stok-read-blank new-newport nestlevel)))
		  (stok-read-combine-obj-and-newport
		   (cons (stok-read-get-obj listobj-and-new-newport)
			 (stok-read-get-obj nextobj-and-new-new-newport))
		   (stok-read-get-port nextobj-and-new-new-newport)))))))
     ((char=? c #\))
      (let ((got-object (stok-read-compose-object uncompleted-object-list)))
	(stok-read-combine-obj-and-newport (cons got-object '()) newport)))
     ((char=? c #\")
      (let ((obj-before-string (stok-read-compose-object uncompleted-object-list)))
	(if (= nestlevel 0)
	    (error 'stok-read-object "Invalid input data" "Invalid input data")
	    (let* ((str-and-new-newport (stok-read-string newport))
		   (str         (stok-read-get-obj str-and-new-newport))
		   (new-newport (stok-read-get-port str-and-new-newport)))
	      (let* ((nextobj-and-new-new-newport (stok-read-blank newport nestlevel))
		     (nextobj (stok-read-get-obj nextobj-and-new-new-newport))
		     (new-new-newport (stok-read-get-port nextobj-and-new-new-newport)))
		(stok-read-combine-obj-and-newport
		 (cons obj-before-string (cons str nextobj))
		 new-new-newport))))))
     ((char=? c #\;)
      (let ((obj-before-comment (stok-read-compose-object uncompleted-object-list)))
	(if (= nestlevel 0)
	    (stok-read-combine-obj-and-newport obj-before-comment newport)
	    (let ((new-newport (stok-read-comment-until-line-end newport)))
	      (let* ((nextobj-and-new-new-newport (stok-read-blank new-newport nestlevel))
		     (nextobj (stok-read-get-obj nextobj-and-new-new-newport))
		     (new-new-newport (stok-read-get-port nextobj-and-new-new-newport)))
		
		(stok-read-combine-obj-and-newport
		 (cons obj-before-comment nextobj)
		 new-new-newport))))))
     (else
      (stok-read-object newport nestlevel (append uncompleted-object-list (list c)))))))


(define (stok-read-character port)
  (let* ((secondchar-and-newport (stok-read-char port))
	 (secondchar (stok-read-get-obj secondchar-and-newport))
	 (newport (stok-read-get-port secondchar-and-newport)))
    (cond
     ((stok-read-blank-char? secondchar)
      (error 'stok-read-character "Error0" "Error0: Too short character~%"))
     ((or (char=? secondchar #\t) (char=? secondchar #\T))
      (stok-read-combine-obj-and-newport #t newport))
     ((or (char=? secondchar #\f) (char=? secondchar #\F))
      (stok-read-combine-obj-and-newport #f newport))
     (else
      (if (char=? secondchar #\\)
	  (let* ((thirdchar-and-new-newport (stok-read-char newport))
		 (thirdchar (stok-read-get-obj thirdchar-and-new-newport))
		 (new-newport (stok-read-get-port thirdchar-and-new-newport)))
            ;;; third char is used directly as a return value
	    (stok-read-combine-obj-and-newport thirdchar new-newport))
	  (error 'stok-read-character "ERROR2: invalid sharp-sign prefix~a~%" secondchar))))))  

(define (stok-read-compose-object charlist)
  (let ((charlen (length charlist))
	(firstchar (car charlist)))
    (cond
     ;;; unsigned integer
     ((stok-read-all-decimal? charlist) (stok-read-compose-integer charlist))
     ;;; negative integer
     ((and
       (>= charlen 2)
       (char=? firstchar #\-)
       (stok-read-all-decimal? (cdr charlist)))
      (- (stok-read-compose-integer (cdr charlist))))
     ((char=? firstchar #\')  (stok-read-compose-quote charlist))
     (else
      (string->symbol (list->string charlist))))))

(define (stok-read-all-decimal? charlist)
  (if (null? charlist)
      #t
      (let ((ch (car charlist)))
	(if (and
	     (char<=? #\0 ch)
	     (char<=? ch #\9))
	    (stok-read-all-decimal? (cdr charlist))
	    #f))))

(define (stok-read-compose-integer charlist1)
  (let loop ((charlist charlist1)
	     (result 0))
    (if (null? charlist)
	result
	(let ((ch (car charlist)))
	  (loop
	   (cdr charlist)
	   (+ (* result 10)
	      (- (char->integer ch) (char->integer #\0))))))))

(define (stok-read-compose-quote charlist)
  (list 'quote (stok-read-compose-object (cdr charlist))))

(define (stok-read-string port1)
  (let loop ((charlist '())
	     (port port1))
    (let* ((c-and-newport (stok-read-char port))
	   (c (stok-read-get-obj c-and-newport))
	   (newport (stok-read-get-port c-and-newport)))
      (cond
       ((char=? c #\")
	(stok-read-combine-obj-and-newport (list->string (reverse charlist)) newport))
       ((char=? c #\\)
	(let* ((next-of-c-and-new-newport (stok-read-char newport))
	       (next-of-c (stok-read-get-obj next-of-c-and-new-newport))
	       (new-newport (stok-read-get-port next-of-c-and-new-newport)))
	  (loop (cons next-of-c charlist) new-newport)))
       (else
	(loop (cons c charlist) newport))))))

(define (stok-read-comment-until-line-end port)
  (let* ((c-and-newport (stok-read-char port))
	 (c (stok-read-get-obj c-and-newport))
	 (newport (stok-read-get-port c-and-newport)))
    (if (= (char->integer c) 10) ; newline
	newport
	(stok-read-comment-until-line-end newport))))

