;-*-SCHEME-*-

;;;
;;; StoK: Scheme to KL1 translator
;;;   Copyright (C) 1997 Yoshihiro Oyama
;;;     <oyama@is.s.u-tokyo.ac.jp>
;;;       December 19, 1997
;;;

;;
;; Calculate free variables in each predicate, to get ready
;; for a coming complex closure conversion
;;

(module klofree)

;
; A toplevel predicate is regarded as a local predicate
; without variables bound outside the local definition.
;

(define (klofree predicate)
  (klofree-top-predicate predicate))

(define (klofree-top-predicate predicate)
  (let ((free-and-bound-vars-and-converted-expr
	 (klofree-local-predicate predicate)))
    (let ((freevars  (klofree-get-freevars  free-and-bound-vars-and-converted-expr))
	  (boundvars (klofree-get-boundvars free-and-bound-vars-and-converted-expr))
	  (converted-expr (klofree-get-expr free-and-bound-vars-and-converted-expr)))
      (let ((truly-freevars (klofree-difference freevars boundvars)))
	(klofree-make-freevars-ed-predicate truly-freevars converted-expr)))))

(define (klofree-local-predicate predicate)
  (let ((keyword (list-ref predicate 0)))
    (cond
     ((equal? keyword 'top-predicate)
      ; a top predicate is regarded as a local predicate
      (klofree-local-normal-predicate predicate))
     ((equal? keyword 'local-predicate)
      (klofree-local-normal-predicate predicate))
     ((equal? keyword 'global-variable)
      (klofree-local-normal-predicate predicate))
     ((equal? keyword 'either-predicate)
      (klofree-local-either-predicate predicate))
     (else
      (error 'klofree-local-predicate "Unsupported predicate type" predicate)))))

(define (klofree-local-normal-predicate predicate)
  (let ((keyword        (list-ref predicate 0))
	(predicate-name (list-ref predicate 1))
	(predicate-args (list-ref predicate 2))
	(guard          (list-ref predicate 3))  ; only no guard case is supported
	(body           (list-ref predicate 4))) ; only one body is allowed
    (let ((free-and-bound-vars-and-converted-expr (klofree-parallel body)))
      (let ((freevars  (klofree-get-freevars  free-and-bound-vars-and-converted-expr))
	    (boundvars (klofree-get-boundvars free-and-bound-vars-and-converted-expr))
	    (converted-expr (klofree-get-expr free-and-bound-vars-and-converted-expr)))
	(let* ((truly-boundvars (append (cons predicate-name predicate-args) boundvars))
	       (truly-freevars (klofree-difference freevars truly-boundvars)))
	  (klofree-compound-free-and-bound-vars-and-expr
	   truly-freevars
	   truly-boundvars
	   (list keyword predicate-name predicate-args guard converted-expr)))))))
	 
(define (klofree-local-either-predicate predicate)
  (let ((keyword        (list-ref predicate 0))
	(predicate-name (list-ref predicate 1))
	(predicate-args (list-ref predicate 2))
	(guard          (list-ref predicate 3))  ; only no guard case is supported
	(body1          (list-ref predicate 4))
	(body2          (list-ref predicate 5)))
    (let ((free-and-bound-vars-and-converted-expr1 (klofree-parallel body1))
	  (free-and-bound-vars-and-converted-expr2 (klofree-parallel body2)))
      (let ((freevars1  (klofree-get-freevars  free-and-bound-vars-and-converted-expr1))
	    (boundvars1 (klofree-get-boundvars free-and-bound-vars-and-converted-expr1))
	   (freevars2  (klofree-get-freevars  free-and-bound-vars-and-converted-expr2))
	   (boundvars2 (klofree-get-boundvars free-and-bound-vars-and-converted-expr2)))
	(let ((converted-expr1 (klofree-get-expr free-and-bound-vars-and-converted-expr1))
	      (converted-expr2 (klofree-get-expr free-and-bound-vars-and-converted-expr2)))
	  (let* ((truly-boundvars
		  (uniq
		   (append
		    (cons predicate-name predicate-args)
		    boundvars1
		    boundvars2)))
		 (truly-freevars
		  (klofree-difference
		   (uniq (append freevars1 freevars2)) truly-boundvars)))
	    (klofree-compound-free-and-bound-vars-and-expr
	     truly-freevars
	     truly-boundvars
	     (list
	      keyword
	      predicate-name
	      predicate-args
	      guard
	      converted-expr1
	      converted-expr2))))))))

	 
(define (klofree-main expr)
  (let ((op (car expr)))
    (cond
     ((equal? op 'primitive) (klofree-primitive expr))
     ((equal? op 'assign)    (klofree-assign expr))
     ((equal? op 'apply)     (klofree-apply expr))
     ((equal? op 'fix)       (klofree-fix expr))
     ((klofree-terminal? op) (klofree-terminal expr))
     (else
      (error 'klofree-main "Unknown primitive" expr)))))

(define (klofree-parallel expr-list1)
  (let loop ((expr-list expr-list1)
	     (freevars-maybe-duplicated  '())
	     (boundvars-maybe-duplicated '())
	     (converted-expr-list '()))
    (if (null? expr-list)
	(klofree-compound-free-and-bound-vars-and-expr
	 (uniq freevars-maybe-duplicated)
	 (uniq boundvars-maybe-duplicated)
	 converted-expr-list)
	(let* ((expr (car expr-list))
	       (free-and-bound-vars-and-converted-expr (klofree-main expr)))
	  (let ((freevars-in-expr
		 (klofree-get-freevars  free-and-bound-vars-and-converted-expr))
		(boundvars-in-expr
		 (klofree-get-boundvars free-and-bound-vars-and-converted-expr))
		(converted-expr
		 (klofree-get-expr      free-and-bound-vars-and-converted-expr)))
	    (loop
	     (cdr expr-list)
	     (append freevars-maybe-duplicated freevars-in-expr)
	     (append boundvars-maybe-duplicated boundvars-in-expr)
	     (append converted-expr-list (list converted-expr))))))))

(define (klofree-primitive expr)
  (let (
; I commented out for more speed
;	(keyword (list-ref expr 0))
	(op      (list-ref expr 1))
	(args    (list-ref expr 2))
	(reply-channel (list-ref expr 3)))
    (let ((freevars (uniq args))
	  (boundvars (list reply-channel)))
      (klofree-compound-free-and-bound-vars-and-expr
       freevars boundvars expr))))

(define (klofree-assign expr)
  (let (
; I commented out for more speed
;	(keyword (list-ref expr 0))
	(source  (list-ref expr 1))
	(dest    (list-ref expr 2)))
    (let ((freevars  (list source))
	  (boundvars (list dest)))
      (klofree-compound-free-and-bound-vars-and-expr freevars boundvars expr))))

(define (klofree-apply expr)
  (let (
; I commented out for more speed
;	(keyword (list-ref expr 0))
	(func    (list-ref expr 1))
	(args    (list-ref expr 2)))
    (let ((in-args (cdr args))
	  (out-args (list (car args))))
      (let ((freevars (uniq (cons func in-args)))
	    (boundvars out-args))
	(klofree-compound-free-and-bound-vars-and-expr freevars boundvars expr)))))

(define (klofree-fix expr)
  (let (
; I commented out for more speed
;	(keyword (list-ref expr 0))
	(binds   (list-ref expr 1))
	(body    (list-ref expr 2)))
    (if (not (= (length binds) 1))
	(error 'klofree-fix "Multiple binds in fix is not supported" expr)
	(let* ((local-predicate (car binds))
	       (local-predicate-name (list-ref local-predicate 1)))
	  (let ((free-and-bound-vars-and-converted-body
		 (klofree-parallel body))
		(free-and-bound-vars-and-converted-local-predicate
		 (klofree-local-predicate local-predicate)))
	    (let ((freevars-in-body
		   (klofree-get-freevars  free-and-bound-vars-and-converted-body))
		  (boundvars-in-body
		   (klofree-get-boundvars free-and-bound-vars-and-converted-body))
		  (converted-body
		   (klofree-get-expr      free-and-bound-vars-and-converted-body))
		  (freevars-in-local-predicate
		   (klofree-get-freevars free-and-bound-vars-and-converted-local-predicate))
		  (boundvars-in-local-predicate
		   (klofree-get-boundvars free-and-bound-vars-and-converted-local-predicate))
		  (converted-local-predicate
		   (klofree-get-expr free-and-bound-vars-and-converted-local-predicate)))
	      (let ((truly-freevars-in-local-predicate
		     (klofree-difference
		      freevars-in-local-predicate
		      (cons local-predicate-name boundvars-in-local-predicate)))
		    (boundvars-in-fix
		     (uniq (cons local-predicate-name boundvars-in-body))))
		(let ((freevars-in-fix
		       (uniq
			(append freevars-in-body truly-freevars-in-local-predicate))))
		  (let ((truly-freevars-in-fix 
			 (klofree-difference freevars-in-fix boundvars-in-fix))
			(truly-boundvars-in-fix boundvars-in-body)
			(truly-freevars-in-local-predicate
			 (klofree-difference
			  freevars-in-local-predicate
			  boundvars-in-local-predicate)))
		    (klofree-compound-free-and-bound-vars-and-expr
		     truly-freevars-in-fix
		     truly-boundvars-in-fix
		     (list
		      'fix
		      (list
		       (klofree-make-freevars-ed-predicate
			truly-freevars-in-local-predicate
			converted-local-predicate))
		      converted-body)))))))))))
		   
;;;
;;;
;;;
(define (klofree-make-freevars-ed-predicate freevars expr)
  (cons freevars expr))
(define (klofree-detach-freevars freevars-ed-predicate)
  (car  freevars-ed-predicate))
(define (klofree-detach-predicate freevars-ed-predicate)
  (cdr  freevars-ed-predicate))

;;;
(define (klofree-compound-free-and-bound-vars-and-expr freevars boundvars expr)
  (list freevars boundvars expr))
(define (klofree-get-freevars freevars-and-boundvars-and-expr)
  (list-ref freevars-and-boundvars-and-expr 0))
(define (klofree-get-boundvars freevars-and-boundvars-and-expr)
  (list-ref freevars-and-boundvars-and-expr 1))
(define (klofree-get-expr freevars-and-boundvars-and-expr)
  (list-ref freevars-and-boundvars-and-expr 2))

;;;
(define (klofree-difference freevars boundvars)
  (list-difference freevars boundvars))

(define (list-difference subtractee-list subtractor-list)
  (if (null? subtractee-list)
      '()
      (if (or
	   (klofree-not-variable? (car subtractee-list))
	   (member (car subtractee-list) subtractor-list))
	  (list-difference (cdr subtractee-list) subtractor-list)
	  (cons
	   (car subtractee-list)
	   (list-difference (cdr subtractee-list) subtractor-list)))))

(define (klofree-not-variable? expr)
  (or
   (number? expr)
   (string? expr)
   (boolean? expr)
   (label-quote? expr)
   (char? expr)))

