;-*-SCHEME-*-

;;;
;;; StoK: Scheme to KL1 translator
;;;   Copyright (C) 1997 Yoshihiro Oyama
;;;     <oyama@is.s.u-tokyo.ac.jp>
;;;       December 19, 1997
;;;

;;
;; This module is the HEART of stok
;;
;; The below function receives a parsed Scheme define expression
;; and generates KL1-like program. Since the KL1-like program
;; has a first-class closure, we cannot generate native KL1 code
;; directly from the program.
;;

(module kernel)

(define (traker scheme-define-expression)
  (let ((keyword  (list-ref scheme-define-expression 0))
	(scm-args (list-ref scheme-define-expression 1))
	(body     (list-ref scheme-define-expression 2)) ; always one body
	(reply-channel (unique-name 'rep)))
    (cond
     ((equal? keyword 'define-function)
      (traker-make-expr-top-predicate
       ; function name == predicate name
       (traker-make-expr-predicate-name (car scm-args))
       ; arguments
       (traker-make-expr-args (cons reply-channel (cdr scm-args)))
       ; #t indicates no guard
       #t
       (traker-make-expr-implicit-parallel (traker-main body reply-channel))))
     ((equal? keyword 'define-global-variable)
      (traker-make-expr-global-variable
       ; function name == predicate name
       (traker-make-expr-predicate-name (car scm-args))
       ; arguments
       (traker-make-expr-args (cons reply-channel (cdr scm-args)))
       ; #t indicates no guard
       #t
       (traker-make-expr-implicit-parallel (traker-main body reply-channel))))
     (else
      (error 'traker "Unsupported predicate type" scm-define-expression)))))


(define (traker-make-expr-implicit-parallel expr)
  (if (traker-explicit-parallel? expr)
      (traker-eliminate-explicit-parallel expr)
      (list expr) ; a list containing only one expr
      ))

(define (traker-explicit-parallel? expr) (equal? (car expr) 'parallel))
(define (traker-eliminate-explicit-parallel expr) (cdr expr))

(define (traker-main expr reply-channel)
  (if (traker-terminal? expr)
      (traker-terminal expr reply-channel)
      (let ((op (car expr)))
	(cond
	 ((equal? op 'primitive) (traker-primop expr reply-channel))
	 ((equal? op 'apply)     (traker-apply expr reply-channel))
	 ((equal? op 'if)        (traker-if expr reply-channel))
	 ((equal? op 'fix)       (traker-fix expr reply-channel))
	 ((equal? op '%quote)     (traker-make-expr-terminal expr reply-channel))
	 (else (error 'traker-main "Unknown keyword" expr))))))

;;;
;;; We should define more formally the terminality.
;;;
(define (traker-terminal? expr)
  (not (pair? expr)))
(define (traker-terminal expr reply-channel)
  (traker-make-expr-terminal expr reply-channel))

(define (traker-fix expr reply-channel)
  (let ((keyword (list-ref expr 0))
	(binds   (list-ref expr 1))
	(body    (list-ref expr 2)))
    (if (not (= (length binds) 1))
	(error 'traker-fix "Mulltiple bind in fix is not supported" binds)
	(let ((bind (car binds)))
	  (traker-make-expr-fix
	   (traker-make-expr-local-binds
	    (list
	     (traker-local-bind bind)))
	   (traker-make-expr-implicit-parallel (traker-main body reply-channel)))))))

(define (traker-local-bind bind)
  (let ((funcname (list-ref bind 0))
	(scm-args (list-ref bind 1))
	(body     (list-ref bind 2)))
    (let ((reply-channel (unique-name 'lfrep)))
      (traker-make-expr-local-predicate
       (traker-make-expr-predicate-name funcname)
       (traker-make-expr-args (cons reply-channel scm-args))
       #t
       (traker-make-expr-implicit-parallel (traker-main body reply-channel))))))

(define (traker-primop expr reply-channel)
  (let ((keyword (list-ref expr 0))
	(op      (list-ref expr 1))
	(args    (cdr (cdr expr))))
    (let ((arglen (length args)))
      (let ((tmp-vars-for-args (unique-names 'rep arglen)))
	(traker-make-expr-parallel
	 (traker-args args tmp-vars-for-args)
	 (traker-make-expr-primitive op tmp-vars-for-args reply-channel))))))

(define (traker-args expr-list vars-to-return-result)
  (if (null? expr-list)
      (traker-make-expr-empty)
      (let ((expr (car expr-list))
	    (var-to-return-result (car vars-to-return-result)))
	(traker-make-expr-parallel
	 (traker-args (cdr expr-list) (cdr vars-to-return-result))
	 (traker-main expr var-to-return-result)))))

(define (traker-quote expr reply-channel)
  (let ((keyword   (list-ref expr 0))
	(atom-expr (list-ref expr 1)))
    ;;; should we give this 'expr' to traker-terminal???
    expr))
       
(define (traker-apply expr reply-channel)
  (let ((keyword  (list-ref expr 0))
	(function (list-ref expr 1))
	(scm-args (cdr (cdr expr)))
	(expr-len (- (length expr) 1))) ; eliminate keyword length
    (let ((tmp-vars-for-expr (unique-names 'rep expr-len)))
      (let ((tmp-var-for-function  (car tmp-vars-for-expr))
	    (tmp-vars-for-scm-args (cdr tmp-vars-for-expr)))
	(traker-make-expr-parallel
	 (traker-args (cdr expr) tmp-vars-for-expr)
	 (traker-make-expr-application
	  (traker-make-expr-predicate-name tmp-var-for-function)
	  reply-channel
	  (traker-make-expr-args tmp-vars-for-scm-args)))))))
	    
(define (traker-if expr reply-channel)
  (let ((keyword   (list-ref expr 0))
	(condexpr  (list-ref expr 1))
	(trueexpr  (list-ref expr 2))
	(falseexpr (list-ref expr 3))
	(bool-channel (unique-name 'brep))
	(new-reply-channel (unique-name 'lbrep))
	(local-pred-name (unique-name 'pred)))
    (traker-make-expr-parallel
     (traker-main condexpr bool-channel)
     (traker-make-expr-fix
      ;;; we make a local function which is executed after
      ;;; receiving the test value.
      (traker-make-expr-local-binds
       (list (traker-make-expr-either-predicate
	      (traker-make-expr-predicate-name local-pred-name)
	      (traker-make-expr-args (list new-reply-channel))
	      #t
	      (traker-make-expr-implicit-parallel
	       (traker-main trueexpr  new-reply-channel))
	      (traker-make-expr-implicit-parallel
	       (traker-main falseexpr new-reply-channel)))))
      ;;; In this part we calculate the test expression
      (traker-make-expr-implicit-parallel
       (traker-make-expr-application
	local-pred-name reply-channel (list bool-channel)))))))

;;;
;;;
;;;
(define (traker-make-expr-args args) args)
;;; Only two-bodied clauses are allowed
(define (traker-make-expr-parallel body1 body2)
  (traker-make-expr-parallel-super (list body1 body2) '()))

(define (traker-make-expr-parallel-super bodylist result)
  (if (null? bodylist)
      (cons 'parallel result)
      (let ((body (car bodylist)))
	(let ((op (car body)))
	  (if (equal? op 'parallel)
	      (let ((bodies (cdr body)))
		(traker-make-expr-parallel-super
		 (append bodies (cdr bodylist))
		 result))
	      (if (equal? op 'die)
		  (traker-make-expr-parallel-super
		   (cdr bodylist) result)
		  (traker-make-expr-parallel-super
		   (cdr bodylist)
		   (append result (list body)))))))))

(define (traker-make-expr-terminal expr reply-channel)
  (list 'assign expr reply-channel))

(define (traker-make-expr-primitive op arglist reply-channel)
  (list 'primitive op arglist reply-channel))

(define (traker-make-expr-top-predicate pred-name args guard body)
  (list 'top-predicate pred-name args guard body))
(define (traker-make-expr-global-variable pred-name args guard body)
  (list 'global-variable pred-name args guard body))
(define (traker-make-expr-either-predicate pred-name args guard body1 body2)
  (list 'either-predicate pred-name args guard body1 body2))
(define (traker-make-expr-local-predicate pred-name args guard body)
  (list 'local-predicate pred-name args guard body))

(define (traker-make-expr-predicate-name expr) expr)

(define (traker-make-expr-empty) '(die))
(define (traker-make-expr-application predname reply-channel args)
  (list 'apply predname (cons reply-channel args)))

(define (traker-make-expr-fix bind body)
  (list 'fix bind body))
(define (traker-make-expr-local-binds local-pred-list) local-pred-list)

