;-*-SCHEME-*-

;;;
;;; StoK: Scheme to KL1 translator
;;;   Copyright (C) 1997 Yoshihiro Oyama
;;;     <oyama@is.s.u-tokyo.ac.jp>
;;;       December 19, 1997
;;;

;;
;; This module provides an interface function for users
;;

(module intface)

;;;
;;; The very function for interface to shell
;;;
(define (intface scheme-filename-string-list1)
  (let loop ((scheme-filename-string-list scheme-filename-string-list1)
	     (makefile-port (open-output-file "_kl1_sources_")))
    (if (null? scheme-filename-string-list)
	(begin
	  (close-output-port makefile-port)
	  #t)
	(let ((source-filename (car scheme-filename-string-list))
	      (dest-filename
	       (path-attach-extension
		(path-eliminate-extension
		 (car scheme-filename-string-list))
		"kl1")))
	  (intface-main source-filename dest-filename)
	  (let ((new-makefile-port
		 (stok-display-string-without-quote
		  (string-append dest-filename " ") makefile-port)))
	    (loop (cdr scheme-filename-string-list) new-makefile-port))))))

;;;
;;; Read one source file and generate one destination file
;;;
(define (intface-main sourcefilename destfilename)
  (let ((input-port  (open-input-file  sourcefilename))
	(output-port (open-output-file destfilename)))
    (stok-display "file opened." (current-output-port))
    (stok-newline (current-output-port))
    (let* ((newports1 (intface-read-this-module-name input-port output-port))
	   (new-input-port1  (intface-get-input-port  newports1))
	   (new-output-port1 (intface-get-output-port newports1)))
      (modinit-reset-export-functions!)
      (let* ((newports2
	      (if (equal? (gencode-get-this-module-name) 'scheme-main)
		  (begin
		    ;;; r4rs module is used by default
		    (modinit-add-export-function! (label-make-module-label 'r4rs 2))
		    (let ((new-input-port-a
			   (intface-read-import-module-name new-input-port1)))
		      (intface-combine-ports new-input-port-a new-output-port1)))
		  newports1))
	     (new-input-port2  (intface-get-input-port  newports2))
	     (new-output-port2 (intface-get-output-port newports2)))
	(let* ((newports3
		(intface-read-expr-and-compile-loop
		 new-input-port2 new-output-port2))
	       (new-input-port3  (intface-get-input-port  newports3))
	       (new-output-port3 (intface-get-output-port newports3)))
	  (close-input-port new-input-port3) ; returns unit
	  (let ((new-output-port4
		 (modinit-generate-initialization-predicate new-output-port3)))
	    (close-output-port new-output-port4)
	    (stok-display "file closed." (current-output-port))
	    (stok-newline (current-output-port))
	    ))))))

;;;
;;; read current-reading module name
;;;
(define (intface-read-this-module-name input-port output-port)
  (let* ((read-expr-and-newport (stok-read input-port))
	 (read-expr       (stok-read-get-obj     read-expr-and-newport))
	 (new-input-port1 (stok-read-get-port read-expr-and-newport)))
    (let ((op (car read-expr)))
      (if (equal? op 'module)
	  (let ((module-name (list-ref read-expr 1)))
	    (gencode-set-this-module-name! module-name)
	    (let* ((new-output-port1
		    (stok-display-string-without-quote ":- module " output-port))
		   (new-output-port2
		    (stok-display-string-without-quote
		     (symbol->string-with-escape module-name) new-output-port1))
		   (new-output-port3
		    (stok-display-string-without-quote "." new-output-port2))
		   (new-output-port4
		    (stok-newline new-output-port3))
		   (new-output-port5
		    (stok-newline new-output-port4)))
	      (intface-combine-ports new-input-port1 new-output-port5)))
	  (error 'read-this-module-name "All files need module declaration" read-expr)))))

;;;
;;; This function reads module names which requires initialization
;;; before scheme-main function is called.
;;; This declaration exists only in scheme-main module.
;;;
;;; The declaration form is like (module-import queue)
;;; It ends with (module-import-end)
;;;
(define (intface-read-import-module-name input-port)
  (let* ((read-expr-and-newport (stok-read input-port))
	 (read-expr       (stok-read-get-obj     read-expr-and-newport))
	 (new-input-port1 (stok-read-get-port read-expr-and-newport)))
    (let ((op (car read-expr)))
      (cond
       ((equal? op 'module-import)
	(let ((import-module-name (list-ref read-expr 1)))
	  (modinit-add-export-function!
	   (label-make-module-label import-module-name 2))
	  (intface-read-import-module-name new-input-port1)))
       ((equal? op 'module-import-end) new-input-port1)
       (else
	(error
	 'read-import-module-name
	 "module schme-main must have module-import declaration"
	 read-expr))))))

;;;
;;; read define expression, compile the expression, and
;;; write the compiled KL1 program to a given port.
;;;
(define (intface-read-expr-and-compile-loop input-port output-port)
  (let* ((read-expr-and-newport (stok-read input-port))
	 (read-expr       (stok-read-get-obj     read-expr-and-newport))
	 (new-input-port1 (stok-read-get-port read-expr-and-newport)))
    (let ((process-definition read-expr))
      (if (eof-object? process-definition)
	  (begin
	    (intface-combine-ports new-input-port1 output-port))
	  (begin
	    (stok-display "Compiling " (current-output-port))
	    (stok-display
	     (let ((secondarg (list-ref process-definition 1)))
	       (if (pair? secondarg)
		   (car secondarg) ; function name
		   secondarg)) ; global variable name
	     (current-output-port))
	    (stok-display "..." (current-output-port))
	    (let ((new-output-port1
		   (intface-compile process-definition output-port)))
	      (stok-display "done." (current-output-port))
	      (stok-newline (current-output-port))
	      (intface-read-expr-and-compile-loop
	       new-input-port1 new-output-port1)))))))

;;;
;;; Compile one Scheme function and output the generated KL1
;;; program to a given port.
;;;
(define (intface-compile scm-define-expr output-port)
  (stok-display "6..." (current-output-port))
  (let ((core-scheme-expr (parse scm-define-expr)))
    (stok-display "5..." (current-output-port))
    (let ((naive-kl1-expr (traker core-scheme-expr)))
      (stok-display "4..." (current-output-port))
      (let ((klofree-ed-kl1-expr (klofree naive-kl1-expr)))
	(stok-display "3..." (current-output-port))
	(let ((klokon-ed-kl1-expr (klokon klofree-ed-kl1-expr)))
	  (stok-display "2..." (current-output-port))
	  (let ((amc-ed-kl1-expr (amc klokon-ed-kl1-expr)))
	    (stok-display "1..." (current-output-port))
	    (let ((new-output-port (gencode output-port amc-ed-kl1-expr)))
	      new-output-port)))))))

(define (intface-combine-ports input-port output-port)
  (cons input-port output-port))
(define (intface-get-input-port  ports) (car ports))
(define (intface-get-output-port ports) (cdr ports))

