;-*-SCHEME-*-

;;;
;;; StoK: Scheme to KL1 translator
;;;   Copyright (C) 1997 Yoshihiro Oyama
;;;     <oyama@is.s.u-tokyo.ac.jp>
;;;       December 19, 1997
;;;

;;
;; KL1 code generator
;;

(module gencode)


(define (gencode-s2k-lib-module-name) "s2klib")
(define *gencode-this-module-name* 'default-module-name)
(define (gencode-get-this-module-name) (global-ref *gencode-this-module-name*))
(define (gencode-set-this-module-name! name) (set! *gencode-this-module-name* name))

;;;
;;; Most of the below functions receive a port and return a new port.
;;; They merely write objects to a received port so that they need not
;;; return a meaningful value such as integer or cell, 
;;; in contrast to many typical Scheme functions.
;;;

(define (gencode port predicates)
  (gencode-predicates port predicates))
    
(define (gencode-predicates port predicates)
  (if (null? predicates)
      ; return a port which is next used by the caller of gencode-predicates
      port
      (let ((predicate (car predicates)))
	(let ((keyword (list-ref predicate 0)))
	  (cond
	   ((equal? keyword 'top-predicate)
	    (let ((newport     (gencode-top-predicate port predicate)))
	      (gencode-predicates newport (cdr predicates))))
	   ((equal? keyword 'local-predicate)
	    (let ((newport     (gencode-local-predicate port predicate)))
	      (gencode-predicates newport (cdr predicates))))
	   ((equal? keyword 'global-variable)
	    (let ((newport     (gencode-global-variable port predicate)))
	      (gencode-predicates newport (cdr predicates))))
	   ((equal? keyword 'either-predicate)
	    (let ((newport     (gencode-either-predicate port predicate)))
	      (gencode-predicates newport (cdr predicates))))
	   (else
	    (error 'gencode-predicates "Unknown predicate type" predicate)))))))

(define (gencode-top-predicate port predicate)
  (gencode-non-either-predicate port predicate))
(define (gencode-local-predicate port predicate)
  (gencode-non-either-predicate port predicate))

(define (gencode-global-variable port predicate)
  (let ((keyword         (list-ref predicate 0))
	(predicate-label (list-ref predicate 1))
	(predicate-args  (list-ref predicate 2))
	(guard           (list-ref predicate 3))
	(body            (list-ref predicate 4)))
    (modinit-add-export-function! predicate-label)
    (let* ((newport1 (gencode-print-predicate-label port predicate-label))
	   (newport2 (gencode-print-predicate-args newport1 predicate-args))
	   (newport3 (gencode-print-tombow newport2))
	   (newport4 (gencode-print-guard newport3 guard))
	   (newport5 (gencode-print-vertical-bar newport4))
	   (newport6 (gencode-clause-list newport5 body))
	   (newport7 (gencode-print-period newport6)))
      (stok-newline newport7))))

(define (gencode-non-either-predicate port predicate)
  (let ((keyword         (list-ref predicate 0))
	(predicate-label (list-ref predicate 1))
	(predicate-args  (list-ref predicate 2))
	(guard           (list-ref predicate 3))
	(body            (list-ref predicate 4)))
    (if (equal? keyword 'top-predicate)
	(modinit-add-export-function! predicate-label))
    (let* ((newport1 (gencode-print-predicate-label port predicate-label))
	   (newport2 (gencode-print-predicate-args newport1 predicate-args))
	   (newport3 (gencode-print-tombow newport2))
	   (newport4 (gencode-print-guard newport3 guard))
	   (newport5 (gencode-print-vertical-bar newport4))
	   (newport6 (gencode-clause-list newport5 body))
	   (newport7 (gencode-print-period newport6)))
      (stok-newline newport7))))

(define (gencode-either-predicate port predicate)
  (let ((keyword         (list-ref predicate 0))
	(predicate-label (list-ref predicate 1))
	(predicate-args  (list-ref predicate 2))
	(guard           (list-ref predicate 3))
	(body1           (list-ref predicate 4))
	(body2           (list-ref predicate 5)))
    (let ((false-predicate-args
	   (gencode-insert-at-second-last predicate-args #f))
	  (true-predicate-args
	   (gencode-insert-at-second-last predicate-args 's2k-any)))
      (let ((either-predicate-printer
	     (lambda (predicate-args clause-list pt)
	       (let* ((newport-sub1 (gencode-print-predicate-label pt predicate-label))
		      (newport-sub2 (gencode-print-predicate-args newport-sub1 predicate-args))
		      (newport-sub3 (gencode-print-tombow newport-sub2))
		      (newport-sub4 (gencode-print-guard newport-sub3 guard))
		      (newport-sub5 (gencode-print-vertical-bar newport-sub4))
		      (newport-sub6 (gencode-clause-list newport-sub5 clause-list))
		      (newport-sub7 (gencode-print-period newport-sub6)))
		 (stok-newline newport-sub7)))))
	(let* ((newport1 (either-predicate-printer false-predicate-args body2 port))
	       (newport2 (gencode-print-otherwise newport1)))
	  (either-predicate-printer true-predicate-args  body1 newport2))))))

(define (gencode-insert-at-second-last lis o)
  (if (null? lis)
      (error 'gencode-insert-at-second-last "PANIC! compiler bug" o)
      (if (null? (cdr lis))
	  (cons o lis)
	  (cons (car lis) (gencode-insert-at-second-last (cdr lis) o)))))

(define (gencode-print-predicate-label port predicate-label-structure)
  (let* ((raw-predicate-label (label-get-predicate-label predicate-label-structure))
	 (kl1-predicate-label-string
	  (cond
	   ((label-predicate-label? predicate-label-structure)
	    (string-append "lab_" (symbol->string-with-escape raw-predicate-label)))
	   ((label-predicate-init-label? predicate-label-structure)
	    (string-append "init_" (symbol->string-with-escape raw-predicate-label)))
	   ((label-predicate-label-ref? predicate-label-structure)
	    (string-append
	     "predicate#("
	     (symbol->string-with-escape (gencode-get-this-module-name))
	     ":lab_"
	     (symbol->string-with-escape raw-predicate-label)
	     "/"
	     (integer->string
	      (label-get-predicate-arglen predicate-label-structure))
	     ")"))
	   ((label-predicate-init-label-ref? predicate-label-structure)
	    (string-append
	     "predicate#("
	     (symbol->string-with-escape (gencode-get-this-module-name))
	     ":init_"
	     (symbol->string-with-escape raw-predicate-label)
	     "/"
	     (integer->string
	      (label-get-predicate-arglen predicate-label-structure))
	     ")"))
	   (else
	    (error
	     'gencode-print-predicate-label
	     "Unknown label type"
	     predicate-label-structure)))))
    (stok-display-string-without-quote kl1-predicate-label-string port)))

(define (gencode-print-terminal port x)
  (cond
   ;;;
   ;;; Dont exchange the order between boolean? and symbol?
   ;;; because a boolean value is expressed as a symbol in KL1
   ((boolean? x)
    (if x
	(stok-display-string-without-quote "true" port)
	(stok-display-string-without-quote "false" port)))
   ;;;
   ((symbol? x)
    (cond
     ((equal? x 's2k-any) (stok-display-string-without-quote "_" port))
     (else
      (let ((escaped-upcased-str (string-upcase (symbol->string-with-escape x))))
	(stok-display-string-without-quote escaped-upcased-str port)))))
   ;;;
   ((label-predicate-label-ref? x) (gencode-print-predicate-label port x))
   ;;;
   ((label-predicate-init-label-ref? x) (gencode-print-predicate-label port x))
   ;;;
   ((integer? x) (stok-display-string-without-quote (integer->string x) port))
   ;;;
   ((string? x) (stok-display-string-with-quote x port))
   ;;;
   ((char? x)
    (stok-display-string-without-quote
     (string-append "0'" (list->string (list x))) port))
   ((label-quote? x)
    ;;; the quoted object is either a symbol or a null
    (let ((not-quoted-object (label-get-quoted-object x)))
      (if (null? not-quoted-object)
	  (stok-display-string-without-quote "[]" port)
	  (stok-display-string-without-quote
	   (string-append "'" (symbol->string not-quoted-object) "'") port))))
   ;;;
   (else (error 'gencode-print-terminal "unsupported data type" x))))


(define (gencode-print-tombow port)
  (stok-display-string-without-quote " :- " port))

(define (gencode-print-period port)
  (let ((newport1 (stok-display-string-without-quote "." port)))
    (stok-newline newport1)))
  
(define (gencode-print-otherwise port)
  (let ((newport1 (stok-display-string-without-quote "otherwise." port)))
    (stok-newline newport1)))
  
(define (gencode-print-guard port guard)
  (if (equal? guard #t)
      (stok-display-string-without-quote "true" port)
      (error 'gencode-print-guard "Guard except true is not supported" guard)))

(define (gencode-print-vertical-bar port)
  (let ((newport1 (stok-display-string-without-quote " | " port)))
    (stok-newline newport1)))

(define (gencode-print-equal port)
  (stok-display-string-without-quote " = " port))

(define (gencode-clause-list port0 clause-list)
  (let loop ((lis clause-list)
	     (print-comma-p #f)
	     (port port0))
    (if (null? lis)
	port
	(let ((x (car lis)))
	  (let ((newport1
		 (if print-comma-p
		     (let ((newport-sub1 (stok-display-string-without-quote "," port)))
		       (stok-newline newport-sub1))
		     port)))
	    (let* ((newport2 (gencode-clause newport1 x)))
	      (loop (cdr lis) #t newport2)))))))

(define (gencode-clause port expr)
  (let ((keyword (list-ref expr 0)))
    (cond
     ((equal? keyword 'primitive) (gencode-primitive port expr))
     ((equal? keyword 'assign)    (gencode-assign port expr))
     ((equal? keyword 'apply)     (gencode-apply port expr))
     ((equal? keyword 'direct-apply) (gencode-direct-apply port expr))
     (else
      (error 'gencode-clause "Unknown keyword" expr)))))

(define (gencode-primitive port expr)
  (let ((op      (list-ref expr 1)))
    (cond
     ((equal? op 'record-ref)  (gencode-record-ref          port expr))
     ((equal? op 'make-record) (gencode-make-record         port expr))
     (else                     (gencode-primitive-nonrecord port expr)))))

(define (gencode-record-ref port expr)
  (let (
; I commented out for more speed
;	(keyword (list-ref expr 0))
;	(op      (list-ref expr 1))
	(record  (list-ref expr 2))
	(content (list-ref expr 3)))
    (let* ((newport1 (gencode-print-terminal port record))
	   (newport2 (gencode-print-equal newport1)))
       (gencode-print-record-ref-list newport2 content))))

(define (gencode-print-apply-args-list port lis)
  (gencode-print-parenthesis-list port lis))
(define (gencode-print-primitive-args-list port lis)
  (gencode-print-parenthesis-list port lis))
(define (gencode-print-make-record-list port lis)
  (gencode-print-bracket-list port lis))
(define (gencode-print-record-ref-list port lis)
  (gencode-print-bracket-list port lis))
(define (gencode-print-predicate-args port lis)
  (gencode-print-parenthesis-list port lis))

(define (gencode-print-parenthesis-list port lis)
  (gencode-print-generic-list port lis "(" ")"))
(define (gencode-print-bracket-list port lis)
  (gencode-print-generic-list port lis "[" "]"))

(define (gencode-print-generic-list port lis1 left-bracket right-bracket)
  (let ((newport-a (stok-display-string-without-quote left-bracket port)))
    (let ((newport-b
	   (let loop ((lis lis1)
		      (print-comma-p #f)
		      (newport1 newport-a))
	     (if (null? lis)
		 newport1
		 (let ((x (car lis)))
		   (let ((newport2
			  (if print-comma-p
			      (stok-display-string-without-quote ", " newport1)
			      newport1)))
		     (let ((newport3 (gencode-print-terminal newport2 x)))
		       (loop (cdr lis) #t newport3))))))))
      (stok-display-string-without-quote right-bracket newport-b))))

(define (gencode-make-record port expr)
  (let (
; I commented out for more speed
;	(keyword (list-ref expr 0))
;	(op      (list-ref expr 1))
	(content (list-ref expr 2))
	(record  (list-ref expr 3)))
    (let* ((newport1 (gencode-print-terminal port record))
	   (newport2 (gencode-print-equal newport1)))
      (gencode-print-make-record-list newport2 content))))

(define (gencode-primitive-nonrecord port expr)
  (let (
; I commented out for more speed
;	(keyword (list-ref expr 0))
	(op      (list-ref expr 1))
	(src     (list-ref expr 2))
	(dest    (list-ref expr 3)))
    (let ((newport1 (gencode-print-primitive-op port op)))
      (gencode-print-primitive-args-list newport1 (cons dest src)))))

(define (gencode-print-primitive-op port op)
  (let ((kl1-op (table-convert-op-from-scheme-to-kl1 op)))
    (stok-display-string-without-quote
     (string-append (gencode-s2k-lib-module-name) ":" kl1-op) port)))

(define (gencode-assign port expr)
  (let (
; I commented out for more speed
;	(keyword (list-ref expr 0))
	(src     (list-ref expr 1))
	(dest    (list-ref expr 2)))
    (if (label-global? src)
	(let ((global-table-key (symbol->string (label-get-global src))))
	  (let ((newport1
		 (stok-display-string-without-quote
		  (string-append
		   (gencode-s2k-lib-module-name) ":" "global_table_ref")
		  port)))
	    (gencode-print-primitive-args-list
	     newport1 (list dest global-table-key (label-get-environment-name)))))
	(let* ((newport1 (gencode-print-terminal port dest))
	       (newport2 (gencode-print-equal newport1)))
	  (gencode-print-terminal newport2 src)))))

(define (gencode-print-global-apply port n)
  (let ((newport1
	 (stok-display-string-without-quote
	  (string-append (gencode-s2k-lib-module-name) ":" "apply") port)))
    (stok-display-string-without-quote (integer->string n) newport1)))

(define (gencode-apply port expr)
  (let (
; I commented out for more speed
;	(keyword (list-ref expr 0))
	(closure (list-ref expr 1))
	(args    (list-ref expr 2)))
    (let ((newport1 (gencode-print-global-apply port (- (length args) 1))))
      (gencode-print-apply-args-list newport1 (cons closure args)))))

(define (gencode-direct-apply port expr)
  (let ((keyword (list-ref expr 0))
	(closure (list-ref expr 1))
	(args    (list-ref expr 2)))
    (let ((newport1 (gencode-print-predicate-label port closure)))
      (gencode-print-apply-args-list newport1 args))))

