;-*-SCHEME-*-

;;;
;;; StoK: Scheme to KL1 translator
;;;   Copyright (C) 1997 Yoshihiro Oyama
;;;     <oyama@is.s.u-tokyo.ac.jp>
;;;       December 19, 1997
;;;

;;
;; Scheme parser and Scheme syntax sugar expander
;;

(module expand)

(define (parse define-expr)
  (parse-define define-expr))

(define (parse-define expr)
  (let ((keyword (list-ref expr 0))
	(args    (list-ref expr 1))
	(bodies  (cdr (cdr expr))))
    (if (pair? args)
	;;; function declaration
	(let ((funcname (car args))
	      (funcargs (cdr args)))
	  (parse-make-expr-define-function
	   (parse-funcname funcname)
	   (parse-funcargs funcargs)
	   (parse-expr (cons 'begin bodies))))
	;;; global variable declaration
	;;; we make a function whose name is the name of the global variable.
	(let ((global-variable-name (list-ref expr 1)))
	  (parse-make-expr-define-global-variable
	   (parse-funcname global-variable-name)
	   (parse-expr (cons 'begin bodies)))))))

(define (parse-make-expr-define-global-variable varname body)
   ; we use a dirty to regard the global variable name
   ; as a function name
  (list 'define-global-variable (list varname) body))

(define (parse-make-expr-define-function funcname funcargs body)
  (list 'define-function (cons funcname funcargs) body))

(define (parse-expr e)
  (if (parse-terminal? e)
      (parse-terminal e)
      (let ((op (car e)))
	(let ((expander (parse-lookup-table op (parse-syntax-sugar-table))))
	  (if expander
	      (parse-expr (expander e))
	      (let ((parser (parse-lookup-table op (parse-table))))
		(if parser
		    (parser e)
		    (parse-app e))))))))

;;; Is x terminal?
;;; Consider vector constant!
;;;   (such as (define (foo x) #(1 2 3)))
(define (parse-terminal? x)
  (not (pair? x)))
    
;;; parse terminal
(define (parse-terminal x) (parse-make-expr-terminal x))
(define (parse-make-expr-terminal x) x)

;;; parse if expression
(define (parse-if expr)
  (let ((expr-length (length expr)))
    (cond
     ((= expr-length 3)
      (let ((keyword    (list-ref expr 0))
	    (test       (list-ref expr 1))
	    (consequent (list-ref expr 2)))
	(parse-make-expr-if
	 (parse-expr test)
	 (parse-expr consequent)
	 (parse-expr 0))))
     ((= expr-length 4)
      (let ((keyword    (list-ref expr 0))
	    (test       (list-ref expr 1))
	    (consequent (list-ref expr 2))
	    (alternate  (list-ref expr 3)))
	(parse-make-expr-if
	 (parse-expr test)
	 (parse-expr consequent)
	 (parse-expr alternate))))
     (else
      (error 'parse-if "Invalid syntax for if" expr)))))

;;; make parse tree of if
(define (parse-make-expr-if condition trueclause falseclause)
  (list 'if condition trueclause falseclause))

;;; parse fix expression
(define (parse-fix expr)
  (let ((keyword (list-ref expr 0))
	(binds   (list-ref expr 1))
	(bodies  (cdr (cdr expr))))
    (let ((parsedbinds (parse-local-funcs binds))
	  (parsedbody  (parse-expr (cons 'begin bodies))))
      (parse-make-expr-fix parsedbinds parsedbody))))

;;; parse local functions in fix expression
;;; This part is a dirty part of our parser.
(define (parse-local-funcs binds)
  (parse-make-expr-local-funcs (map parse-local-func binds)))

;;; parse local function 
(define (parse-local-func expr)
  (let ((funcname (list-ref expr 0))
	(funcargs (list-ref expr 1))
	(bodies (cdr (cdr expr))))
    (parse-make-expr-local-func
     (parse-funcname funcname)
     (parse-funcargs funcargs)
     (parse-expr (cons 'begin bodies)))))

;;; make parse tree of fix
(define (parse-make-expr-fix binds body)
  (list 'fix binds body))
(define (parse-make-expr-local-func func args body)
  (list func args body))
(define (parse-make-expr-local-funcs binds) binds)

;;; Correct parsings are omitted here.
(define (parse-funcname funcname) (parse-make-expr-funcname funcname))
(define (parse-funcargs funcargs) (parse-make-expr-funcargs funcargs))
(define (parse-make-expr-funcname funcname) funcname)
(define (parse-make-expr-funcargs funcargs) funcargs)

;;; parse and expand function call
(define (parse-app e)
  (let ((operator (car e))
        (operand  (cdr e)))
    (cond
     ;;; operator is primitive
     ((member operator (table-primitive-ops))
      (parse-make-expr-primitive operator (map parse-expr operand)))
     ;;; operator is not primitive --- user-defined procedure application
     (else
      (let ((parsed-func (parse-expr operator))
	    (parsed-args (map parse-expr operand)))
	(parse-make-expr-app parsed-func parsed-args))))))

;;;  make parse tree of application
(define (parse-make-expr-app operator operand)
  (cons 'apply (cons operator operand)))
;;;  make parse tree of primitive
(define (parse-make-expr-primitive operator operand)
  (cons 'primitive (cons operator operand)))

(define (parse-quote quote-expr)
  (let ((keyword   (list-ref quote-expr 0))
	(atom-expr (list-ref quote-expr 1)))
    (parse-make-expr-quote atom-expr)))

(define (parse-make-expr-quote atom-expr)
  (list '%quote atom-expr))

;;; ================
;;; expand fucntions
;;; ================

(define (expand-quote expr)
  (let loop ((body (list-ref expr 1)))
    (cond
     ((pair? body)
      (list 'cons (loop (car body))  (loop (cdr body))))
     ((or
       (boolean? body)
       (char? body)
       (number? body)
       (string? body))
      body)
     ((or
       (symbol? body)
       (null? body))
      (list '%quote body))
     (else
      ;;; In this case (procedure? body) is true.
      ;;; never reached if this compiler has no bug.
      (error 'expand-quote "PANIC! compiler bug" expr)))))

(define (expand-lambda expr)
  (let ((keyword (list-ref expr 0))
	(args    (list-ref expr 1))
	(bodies  (cdr (cdr expr))))
    (let ((newfunc (unique-name 'lambdatmp)))
      (list
       'fix
       (list (cons newfunc (cons args bodies)))
       newfunc))))

(define (expand-set! expr)
  (let ((keyword (list-ref expr 0))
	(var     (list-ref expr 1))
	(val     (list-ref expr 2)))
    (let ((escaped-var (symbol->string-with-escape var)))
      (list '%global-file-set escaped-var val))))

(define (expand-global-ref expr)
  (let ((keyword (list-ref expr 0))
	(var     (list-ref expr 1)))
    (let ((escaped-var (symbol->string-with-escape var)))
      (list '%global-file-ref escaped-var))))

(define (expand-cond expr)
  (let ((keyword    (car expr))
	(clauselist (cdr expr)))
    (let ((clauselist-len (length clauselist)))
      (cond
       ((= clauselist-len 1)
	(let ((last-clause (car clauselist)))
	  (let ((test      (car last-clause))
		(expr-list (cdr last-clause)))
	    (if (equal? test 'else)
		(cons 'begin expr-list)
		(list 'if test (cons 'begin expr-list) 0)))))
       ((> clauselist-len 1)
	(let ((head-clause (car clauselist)))
	  (let ((test      (car head-clause))
		(expr-list (cdr head-clause)))
	    (list 'if test (cons 'begin expr-list) (cons 'cond (cdr clauselist))))))
       (else
	(error 'expand-cond "Invalid syntax for cond" expr))))))

(define (expand-and expr)
  (let ((keyword  (car expr))
	(and-args (cdr expr)))
    (if (null? and-args)
	#t
	(if (null? (cdr and-args))
	    (car and-args)
	    (list
	     'if
	     (car and-args)
	     (cons 'and (cdr and-args))
	     (car and-args))))))
    
(define (expand-or expr)
  (let ((keyword (car expr))
	(or-args (cdr expr)))
    (if (null? or-args)
	#f
	(if (null? (cdr or-args))
	    (car or-args)
	    (list
	     'if
	     (car or-args)
	     (car or-args)
	     (cons 'or (cdr or-args)))))))

(define (expand-let expr)
  (let ((expr-length (length expr)))
    (if (< expr-length 3)
	(error 'parse-let "Invalid syntax for let" expr)
	(let ((second-subexpr (list-ref expr 1)))
	  (if (pair? second-subexpr)
	      (expand-normal-let expr)
	      (expand-named-let  expr))))))

(define (expand-normal-let expr)
  (let ((keyword (list-ref expr 0))
	(binds   (list-ref expr 1))
	(bodies  (cdr (cdr expr))))
    ;;; making named let
    ;;;  (let ((tmpfuncname 'lettmp))
    ;;;    (list 'let tmpfuncname binds (cons 'begin bodies)))
    ;;; No, instead of translating into named let,
    ;;; we expand let to fix (to get efficient code).
    (let ((unique-localfuncname (unique-name 'lettmp))
	  (boundvars  (map car binds))
	  (initvalues (map cadr binds)))
      (list
       'fix
       (list (cons unique-localfuncname (cons boundvars bodies)))
       (cons unique-localfuncname initvalues)))
    ))

(define (expand-named-let expr)
  (let ((keyword       (list-ref expr 0))
	(localfuncname (list-ref expr 1))
	(binds         (list-ref expr 2))
	(bodies        (cdr (cdr (cdr expr)))))
    (let ((unique-localfuncname (unique-name localfuncname))
	  (boundvars  (map car binds))
	  (initvalues (map cadr binds)))
      (list
       'fix
       (list
	(list
	 unique-localfuncname
	 boundvars
	 (list
	  (cons 'lambda (cons (list localfuncname) bodies))
	  unique-localfuncname)))
       (cons unique-localfuncname initvalues)))))

(define (expand-let* expr)
  (let ((keyword (list-ref expr 0))
	(binds   (list-ref expr 1))
	(bodies  (cdr (cdr expr))))
    (if (null? binds)
	(cons 'begin bodies)
	(list
	 'let
	 (list (car binds))
	 (cons 'let* (cons (cdr binds) bodies))))))

(define (expand-begin expr)
  (let ((keyword   (car expr))
	(expr-list (cdr expr)))
    (let ((expr-list-len (length expr-list)))
      (cond
       ((= expr-list-len 1) (car expr-list))
       ((> expr-list-len 1)
	(let ((d (unique-name 'd)))
	  (cons
	   'let
	   (cons
	    (list (list d (car expr-list)))
	    (cdr expr-list)))))
       (else
	(error 'expand-begin "Invalid syntax for begin" expr))))))

(define (expand-not expr) (cons '%not (cdr expr)))
(define (expand-boolean? expr) (cons '%boolean? (cdr expr)))
(define (expand-equal? expr) (cons '%equal? (cdr expr)))
(define (expand-pair? expr) (cons '%pair? (cdr expr)))
(define (expand-cons expr) (cons '%cons (cdr expr)))
(define (expand-car expr) (cons '%car (cdr expr)))
(define (expand-cdr expr) (cons '%cdr (cdr expr)))
(define (expand-cadr expr) (cons '%cadr (cdr expr)))
(define (expand-caddr expr) (cons '%caddr (cdr expr)))
(define (expand-null? expr) (cons '%null? (cdr expr)))
(define (expand-list expr)
  (let ((op   (car expr))
	(args (cdr expr)))
    (expand-foldr-prims op '%cons args (list 'quote '()) #t)))
(define (expand-length expr) (cons '%length (cdr expr)))
(define (expand-append expr)
  (let ((op   (car expr))
	(args (cdr expr)))
    (expand-foldr-prims op '%append args (list 'quote '()) #f)))
(define (expand-list-ref expr)
  (let ((lis   (list-ref expr 1))
	(index (list-ref expr 2)))
    ;;; optimize when index is constant
    (if (integer? index)
	(cond
	 ((= index 0) (list '%car   lis))
	 ((= index 1) (list '%cadr  lis))
	 ((= index 2) (list '%caddr lis))
	 (else (list '%list-ref lis index)))
	(list '%list-ref lis index))))
(define (expand-member expr) (cons '%member (cdr expr)))
(define (expand-symbol? expr) (cons '%symbol? (cdr expr)))

(define (expand-symbol->string expr) (cons '%symbol->string (cdr expr)))
(define (expand-string->symbol expr) (cons '%string->symbol (cdr expr)))
(define (expand-number? expr) (cons '%number? (cdr expr)))
(define (expand-integer? expr) (cons '%integer? (cdr expr)))

;;; Ugh!: arbitrary number of arguments should be received
(define (expand-= expr) (cons '%= (cdr expr)))
(define (expand-> expr) (cons '%> (cdr expr)))
(define (expand-< expr) (cons '%< (cdr expr)))
(define (expand->= expr) (cons '%>= (cdr expr)))
(define (expand-<= expr) (cons '%<= (cdr expr)))

(define (expand-+ expr)
  (let ((op   (car expr))
	(args (cdr expr)))
    (expand-foldr-prims op '%+ args 0 #f)))
(define (expand-* expr)
  (let ((op   (car expr))
	(args (cdr expr)))
    (expand-foldr-prims op '%* args 0 #f)))

(define (expand-- expr)
  (expand---or-quotient-or-remainder expr '%- 0))
(define (expand-quotient expr)
  (expand---or-quotient-or-remainder expr '%quotient 1))
(define (expand-remainder expr)
  (expand---or-quotient-or-remainder expr '%remainder 0))
(define (expand---or-quotient-or-remainder expr newop atom-elem)
  (let ((op   (car expr))
	(args (cdr expr)))
    (let ((args-len (length args)))
      (cond
       ((= args-len 2) (cons newop args))
       ((> args-len 2) (expand-foldl-prims op newop args atom-elem #f))
       ((= args-len 1) (list newop atom-elem (car args)))
       (else
	(error 'expand-- "Invalid args for - or /" expr))))))

(define (expand-char? expr) (cons '%char? (cdr expr)))
(define (expand-char=? expr) (cons '%char=? (cdr expr)))
(define (expand-char<? expr) (cons '%char<? (cdr expr)))
(define (expand-char>? expr) (cons '%char>? (cdr expr)))
(define (expand-char<=? expr) (cons '%char<=? (cdr expr)))
(define (expand-char>=? expr) (cons '%char>=? (cdr expr)))

(define (expand-char->integer expr) (cons '%char->integer (cdr expr)))
(define (expand-integer->char expr) (cons '%integer->char (cdr expr)))

(define (expand-string? expr) (cons '%string? (cdr expr)))

(define (expand-make-string expr) (cons '%make-string (cdr expr)))
(define (expand-string-length expr) (cons '%string-length (cdr expr)))
(define (expand-string-ref expr) (cons '%string-ref (cdr expr)))
(define (expand-string-set expr) (cons '%string-set (cdr expr)))
(define (expand-string-append expr)
  (let ((op   (car expr))
	(args (cdr expr)))
    (expand-foldr-prims op '%string-append args "" #f)))

(define (expand-procedure? expr) (cons '%procedure? (cdr expr)))
(define (expand-current-input-port expr) (cons '%current-input-port (cdr expr)))
(define (expand-current-output-port expr) (cons '%current-output-port (cdr expr)))
(define (expand-open-input-file expr) (cons '%open-input-file (cdr expr)))
(define (expand-open-output-file expr) (cons '%open-output-file (cdr expr)))
(define (expand-close-input-port expr) (cons '%close-input-port (cdr expr)))
(define (expand-close-output-port expr) (cons '%close-output-port (cdr expr)))
(define (expand-stok-read-char expr) (cons '%stok-read-char (cdr expr)))
(define (expand-eof-object? expr) (cons '%eof-object? (cdr expr)))
(define (expand-stok-newline expr) (cons '%stok-newline (cdr expr)))
(define (expand-stok-write-char expr) (cons '%stok-write-char (cdr expr)))
(define (expand-stok-display-string-with-quote expr)
  (cons '%stok-display-string-with-quote (cdr expr)))
(define (expand-stok-display-string-without-quote expr)
  (cons '%stok-display-string-without-quote (cdr expr)))
(define (expand-error expr) (cons '%error (cdr expr)))
(define (expand-exit expr) (cons '%exit (cdr expr)))

(define (expand-isqrt expr) (cons '%isqrt (cdr expr)))

;;;
;;;
;;;

(define (parse-lookup-table keysym table)
  (if (null? table)
      #f
      (if (equal? keysym (car (car table)))
	  (cdr (car table))
	  (parse-lookup-table keysym (cdr table)))))

(define (parse-table)
  (list
   (cons 'if  parse-if)
   (cons 'fix parse-fix)
   (cons '%quote parse-quote)
   ))

(define (parse-syntax-sugar-table)
  (list
   (cons 'quote      expand-quote)
   (cons 'lambda     expand-lambda)
   ; if
   (cons 'set!       expand-set!)
   (cons 'global-ref expand-global-ref)
   (cons 'cond       expand-cond)
   (cons 'and        expand-and)
   (cons 'or         expand-or)
   (cons 'let        expand-let) ; includingnamed-let
   (cons 'let*       expand-let*)
   (cons 'begin      expand-begin)
   ; define
   (cons 'not      expand-not)
   (cons 'boolean?     expand-boolean?)
   (cons 'equal?     expand-equal?)
   (cons 'pair?     expand-pair?)
   (cons 'cons     expand-cons)
   (cons 'car     expand-car)
   (cons 'cdr     expand-cdr)
   (cons 'null? expand-null?)
   (cons 'list       expand-list)
   (cons 'length expand-length)
   (cons 'append expand-append)
   ; reverse
   (cons 'list-ref expand-list-ref)
   (cons 'member expand-member)
   ; assoc
   (cons 'symbol? expand-symbol?)
   (cons 'symbol->string expand-symbol->string)
   (cons 'string->symbol expand-string->symbol)
   (cons 'number? expand-number?)
   (cons 'integer? expand-integer?)
   (cons '= expand-=)
   (cons '< expand-<)
   (cons '> expand->)
   (cons '<= expand-<=)
   (cons '>= expand->=)
   (cons '+ expand-+)
   (cons '- expand--)
   (cons '* expand-*)
   (cons 'quotient expand-quotient)
   (cons 'remainder expand-remainder)
   ; integer->string
   ; string->integer
   (cons 'char? expand-char?)
   (cons 'char=? expand-char=?)
   (cons 'char<? expand-char<?)
   (cons 'char>? expand-char>?)
   (cons 'char<=? expand-char<=?)
   (cons 'char>=? expand-char>=?)
   (cons 'char->integer expand-char->integer)
   (cons 'integer->char expand-integer->char)
   ; char-upcase
   ; char-downcase
   (cons 'string? expand-string?)
   (cons 'make-string expand-make-string)
   (cons 'string-length expand-string-length)
   (cons 'string-ref expand-string-ref)
   (cons 'string-set expand-string-set)
   (cons 'string-append expand-string-append)
   ; string->list
   ; list->string
   (cons 'procedure? expand-procedure?)
   ; map
   (cons 'current-input-port expand-current-input-port)
   (cons 'current-output-port expand-current-output-port)
   (cons 'open-input-file expand-open-input-file)
   (cons 'open-output-file expand-open-output-file)
   (cons 'close-input-port expand-close-input-port)
   (cons 'close-output-port expand-close-output-port)
   ; stok-read
   (cons 'stok-read-char expand-stok-read-char)
   (cons 'eof-object? expand-eof-object?)
   ; 'stok-display
   (cons 'stok-newline expand-stok-newline)
   (cons 'stok-write-char expand-stok-write-char)
   ; stok-read-get-obj
   ; stok-read-get-port
   (cons 'stok-display-string-with-quote expand-stok-display-string-with-quote)
   (cons 'stok-display-string-without-quote expand-stok-display-string-without-quote)
   (cons 'error expand-error)
   (cons 'exit expand-exit)

   (cons 'isqrt expand-isqrt)
   ))

(define (expand-foldr-prims op new-op list-args terminator always-need-terminatorp)
  (cond
   ((null? list-args) terminator)
   ((null? (cdr list-args))
    (if always-need-terminatorp
	(list new-op (car list-args) terminator)
	(car list-args)))
   (else
    (list
     new-op
     (car list-args)
     (expand-foldr-prims
      op
      new-op
      (cdr list-args)
      terminator
      always-need-terminatorp)))))

(define (expand-foldl-prims op new-op list-args terminator always-need-terminatorp)
  (let loop ((reversed-list-args (reverse list-args)))
    (cond
     ((null? reversed-list-args) terminator)
     ((null? (cdr reversed-list-args))
      (if always-need-terminatorp
	  (list new-op terminator (car reversed-list-args))
	  (car reversed-list-args)))
     (else
      (list
       new-op
       (loop (cdr reversed-list-args))
       (car reversed-list-args))))))

