;-*-SCHEME-*-

;;;
;;; StoK: Scheme to KL1 translator
;;;   Copyright (C) 1997 Yoshihiro Oyama
;;;     <oyama@is.s.u-tokyo.ac.jp>
;;;       December 19, 1997
;;;

;;
;; Abstract KL1 code generator
;;  This generator simply flattens local predicates by
;;  eliminating fix expression.
;; 

(module amc)

(define (amc-cons unprocessed-predicates expr)
  (cons unprocessed-predicates expr))
(define (amc-car unprocessed-predicates-and-expr)
  (car unprocessed-predicates-and-expr))
(define (amc-cdr unprocessed-predicates-and-expr)
  (cdr unprocessed-predicates-and-expr))

(define (amc top-predicate)
  (let ((unprocessed-predicates (list top-predicate)))
    ;;; a top predicate is regarded as a local predicate
    (amc-predicates unprocessed-predicates)))

(define (amc-predicates unprocessed-predicates)
  (if (null? unprocessed-predicates)
      '()
      (let ((unprocessed-predicate (car unprocessed-predicates)))
	(let ((newly-added-unprocessed-predicates-and-converted-predicate
	       (amc-predicate unprocessed-predicate)))
	  (let ((newly-added-unprocessed-predicates
		 (amc-car newly-added-unprocessed-predicates-and-converted-predicate))
		(converted-predicate
		 (amc-cdr newly-added-unprocessed-predicates-and-converted-predicate)))
	    (cons
	     converted-predicate
	     (amc-predicates
	      (append
	       newly-added-unprocessed-predicates
	       (cdr unprocessed-predicates)))))))))

(define	(amc-predicate predicate)
  (let ((keyword (car predicate)))
    (cond
     ;;; a top predicate is regarded as a local predicate
     ((equal? keyword 'top-predicate)
      (amc-local-predicate predicate))
     ((equal? keyword 'local-predicate)
      (amc-local-predicate predicate))
     ;;; a global variable predicate is regarded as a local predicate
     ((equal? keyword 'global-variable)
      (amc-local-predicate predicate))
     ((equal? keyword 'either-predicate)
      (amc-either-predicate predicate))
     (else
      (error 'amc-predicate "Unknown predicate type" predicate)))))

(define (amc-local-predicate predicate)
  (let ((keyword        (list-ref predicate 0))
	(predicate-name (list-ref predicate 1))
	(predicate-args (list-ref predicate 2))
	(guard          (list-ref predicate 3))
	(body           (list-ref predicate 4)))
    (let ((unprocessed-predicates-and-converted-body (amc-parallel body)))
      (let ((unprocessed-predicates
	     (amc-car unprocessed-predicates-and-converted-body))
	    (converted-body
	     (amc-cdr unprocessed-predicates-and-converted-body)))
	(amc-cons
	 unprocessed-predicates
	 (list
	  keyword
	  predicate-name
	  predicate-args
	  guard
	  converted-body))))))

(define (amc-either-predicate predicate)
  (let ((keyword        (list-ref predicate 0))
	(predicate-name (list-ref predicate 1))
	(predicate-args (list-ref predicate 2))
	(guard          (list-ref predicate 3))
	(body1          (list-ref predicate 4))
	(body2          (list-ref predicate 5)))
    (let ((unprocessed-predicates-and-converted-body1 (amc-parallel body1))
	  (unprocessed-predicates-and-converted-body2 (amc-parallel body2)))
      (let ((unprocessed-predicates1
	     (amc-car unprocessed-predicates-and-converted-body1))
	    (converted-body1
	     (amc-cdr unprocessed-predicates-and-converted-body1))
	    (unprocessed-predicates2
	     (amc-car unprocessed-predicates-and-converted-body2))
	    (converted-body2
	     (amc-cdr unprocessed-predicates-and-converted-body2)))
	(amc-cons
	 (append unprocessed-predicates1 unprocessed-predicates2)
	 (list
	  keyword
	  predicate-name
	  predicate-args
	  guard
	  converted-body1
	  converted-body2))))))

(define (amc-parallel expr-list1)
  (let loop
      ((expr-list expr-list1)
       (converted-expr-list '())
       (unprocessed-predicates '()))
    (if (null? expr-list)
	(amc-cons unprocessed-predicates converted-expr-list)
	(let ((expr (car expr-list)))
	  (let ((newly-added-unprocessed-predicates-and-converted-expr
		 (amc-main expr)))
	    (let ((newly-added-unprocessed-predicates
		   (amc-car newly-added-unprocessed-predicates-and-converted-expr))
		  (converted-expr ; notice that converted expr is expr-list
		   (amc-cdr newly-added-unprocessed-predicates-and-converted-expr)))
	      (loop
	       (cdr expr-list)
	       (append converted-expr-list converted-expr)
	       (append unprocessed-predicates newly-added-unprocessed-predicates))))))))

(define (amc-main expr)
  (let ((op (car expr))) ; should we provide a wrap function?
    (cond
     ((equal? op 'primitive) (amc-primitive expr))
     ((equal? op 'assign)    (amc-assign expr))
     ((equal? op 'apply)     (amc-apply expr))
     ((equal? op 'fix)       (amc-fix expr))
     ((equal? op 'direct-apply) (amc-apply expr))
     (else
      (error 'amc-main "Unknown operator" expr)))))

;;; not touched
(define (amc-primitive expr) (amc-cons '() (list expr)))
(define (amc-assign expr)    (amc-cons '() (list expr)))
(define (amc-apply expr)     (amc-cons '() (list expr)))

(define (amc-fix expr)
  (let ((keyword (list-ref expr 0))
	(binds   (list-ref expr 1))
	(body    (list-ref expr 2)))
    (if (not (= (length binds) 1))
	(error 'amc-fix "Multiple local function bind in fix not supported" expr)
	(let ((bind (car binds)))
	  (let ((newly-added-unprocessed-predicates-and-converted-expr
		 (amc-parallel body)))
	    (let ((newly-added-unprocessed-predicates
		   (amc-car newly-added-unprocessed-predicates-and-converted-expr))
		  (converted-expr
		   (amc-cdr newly-added-unprocessed-predicates-and-converted-expr)))
	      (amc-cons
	       (cons bind newly-added-unprocessed-predicates)
	       converted-expr)))))))

