/* huiMAccess.c;
 *	Monitor Access module for Helios User Interface; 
 *		created on September 22th, 1994.
 *
 * $Id: huiMAccess.c,v 1.9 1995/03/02 09:25:11 k3sato Exp $
 */

#include <stdio.h>
#include <sys/time.h>
#include <string.h>

#include <helios/SMessage.h>
#include "helios.h"
#include "huiMAccess.h"
#include "huiMessage.h"


static struct timeval		huiTimeout = {(long)25, (long)0};

HuiBoolean huiSetupTraceMsg();
HuiBoolean huiGetMonMsgType();

typedef struct _HuiMonMsgTypeList {
    HuiMonMsgType	msgType;
    char		*str;
} HuiMonMsgTypeList;

static HuiMonMsgTypeList MonMsgTypeList[] = {
    {HuiMonMsg_UI2MonitorMsg,		"Message type from UI to Monitor"},
    {HuiMonMsg_UserInfo,		"UserInfo"},
    {HuiMonMsg_AgentInfo,		"AgentInfo"},
    {HuiMonMsg_AgentInfoEnd,		"End of AgentInfo"},

    {HuiMonMsg_TraceInfo,		"Trace information from UI to Monitor"},
    {HuiMonMsg_SMessageInfo,		"S-Message information"},
    {HuiMonMsg_MTraceInfo,		"Message trace information"},
    {HuiMonMsg_ATraceInfo,		"Agent trace information"},
    {HuiMonMsg_AgentStatusInfo,		"Agent status information"},

    {HuiMonMsg_AskMonitorStatus,	"Ask monitor Status from UI"},
    {HuiMonMsg_AskTraceSpeed,		"Ask Trace speed of monitor"},


    {HuiMonMsg_Monitor2UIMsg,		"Message type from Monitor to UI"},
    {HuiMonMsg_MonitorStandby,		"Monitor process stands by"},
    {HuiMonMsg_AgentTreeDispStandby,	"Agent tree display stands by"},
    {HuiMonMsg_AskForNextMsg,		"Ask for next message"},
    {HuiMonMsg_WaitForMsgFromUI,	"Wait for message from UI"},

    {HuiMonMsg_MonitorStatus,		"Monitor Status"},
    {HuiMonMsg_TraceSpeedValue,		"Trace speed of monitor"},

    {HuiMonMsg_MonitorErrorMsg,		"Monitor Error Message"},
    {HuiMonMsg_ErrorInMsgFromUI,	"Error in message received from UI"},

    {HuiMonMsg_TestMsgUI2Monitor,	"Test Message from UI to monitor"},
    {HuiMonMsg_TestMsgMonitor2UI,	"Test Message from monitor to UI"},

    {NULL,			NULL},
};

char *huiMonMsgType2Str(msgType)
    HuiMonMsgType	msgType;
{
    HuiMonMsgTypeList	*currList;

    for (currList = MonMsgTypeList
	 ; currList->str != NULL
	 ; currList++) {
	if (currList->msgType == msgType) {
	    return currList->str;
	}
    }
    return "";
}

/* $B%b%K%?!<$X%a%C%;!<%8$rAw$k(B
 */
HuiBoolean huiSendMsg2Monitor(hAccessInfo, msg)
    HHuiAccessInfo	hAccessInfo;
    char		*msg;
{
    HuiMonitorInfo	*mInfo;
    int			pipeFd;

    fd_set		mask;
    fd_set		writeOk;
    int			width;
    int			retry;
    int			length;

    HuiErrorMsgType	eMsgType;

    if ((length = strlen(msg) + 1) >= HuiTextBufLen_Lim) {
	eMsgType = HuiEMsg_TextLenLimitOver;
	goto Error_Rtn;
    }

    mInfo = &(hAccessInfo->monitor);

    if (mInfo->state == HuiOff) {
	eMsgType = HuiEMsg_NotRunMonitorProcess;
	goto Error_Rtn;
    }

    pipeFd = mInfo->pipeOut;

    FD_ZERO(&mask);
    FD_SET(pipeFd, &mask);
    width = pipeFd + 1;

    for (retry = HuiMaxNumOfRetry; retry > 0; retry--) {
	/* select $B$NA0$K(B writeOk $B$r=i4|2=$9$k!#(B
	 */
	writeOk = mask;
	select(width, NULL, (fd_set *)&writeOk, NULL, &huiTimeout);

	/* $B%=%1%C%H$+$iFI$_9~$_2D$+!)(B
	 */
	if (FD_ISSET(pipeFd, &writeOk)) {
	    if ((length = write(pipeFd, msg, length)) == ERROR_FLAG) {
		HuiPerror();
		eMsgType = HuiEMsg_PipeWriteError;
		goto Error_Rtn;
	    }
	    return HdTrue;
	}
    }

    eMsgType = HuiEMsg_OverMaxNumOfRetryToWritePipe;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* $B%b%K%?!<$+$i%a%C%;!<%8$r<u$1<h$k(B
 */
HuiBoolean huiReceiveMsgFromMonitor(hAccessInfo, msg)
    HHuiAccessInfo	hAccessInfo;
    char		*msg;
{
    HuiMonitorInfo	*mInfo;
    int			pipeFd;
    fd_set		mask;
    fd_set		readOk;
    int			width;
    int			retry;
    int			length;
    HuiErrorMsgType	eMsgType;


    mInfo = &(hAccessInfo->monitor);

    if (mInfo->state == HuiOff) {
	eMsgType = HuiEMsg_NotRunMonitorProcess;
	goto Error_Rtn;
    }

    pipeFd = mInfo->pipeIn;

    FD_ZERO(&mask);
    FD_SET(pipeFd, &mask);
    width = pipeFd + 1;

    for (retry = HuiMaxNumOfRetry; retry > 0; retry--) {
	/* select $B$NA0$K(B readOk $B$r=i4|2=$9$k!#(B
	 */
	readOk = mask;
	select(width, (fd_set *)&readOk, NULL, NULL, &huiTimeout);

	/* $B%=%1%C%H$+$iFI$_9~$_2D$+!)(B
	 */
	if (FD_ISSET(pipeFd, &readOk)) {
	    if ((length = read(pipeFd, msg, HuiTextBufLen_Lim)) == ERROR_FLAG) {
		HuiPerror();
		eMsgType = HuiEMsg_PipeReadError;
		goto Error_Rtn;
	    }
	    if (length >= HuiTextBufLen_Lim) {
		eMsgType = HuiEMsg_TextLenLimitOver;
		goto Error_Rtn;
	    }
	    if (length == 0) {
		eMsgType = HuiEMsg_NoPipeData;
		goto Error_Rtn;
	    }
	    msg[length] = '\0';
	    return HuiTrue;
	}
    }

    eMsgType = HuiEMsg_OverMaxNumOfRetryToReadPipe;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* S $B%a%C%;!<%8$r(B monitor $B$KAw$k(B
 */
HuiBoolean huiSendSMsg2Monitor(hAccessInfo, sMsg)
    HHuiAccessInfo	hAccessInfo;
    SMessage		sMsg;
{
    HuiMonMsgType	monMsgType;
    int			traceSpeed;
    HuiRecvBuf		msgBuf;
    HuiErrorMsgType	eMsgType;

    if (huiSetupTraceMsg(msgBuf, sMsg) == HuiFalse) {
	eMsgType = HuiEMsg_CSMGetTypeError;
	goto Error_Rtn;
    }

    if (huiSendMsg2Monitor(hAccessInfo, msgBuf) == HuiFalse) {
	eMsgType = HuiEMsg_SendMsg2MonitorError;
	goto Error_Rtn;
    }

    if (huiReceiveMsgFromMonitor(hAccessInfo, msgBuf) == HuiFalse) {
	eMsgType = HuiEMsg_ReceiveMsgFromMonitorError;
	goto Error_Rtn;
    }

    huiPrintf("received data from monitor:\n\t%s\n", msgBuf);

    if (huiSetupTraceSpeedFromMsg(hAccessInfo, msgBuf) == HuiFalse) {
	if (huiGetMonMsgType(hAccessInfo, msgBuf, &monMsgType) == HuiFalse) {
	    eMsgType = HuiEMsg_GetMonMsgTypeError;
	    goto Error_Rtn;
	}

	if (monMsgType != HuiMonMsg_WaitForMsgFromUI) {
	    eMsgType = HuiEMsg_ErrorInMonitor;
	    goto Error_Rtn;
	}
    }

    traceSpeed = hAccessInfo->monitor.traceSpeed;

    if (traceSpeed > 0) {
	if (traceSpeed < 10) {
	    traceSpeed = 10 - traceSpeed;
	    sleep(traceSpeed);
	}
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Setup Trace Message for the message to Monitor
 */
HuiBoolean huiSetupTraceMsg(buf, sMsg)
    char		*buf;
    SMessage		sMsg;
{
    SMType		sMType;
    char		*replyFrom;
    char		*replyState, *replyMsg;
    HuiErrorMsgType	eMsgType;

    if ((sMType = CSMGetType(sMsg)) == NULL) {
	eMsgType = HuiEMsg_CSMGetTypeError;
	goto Error_Rtn;
    }

    /* Agent ID of sender. (ex. "0.1.9")
     */
    replyFrom = CSMGetFromAidStr(sMsg);

    replyState = CSMGetStatusStr(sMsg);

    /* message.
     */
    replyMsg = CSMGetDataStr(sMsg);

    /*
      sprintf(buf, "%d %s %s %s",
	    (sMType->debug != NULL) ? HuiMonMsg_MTraceInfo : HuiMonMsg_ATraceInfo,
	    replyFrom, replyTo, replyState);
     */

    sprintf(buf, "%d\n%s", HuiMonMsg_SMessageInfo, replyMsg);

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Setup Trace Speed of Monitor
 */
HuiBoolean huiSetupTraceSpeedFromMsg(hAccessInfo, msg)
    HHuiAccessInfo	hAccessInfo;
    char		*msg;
{
    HuiMonMsgType	monMsgType;
    int			traceSpeed;
    int			ret;
    HuiErrorMsgType	eMsgType;

    ret = sscanf(msg, "%d %d", &monMsgType, &traceSpeed);
    if (ret == 0 || ret == EOF) {
	eMsgType = HuiEMsg_NotTraceSpeedValueMsgType;
	goto Error_Rtn;
    }

    if (monMsgType != HuiMonMsg_TraceSpeedValue) {
	eMsgType = HuiEMsg_NotTraceSpeedValueMsgType;
	goto Error_Rtn;
    }

    hAccessInfo->monitor.traceSpeed = traceSpeed;

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Get Monitor Message Type from Message
 */
HuiBoolean huiGetMonMsgType(hAccessInfo, msg, monMsgType)
    HHuiAccessInfo	hAccessInfo;
    char		*msg;
    HuiMonMsgType	*monMsgType;
{
    int			ret;
    HuiErrorMsgType	eMsgType;

    ret = sscanf(msg, "%d", monMsgType);
    if (ret == 0 || ret == EOF) {
	eMsgType = HuiEMsg_NoMonMsgTypeInMsg;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*  HuiMonMsgType $B$r(B monitor $B$KAw$k(B
 */
HuiBoolean huiSendMonMsgType2Monitor(hAccessInfo, monMsgType)
    HHuiAccessInfo	hAccessInfo;
    HuiMonMsgType	monMsgType;
{
    HuiWord		msgBuf;
    HuiErrorMsgType	eMsgType;
    char		*ret;

    ret = sprintf(msgBuf, "%d \n", monMsgType);
    if (ret == NULL) {
	eMsgType = HuiEMsg_SprintfError;
	goto Error_Rtn;
    }

    if (huiSendMsg2Monitor(hAccessInfo, msgBuf) == HuiFalse) {
	eMsgType = HuiEMsg_SendMsg2MonitorError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*  HuiMonMsgType $B$r(B monitor $B$+$i<u$1<h$k(B
 */
HuiBoolean huiReceiveMonMsgTypeFromMonitor(hAccessInfo, monMsgType)
    HHuiAccessInfo	hAccessInfo;
    HuiMonMsgType	*monMsgType;
{
    HuiRecvBuf		msgBuf;
    HuiErrorMsgType	eMsgType;

    if (huiReceiveMsgFromMonitor(hAccessInfo, msgBuf) == HuiFalse) {
	eMsgType = HuiEMsg_ReceiveMsgFromMonitorError;
	goto Error_Rtn;
    }

    if (huiGetMonMsgType(hAccessInfo, msgBuf, monMsgType) == HuiFalse) {
	eMsgType = HuiEMsg_GetMonMsgTypeError;
	goto Error_Rtn;
    }

    huiPrintf("Message type from monitor ; %s\n", huiMonMsgType2Str(*monMsgType));

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}
