/* huiDAccess.c;
 *	Daemon access module for Helios User Interface,
 *		created on July 22nd, 1994.
 *
 *	modification history:	
 *	(1) add the functions to send the message HdMsg_StartAppGui
 *		on September 1st in '94.
 *	(2) add the functions to receive the agentId and sin_port for user
 *		from heliosd, on October 3rd in '94.
 *	(3) add the functions to send the message HdMsg_ResetAgent
 *		on October 11th in '94.
 *	(4) add the functions to send the message HdMsg_GetParaAgent
 *		on November 14th in '94.
 *
 * $Id: huiDAccess.c,v 2.1 1994/12/12 06:27:19 k3sato Exp $
 */

#include <stdio.h>
#include <pwd.h>

#include <helios/hdMessage.h>
#include <helios/hDaemonLib.h>

#include "helios.h"
#include "huiMessage.h"
#include "huiMemory.h"
#include <sys/socket.h>

static HAgentInfo huiGetUserInfo();
static HuiBoolean huiSetupAgentID2UserInfo();
static HuiBoolean huiSetupHostName2UserInfo();
static HuiBoolean huiSetupAgentName2UserInfo();
static HuiBoolean huiSetupSockInfo2UserInfo();
static HuiBoolean huiSendUserInfo2Daemon();
static HuiBoolean huiSetupUserInfoMsg2Daemon();

HuiBoolean huiResetAgent();

/* ==========================================================
 *	Functions to send the message of HdMsg_SetupUserInfo
 *	to Daemon
 * ========================================================== */
/*
 * $B%f!<%6!&%$%s%?%U%'!<%9>pJs$r%G!<%b%s$KDLCN$9$k4X?t(B
 */
HAgentInfo huiSetupUserInfo()
{
    HAgentInfo		userInfo;
    HuiErrorMsgType	eMsgType;

    if ((userInfo = huiGetUserInfo()) == NULL) {
	eMsgType = HuiEMsg_GetUserInfoError;
	goto Error_Rtn;
    }

    /* $B%f!<%6>pJs$r%G!<%b%s$XAw$k(B
     */
    if (huiSendUserInfo2Daemon(userInfo) == HuiFalse) {
	eMsgType = HuiEMsg_SendUserInfo2Daemon;
	goto Error_Rtn;
    }

    return userInfo;

 Error_Rtn:
    HuiFreeUserInfo(&userInfo);
    huiPutsErrorMsg(eMsgType);
    return NULL;
}

/* Set user interface information to the struct of HuiUserInfo;
 *
 *	char		*agentID;
 *	char		*agentName;
 *	SocketInfo	sockInfo;
 *	int		pid;
 */
static HAgentInfo huiGetUserInfo()
{
    HAgentInfo		userInfo;
    HuiErrorMsgType	eMsgType;

    if ((userInfo = (HAgentInfo)HuiMalloc(sizeof(AgentInfo))) == NULL) {
	eMsgType = HuiEMsg_HuiMallocError;
	goto Error_Rtn;
    }

    if ((userInfo->agentPath = HuiMallocString("helios")) == NULL) {
	eMsgType = HuiEMsg_HuiMallocStringError;
	goto Error_Rtn;
    }

    /* $B%f!<%6!&%$%s%?%U%'!<%9$N(B agent $BL>$N@_Dj(B
     */
    if (huiSetupAgentName2UserInfo(userInfo) == HuiFalse) {
	eMsgType = HuiEMsg_SetupAgentName2UserInfo;
	goto Error_Rtn;
    }

    /* $B%=%1%C%H>pJs$N@_Dj(B
     */
    if (huiSetupSockInfo2UserInfo(userInfo) == HuiFalse) {
	eMsgType = HuiEMsg_SetupSockInfo2UserInfo;
	goto Error_Rtn;
    }

    /* pid $B$N@_Dj(B */
    userInfo->pid = getpid();

    return userInfo;

 Error_Rtn:
    HuiFreeUserInfo(&userInfo);
    huiPutsErrorMsg(eMsgType);
    return NULL;
}

/* $B%f!<%6!&%$%s%?%U%'!<%9$N(B agent ID $B$N@_Dj(B
 */
static HuiBoolean huiSetupAgentID2UserInfo(userInfo, agentId)
    HAgentInfo		userInfo;
    char		*agentId;
{
    HuiErrorMsgType	eMsgType;

    if ((userInfo->agentId = HuiMallocString(agentId)) == NULL) {
	eMsgType = HuiEMsg_HuiMallocStringError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* $B%f!<%6!&%$%s%?%U%'!<%9$r5/F0$7$?%^%7%s$N%[%9%HL>$N3MF@(B
 */
static HuiBoolean huiSetupHostName2UserInfo(sockInfo)
    HSocketInfo		sockInfo;

{
    char		hostname[HostNameLen_Lim];
    HuiErrorMsgType	eMsgType;

    /* $B%f!<%6!&%$%s%?%U%'!<%9$r5/F0$7$?%^%7%s$N%[%9%HL>(B */
    if (gethostname(hostname, HostNameLen_Lim) == ERROR_FLAG) {
	eMsgType = HuiEMsg_NotFindHostName;
	HuiPerror();
	goto Error_Rtn;
    }

    if ((sockInfo->hostName = HuiMallocString(hostname)) == NULL) {
	eMsgType = HuiEMsg_HuiMallocStringError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* $B%f!<%6!&%$%s%?%U%'!<%9$N(B agent $BL>(B $B$H$7$F!"(B
 * $B%f!<%6$N(B login name + process Id $B$r@_Dj$9$k!#(B
 */
static HuiBoolean huiSetupAgentName2UserInfo(userInfo)
    HAgentInfo		userInfo;
{
    struct passwd	*userpwd;
    HuiWord		agentName;
    HuiErrorMsgType	eMsgType;

    /* $B%f!<%6!&%$%s%?%U%'!<%9$r5/F0$7$?%f!<%6$N(B login $BL>$N3MF@(B */
    if ((userpwd = getpwuid(getuid())) == NULL) {
	eMsgType = HuiEMsg_NotGetLoginNameOfUser;
	goto Error_Rtn;
    }

    /*
    sprintf(agentName, "%s_%d", userpwd->pw_name, getpid());
    */
    sprintf(agentName, "%s", userpwd->pw_name);

    if ((userInfo->agentName = HuiMallocString(agentName)) == NULL) {
	eMsgType = HuiEMsg_HuiMallocStringError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* $B%=%1%C%H>pJs$N@_Dj(B
 */
static HuiBoolean huiSetupSockInfo2UserInfo(userInfo)
    HAgentInfo		userInfo;
{
    HSocketInfo		sockInfo;
    HuiErrorMsgType	eMsgType;

    sockInfo = &(userInfo->sockInfo);

    if (huiSetupHostName2UserInfo(sockInfo) == HuiFalse) {
	eMsgType = HuiEMsg_SetupHostName2UserInfo;
	goto Error_Rtn;
    }

    sockInfo->sin_family = AF_INET;

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* $B%f!<%6>pJs$r%G!<%b%s$XAw$j!"(B
 * $B%f!<%6$N(B agentId $B$r%G!<%b%s$+$i<u$1<h$k!#(B
 */
static HuiBoolean huiSendUserInfo2Daemon(userInfo)
    HAgentInfo		userInfo;
{
    HeliosdMsgType	msgType;
    HuiText		msg;
    HuiWord		agentId;
    u_int		sin_port;
    HuiErrorMsgType	eMsgType;
    int			ret;

    if (huiSetupUserInfoMsg2Daemon(userInfo, msg) == HuiFalse) {
	eMsgType = HuiEMsg_SetupUserInfoMsg2Daemon;
	goto Error_Rtn;
    }

    /*	create socket to heliosd
     */
    if (hdCreateSocket() == HdFalse) {
	eMsgType = HuiEMsg_HdCreateSocketError;
	goto Error_Rtn;
    }

    if (hdSendMessage(HdMsg_SetupUserInfo, msg) == HdFalse) {
	eMsgType = HuiEMsg_HdSendMessageError;
	goto Close_Error_Rtn;
    }

    if (hdReceiveMessage(msg) == HdFalse) {
	eMsgType = HuiEMsg_HdReceiveMessageError;
	goto Close_Error_Rtn;
    }

    ret = sscanf(msg, "%d %s %u", &msgType, agentId, &sin_port);

    if (ret == 0 || ret == EOF || msgType != HdMsg_UserAgentId) {
	eMsgType = HuiEMsg_HdReceivedDataError;
	goto Close_Error_Rtn;
    }

    /*	close socket to heliosd
     */
    hdCloseSocket();

    /* $B%f!<%6!&%$%s%?%U%'!<%9$N(B agent ID $B$N@_Dj(B */
    if (huiSetupAgentID2UserInfo(userInfo, agentId) == HuiFalse) {
	eMsgType = HuiEMsg_SetupAgentID2UserInfo;
	goto Error_Rtn;
    }

    userInfo->sockInfo.sin_port = sin_port;

    return HuiTrue;

 Close_Error_Rtn:
    hdCloseSocket();

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Setup user info message to helios daemon.
 *
 *  component of user data;
 *	<agentId>:		# received data from heliosd
 *	<user login name>:	# strPtr[0]
 *	<host name>:		# strPtr[1]
 *	<sin_family>:		# strPtr[2]
 *	<sin_port>:		# received data from heliosd
 *	<pid>			# strPtr[3]
 */
static HuiBoolean huiSetupUserInfoMsg2Daemon(userInfo, msg)
    HAgentInfo		userInfo;
    char		*msg;
{
    HuiErrorMsgType	eMsgType;

    if (userInfo->agentName == NULL ||    /* userInfo->agentID == NULL ||  */
	userInfo->sockInfo.hostName == NULL) {
	eMsgType = HuiEMsg_NullPointer;
	goto Error_Rtn;
    }

    sprintf(msg, "%s:%s:%d:%d",   /* "%s:%s:%s:%d:%d:%d", */
	    /* userInfo->agentID, */
	    userInfo->agentName,
	    userInfo->sockInfo.hostName,
	    userInfo->sockInfo.sin_family,
	    /* userInfo->sockInfo.sin_port, */
	    userInfo->pid);

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* ==========================================================
 *	Functions to send the message of HdMsg_StartAppGui
 *	to Daemon
 * ========================================================== */
/*
 * $B%"%W%j%1!<%7%g%sBP1~$N(B GUI $B$r5/F0$9$k%a%C%;!<%8$r(B
 * $B%G!<%b%s$KDLCN$9$k4X?t(B
 */
HuiBoolean huiStartAppGui(agentName)
    char		*agentName;
{
    HuiErrorMsgType	eMsgType;

    /*	create socket to heliosd
     */
    if (hdCreateSocket() == HdFalse) {
	eMsgType = HuiEMsg_HdCreateSocketError;
	goto Error_Rtn;
    }

    if (hdSendMessage(HdMsg_StartAppGui, agentName) == HdFalse) {
	eMsgType = HuiEMsg_HdSendMessageError;
	goto Error_Rtn;
    }

    /*	close socket to heliosd
     */
    hdCloseSocket();

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* ==========================================================
 *	Functions to send the message of HdMsg_ResetAgent
 *	to Daemon
 * ========================================================== */
/*
 * connect $B$5$l$F$$$k%(!<%8%'%s%H$r%j%;%C%HMW5a$9$k%a%C%;!<%8$r(B
 * $B%G!<%b%s$KDLCN$9$k4X?t(B
 */
HuiBoolean huiResetAgent()
{
    HuiErrorMsgType	eMsgType;

    /*	create socket to heliosd
     */
    if (hdCreateSocket() == HdFalse) {
	eMsgType = HuiEMsg_HdCreateSocketError;
	goto Error_Rtn;
    }

    if (hdSendMessage(HdMsg_ResetAgent, "") == HdFalse) {
	eMsgType = HuiEMsg_HdSendMessageError;
	goto Error_Rtn;
    }

    /*	close socket to heliosd
     */
    hdCloseSocket();

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* ==========================================================
 *   Functions to send the message of HdMsg_GetParaAgent
 *   to Daemon
 * ========================================================== */
/*
 * Parametric agent $B$N@8@.$rMW5a$9$k%a%C%;!<%8$r(B
 * $B%G!<%b%s$KDLCN$9$k4X?t(B
 */
HuiBoolean huiGetParaAgent(hAccessInfo, parentInfo)
    HHuiAccessInfo	hAccessInfo;
    HAgentInfo		parentInfo;
{
    HSocketInfo		sockInfo;
    HuiErrorMsgType	eMsgType;

    /*	create socket to heliosd
     */
    if (hdCreateSocket() == HdFalse) {
	eMsgType = HuiEMsg_HdCreateSocketError;
	goto Error_Rtn;
    }

    if (hdSendMessage(HdMsg_GetParaAgent, parentInfo->agentId) == HdFalse) {
	eMsgType = HuiEMsg_HdSendMessageError;
	goto Close_Error_Rtn;
    }

    if (hdGetSockaddrFromDaemon(&sockInfo) == HdFalse) {
	eMsgType = HdEMsg_GetSockaddrFromDaemonError;
	goto Close_Error_Rtn;
    }

    /*	close socket to heliosd
     */
    hdCloseSocket();

    parentInfo->sockInfo.hostName = sockInfo->hostName;
    parentInfo->sockInfo.sin_family = sockInfo->sin_family;
    parentInfo->sockInfo.sin_port = sockInfo->sin_port;

    HuiFree(sockInfo);

    return HuiTrue;

 Close_Error_Rtn:
    hdCloseSocket();

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}
