/* huiCommon.c;
 *	Common module for helios user interface commands
 *		created on June 9th, 1994.
 *
 *	modification history:	
 *	(1) modified huiShowHelp() so as to add the command name to its argument
 *		on October 24th, 1994.
 *	(2) add huiGetAgentId() and huiGetMsgId()
 *		on October 24th, 1994.
 *	(3) add huiGetMsgStatus()
 *		on October 31st, 1994.
 *
 * $Id: huiCommon.c,v 2.1 1995/03/02 09:25:11 k3sato Exp $
 */

#include <stdio.h>
#include <signal.h>
#include <malloc.h>
#include <string.h>

#include "helios.h"
#include "huiAgentInfo.h"
#include "huiMessage.h"
#include "huiCommon.h"
#include "huiMemory.h"

/* defined in helios.c
 */
extern FILE	*HuiLogFd;

/* defined in huiCmdAnalysis.c
 */
extern int	HuiCmdLine;
extern char	*HuiUnknownCmdFrmt;

extern HuiCmdList tlCmdList[];
extern HuiCmdList alCmdList[];

static HHuiTraceSwitch huiGetTraceSwitch();
static HHuiTraceSwitch huiMakeTraceSwitch();
static HuiBoolean huiAppendTraceSwitch();

static HuiBoolean huiTextDelimitCheck();
static HuiBoolean huiStdinEofCheck();
static char huiUpperChar();

static void huiShowAllHelp();
static void huiShowhelpMsg();

int huiGetTuple();

/*
 * Get Agent Id from comman line
 */
HuiBoolean huiGetAgentId(hAccessInfo, agentId)
    HHuiAccessInfo	hAccessInfo;
    char		*agentId;
{
    int			len;
    char		c;
    HuiSwitchState	periodFlag = HuiOff;
    HuiSwitchState	endFlag = HuiOff;
    HuiErrorMsgType	eMsgType;

    huiSkipCrSpace();

    for (len = 0; endFlag == HuiOff;) {
	agentId[len++] =
	  c = huiGetChar();

	if (len >= HuiWordBufLen_Lim) {
	    eMsgType = HuiEMsg_TooLongWord;
	    goto Error_Rtn;
	}

	switch (c) {
	case HuiChar_Period:
	    if (periodFlag == HuiOff) {
		periodFlag = HuiOn;
		continue;
	    } else {
		eMsgType = HuiEMsg_IllegalAgentId;
		goto Error_Rtn;
	    }
	    break;

	case HuiChar_Space:
	    if (periodFlag == HuiOn) {
		eMsgType = HuiEMsg_IllegalAgentId;
		goto Error_Rtn;
	    } else {
		endFlag = HuiOn;
	    }
	    break;

	case HuiChar_Underscore:
	    if (periodFlag == HuiOn) {
		eMsgType = HuiEMsg_IllegalAgentId;
		goto Error_Rtn;
	    } else {
		huiUnGetChar(c);
		endFlag = HuiOn;
	    }
	    break;

	case HuiChar_CR:
	    if (periodFlag == HuiOn) {
		--len;
		huiUnGetChar(HuiChar_CR);
		huiUnGetChar(HuiChar_Period);
	    }
	    endFlag = HuiOn;
	    break;

	default:
	    if ('0' <= c && c <= '9') {
		periodFlag = HuiOff;
		continue;
	    }

	    eMsgType = HuiEMsg_IllegalAgentId;
	    goto Error_Rtn;
	}
    }

    if (len < 2) {
	eMsgType = HuiEMsg_NotAgentId;
	goto Error_Rtn;
    }

    agentId[--len] = '\0';

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * Get Message Id from comman line
 *
 *	<message Id> ::= <user Id>.<agent Id>_<sequence number>
 */
HuiBoolean huiGetMsgId(hAccessInfo, msgId)
    HHuiAccessInfo	hAccessInfo;
    char		*msgId;
{
    HuiWordType		wtype;
    int			len;
    char		c;
    HuiErrorMsgType	eMsgType;

    /* get the part of UserId in message Id
     */
    wtype = huiGetWord(msgId);
    if (wtype != HuiWT_String && wtype != HuiWT_OneChar) {
	eMsgType = HuiEMsg_NotMsgId;
	goto Error_Rtn;
    }

    /* check a period between user Id and agent Id
     */
    if (huiCmdEndCheck(NULL) == HuiFalse) {
	eMsgType = HuiEMsg_IllegalMsgId;
	goto Error_Rtn;
    }

    len = strlen(msgId);
    msgId[len++] = HuiChar_Period;

    /* get the part of AgentId in message Id
     */
    if (huiGetAgentId(hAccessInfo, &msgId[len]) == HuiFalse) {
	eMsgType = HuiEMsg_GetAgentIdError;
	goto Error_Rtn;
    }

    /* check underscore after agent Id in message Id 
     */
    len = strlen(msgId);

    if ((c = huiGetChar()) != HuiChar_Underscore) {
	eMsgType = HuiEMsg_IllegalMsgId;
	goto Error_Rtn;
    }

    msgId[len++] = c;

    /* get the sequence number of message in message Id
     */
    for (;;) {
	if (len >= HuiWordBufLen_Lim) {
	    eMsgType = HuiEMsg_TooLongWord;
	    goto Error_Rtn;
	}

	c = huiGetChar();
	if (c < '0' || '9' < c)
	  break;

	msgId[len++] = c;
    }

    msgId[len] = '\0';

    if (c != HuiChar_Space) {
	huiUnGetChar(c);
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * Get Message Status from comman line
 *
 *	<status> ::= <text>
 */
HuiBoolean huiGetMsgStatus(hAccessInfo, msgStatus)
    HHuiAccessInfo	hAccessInfo;
    char		*msgStatus;
{
    HuiWordType		wtype;
    HuiErrorMsgType	eMsgType;

    wtype = huiGetWord(msgStatus);
    if (wtype != HuiWT_String && wtype != HuiWT_OneChar) {
	eMsgType = HuiEMsg_NotMsgStatus;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * Setup message trace switch for the agent name
 */
HHuiTraceSwitch huiSetupMTraceSwitch(hAccessInfo, agentName, switchState)
    HHuiAccessInfo	hAccessInfo;
    char		*agentName;
    HuiSwitchState	switchState;
{
    HHuiTraceSwitch	traceSwitch;
    HuiErrorMsgType	eMsgType;

    if ((traceSwitch = huiGetTraceSwitch(hAccessInfo->traceInfo.traceSwitch, agentName))
	!= NULL) {
	if (traceSwitch->mTrace == switchState) {
	    eMsgType = HuiEMsg_AlreadySetMTraceSwitch;
	    goto Error_Rtn;
	}

	traceSwitch->mTrace = switchState;

    } else {
	if ((traceSwitch = huiMakeTraceSwitch(hAccessInfo, agentName)) == NULL) {
	    eMsgType = NULL;
	    goto Error_Rtn;
	}

	traceSwitch->mTrace = switchState;
    }

    return traceSwitch;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 * Setup agent trace switch for the agent name
 */
HHuiTraceSwitch huiSetupATraceSwitch(hAccessInfo, agentName, switchState)
    HHuiAccessInfo	hAccessInfo;
    char		*agentName;
    HuiSwitchState	switchState;
{
    HHuiTraceSwitch	traceSwitch;
    HuiErrorMsgType	eMsgType;

    if (agentName == NULL) {
	eMsgType = HuiEMsg_CharPointerIsNull;
	goto Error_Rtn;
    }

    if (*agentName == NULL){
	if (hAccessInfo->level == HuiAccess2AgentLevel) {
	    agentName = hAccessInfo->agentName;
	} else {
	    eMsgType = HuiEMsg_NotAgentName;
	    goto Error_Rtn;
	}
    }

    if ((traceSwitch = huiGetTraceSwitch(hAccessInfo->traceInfo.traceSwitch, agentName))
	!= NULL) {
	if (traceSwitch->aTrace == switchState) {
	    eMsgType = HuiEMsg_AlreadySetATraceSwitch;
	    goto Error_Rtn;
	}

	traceSwitch->aTrace = switchState;

    } else {
	if ((traceSwitch = huiMakeTraceSwitch(hAccessInfo, agentName)) == NULL) {
	    eMsgType = NULL;
	    goto Error_Rtn;
	}

	traceSwitch->aTrace = switchState;
    }

    return traceSwitch;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 * Get HuiTraceSwitch for a agentName
 */
static HHuiTraceSwitch huiGetTraceSwitch(traceSwitch, agentName)
    HHuiTraceSwitch	traceSwitch;
    char		*agentName;
{
    for (; traceSwitch != NULL
	 ; traceSwitch = traceSwitch->next) {
	if (strcmp(traceSwitch->agent->agentName, agentName) == 0) {
	    return traceSwitch;
	}
    }

    return NULL;
}

/*
 * Make HuiTraceSwitch's area for a agentName
 */
static HHuiTraceSwitch huiMakeTraceSwitch(hAccessInfo, agentName)
    HHuiAccessInfo	hAccessInfo;
    char		*agentName;
{
    HAgentInfo		hAgentInfo;
    HHuiTraceSwitch	traceSwitch;
    HuiErrorMsgType	eMsgType;

    if ((hAgentInfo = huiSearchAgentInfo(hAccessInfo->allAgentInfo, agentName))
	== NULL) {
	eMsgType = HuiEMsg_NotDefinedAgentName;
	goto Error_Rtn;
    }

    if ((traceSwitch = (HHuiTraceSwitch)HuiMalloc(sizeof(HuiTraceSwitch))) == NULL) {
	eMsgType = HuiEMsg_HuiMallocError;
	goto Error_Rtn;
    }

    traceSwitch->agent = hAgentInfo;

    if (huiAppendTraceSwitch(&hAccessInfo->traceInfo.traceSwitch, traceSwitch) == HuiFalse) {
	eMsgType = HuiEMsg_AppendTraceSwitchError;
	goto Error_Rtn;
    }

    return traceSwitch;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 * Append traceSwitch to all traceSwitches
 */
static HuiBoolean huiAppendTraceSwitch(allTraceSwitches, traceSwitch)
    HHuiTraceSwitch	*allTraceSwitches;
    HHuiTraceSwitch	traceSwitch;
{
    for (; *allTraceSwitches != NULL; allTraceSwitches = &((*allTraceSwitches)->next))
      ;

    *allTraceSwitches = traceSwitch;

    return HuiTrue;
}

/*
 *	Show Help for Command
 */
void huiShowHelp(hAccessInfo, cmdName)
    HHuiAccessInfo	hAccessInfo;
    char		*cmdName;
{
    HuiCmdList	*cmdList;
    int		length;

    cmdList = (hAccessInfo->level == HuiAccess2TopLevel)
      ? tlCmdList : alCmdList;

    if (cmdName == NULL || *cmdName == '\0') {
	huiShowAllHelp(cmdList);
    } else {
	length = strlen(cmdName);
	for (; cmdList != NULL; cmdList++) {
	    if (length == cmdList->len
		&& strcmp(cmdList->cmd, cmdName) == 0) {

		huiShowCmdUsage(cmdList);
		huiShowhelpMsg(cmdList);
		return;
	    }
	}
	huiPrintf(HuiUnknownCmdFrmt, cmdName);
    }
}

/*
 *	Show command usage for all command of current level
 */
static void huiShowAllHelp(cmdList)
    HuiCmdList	*cmdList;
{
    for (; cmdList->usage != NULL; cmdList++) {
	huiPrintf("\t%s.\n", cmdList->usage);
    }
}

/*
 *	Show Command Usage
 */
void huiShowCmdUsage(cmd)
    HuiCmdList	*cmd;
{
    huiPrintf("Usage:-> %s.\n", cmd->usage);
}

/*
 *	Show help message for one command
 */
static void huiShowhelpMsg(cmd)
    HuiCmdList	*cmd;
{
    huiPrintf("Description:\n\t%s.\n", cmd->helpMsg);
}

/*
 * Check if the command is terminated correctly.
 */
HuiBoolean huiCmdEndCheck(ch)
    int			ch;
{
    HuiWordType		wtype;
    HuiWord		wordBuf;
    HuiErrorMsgType	eMsgType;
    HuiBoolean		ret = HuiFalse;

    if (ch == NULL) {
        wtype = huiGetWord(wordBuf);
	if (wtype == HuiWT_Delimit
	    && wordBuf[0] == HuiChar_Period) ret = HuiTrue;
    } else if (ch == HuiChar_Period) {
	ret = HuiTrue;
    }

    if (ret == HuiTrue) {
	if ((ch = getc(stdin)) != HuiChar_CR) huiUnGetChar(ch);
    } else if (ch == NULL) {
	eMsgType = HuiEMsg_NotCmdTerminated;
	goto Error_Rtn;
    }

    return ret;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * Check if given or next character is comma.
 */
HuiBoolean huiCommaCheck(ch)
    int			ch;
{
    HuiWordType		wtype;
    HuiWord		wordBuf;
    HuiBoolean		ret = HuiFalse;

    if (ch == NULL) {
        wtype = huiGetWord(wordBuf);
	if (wtype == HuiWT_Delimit
	    && wordBuf[0] == HuiChar_Comma) ret = HuiTrue;
    } else if (ch == HuiChar_Comma) {
	ret = HuiTrue;
    }

    return ret;
}

/* =============================================================
 * Get Agent Declaration;
 *
 *  <agent declaration> := <agent name>( [ <argument list> ] )
 *
 *  <argument list> := <argument> [{ , <argument> }]
 *
 *  <argument> := <alpha-numeric> | "<string>"
 * ============================================================= */
int huiGetFuncDecl(text)
    char		*text;
{
    int			n;
    int			textLen;
    HuiWordType		wtype;
    HuiErrorMsgType	eMsgType;

    wtype = huiGetWord(text);
    if (wtype != HuiWT_String && wtype != HuiWT_OneChar) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    textLen = strlen(text);

    if ((n = huiGetTuple(text + textLen)) == 0) {
	eMsgType = HuiEMsg_NotTupleInDecl;
	goto Error_Rtn;
    }

    return textLen + n;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return 0;
}

/* =============================================================
 * Get Tuple with parenthesis;
 *
 *  ( <argument> [{ , <argument> }] )
 *
 *  <argument> := <alpha-numeric> | "<string>"
 * ============================================================= */
int huiGetTuple(text)
    char		*text;
{
    int			n;
    int			textLen;
    char		*currTextPtr;
    HuiWordType		wtype;
    HuiErrorMsgType	eMsgType;

    wtype = huiGetWord(text);
    if (wtype != HuiWT_Delimit || *text != HuiChar_LeftParen) {
	eMsgType = HuiEMsg_NotLeftParanOfTuple;
	goto Error_Rtn;
    }

    textLen = strlen(text);
    currTextPtr = text + textLen;

    for (;;) {
	if (textLen >= HuiTextBufLen_Lim) {
	    eMsgType = HuiEMsg_TextLenLimitOver;
	    goto Error_Rtn;
	}

	if ((n = huiGetText(currTextPtr)) == 0) {
	    wtype = huiGetWord(currTextPtr);
	    if (wtype != HuiWT_Delimit || *currTextPtr != HuiChar_RightParen) {
		eMsgType = HuiEMsg_NotRightParanOfTuple;
		goto Error_Rtn;
	    }
	    textLen += strlen(currTextPtr);
	    break;
	}

	textLen += n;
	currTextPtr += n;

	if (huiGetWord(currTextPtr) == HuiWT_Delimit) {

	    n = strlen(currTextPtr);
	    textLen += n;

	    if (*currTextPtr == HuiChar_Comma) {
		currTextPtr += n;
		continue;
	    }

	    if (*currTextPtr == HuiChar_RightParen) {
		break;
	    } else {
		eMsgType = HuiEMsg_StrangeState;
		goto Error_Rtn;
	    }
	} else {
	    eMsgType = HuiEMsg_StrangeState;
	    goto Error_Rtn;
	}
    }

    return textLen;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return 0;
}

/* ========================================
 * Get text; one statement
 *	of starting
 *		with double quotation
 *		     or single quotation
 *		     or not NULL char
 *	and of ending
 *		with double quotation
 *		  or single quotation
 *		  or comma or semicolon.
 * ======================================== */
int huiGetText(text)
    char		*text;
{
    int			ch;
    int			charNum;
    HuiSwitchState	sQuotFlag, dQuotFlag;
    HuiSwitchState	escFlag;
    HuiSwitchState	endFlag;
    
    huiSkipCrSpace();
    
    sQuotFlag = dQuotFlag = HuiOff;

    if ((ch = huiGetChar()) == HuiChar_DoubleQuot) {
	dQuotFlag = HuiOn;
	ch = huiGetChar();
    } else if (ch == HuiChar_SingleQuot) {
	sQuotFlag = HuiOn;
	ch = huiGetChar();
    }

    for (charNum = 0, escFlag = HuiOff, endFlag = HuiOff;;) {
	switch(ch) {
	case EOF:
	    clearerr(stdin);
	    return ch;
	    /*
	      endFlag = HuiOn;
	      */
	    break;
	case HuiChar_EscCode:
	    text[charNum++] = ch;
	    escFlag = (escFlag == HuiOn) ? HuiOff : HuiOn;
	    break;
	case HuiChar_SingleQuot:
	    if (escFlag == HuiOn) {
		escFlag = HuiOff;
		text[charNum++] = ch;
	    } else if (sQuotFlag == HuiOn) {
		ch = huiGetChar();
		if (huiDelimitCheck(ch) == HuiFalse) {
		    huiPutsErrorMsg(HuiEMsg_NoDelimit);
		    charNum = 0;
		} else {
		    huiUnGetChar(ch);
		}
	    } else {
		huiPutsErrorMsg(HuiEMsg_SingleQuot);
		charNum = 0;
	    }
	    endFlag = HuiOn;
	    break;
	case HuiChar_DoubleQuot:
	    if (escFlag == HuiOn) {
		escFlag = HuiOff;
		text[charNum++] = ch;
	    } else if (dQuotFlag == HuiOn) {
		ch = huiGetChar();
		if (huiDelimitCheck(ch) == HuiFalse) {
		    huiPutsErrorMsg(HuiEMsg_NoDelimit);
		    charNum = 0;
		} else {
		    huiUnGetChar(ch);
		}
	    } else {
		huiPutsErrorMsg(HuiEMsg_DoubleQuot);
		charNum = 0;
	    }
	    endFlag = HuiOn;
	    break;
	default:
	    if (sQuotFlag == HuiOn || dQuotFlag == HuiOn ||
		escFlag == HuiOn ||
		huiTextDelimitCheck(ch) == HuiFalse) {
		text[charNum++] = ch;
		escFlag = HuiOff;
	    } else {
		huiUnGetChar(ch);
		endFlag = HuiOn;
	    }
	    break;
	}
	if (endFlag == HuiOn) break;
	ch = huiGetChar();
    }
    
    text[charNum] = '\0';
    return charNum;
}

/*
 * Check if the given character is a text delimiter.
 */
static HuiBoolean huiTextDelimitCheck(ch)
    char	ch;
{
    if (ch == EOF || ch == HuiChar_Space ||
	ch == HuiChar_LeftParen || ch == HuiChar_RightParen ||
	ch == HuiChar_Comma || ch == HuiChar_Semicolon) {
	return HuiTrue;
    } else {
	return HuiFalse;
    }
}

/* ==================================
 * Get one word and also
 *	get a delimiter as a word.
 * ================================== */
HuiWordType huiGetWord(str)
    char	*str;
{
    int		len;
    char	c;
    HuiWordType	wtype;


    huiSkipCrSpace();

    for (len = 0, wtype = HuiWT_String;;) {
	c = huiGetChar();
	if (huiDelimitCheck(c) == HuiTrue)
	  break;
	else
	  str[len++] = c;

	if (len >= HuiWordBufLen_Lim) {
	    wtype = HuiWT_TooLongWordError;
	    goto Error_Rtn;
	}
    }

    if (len == 0) {
	if (c == HuiChar_CR ||
	    c == HuiChar_Comma ||
	    c == HuiChar_Period ||
	    c == HuiChar_Colon ||
	    c == HuiChar_Semicolon ||
	    c == HuiChar_EqualCode ||
	    c == HuiChar_LeftParen ||
	    c == HuiChar_RightParen) {
	    str[len++] = c;
	    wtype = HuiWT_Delimit;
	} else if (c == EOF) {
	    wtype = HuiWT_EOF;
	} else {
	    wtype = HuiWT_NullString;
	}
    } else {
	if (c != HuiChar_Space) huiUnGetChar(c);
    }

    if (len == 1 && wtype != HuiWT_Delimit) wtype = HuiWT_OneChar;

 Error_Rtn:

    str[len] = '\0';
    return wtype;
}

/*
 * UnGet one word into standard input buffer.
 */
void huiUnGetWord(str)
    char	*str;
{
    int	len = 0;

    if ((len = strlen(str)) > 0) {
	huiUnGetChar(HuiChar_Space);
	for (len-- ; len >= 0; len--) {
	    huiUnGetChar(str[len]);
	}
    }

    return;
}

/*
 * skip space in standard input
 */
void huiSkipCrSpace()
{
    int	c;
    
    while ((c = huiGetChar()) == HuiChar_Space || c == HuiChar_CR);
    
    huiUnGetChar(c);
}

/*
 * get one character from standard input
 */
int huiGetChar()
{
    int	c;
    
    if ((c = getc(stdin)) == EOF) {
	clearerr(stdin);
    }

    if (c == HuiChar_CR) {
	huiPrintf("%3u : ", ++HuiCmdLine);
	fflush(stdout);
    }

    return c;
}

/*
 * unget one character into standard input
 */
void huiUnGetChar(ch)
    int	ch;
{
    ungetc(ch, stdin);
}

/*
 * skip until the EOF of input buffer is got,
 */
void huiSkipToNextStmt()
{
    while (huiStdinEofCheck() == HuiFalse) {
	getc(stdin);
    }
    return;
}

/*
 * skip until the one character(not SPACE or not CR) is got
 *	in input buffer.
 */
void huiSkipToNextChar()
{
    int	ch;

    while (huiStdinEofCheck() == HuiFalse) {
	if ((ch = getc(stdin)) != HuiChar_Space && ch != HuiChar_CR) {
	    huiUnGetChar(ch);
	    break;
	}
    }
    return;
}

/*
 * check the end of stdin buffer
 */
static HuiBoolean huiStdinEofCheck()
{
    return (stdin->_cnt <= 0)
      ? HuiTrue : HuiFalse;
}

/*
 * Check if the given character is a delimiter.
 */
HuiBoolean huiDelimitCheck(ch)
    char	ch;
{
    return (ch == EOF ||
	    ch == HuiChar_Space ||
	    ch == HuiChar_CR ||
	    ch == HuiChar_Tab ||
	    ch == HuiChar_EqualCode ||
	    ch == HuiChar_LeftParen ||
	    ch == HuiChar_RightParen ||
	    ch == HuiChar_Comma ||
	    ch == HuiChar_Period ||
	    ch == HuiChar_Colon ||
	    ch == HuiChar_Semicolon)
      ? HuiTrue : HuiFalse;
}

/*
 * Compare two strings
 *	without caring upper or lower case.
 */
HuiBoolean huiStrCmp(target, srch_str)
    char	*target;
    char	*srch_str;
{
    int		srch_len, diff_len;
    int		cnt;
    
    srch_len = strlen(srch_str);
    diff_len = strlen(target) - srch_len;
    
    if (diff_len != 0) return HuiFalse;
    
    for (cnt = 0; cnt < srch_len; cnt++) {
	if (huiUpperChar(target[cnt]) != huiUpperChar(srch_str[cnt])) {
	    break;
	}
    }
    if (cnt >= srch_len) return HuiTrue;
    
    return HuiFalse;
}

/*
 * transform one character
 *	into one upper case character.
 */
static char huiUpperChar(c)
    char	c;
{
    if ((c < 'a') || (c > 'z')) return(c);
    
    /* case c in ('a', ..., 'z') */
    return(c - 'a' + 'A');
}

/*
 *	file handle function
 */
void HuiFdClose(fd)
    FILE	*fd;
{
    if (fd != NULL) {
	close(fd);
    }
}

/*
 *  Quit Monitor process
 */
HuiBoolean huiQuitMonitor(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiMonitorInfo	*mInfo;
    HuiErrorMsgType	eMsgType;

    mInfo = &(hAccessInfo->monitor);
    if (mInfo->state == HuiOn) {

	if (kill(mInfo->pid, SIGQUIT) != -1) {
	    wait(NULL);
	    huiPrintf("killed monitor process.\n");
	} else {
	    HuiPerror();
	    eMsgType = HuiEMsg_NotRunningMonitorProcess;
	    goto Error_Rtn;
	}

	close(mInfo->pipeIn);
	close(mInfo->pipeOut);

	memset(mInfo, 0, sizeof(HuiMonitorInfo));
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

void huiPutsLine(length, ch)
    int		length;
    char	ch;
{
    char	*bufPtr;
    HuiWord	buf;

    bufPtr = buf;

    if (length >= HuiWordBufLen_Lim) {
	length = HuiWordBufLen_Lim - 1;
    }

    if (ch == NULL) {
	ch = HuiChar_Minus;
    }

    while (length-- > 0) {
	*bufPtr++ = ch;
    }

    *bufPtr = HuiChar_NULL;

    huiPrintf("%s\n", buf);
}


char *huiNextBufPtr(bufPtr, num)
    char	*bufPtr;
    int		num;
{
    while (num-- > 0 && *bufPtr++ != HuiChar_NULL)
      ;

    return bufPtr;
}
