/* huiALevel.c;
 *	Helios User Interface;
 *		Agent Level command process module
 *			created on June 9th, 1994.
 *
 *	modification history:	
 *	(1) add "wake" command
 *		on September 16th in '94.
 *	(2) add "monitor" command
 *		on September 22nd in '94.
 *	(3) add "sleep" and "reset" command
 *		on October 11th in '94.
 *	(4) add "reply" command
 *		on October 25th in '94.
 *	(5) add "test_trace" command
 *		on November 29th in '94.
 *
 * $Id: huiALevel.c,v 2.0 1995/03/02 09:24:49 k3sato Exp $
 */
#include <stdio.h>

#include <helios/hDaemonLib.h>
#include "helios.h"
#include "huiALevel.h"
#include "huiAgentInfo.h"
#include "huiMessage.h"
#include "huiMsgManage.h"
#include "huiCommon.h"
#include "huiKeyWord.h"
#include "huiMemory.h"

/* defined in agent.c
 */
extern void invalidate_agent();

/* defined in huiDAccess.c
 */
extern HuiBoolean huiResetAgent();

/* defined in huiAAccess.c
 */
extern HuiBoolean huiMsgSend2Agent();
extern HuiBoolean huiMsgReply2Agent();
extern HuiBoolean huiPutsWarningMsg();
extern HuiBoolean huiSendAgentTraceSwitch();

/* defined in huiMAccess.c
 */
extern HuiBoolean huiSendMonMsgType2Monitor();
extern HuiBoolean huiReceiveMonMsgTypeFromMonitor();

/* defined in huiTestTrace.c
 */
extern HuiBoolean huiTestHandleDbgInfoFromAgent();


static HuiBoolean huiAlHelpSetup();
static HuiBoolean huiAlLogSetup();
static HuiBoolean huiAlSendSetup();
static HuiBoolean huiAlReceiveSetup();
static HuiBoolean huiAlReplySetup();
static char *huiMsgStatus2Str();

static HuiBoolean huiAlMTraceSetup();
static HuiBoolean huiAlATraceSetup();

static HuiBoolean huiAlWakeSetup();
static HuiBoolean huiAlTestTraceSetup();

/* ==========================================================
 *	$B%(!<%8%'%s%H!&%l%Y%k$N%3%^%s%I!&%X%k%W>pJs$NI=<((B
 * ========================================================== */
HuiBoolean huiAlHelp(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiWord		cmdName;

    if (huiAlHelpSetup(hAccessInfo, cmdName) == HuiFalse) {
	goto Error_Rtn;
    }

    huiShowHelp(hAccessInfo, cmdName);

    return HuiTrue;

 Error_Rtn:
    return HuiFalse;
}

/*
 * command name $B$N3MF@$H!"%3%^%s%I9T=*N;H=Dj(B
 */
static HuiBoolean huiAlHelpSetup(hAccessInfo, cmdName)
    HHuiAccessInfo	hAccessInfo;
    char		*cmdName;
{
    HuiWordType		wtype;
    HuiErrorMsgType	eMsgType;

    wtype = huiGetWord(cmdName);
    if (wtype == HuiWT_String || wtype == HuiWT_OneChar) {
	if (huiCmdEndCheck(NULL) == HuiFalse) {
	    eMsgType = HuiEMsg_NotCmdTerminated;
	    goto Error_Rtn;
	}
    } else {
	if (huiCmdEndCheck(*cmdName) == HuiFalse) {
	    eMsgType = HuiEMsg_NotCmdTerminated;
	    goto Error_Rtn;
	}
	*cmdName = '\0';
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    huiShowCmdUsage(hAccessInfo->currCmd);
    return HuiFalse;
}

/* =========================================
 *	$B%(!<%8%'%s%H$H$N@\B3$r=*N;$7$F!"(B
 *	$B%H%C%W!&%l%Y%k$XLa$k!#(B
 * ========================================= */
HuiBoolean huiAlTop(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiErrorMsgType	eMsgType;

    if (huiCmdEndCheck(NULL) == HuiFalse) {
	huiShowCmdUsage(hAccessInfo->currCmd);
	goto Error_Rtn;
    }

    if (huiPutsWarningMsg(HuiWMsg_Confirm2QuitAgnet) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    HuiFreeString(&(hAccessInfo->agentName));
    HuiFreeString(&(hAccessInfo->sockAddr));

    if (huiQuitMonitor(hAccessInfo) == HuiFalse) {
	eMsgType = HuiEMsg_QuitMonitorError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* =========================================
 *	$B%a%C%;!<%8$r%(!<%8%'%s%H$XAwIU(B
 *
 *	S-Message $B$X$N9=@.$O!"(B
 *		huiAAccess.c
 *	$B$G9T$&!#(B
 * ========================================= */
HuiBoolean huiAlSend(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HHuiMsgManageInfo	msgManageInfo;
    HuiErrorMsgType	eMsgType;

    if (huiAlSendSetup(hAccessInfo, &msgManageInfo) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    msgManageInfo->status = HuiMsgStatus_Hold;

    if (huiMsgSend2Agent(hAccessInfo, msgManageInfo) == HuiFalse) {
	eMsgType = HuiEMsg_HuiMsgSend2AgentError;
	goto Error_Rtn;
    }

    msgManageInfo->status = HuiMsgStatus_Sent;

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * message text $B$N3MF@$H!"%3%^%s%I9T=*N;H=Dj(B
 */
static HuiBoolean huiAlSendSetup(hAccessInfo, hMsgManageInfo)
    HHuiAccessInfo	hAccessInfo;
    HHuiMsgManageInfo	*hMsgManageInfo;
{
    HuiSwitchState	usageFlag = HuiOn;
    HuiErrorMsgType	eMsgType;

    if (huiGetHMessage(hAccessInfo, hMsgManageInfo) == HuiFalse) {
	eMsgType = HuiEMsg_HuiGetHMessageError;
	goto Error_Rtn;
    }

    if (huiAppendMsgManageInfo(hAccessInfo, *hMsgManageInfo) == HuiFalse) {
	eMsgType = HuiEMsg_HuiAppendMsgManageInfoError;
	goto Error_Rtn_NoUsage;
    }

    if (huiShowMsgId(hAccessInfo, *hMsgManageInfo) == HuiFalse) {
	eMsgType = HuiEMsg_HuiShowMsgIdError;
	goto Error_Rtn_NoUsage;
    }

    return HuiTrue;

 Error_Rtn_NoUsage:
    usageFlag = HuiOff;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    if (usageFlag == HuiOn) {
	huiShowCmdUsage(hAccessInfo->currCmd);
    }
    return HuiFalse;
}

/* =========================================
 *	$B%(!<%8%'%s%H$+$iAw$i$l$F$-$?(B
 *	$B%a%C%;!<%8$rI=<($9$k(B
 * ========================================= */
HuiBoolean huiAlReceive(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiWord		msgID;
    HuiSwitchState	usageFlag = HuiOn;
    HuiErrorMsgType	eMsgType;

    if (huiAlReceiveSetup(hAccessInfo, msgID) == HuiFalse) {
	eMsgType = HuiEMsg_HuiAlReceiveSetupError;
	goto Error_Rtn;
    }

    if (huiShowReceivedMsg(hAccessInfo, msgID) == HuiFalse) {
	eMsgType = HuiEMsg_ShowReceivedMsgError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    if (usageFlag == HuiTrue)
      huiShowCmdUsage(hAccessInfo->currCmd);
    return HuiFalse;
}

/*
 * Message ID  $B$N3MF@$H!"%3%^%s%I9T=*N;H=Dj(B
 */
static HuiBoolean huiAlReceiveSetup(hAccessInfo, msgId)
    HHuiAccessInfo	hAccessInfo;
    char		*msgId;
{
    HuiWordType		wtype;
    int			len;
    char		c;
    HuiSwitchState	usageFlag = HuiOn;
    HuiErrorMsgType	eMsgType;

    /* get the part of UserId in message Id
     */
    wtype = huiGetWord(msgId);
    if (wtype == HuiWT_Delimit) {
	if (huiCmdEndCheck(msgId[0]) == HuiFalse) {
	    eMsgType = HuiEMsg_NotCmdTerminated;
	    goto Error_Rtn;
	} else {
	    msgId[0] = '\0';
	    goto Good_Rtn;
	}
    }

    if (huiJudgeKeyWord(HuiAllKeyWordList, msgId) == HuiTrue) {
	goto CheckEnd_Rtn;
    }

    /* check a period between user Id and agent Id
     */
    if (huiCmdEndCheck(NULL) == HuiFalse) {
	eMsgType = HuiEMsg_IllegalMsgId;
	goto Error_Rtn;
    }

    len = strlen(msgId);
    msgId[len++] = HuiChar_Period;

    /* get the part of AgentId in message Id
     */
    if (huiGetAgentId(hAccessInfo, &msgId[len]) == HuiFalse) {
	eMsgType = HuiEMsg_GetAgentIdError;
	goto Error_Rtn;
    }

    /* check underscore after agent Id in message Id 
     */
    len = strlen(msgId);

    if ((c = huiGetChar()) != HuiChar_Underscore) {
	eMsgType = HuiEMsg_IllegalMsgId;
	goto Error_Rtn;
    }

    msgId[len++] = c;

    /* get the sequence number of message in message Id
     */
    for (;;) {
	if (len >= HuiWordBufLen_Lim) {
	    eMsgType = HuiEMsg_TooLongWord;
	    goto Error_Rtn;
	}

	c = huiGetChar();
	if (c < '0' || '9' < c)
	  break;

	msgId[len++] = c;
    }

    msgId[len] = '\0';

    if (c != HuiChar_Space) {
	huiUnGetChar(c);
    }

 CheckEnd_Rtn:
    if (huiCmdEndCheck(NULL) == HuiFalse) {
	eMsgType = HuiEMsg_NotCmdTerminated;
	goto Error_Rtn;
    }

 Good_Rtn:
    return HuiTrue;

/* Error_Rtn_NoUsage:
    usageFlag = HuiOff;
*/

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    if (usageFlag == HuiOn) {
	huiShowCmdUsage(hAccessInfo->currCmd);
    }
    return HuiFalse;
}

/* =========================================
 *	$B%(!<%8%'%s%H$+$iAw$i$l$F$-$?(B
 *	ASK $B%a%C%;!<%8$KBP$9$k2sEz$rAw$k(B
 * ========================================= */
HuiBoolean huiAlReply(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HHuiMsgManageInfo	msgManageInfo;
    HuiErrorMsgType	eMsgType;

    if (huiAlReplySetup(hAccessInfo, &msgManageInfo) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    if (huiMsgReply2Agent(hAccessInfo, msgManageInfo) == HuiFalse) {
	eMsgType = HuiEMsg_HuiMsgReply2AgentError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * Message ID $B!"(BMessage Status $B5Z$S(B $B!c2sEz!d(B $B$N3MF@$H!"(B
 * $B%3%^%s%I9T=*N;H=Dj(B
 */
static HuiBoolean huiAlReplySetup(hAccessInfo, hMsgManageInfo)
    HHuiAccessInfo	hAccessInfo;
    HHuiMsgManageInfo	*hMsgManageInfo;
{
    HuiSwitchState	usageFlag = HuiOn;
    HuiErrorMsgType	eMsgType;

    if (huiGetReplyHMsg(hAccessInfo, hMsgManageInfo) == HuiFalse) {
	eMsgType = HuiEMsg_HuiGetHMessageError;
	goto Error_Rtn;
    }

    if (huiShowMsgId(hAccessInfo, *hMsgManageInfo) == HuiFalse) {
	eMsgType = HuiEMsg_HuiShowMsgIdError;
	goto Error_Rtn_NoUsage;
    }

    return HuiTrue;

 Error_Rtn_NoUsage:
    usageFlag = HuiOff;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    if (usageFlag == HuiOn) {
	huiShowCmdUsage(hAccessInfo->currCmd);
    }
    return HuiFalse;
}

/* ====================================================
 *	$B%(!<%8%'%s%H$XAwIU$7$?%a%C%;!<%8$G!"(B
 *	$B$^$@2sEz$r2hLL$K=PNO$7$F$$$J$$%a%C%;!<%8$r(B
 *	$B2hLL$KI=<($9$k(B
 * ==================================================== */
HuiBoolean huiAlShowMsg(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HHuiMsgManageInfo	currInfo;
    HuiMsgManageStatus	status;
    HuiSwitchState	usageFlag = HuiOn;
    HuiErrorMsgType	eMsgType;

    if (huiCmdEndCheck(NULL) == HuiFalse) {
	goto Error_Rtn;
    }

    huiPrintf(" status\t MsgID\t\t To\t message\t\t Reply Status\n");
    huiPutsLine(50, HuiChar_EqualCode);

    for (currInfo = hAccessInfo->msgManageInfo
	 ; currInfo != NULL
	 ; currInfo = currInfo->next) {
	if ((status = currInfo->status) != HuiMsgStatus_Empty
	    && status != HuiMsgStatus_Filed) {
	    huiPrintf("%s\t %s\t %s\t %s\t %s\n",
		      huiMsgStatus2Str(status),
		      currInfo->msgID,
		      (currInfo->toAid == NULL)
		      ? currInfo->funcName : currInfo->toAid,
		      currInfo->method,
		      currInfo->msgStatus);
	}
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    if (usageFlag == HuiTrue)
      huiShowCmdUsage(hAccessInfo->currCmd);
    return HuiFalse;
}

typedef struct _HuiMsgManageStatusList {
    HuiMsgManageStatus	status;
    char		*str;
} HuiMsgManageStatusList;

static HuiMsgManageStatusList msgManageStatusList[] = {
    {HuiMsgStatus_Hold,		"Hold"},
    {HuiMsgStatus_Sent,		"Sent"},
    {HuiMsgStatus_Waiting,	"Waiting"},
    {HuiMsgStatus_Replied,	"Replied"},
    {HuiMsgStatus_Shown,	"Shown"},
    {HuiMsgStatus_Filed,	"Filed"},
    {HuiMsgStatus_Asked,	"Asked"},
    {HuiMsgStatus_AskedReplied,	"AskedReplied"},
    {NULL,			NULL},
};

static char *huiMsgStatus2Str(msgStatus)
    HuiMsgManageStatus	msgStatus;
{
    HuiMsgManageStatusList	*currList;

    for (currList = msgManageStatusList
	 ; currList->str != NULL
	 ; currList++) {
	if (currList->status == msgStatus) {
	    return currList->str;
	}
    }
    return "";
}

/* ======================================
 *	$B%m%0!&%U%!%$%kL>$N;XDj(B
 *	$B5Z$S!"%m%0!&%U%!%$%k$N%*!<%W%s(B
 * ====================================== */
HuiBoolean huiAlLog(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    FILE		*fd;
    HuiErrorMsgType	eMsgType;

    if (huiAlLogSetup(hAccessInfo) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    if ((fd = fopen(hAccessInfo->logFileName, "w+")) == NULL) {
	eMsgType = HuiEMsg_Fail2OpenFile2Write;
	goto Error_Rtn;
    }

    HuiFdClose(hAccessInfo->logFd);
    hAccessInfo->logFd = fd;

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * log file name $B$N3MF@$H!"%3%^%s%I9T=*N;H=Dj(B
 */
static HuiBoolean huiAlLogSetup(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiWordType		wtype;
    HuiWord		wordBuf;
    HuiSwitchState	usageFlag = HuiOn;
    HuiErrorMsgType	eMsgType;

    wtype = huiGetWord(wordBuf);
    if (wtype == HuiWT_String || wtype == HuiWT_OneChar) {
	if (huiCmdEndCheck(NULL) == HuiFalse) {
	    eMsgType = HuiEMsg_NotCmdTerminated;
	    goto Error_Rtn;
	}
    } else {
	eMsgType = HuiEMsg_NotLogFileName;
	goto Error_Rtn;
    }

    if (huiPutsWarningMsg(HuiWMsg_Confirm2ChangeLog) == HuiTrue) {
	HuiFreeString(&(hAccessInfo->logFileName));
	if ((hAccessInfo->logFileName = HuiMallocString(wordBuf)) == NULL) {
	    eMsgType = HuiEMsg_HuiMallocStringError;
	    goto Error_Rtn_NoUsage;
	}
    } else {
	return HuiFalse;
    }

    return HuiTrue;

 Error_Rtn_NoUsage:
    usageFlag = HuiOff;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    if (usageFlag == HuiOn) {
	huiShowCmdUsage(hAccessInfo->currCmd);
    }
    return HuiFalse;
}

/* ===============================================
 *	$B%(!<%8%'%s%H!&%l%Y%k$+$i$N(B
 *	$B%X%j%*%9!&%f!<%6%$%s%?%U%'!<%9$N=*N;(B
 * =============================================== */
HuiBoolean huiAlBye(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiErrorMsgType	eMsgType;

    if (huiCmdEndCheck(NULL) == HuiFalse) {
	huiShowCmdUsage(hAccessInfo);
	goto Error_Rtn;
    }

    if (huiQuitMonitor(hAccessInfo) == HuiFalse) {
	eMsgType = HuiEMsg_QuitMonitorError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* ===============================================
 *  Show function name of connectd agent
 * =============================================== */
HuiBoolean huiAlShowFunctions(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{

    huiPutsFuncDir(hAccessInfo->connectedAgent->funcDir);

    return HuiTrue;
}

/* ===============================================
 *  Show method of connected agent
 * =============================================== */
HuiBoolean huiAlShowMethods(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{

    huiPutsMethodDir(hAccessInfo->connectedAgent->methodDir);

    return HuiTrue;
}

/* ===============================================
 *
 * =============================================== */
/*
HuiBoolean huiAlShowStructure(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{

    return HuiTrue;

 Error_Rtn:
    return HuiFalse;
}
*/

/* ===============================================
 *
 * =============================================== */
/*
HuiBoolean huiAlShowLocation(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{

    return HuiTrue;

 Error_Rtn:
    return HuiFalse;
}
*/

/* ===============================================
 *
 * =============================================== */
/*
HuiBoolean huiAlShowStatus(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{

    return HuiTrue;

 Error_Rtn:
    return HuiFalse;
}
*/

/* ======================================
 *	$B%a%C%;!<%8!&%H%l!<%9(B
 * ====================================== */
HuiBoolean huiAlMTrace(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiErrorMsgType	eMsgType;

    if (huiAlMTraceSetup(hAccessInfo) == HuiFalse) {
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * ON/OFF $B%U%i%0$N3MF@$H!"%3%^%s%I9T=*N;H=Dj(B
 */
static HuiBoolean huiAlMTraceSetup(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiKeyWordType 	kwType;
    HuiErrorMsgType	eMsgType;


    kwType = huiGetKeyWord(HuiOnOffKeyWordList);

    switch (kwType) {
    case HuiKW_On:
    case HuiKW_Off:
	if (huiCmdEndCheck(NULL) == HuiFalse) {
	    eMsgType = HuiEMsg_NotCmdTerminated;
	    goto Error_Rtn;
	}

	hAccessInfo->traceInfo.allMsgTrace = (HuiKW_On) ? HuiOn : HuiOff;
	break;
    case HuiKW_NotUsed:
	eMsgType = HuiEMsg_NotOnOffSwitch;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    huiShowCmdUsage(hAccessInfo->currCmd);
    return HuiFalse;
}

/* ======================================
 *	$B%(!<%8%'%s%H!&%H%l!<%9(B
 * ====================================== */
HuiBoolean huiAlATrace(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiWord		wordBuf;
    HuiSwitchState	switchState;
    HHuiTraceSwitch	traceSwitch;
    HuiErrorMsgType	eMsgType;

    if (huiAlATraceSetup(hAccessInfo, wordBuf, &switchState) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    /* check agent trace switch for the agent name
     */
    if ((traceSwitch = huiSetupATraceSwitch(hAccessInfo, wordBuf, switchState))
	== NULL) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    /* send control message to the agent
     */
    if (huiSendAgentTraceSwitch(hAccessInfo, traceSwitch) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * agent name$B!"5Z$S(B ON/OFF $B%U%i%0$N3MF@$H!"(B
 * $B%3%^%s%I9T=*N;H=Dj(B
 */
static HuiBoolean huiAlATraceSetup(hAccessInfo, agentName, switchState)
    HHuiAccessInfo	hAccessInfo;
    char		*agentName;
    HuiSwitchState	*switchState;
{
    HuiWordType		wtype;
    HuiWord		wordBuf;
    HuiKeyWordType 	kwType;
    HuiErrorMsgType	eMsgType;

    wtype = huiGetWord(agentName);
    if (wtype != HuiWT_String && wtype != HuiWT_OneChar) {
	eMsgType = HuiEMsg_NotOnOffSwitch;
	goto Error_Rtn;
    }

    wtype = huiGetWord(wordBuf);
    if (wtype != HuiWT_Delimit) {
	eMsgType = HuiEMsg_NoDelimit;
	goto Error_Rtn;
    }

    if (huiCommaCheck(wordBuf[0]) == HuiFalse) {
	if (huiCmdEndCheck(wordBuf[0]) == HuiFalse) {
	    eMsgType = HuiEMsg_NotCmdTerminated;
	    goto Error_Rtn;
	}

	kwType = huiCheckKeyWord(HuiOnOffKeyWordList, agentName);
	agentName[0] = '\0';
    } else {
	if ((kwType = huiGetKeyWord(HuiOnOffKeyWordList)) != HuiKW_NotUsed) {
	    if (huiCmdEndCheck(NULL) == HuiFalse) {
		eMsgType = HuiEMsg_NotCmdTerminated;
		goto Error_Rtn;
	    }
	}
    }

    switch (kwType) {
    case HuiKW_On:
    case HuiKW_Off:
	*switchState = (kwType == HuiKW_On) ? HuiOn : HuiOff;
	break;
    default:
	eMsgType = HuiEMsg_NotOnOffSwitch;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    huiShowCmdUsage(hAccessInfo->currCmd);
    return HuiFalse;
}

/* ======================================
 *	exec Monitor process
 * ====================================== */
HuiBoolean huiAlMonitor(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    int			pid;
    int			pipeIn[2];
    int			pipeOut[2];
    HuiMonMsgType	monMsgType;
    HuiErrorMsgType	eMsgType;


    if (huiCmdEndCheck(NULL) == HuiFalse) {
	huiShowCmdUsage(hAccessInfo->currCmd);
	eMsgType = NULL;
	goto Error_Rtn;
    }

    if (hAccessInfo->monitor.state == HuiOn) {
	eMsgType = HuiEMsg_AlreadyRunMonitorProcess;
	goto Error_Rtn;
    }

    if (pipe(pipeIn) == -1) {
	HuiPerror();
	eMsgType = NULL;
	goto Error_Rtn;
    }

    if (pipe(pipeOut) == -1) {
	HuiPerror();
	eMsgType = NULL;
	goto Error_Rtn;
    }

    /* $B;R%W%m%;%9!J%b%K%?!<!&%W%m%;%9!K$N5/F0(B
     */
    if ((pid = fork()) == 0) {	/* $B$3$3$+$i;R%W%m%;%9(B */
	char	*argv[4];

	/* $B?F$N(B read $BB&$N(B pipe $B$r(B close
	 */
	close(pipeIn[0]);

	/* $B?F$N(B write $BB&$N(B pipe $B$r(B close
	 */
	close(pipeOut[1]);

	/* $B?F$+$i$N(B read $BB&$N(B pipe $B$rI8=`F~NO$K%;%C%H(B
	 */
	close(0);
	dup(pipeOut[0]);
	close(pipeOut[0]);

	/* $B?F$X$N(B write $BB&$N(B pipe $B$rI8=`=PNO$K%;%C%H(B
	 */
	close(1);
	dup(pipeIn[1]);
	close(pipeIn[1]);

	argv[0] = "huiMonitor";
	argv[1] = hAccessInfo->agentName;
	argv[2] = NULL;
	argv[3] = NULL;

	execv(argv[0], argv);

	exit(0);

	/* $B;R%W%m%;%9=*$j(B
	 */
    } else if (pid == ERROR_FLAG) {
	eMsgType = HuiEMsg_CreateMonitorProcessError;
	HuiPerror();
	goto Error_Rtn;
    }

    /*
     *	sleep time for creating the above subprocess
     *	and for preparing socket in the subprocess.
     */
    sleep(HUIMONITOR_SLEEP_SPAN);

    /* $B;R$+$i?F$X$N=PNO(B pipe $B$N?FB&$N(B write fd $B$r(B close
     */
    close(pipeIn[1]);

    /* $B?F$+$i;R$X$N=PNO(B pipe $B$N;RB&$N(B read fd $B$r(B close
     */
    close(pipeOut[0]);

    /* $B;R%W%m%;%9$X$N=PNOB&$N(B pipe $B$r3NJ](B
     */
    hAccessInfo->monitor.pipeIn = pipeIn[0];
    hAccessInfo->monitor.pipeOut = pipeOut[1];

    /* $B;R%W%m%;%9$N(B process ID $B$r3NJ](B
     */
    hAccessInfo->monitor.pid = pid;
    hAccessInfo->monitor.state = HuiOn;
    hAccessInfo->traceInfo.allMsgTrace = HuiOn;

    /* wait for Monitor standby message from monitor process
     */
    if (huiReceiveMonMsgTypeFromMonitor(hAccessInfo, &monMsgType) == HuiFalse) {
	eMsgType = HuiEMsg_ReceiveMonMsgTypeFromMonitorError;
	goto Error_Rtn;
    }

    if (monMsgType != HuiMonMsg_MonitorStandby) {
	eMsgType = HuiEMsg_NotMonitorStandby;
	goto Error_Rtn;
    }

    hAccessInfo->monitor.status = HuiMonStatus_Standby;

    /* send UserInfo to monitor.
     */
    if (huiSendOneAgentInfo2Monitor(hAccessInfo, HuiMonMsg_UserInfo, hAccessInfo->userInfo)
	== HuiFalse) {
	eMsgType = HuiEMsg_SendUserInfo2MonitorFail;
	goto Error_Rtn;
    }

    /* send AgentInfo to monitor.
     */
    if (huiSendAgentInfo2Monitor(hAccessInfo, hAccessInfo->connectedAgent) == HuiFalse) {
	eMsgType = HuiEMsg_SendAgentInfo2MonitorError;
	goto Error_Rtn;
    }

    /* wait for AgentTreeDispStandby message from monitor process
     */
    if (huiReceiveMonMsgTypeFromMonitor(hAccessInfo, &monMsgType) == HuiFalse) {
	eMsgType = HuiEMsg_ReceiveMonMsgTypeFromMonitorError;
	goto Error_Rtn;
    }

    if (monMsgType != HuiMonMsg_AgentTreeDispStandby) {
	eMsgType = HuiEMsg_NotAgentTreeDispStandby;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* =======================================
 *	$B;XDj$5$l$?(B Agent ID $B$KBP1~$9$k(B
 *	agent process $B$N5/F0(B
 * ======================================= */
HuiBoolean huiAlWake(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiWord		agentId;
    struct sockaddr_in	*sockAddr;
    HuiErrorMsgType	eMsgType;

    if (huiAlWakeSetup(hAccessInfo, agentId) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    if ((sockAddr = get_agent(agentId)) == NULL) {
	eMsgType = HuiEMsg_GetAgentError;
	goto Error_Rtn;
    }

    HuiFree(sockAddr);

    huiPutsGoodMsg(HuiGMsg_WakeedUpAgent);

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * agent Id $B$N3MF@$H!"%3%^%s%I9T=*N;H=Dj(B
 */
static HuiBoolean huiAlWakeSetup(hAccessInfo, agentId)
    HHuiAccessInfo	hAccessInfo;
    char		*agentId;
{
    HuiErrorMsgType	eMsgType;

    if (huiGetAgentId(hAccessInfo, agentId) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    if (huiCmdEndCheck(NULL) == HuiFalse) {
	eMsgType = HuiEMsg_NotCmdTerminated;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    huiShowCmdUsage(hAccessInfo->currCmd);
    return HuiFalse;

}

/* =======================================
 *	$B;XDj$5$l$?(B Agent ID $B$KBP1~$9$k(B
 *	agent process $B$N:o=|(B
 * ======================================= */
HuiBoolean huiAlSleep(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiWord		agentId;
    HuiErrorMsgType	eMsgType;

    if (huiAlWakeSetup(hAccessInfo, agentId) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    invalidate_agent(agentId);

    if (delete_agent(agentId) == HuiFalse) {
	eMsgType = HuiEMsg_DeleteAgentError;
	goto Error_Rtn;
    }

    huiPutsGoodMsg(HuiGMsg_SleepAgent);

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* =======================================
 *  connect $B$5$l$F$$$k%(!<%8%'%s%H$r(B
 *  $B%j%;%C%HMW5a$9$k%a%C%;!<%8$r(B daemon $B$K(B
 *  $BAw$k(B
 * ======================================= */
HuiBoolean huiAlReset(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiErrorMsgType	eMsgType;

    if (huiCmdEndCheck(NULL) == HuiFalse) {
	huiShowCmdUsage(hAccessInfo->currCmd);
	goto Error_Rtn;
    }

    if (huiResetAgent() == HdFalse) {
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* ============================================
 *  $B;XDj$5$l$?%U%!%$%k$+$i%F%9%H!&%G!<%?$r(B
 *  $BFI$_9~$_!"%b%K%?!<$KAw?.$9$k%F%9%H%3%^%s%I(B
 * ============================================ */
HuiBoolean huiAlTestTrace(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;
{
    HuiWord		fileName;
    HuiErrorMsgType	eMsgType;

    if (huiAlTestTraceSetup(hAccessInfo, fileName) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    if (huiTestHandleDbgInfoFromAgent(hAccessInfo, fileName) == HuiFalse) {
	eMsgType = HuiEMsg_TestHandleDbgInfoFromAgentError;
	goto Error_Rtn;
    }
									  
    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * file name $B$N3MF@$H!"%3%^%s%I9T=*N;H=Dj(B
 */
static HuiBoolean huiAlTestTraceSetup(hAccessInfo, fileName)
    HHuiAccessInfo	hAccessInfo;
    char		*fileName;
{
    HuiWordType		wtype;
    HuiSwitchState	usageFlag = HuiOn;
    HuiErrorMsgType	eMsgType;

    wtype = huiGetWord(fileName);
    if (wtype == HuiWT_String || wtype == HuiWT_OneChar) {
	if (huiCmdEndCheck(NULL) == HuiFalse) {
	    eMsgType = HuiEMsg_NotCmdTerminated;
	    goto Error_Rtn;
	}
    } else {
	if (huiCmdEndCheck(*fileName) == HuiFalse) {
	    eMsgType = HuiEMsg_NotCmdTerminated;
	    goto Error_Rtn;
	}

	sprintf(fileName, HuiTestTraceFileNameFrmt, hAccessInfo->agentName);
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    if (usageFlag == HuiOn) {
	huiShowCmdUsage(hAccessInfo->currCmd);
    }
    return HuiFalse;
}
