/* helios.c;
 *
 *	Helios User Interface main program
 *		created on May 25th in '94.
 *
 *	modification history:	
 *	(1) add event handler for stdin; register_fd(0, FD_EV_READ)
 *		on September 26th in '94.
 *	(2) add huiCatchupSigCld()
 *		on October 27th in '94.
 *
 * $Id: helios.c,v 1.10 1995/03/02 09:25:11 k3sato Exp $
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/wait.h>

#include <helios/event.h>
#include <helios/comm.h>
#include "helios.h"
#include "huiCmdAnalysis.h"
#include "huiMessage.h"
#include "huiCommon.h"
#include "huiVersion.h"

/* defined in hDaemonLib.c
 */
extern HdBoolean heliosdInit();
extern HAgentInfo get_agentInfo();

/* defined in huiDAccess.c
 */
extern HAgentInfo huiSetupUserInfo();


/* Helios Access information
 */
HuiAccessInfo	*huiAccessInfo;

FILE	*HuiLogFd;

static HuiBoolean huiInitial();
static void huiCatchupSigCld();

int main(argc, argv)
    int			argc;
    char		*argv[];
{
    HuiAccessInfo	accessInfo;
    HuiCmdList		*cmd;
    HuiErrorMsgType	eMsgType;
    HuiBoolean		ret;

    if (huiInitial(&accessInfo) == HuiFalse) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    huiAccessInfo = &accessInfo;

    /* opening message
     */
    printf("\n\t%s \t%s%s\n",
	   HELIOS_VER_HEADER, HELIOS_VER_NUM, HELIOS_VER_DATE);

    for (;;) {
	accessInfo.currCmd =
	  cmd = huiGetCmd(&accessInfo);	/* get command */

	if (cmd != NULL) {
	    ret = (*cmd->func)(&accessInfo);

	    if (ret == HuiTrue) {
		switch(cmd->type) {
		case HuiTL_CONNECT:
		    accessInfo.level = HuiAccess2AgentLevel;
		    break;
		case HuiAL_TOP:
		    accessInfo.level = HuiAccess2TopLevel;
		    break;
		case HuiTL_BYE:
		case HuiAL_BYE:
		    goto Good_Bye;
		}
		huiSkipToNextChar();
	    } else {
		huiSkipToNextStmt();
	    }
	}
    }

 Good_Bye:
    exit(0);

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    exit(eMsgType);
}

/*
 *      Initial of helios
 */
static HuiBoolean huiInitial(hAccessInfo)
    HHuiAccessInfo	hAccessInfo;    
{
    HuiErrorMsgType	eMsgType;

    /*	initialize to access to heliosd
     */
    if (heliosdInit() == HdFalse) {
	eMsgType = HuiEMsg_HeliosdInitError;
	goto Error_Rtn;
    }

    /* initialize of log file descriptor
     */
    HuiLogFd = NULL;

    /* initialize of hAccessInfo
     */
    memset(hAccessInfo, 0, sizeof(HuiAccessInfo));
    hAccessInfo->level = HuiAccess2TopLevel;

    /* initialize of user interface information
     */
    if ((hAccessInfo->userInfo = huiSetupUserInfo()) == NULL) {
	eMsgType = HuiEMsg_SetupUsrInfoError;
	goto Error_Rtn;
    }

    /*
     *	get all agent informaition from heliosd
     */
    if ((hAccessInfo->allAgentInfo = get_agentInfo(NULL)) == NULL) {
	eMsgType = HuiEMsg_GetAgentInfoError;
	goto Error_Rtn;
    }

    /*
     *	initialize comm part.
     *	also execute heliosdInit().
     */
    if (init_comm(hAccessInfo->userInfo->sockInfo.sin_port, 1) != 0) {
	eMsgType = HuiEMsg_InitCommError;
	goto Error_Rtn;
    }

    /* register stdin to helios event handler.
     */
    register_fd(0, FD_EV_READ);

    hAccessInfo->monitor.traceSpeed = HuiMonitorMaxSpeed;

    /*	$B;R%W%m%;%9$+$iAw$i$l$F$/$k%7%0%J%k$r=hM}$9$k(B
     *	$B4X?t$N@_Dj(B
     */
    signal(SIGCLD, huiCatchupSigCld);

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* ============================================================================
 *	signal handler
 *
 *	argument: meaning
 *	sig	$B%7%0%J%kHV9f(B
 *	code	$BDI2C$N>\:Y$rDs6!$9$kFCDj$N%7%0%J%k$N%Q%i%a!<%?(B
 *	scp	sigcontext $B9=B$BN!J(Bsignal.h $B$KDj5A$5$l$k!K$r;X$9%]%$%s%?$G$"$j!"(B
 *		$B%7%0%J%kA0$+$i$N%3%s%F%-%9%H$r%j%9%H%"$9$k$?$a$K;HMQ$7$^$9!#(B
 *	addr	$BDI2C$N%"%I%l%9>pJs(B
 * ============================================================================ */
/*
 *	$B;R%W%m%;%9$,(B stop $B0?$O!"(Bexit $B$7$?;~$K?F%W%m%;%9$K(B
 *	$BAw$i$l$k%7%0%J%k$r=hM}$9$k4X?t(B
 */
static void huiCatchupSigCld(sig, code, scp, addr)
    int			sig, code;
    struct sigcontext	*scp;
    char		*addr;
{
    HuiMonitorInfo	*monitor;
    int			pid;
    int			status;
    int			mask;

    mask = sigblock(sigmask(SIGCLD));

    if ((pid = wait(&status)) != -1) {
	huiPrintf("exit a child process of helios(pid = %d).\n"
		  , pid);

	monitor = &(huiAccessInfo->monitor);
	if (monitor->pid == pid) {
	    monitor->state = HuiOff;
	    close(monitor->pipeIn);
	    close(monitor->pipeOut);
	    monitor->pid = 0;
	    huiPrintf("Helios monitor process exited.\n");
	}
    }

    sigsetmask(mask);
    return;
}
