/* huiMonitor.c
 *
 *	Main program of monitor for Helios User Interface
 *		created on September 6th in 94.
 *
 *	modification history:	
 *	(1) add huiSetReadFileHandler(), huiResetReadFileHandler()
 *		on October 27th in '94.
 *
 * $Id: huiMonitor.c,v 2.0 1995/03/03 03:17:19 k3sato Exp $
 */

#include <stdio.h>
#include <signal.h>

#include "huiMonitor.h"
#include "huiMessage.h"
#include "huiPAccess.h"
#include "huiTclMessage.h"

/* define in huiAgentDailog.c
 */
extern HuiBoolean huiTclCreateCmds();

/* define in huiAgentTree.c
 */
extern HuiBoolean huiSetupAgentTreeInfo();

/* define in huiMonTrace.c
 */
extern HuiBoolean huiShowMsgTrace();
extern HuiBoolean huiShowAgentTrace();

/* define in huiPAccess.c
 */
extern HuiBoolean huiGetMonMsgType();

FILE	*HuiLogFd;

static HuiBoolean huiMonitorInit();

void huiSetReadFileHandler();
void huiResetReadFileHandler();

static void huiMonitorMsgProc();
static void huiMonCatchupSigQuit();


int main(argc, argv)
    int			argc;
    char		*argv[];
{
    HuiMonitorCtlInfo	monCtlInfo;
    Tcl_Interp		*interp;
    Tk_Window		mainWindow;	/* The main window for the application.  If
					 * NULL then the application no longer
					 * exists. */
    HuiWord		wordBuf;

    if (argc < 2) {
	fprintf(stderr,
		"̾ˤޤ\n\tUsage : %s <agent name>\n",
		argv[0]);
	exit(100);
    }

    /*
     *	QUIT ʥ (= SIGQUIT) ؿ
     */
    signal(SIGQUIT, huiMonCatchupSigQuit);

    memset(&monCtlInfo, 0, sizeof(HuiMonitorCtlInfo));

    monCtlInfo.pipeIn = 0;
    monCtlInfo.interp =
      interp = Tcl_CreateInterp();

    /*
     * Initialize the Tk application.
     */
    sprintf(wordBuf, "huiMonitor for %.100s", argv[1]);

    mainWindow = Tk_CreateMainWindow(interp, NULL, wordBuf, "Tk");
    if (mainWindow == NULL) {
	goto TclError_Rtn;
    }

    /* Initialize of huiMonitor
    */
    if (huiMonitorInit(&monCtlInfo, mainWindow) == HuiFalse) {
	goto TclError_Rtn;
    }

    if (Tcl_EvalFile(interp, "huiMonitor.tcl") != TCL_OK) {
	fprintf(stderr, "fail in Tcl_EvalFile\n");
	goto TclError_Rtn;
    }
    /*
    fprintf(stderr, "success in Tcl_EvalFile\n");
    */

    /* get AgentTreeInfo from user interface
     */
    if (huiSetupAgentTreeInfo(&monCtlInfo) == HuiFalse) {
	fprintf(stderr, "fail in huiSetupAgentTreeInfo().\n");
	goto Error_Rtn;
    }

    /*
     * Loop infinitely, waiting for commands to execute.  When there
     * are no windows left, Tk_MainLoop returns and we exit.
     */

    Tk_MainLoop();

    /*
     * Don't exit directly, but rather invoke the Tcl "exit" command.
     * This gives the application the opportunity to redefine "exit"
     * to do additional cleanup.
     */

    Tcl_Eval(interp, "exit");

 TclError_Rtn:
    huiPutsTclResult(interp);

 Error_Rtn:

    Tcl_DeleteInterp(interp);
    exit(1);
}

/*
 * Initialize the huiMonitor
 */
static HuiBoolean huiMonitorInit(monCtlInfo, mainWindow)
    HuiMonitorCtlInfo	*monCtlInfo;
    Tk_Window		mainWindow;
{
    Tcl_Interp		*interp;
    XFontStruct		*fonts;
    char		*errorStr;

    HuiLogFd = NULL;

    interp = monCtlInfo->interp;

    /*
    Tk_GeometryRequest(mainWindow, 200, 200);
    */
    if (Tcl_VarEval(interp, "wm minsize . 100 100", NULL) != TCL_OK) {
	errorStr = "Tcl_VarEval";
	goto TclError_Rtn;
    }

    /*
    if (Tcl_SetVar(interp, "tcl_interactive", "0", TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL) {
	errorStr = "Tcl_SetVar";
	goto TclError_Rtn;
    }
    */

    /*
     * Invoke application-specific initialization.
     */
    /*
    if (Tcl_AppInit(interp) != TCL_OK) {
	errorStr = "Tcl_AppInit";
	goto TclError_Rtn;
    }
    */

    if (Tcl_Init(interp) == TCL_ERROR) {
	errorStr = "Tcl_Init";
	goto TclError_Rtn;
    }

    if (Tk_Init(interp) == TCL_ERROR) {
	errorStr = "Tk_Init";
	goto TclError_Rtn;
    }

    /* Ƥpipe (stdin) ե롦ϥɥ顼ȤϿ
     */
    huiSetReadFileHandler(monCtlInfo);

    fonts = Tk_GetFontStruct(interp, mainWindow, "kanji24");

    /*
    fprintf(stderr, "fonts name : %s\n", Tk_NameOfFontStruct(fonts));
    */

    huiTclCreateCmds(monCtlInfo);

    return HuiTrue;

 TclError_Rtn:
    fprintf(stderr, "%s failed: %s\n", errorStr, interp->result);

    return HuiFalse;
}

void huiSetReadFileHandler(monCtlInfo)
    HuiMonitorCtlInfo	*monCtlInfo;
{
    Tk_CreateFileHandler(monCtlInfo->pipeIn, TK_READABLE,
			 (Tk_FileProc *)huiMonitorMsgProc, (ClientData) monCtlInfo);
}

void huiResetReadFileHandler(monCtlInfo)
    HuiMonitorCtlInfo	*monCtlInfo;
{
    Tk_CreateFileHandler(monCtlInfo->pipeIn, 0,
			 (Tk_FileProc *)huiMonitorMsgProc, (ClientData) monCtlInfo);
}

static void huiMonitorMsgProc(monCtlInfo, mask)
    HuiMonitorCtlInfo	*monCtlInfo;
    int			mask;
{
    HuiText		msgBuf;
    HuiMonMsgType	monMsgType;
    HuiErrorMsgType	eMsgType;

    huiResetReadFileHandler(monCtlInfo);

    if (huiGetMsgFromPipe(msgBuf) == HuiFalse) {
	eMsgType = HuiEMsg_GetMsgFromPipeError;
	goto Error_Rtn;
    }

    fprintf(stderr, "Data received from pipe : %s\n", msgBuf);

    if (huiGetMonMsgType(monCtlInfo, msgBuf, &monMsgType) == HuiFalse) {
	eMsgType = HuiEMsg_ReceivedDataFromPipeError;
	goto Error_Rtn;
    }

    switch (monMsgType) {
    case HuiMonMsg_DebugInfo:
    case HuiMonMsg_SMessageInfo:
    case HuiMonMsg_MTraceInfo:
	if (huiShowMsgTrace(monCtlInfo, msgBuf) == HuiFalse) {
	    eMsgType = HuiEMsg_ShowMsgTraceError;
	    goto Error_Rtn;
	}
	break;

    case HuiMonMsg_ATraceInfo:
	if (huiShowAgentTrace(monCtlInfo, msgBuf) == HuiFalse) {
	    eMsgType = HuiEMsg_ShowAgentTraceError;
	    goto Error_Rtn;
	}
	break;

    case HuiMonMsg_AskTraceSpeed:
	break;
    default:
	eMsgType = HuiEMsg_ReceivedDataFromPipeError;
	goto Error_Rtn;
    }

    if (huiPutMonSpeed2UI(monCtlInfo, monCtlInfo->traceSpeed) == HuiFalse) {
	eMsgType = HuiEMsg_PutMonSpeed2UIError;
	goto Error_Rtn;
    }

    huiSetReadFileHandler(monCtlInfo);

    return;

 Error_Rtn:
    huiSetReadFileHandler(monCtlInfo);

    huiPutsErrorMsg(eMsgType);

    if (huiPutErrorMsg2Pipe(eMsgType) == HuiFalse) {
	huiPutsErrorMsg(HuiEMsg_PutMsg2PipeError);
    }

    return;
}

/* ===========================================
 *  QUIT ʥ(= SIGQUIT)ؿ
 * =========================================== */
static void huiMonCatchupSigQuit(sig, code, scp, addr)
    int			sig, code;
    struct sigcontext	*scp;
    char		*addr;
{
    /* close socket.
     */
    close(0);
    close(1);

    exit(sig);
}
