/* huiMonTrace.c
 *	a module to handle trace informations
 *		-- message trace and agent trace --
 *		created on October 6th in 94.
 *
 * $Id: huiMonTrace.c,v 1.10 1995/03/07 12:15:42 k3sato Exp $
 */

#include <stdio.h>
#include <string.h>

#include "huiMonitor.h"
#include "huiMessage.h"
#include "huiTclMessage.h"
#include "huiTclLib.h"
#include "huiMemory.h"
#include "huiAgentProcess.h"
#include "huiMonKeyWord.h"

/* defined in huiAgentTree.c
 */
extern HHuiAgentTreeInfo huiSearchTreeInfo();

/* defined in huiAgentTreeInfo.c
 */
extern HHuiAgentTreeInfo huiGetSingleAgentTreeInfo();

HuiBoolean huiGetMsgTraceInfo();
HuiBoolean huiResetTracePath();
HuiBoolean huiResetOnePath();
HuiBoolean huiDrawTracePath();
HuiBoolean huiDrawOnePath();
HHuiAgentTreeInfo huiGetCommonAgentInfo();
static int huiGetCommonAidStrPtr();
HuiBoolean huiChgColorOfLeafOutline();

HuiBoolean huiDrawTracePathInPModel();
HuiBoolean huiMoveItemInPModel();
HuiBoolean huiChgColorOfAgentPArea();
int huiCountAcceptedAskOfAgentProcess();

HuiBoolean huiMonGetKeyWordValue();
void HuiFreeMsgTraceInfo();

/* ==============================================
 * $B%a%C%;!<%8!&%H%l!<%9$N>pJs$+$i(B
 * $B%a%C%;!<%8$NN.$l$rLp0u$G%b%K%?!<>e$KI=<($9$k(B
 * ============================================== */
HuiBoolean huiShowMsgTrace(monCtlInfo, msgBuf)
    HuiMonitorCtlInfo	*monCtlInfo;
    char		*msgBuf;
{
    HuiErrorMsgType	eMsgType;

    /* reset the current trace path
     */
    if (huiResetTracePath(monCtlInfo) == HuiFalse) {
	eMsgType = HuiEMsg_ResetMTracePathError;
	goto Error_Rtn;
    }

    if (huiGetMsgTraceInfo(monCtlInfo, msgBuf) == HuiFalse) {
	eMsgType = HuiEMsg_GetMsgTraceInfoError;
	goto Error_Rtn;
    }

    if (huiDrawTracePath(monCtlInfo,
			 monCtlInfo->mTraceInfo.fromInfo,
			 monCtlInfo->mTraceInfo.toInfo) == HuiFalse) {
	eMsgType = HuiEMsg_DrawTracePathError;
	goto Error_Rtn;
    }

    if (huiDrawTracePathInPModel(monCtlInfo, &(monCtlInfo->mTraceInfo)) == HuiFalse) {
	eMsgType = HuiEMsg_DrawTracePathInPModelError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * get msgTraceInfo from received data
 */
HuiBoolean huiGetMsgTraceInfo(monCtlInfo, msgBuf)
    HuiMonitorCtlInfo	*monCtlInfo;
    char		*msgBuf;
{
    HHuiMsgTraceInfo	mTraceInfo;
    HuiWord		wordBuf;
    HHuiAgentTreeInfo	rcvAidInfo;
    HHuiAgentTreeInfo	fromInfo;
    HHuiAgentTreeInfo	toInfo;
    HuiErrorMsgType	eMsgType;

    mTraceInfo = &(monCtlInfo->mTraceInfo);

    HuiFreeMsgTraceInfo(mTraceInfo);

    if (huiMonGetKeyWordValue(msgBuf, "Message Type", wordBuf) == HuiFalse) {
	eMsgType = HuiEMsg_MonGetKeyWordValueError;
	goto Error_Rtn;
    }

    mTraceInfo->msgType =
      (strcmp(wordBuf, "ASK") == 0) ? HuiSMsgType_Ask :
	(strcmp(wordBuf, "REPLY") == 0) ? HuiSMsgType_Reply : HuiSMsgType_Control;

    /* Search AgentTreeInfo for msg received Agent
     */
    if (huiMonGetKeyWordValue(msgBuf, "Received AgentID", wordBuf) == HuiFalse) {
	eMsgType = HuiEMsg_MonGetKeyWordValueError;
	goto Error_Rtn;
    }

    if ((rcvAidInfo = huiSearchTreeInfo(monCtlInfo->treeInfo, wordBuf)) == NULL) {
	eMsgType = HuiEMsg_SearchTreeInfoError;
	goto Error_Rtn;
    }

    /* Search AgentTreeInfo for To-agentId
     */
    if (huiMonGetKeyWordValue(msgBuf, "To", wordBuf) == HuiFalse) {
	eMsgType = HuiEMsg_MonGetKeyWordValueError;
	goto Error_Rtn;
    }
    mTraceInfo->to = HuiMallocString(wordBuf);

    /*
    if ((toInfo = huiSearchTreeInfo(monCtlInfo->treeInfo, wordBuf)) == NULL) {
	eMsgType = HuiEMsg_SearchTreeInfoError;
	goto Error_Rtn;
    }
    */
    toInfo = rcvAidInfo;

    /* Search AgentTreeInfo for from-agent
     */
    if (huiMonGetKeyWordValue(msgBuf, "From", wordBuf) == HuiFalse) {
	eMsgType = HuiEMsg_MonGetKeyWordValueError;
	goto Error_Rtn;
    }
    mTraceInfo->from = HuiMallocString(wordBuf);

    switch (mTraceInfo->msgType) {
    case HuiSMsgType_Reply:
	if (huiCompKeyWord(HuiSpecialMsgFuncNameList, wordBuf) == HuiTrue
	    || (wordBuf[0] == HuiChar_Ampersand)
	    || (wordBuf[0] == HuiChar_Minus)) {

	    fromInfo = (rcvAidInfo->type == HdAT_Complex) ?
	      rcvAidInfo : toInfo->parent;
	    break;
	}

    case HuiSMsgType_Control:
    case HuiSMsgType_Ask:
	if (huiCompKeyWord(HuiSpecialMsgFuncNameList, mTraceInfo->to)
	    == HuiTrue) {
	    fromInfo = toInfo->parent;
	} else if ((fromInfo = huiSearchTreeInfo(monCtlInfo->treeInfo, wordBuf))
	    == NULL) {
	    eMsgType = HuiEMsg_SearchTreeInfoError;
	    goto Error_Rtn;
	}

	break;

    }

    mTraceInfo->rcvAidInfo = rcvAidInfo;
    mTraceInfo->toInfo = toInfo;
    mTraceInfo->fromInfo = fromInfo;

    if (fromInfo == toInfo) {
	mTraceInfo->toInfo = 
	  toInfo = fromInfo->parent;
    }

    if (huiMonGetKeyWordValue(msgBuf, "Message ID", wordBuf) == HuiFalse) {
	eMsgType = HuiEMsg_MonGetKeyWordValueError;
	goto Error_Rtn;
    }
    mTraceInfo->msgId = HuiMallocString(wordBuf);

    if (huiMonGetKeyWordValue(msgBuf, "Tid/Status", wordBuf) == HuiFalse) {
	eMsgType = HuiEMsg_MonGetKeyWordValueError;
	goto Error_Rtn;
    }
    mTraceInfo->status = HuiMallocString(wordBuf);

    if (huiMonGetKeyWordValue(msgBuf, "Method", wordBuf) == HuiFalse) {
	eMsgType = HuiEMsg_MonGetKeyWordValueError;
	goto Error_Rtn;
    }
    mTraceInfo->method = HuiMallocString(wordBuf);

    if (huiMonGetKeyWordValue(msgBuf, "Data", wordBuf) == HuiFalse) {
	eMsgType = HuiEMsg_MonGetKeyWordValueError;
	goto Error_Rtn;
    }
    mTraceInfo->msgData = HuiMallocString(wordBuf);

    switch (mTraceInfo->msgType) {
    case HuiSMsgType_Ask:
	toInfo->acceptedAsk++;
	break;
    case HuiSMsgType_Reply:
	if (--(fromInfo->acceptedAsk) < 0) {
	    fromInfo->acceptedAsk = 0;
	}
	break;
    default:
	break;
    }

    return HuiTrue;

 Error_Rtn:
    HuiFreeMsgTraceInfo(&(monCtlInfo->mTraceInfo));
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * $B%a%C%;!<%8!&%H%l!<%9!&%Q%9$NI=<($r%j%;%C%H$9$k(B
 */
HuiBoolean huiResetTracePath(monCtlInfo)
    HuiMonitorCtlInfo	*monCtlInfo;
{
    Tcl_Interp		*interp;
    HHuiAgentTreeInfo	parentInfo;
    HHuiAgentTreeInfo	fromInfo;
    HHuiAgentTreeInfo	toInfo;
    HuiErrorMsgType	eMsgType;

    interp = monCtlInfo->interp;

    if (((parentInfo = monCtlInfo->mTraceInfo.parentInfo) == NULL) ||
	((fromInfo = monCtlInfo->mTraceInfo.fromInfo) == NULL) ||
	((toInfo = monCtlInfo->mTraceInfo.toInfo) == NULL)) {
	return HuiTrue;
    }

    if (huiChgColorOfLeafOutline(interp, fromInfo, "WaitMsgColor") == HuiFalse) {
	eMsgType = HuiEMsg_ChgColorOfLeafOutlineError;
	goto Error_Rtn;
    }

    for (; fromInfo != NULL && fromInfo != parentInfo
	 ; fromInfo = fromInfo->parent) {

	if (huiResetOnePath(interp, fromInfo, fromInfo->parent)
	    == HuiFalse) {
	    eMsgType = HuiEMsg_ResetOnePathError;
	    goto Error_Rtn;
	}
    }

    for (; toInfo != NULL && toInfo != parentInfo
	 ; toInfo = toInfo->parent) {

	if (huiResetOnePath(interp, toInfo->parent, toInfo)
	    == HuiFalse) {
	    eMsgType = HuiEMsg_ResetOnePathError;
	    goto Error_Rtn;
	}
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * $B%a%C%;!<%8$NN.$l$r<($9Lp0u$r=|$-?'$r85$KLa$9(B
 */
HuiBoolean huiResetOnePath(interp, fromInfo, toInfo)
    Tcl_Interp		*interp;
    HHuiAgentTreeInfo	fromInfo;
    HHuiAgentTreeInfo	toInfo;
{
    int			lineConfig;
    HuiText		script;
    HuiErrorMsgType	eMsgType;

    /* decide the line to reset
     */
    if (fromInfo->posX < toInfo->posX) {
	lineConfig = toInfo->lineConfig;
    } else {
	lineConfig = fromInfo->lineConfig;
    }

    sprintf(script,
	    "$lCanvas itemconfig %d -fill $WaitMsgColor -arrow none ; $lCanvas lower %d",
	    lineConfig, lineConfig);

    if (Tcl_Eval(interp, script) != TCL_OK) {
	eMsgType = HuiEMsg_ResetMTracePathError;
	goto TclError_Rtn;
    }

    return HuiTrue;

 TclError_Rtn:
    huiPutsTclResult(interp);

    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * $B%a%C%;!<%8$NN.$l$rLp0u$G%b%K%?!<>e$KI=<($9$k(B
 */
HuiBoolean huiDrawTracePath(monCtlInfo, fromInfo, toInfo)
    HuiMonitorCtlInfo	*monCtlInfo;
    HHuiAgentTreeInfo	fromInfo;
    HHuiAgentTreeInfo	toInfo;
{
    Tcl_Interp		*interp;
    HHuiAgentTreeInfo	parentInfo;
    HHuiAgentTreeInfo	tempInfo;
    char		*fillColor;
    HuiErrorMsgType	eMsgType;

    interp = monCtlInfo->interp;

    if ((parentInfo = huiGetCommonAgentInfo(fromInfo, toInfo)) == NULL) {
	eMsgType = HuiEMsg_GetCommonAgentInfoError;
	goto Error_Rtn;
    }

    monCtlInfo->mTraceInfo.parentInfo = parentInfo;

    fillColor = (monCtlInfo->mTraceInfo.msgType == HuiSMsgType_Ask)
      ? "AskMsgColor" : "ReplyMsgColor";

    for (tempInfo = fromInfo; tempInfo != parentInfo; tempInfo = tempInfo->parent) {
	if (huiDrawOnePath(interp, tempInfo, tempInfo->parent, fillColor)
	    == HuiFalse) {
	    eMsgType = HuiEMsg_DrawOnePathError;
	    goto Error_Rtn;
	}
    }

    for (tempInfo = toInfo; tempInfo != parentInfo; tempInfo = tempInfo->parent) {
	if (huiDrawOnePath(interp, tempInfo->parent, tempInfo, fillColor)
	    == HuiFalse) {
	    eMsgType = HuiEMsg_DrawOnePathError;
	    goto Error_Rtn;
	}
    }

    if (huiChgColorOfLeafOutline(interp, fromInfo, fillColor) == HuiFalse) {
	eMsgType = HuiEMsg_ChgColorOfLeafOutlineError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * from-agent $B$H(B to-agent $B$K6&DL$N?F$N(B agent tree $B$N>pJs$rF@$k(B
 */
HHuiAgentTreeInfo huiGetCommonAgentInfo(fromInfo, toInfo)
    HHuiAgentTreeInfo	fromInfo;
    HHuiAgentTreeInfo	toInfo;
{
    int			i, maxLen;
    int			fromLen, toLen;
    char		*fromAid, *toAid;
    char		fromCh, toCh;
    HHuiAgentTreeInfo	commonInfo;
    HuiWord		commonAid;

    fromAid = fromInfo->agentId;
    toAid = toInfo->agentId;

    fromLen = strlen(fromAid);
    toLen = strlen(toAid);

    if (fromLen < toLen) {
	maxLen = fromLen;
	commonInfo = fromInfo;
    } else {
	maxLen = toLen;
	commonInfo = toInfo;
    }

    strncpy(commonAid, fromAid, maxLen);

    for (i = 0
	 ; i < maxLen && (*fromAid == *toAid)
	 ; i++, fromAid++, toAid++)
      ;

    fromCh = *fromAid;
    toCh = *toAid;

    switch (fromCh) {
    case HdChar_NULL:
	switch (toCh) {
	case HdChar_NULL:
	case HdChar_Period:
	    return commonInfo;

	default:
	    i = huiGetCommonAidStrPtr(i, toAid);
	}
	break;

    case HdChar_Period:
	switch (toCh) {
	case HdChar_NULL:
	    return commonInfo;

	case HdChar_Period:
	    i--;
	    break;

	default:
	    i = huiGetCommonAidStrPtr(i, fromAid);
	    break;
	}
    default:
	i = huiGetCommonAidStrPtr(i, fromAid);
	break;
    }

    if (i == 0) {
	return (fromInfo->posX < toInfo->posX) ? fromInfo : toInfo;
    }

    commonAid[i] = '\0';

    for (; commonInfo != NULL; commonInfo = commonInfo->parent) {
	if (strcmp(commonAid, commonInfo->agentId) == 0) {
	    return commonInfo;
	}
    }

    return NULL;
}

static int huiGetCommonAidStrPtr(ptr, str)
    int		ptr;
    char	*str;
{	
    while ((ptr > 0)
	   && (*str != HuiChar_NULL)
	   && (*str != HuiChar_Period)) {
	str--;
	ptr--;
    }

    return ptr;
}

/*
 * $B%a%C%;!<%8$NN.$l$N#1$D$rLp0u$G%b%K%?!<>e$KI=<($9$k(B
 */
HuiBoolean huiDrawOnePath(interp, fromInfo, toInfo, fillColor)
    Tcl_Interp		*interp;
    HHuiAgentTreeInfo	fromInfo;
    HHuiAgentTreeInfo	toInfo;
    char		*fillColor;
{
    int			lineConfig;
    char		*direction;
    HuiText		script;
    HuiErrorMsgType	eMsgType;

    /* decide the line to redraw with arrow
     * and the arrow direction
     */
    if (fromInfo->posX < toInfo->posX) {
	lineConfig = toInfo->lineConfig;
	direction = "last";
    } else {
	lineConfig = fromInfo->lineConfig;
	direction = "first";
    }

    sprintf(script, "$lCanvas itemconfig %d -fill $%s -arrow %s ; $lCanvas raise %d",
	    lineConfig, fillColor, direction, lineConfig);

    if (Tcl_Eval(interp, script) != TCL_OK) {
	eMsgType = HuiEMsg_DrawMTraceArrowError;
	goto TclError_Rtn;
    }

    return HuiTrue;

 TclError_Rtn:
    huiPutsTclResult(interp);

    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * Agent Leaf $B$NOH$N?'$r;XDj$5$l$??'$KBX$($k(B
 */
HuiBoolean huiChgColorOfLeafOutline(interp, treeInfo, color)
    Tcl_Interp		*interp;
    HHuiAgentTreeInfo	treeInfo;
    char		*color;
{
    HuiText		script;
    HuiErrorMsgType	eMsgType;

    sprintf(script, "$lCanvas itemconfig %d -outline $%s; update idletasks",
	    treeInfo->leafConfig, color);

    if (Tcl_Eval(interp, script) != TCL_OK) {
	eMsgType = HuiEMsg_ChgColorOfLeafOutlineTclError;
	goto TclError_Rtn;
    }

    return HuiTrue;

 TclError_Rtn:
    huiPutsTclResult(interp);

    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * Draw message Trace Path in process model
 */
HuiBoolean huiDrawTracePathInPModel(monCtlInfo, mTraceInfo)
    HuiMonitorCtlInfo	*monCtlInfo;
    HHuiMsgTraceInfo	mTraceInfo;
{
    Tcl_Interp		*interp;
    HHuiAgentTreeInfo	fromInfo;
    HHuiAgentTreeInfo	toInfo;
    HHuiAgentTreeInfo	tempInfo;
    char		*color;
    float		x1, y1, x2, y2;
    HuiText		script;
    HuiErrorMsgType	eMsgType;

    if ((fromInfo = mTraceInfo->fromInfo) == NULL ||
	(toInfo = mTraceInfo->toInfo) == NULL) {
	return HuiTrue;
    }

    interp = monCtlInfo->interp;

    color = (mTraceInfo->msgType == HuiSMsgType_Ask)
      ? "AskMsgColor" : "ReplyMsgColor";

    x1 = fromInfo->pModelX;
    if ((y1 = fromInfo->pModelY) == (float)0) {
	if ((tempInfo = huiGetSingleAgentTreeInfo(fromInfo)) == NULL) {
	    eMsgType = HuiEMsg_GetSingleAgentTreeInfoError;
	    goto Error_Rtn;
	}
	if ((y1 = tempInfo->pModelY) == (float)0) {
	    eMsgType = HuiEMsg_DataErrorInAgentTreeInfo;
	    goto Error_Rtn;
	}
	fromInfo->pModelY = y1;
    }

    x2 = x1 + strlen(fromInfo->agentName) * HuiMonOneCharWidth;
    y2 = y1 + HuiMonOneCharHeight;

    if (huiMoveItemInPModel(interp, mTraceInfo->fromConfig, x1, y1, x2, y2)
	== HuiFalse) {
	eMsgType = HuiEMsg_MoveItemInPModelError;
	goto TclError_Rtn;
    }

    x1 = toInfo->pModelX;
    if ((y1 = toInfo->pModelY) == (float)0) {
	if ((tempInfo = huiGetSingleAgentTreeInfo(toInfo)) == NULL) {
	    eMsgType = HuiEMsg_GetSingleAgentTreeInfoError;
	    goto Error_Rtn;
	}
	if ((y1 = tempInfo->pModelY) == (float)0) {
	    eMsgType = HuiEMsg_DataErrorInAgentTreeInfo;
	    goto Error_Rtn;
	}
	toInfo->pModelY = y1;
    }

    x2 = x1 + strlen(toInfo->agentName) * HuiMonOneCharWidth;
    y2 = y1 + HuiMonOneCharHeight;

    /*
    if (huiMoveItemInPModel(interp, mTraceInfo->toConfig, x1, y1, x2, y2)
	== HuiFalse) {
	eMsgType = HuiEMsg_MoveItemInPModelError;
	goto TclError_Rtn;
    }
    */

    x1 = fromInfo->pModelX + strlen(fromInfo->agentName) * HuiMonOneCharWidth / 2;
    x2 = toInfo->pModelX + strlen(toInfo->agentName) * HuiMonOneCharWidth / 2;

    if (fromInfo->pModelY < toInfo->pModelY) {
	y1 = fromInfo->pModelY + HuiMonOneCharHeight;
	y2 = toInfo->pModelY;
    } else {
	y1 = fromInfo->pModelY;
	y2 = toInfo->pModelY + HuiMonOneCharHeight;
    }

    if (huiMoveItemInPModel(interp, mTraceInfo->arrowConfig, x1, y1, x2, y2)
	== HuiFalse) {
	eMsgType = HuiEMsg_MoveItemInPModelError;
	goto TclError_Rtn;
    }

    sprintf(script, "$pCanvas itemconfig %d -outline $%s; $pCanvas itemconfig %d -fill $%s; $pCanvas itemconfig %d -outline $%s; update idletasks",
	    mTraceInfo->fromConfig, color,
	    mTraceInfo->arrowConfig, color,
	    mTraceInfo->toConfig, color
	    );

    if (Tcl_Eval(interp, script) != TCL_OK) {
	eMsgType = HuiEMsg_ChgColorOfMsgFlowLineInPModelError;
	goto TclError_Rtn;
    }

    if (huiChgColorOfAgentPArea(interp, fromInfo) == HuiFalse) {
	eMsgType = HuiEMsg_ChgColorOfAgentPAreaError;
	goto TclError_Rtn;
    }

    if (huiChgColorOfAgentPArea(interp, toInfo) == HuiFalse) {
	eMsgType = HuiEMsg_ChgColorOfAgentPAreaError;
	goto TclError_Rtn;
    }

    return HuiTrue;

 TclError_Rtn:
    huiPutsTclResult(interp);

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * Move item on pCanvas by changing its coordinates
 */
HuiBoolean huiMoveItemInPModel(interp, fd, x1, y1, x2, y2)
    Tcl_Interp		*interp;
    int			fd;
    float		x1, y1, x2, y2;
{
    HuiText		script;
    HuiErrorMsgType	eMsgType;

    sprintf(script, "$pCanvas coords %i %.1fm %.1fm %.1fm %.1fm; $pCanvas raise %d",
	    fd, x1, y1, x2, y2, fd);

    if (Tcl_Eval(interp, script) != TCL_OK) {
	eMsgType = HuiEMsg_ChgCoordsOfItemError;
	goto TclError_Rtn;
    }

    return HuiTrue;

 TclError_Rtn:
    huiPutsTclResult(interp);
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 * Change the color of Agent process Area in pCanvas
 */
HuiBoolean huiChgColorOfAgentPArea(interp, treeInfo)
    Tcl_Interp		*interp;
    HHuiAgentTreeInfo	treeInfo;
{
    int			processConfig;
    HHuiAgentTreeInfo	tempInfo;
    char		*status;
    int			acceptedAsk;
    HuiText		script;
    HuiErrorMsgType	eMsgType;

    if ((processConfig = treeInfo->processConfig) == 0) {
	if ((tempInfo = huiGetSingleAgentTreeInfo(treeInfo)) == NULL) {
	    eMsgType = HuiEMsg_GetSingleAgentTreeInfoError;
	    goto Error_Rtn;
	}
	processConfig = tempInfo->processConfig;
    }
    /* count of acceptedAsk
     */
    acceptedAsk = huiCountAcceptedAskOfAgentProcess(treeInfo);

    status = (acceptedAsk == 0) ? "Active" : "Asked";

    sprintf(script, "$pCanvas itemconfig %i -fill $ColStatus(%s)",
	    processConfig, status);

    if (Tcl_Eval(interp, script) != TCL_OK) {
	eMsgType = HuiEMsg_ChgFillColorOfItemError;
	goto TclError_Rtn;
    }

    return HuiTrue;

 TclError_Rtn:
    huiPutsTclResult(interp);

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

int huiCountAcceptedAskOfAgentProcess(treeInfo)
    HHuiAgentTreeInfo	treeInfo;    
{
    HHuiAgentTreeInfo	parentInfo;    
    int			acceptedAsk;

    acceptedAsk = treeInfo->acceptedAsk;

    if ((parentInfo = treeInfo->parent) != NULL
	&& parentInfo->child == treeInfo) {

	for (; parentInfo != NULL; parentInfo = parentInfo->next) {
	    acceptedAsk += parentInfo->acceptedAsk;
	}
    }

    return acceptedAsk;
}

/* ==============================================
 * $B%(!<%8%'%s%H!&%H%l!<%9$N>pJs$+$i(B
 * $B%(!<%8%'%s%H!&%9%F%$%?%9$r%b%K%?!<>e$KI=<($9$k(B
 * ============================================== */
HuiBoolean huiShowAgentTrace(monCtlInfo, msgBuf)
    HuiMonitorCtlInfo	*monCtlInfo;
    char		*msgBuf;
{
/*    HuiErrorMsgType	eMsgType;
*/

    return HuiTrue;

/* Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
*/
}

/* ========================
 *  Common routine
 * ======================== */
/*
 *	<keyWord> = <value>
 */
HuiBoolean huiMonGetKeyWordValue(msgBuf, keyWord, value)
    char	*msgBuf;
    char	*keyWord;
    char	*value;
{
    char		*tempKey;
    HuiErrorMsgType	eMsgType;

    for (; *msgBuf != '\0';) {

	for (; *msgBuf == HuiChar_Space ; msgBuf++)
	  ;

	for (tempKey = keyWord
	     ; (*tempKey != '\0') && (*tempKey == *msgBuf)
	     ; tempKey++, msgBuf++)
	  ;

	if (*tempKey == '\0') {

	    for (; *msgBuf == HuiChar_Space ; msgBuf++)
	      ;

	    if (*msgBuf == HuiChar_EqualCode) {
	    
		for (msgBuf++; *msgBuf == HuiChar_Space ; msgBuf++)
		  ;

		for (; *msgBuf != '\0'
		     && *msgBuf != HuiChar_CR;) {
		    *value++ = *msgBuf++;
		}

		*value = '\0';

		return HuiTrue;
	    }
	}

	if ((msgBuf = strchr(msgBuf, HuiChar_CR)) == NULL) {
	    eMsgType = HuiEMsg_ReceivedDataFromPipeError;
	    goto Error_Rtn;
	}
	msgBuf++;
    }

    eMsgType = HuiEMsg_ReceivedDataFromPipeError;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

void HuiFreeMsgTraceInfo(msgTraceInfo)
    HHuiMsgTraceInfo	msgTraceInfo;
{
    if (msgTraceInfo == NULL) return;

    HuiFreeString(&(msgTraceInfo->msgId));
    HuiFreeString(&(msgTraceInfo->from));
    HuiFreeString(&(msgTraceInfo->to));
    HuiFreeString(&(msgTraceInfo->status));
    HuiFreeString(&(msgTraceInfo->method));
    HuiFreeString(&(msgTraceInfo->msgData));
}
