/* huiAgentTree.c
 * *	program to diplay the Agent Tree for Logical model of Agent
 *		created on September 27th in 94.
 *
 * $Id: huiAgentTree.c,v 1.10 1995/03/02 09:27:32 k3sato Exp $
 */

#include "huiMonitor.h"
#include "huiMessage.h"
#include "huiTclLib.h"
#include "huiTclMessage.h"

/* defined in huiMonitor.c
 */
extern HdBoolean huiGetMsgFromPipe();
extern HdBoolean huiPutMsg2Pipe();

/* defined in huiAgentTreeInfo.c
 */
extern HuiBoolean huiGetAgentTreeInfo();

/* defined in huiAgentProcess.c
 */
extern HuiBoolean huiSetupAgentInfoInPModel();
extern HuiBoolean huiDrawAgentInPModel();

HuiBoolean huiDispAgentTreeInfo();
HuiBoolean huiSetupPosOfAgentLeaf();
HuiBoolean huiDrawAgentTree();
HuiBoolean huiDrawCanvas();
HuiBoolean huiDrawAgentLeaf();
HuiBoolean huiDrawAgentBranch();
HuiBoolean huiTclSetAgentLeafSize();

HHuiAgentTreeInfo huiSearchTreeInfo();
HHuiAgentTreeInfo huiSearchTreeInfoForName();
HHuiAgentTreeInfo huiSearchTreeInfoForId();
HHuiAgentTreeInfo huiSearchTreeInfoForFunc();


/*
 *  Setup AgentTreeInfo
 */
HuiBoolean huiSetupAgentTreeInfo(monCtlInfo)
    HuiMonitorCtlInfo	*monCtlInfo;
{
    HuiText		msgBuf;
    HuiErrorMsgType	eMsgType;

    /* send a setup ok message to UI.
     */
    sprintf(msgBuf, "%d \n", HuiMonMsg_MonitorStandby);
    if (huiPutMsg2Pipe(msgBuf) == HuiFalse) {
	eMsgType = HuiEMsg_PutMsg2PipeError;
	goto Error_Rtn;
    }

    if (huiGetAgentTreeInfo(monCtlInfo) == HuiFalse) {
	eMsgType = HuiEMsg_GetAgentTreeInfoError;
	goto Error_Rtn;
    }

    if (huiTclSetAgentLeafSize(monCtlInfo->interp,
			       (float)HuiAgentLeafWidth, (float)HuiAgentLeafHeight)
	== HuiFalse) {
	eMsgType = HuiEMsg_TclSetAgentLeafSizeError;
	goto Error_Rtn;
    }

    if (huiDispAgentTreeInfo(monCtlInfo) == HuiFalse) {
	eMsgType = HuiEMsg_DispAgentTreeInfoError;
	goto Error_Rtn;
    }

    /* send a message of setting up agent tree info to UI.
     */
    sprintf(msgBuf, "%d \n", HuiMonMsg_AgentTreeDispStandby);

    if (huiPutMsg2Pipe(msgBuf) == HuiFalse) {
	eMsgType = HuiEMsg_PutMsg2PipeError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Display AgetTreeInfo on Tk-canvas
 */
HuiBoolean huiDispAgentTreeInfo(monCtlInfo)
    HuiMonitorCtlInfo	*monCtlInfo;
{
    Tcl_Interp		*interp;
    HuiErrorMsgType	eMsgType;

    interp = monCtlInfo->interp;

    monCtlInfo->canvasWidth = monCtlInfo->canvasHeight = 0;

    if (huiSetupPosOfAgentLeaf(monCtlInfo, monCtlInfo->treeInfo) == HuiFalse) {
	eMsgType = HuiEMsg_SetupPosOfAgentLeafError;
	goto Error_Rtn;
    }

    if (huiSetupAgentInfoInPModel(monCtlInfo, monCtlInfo->treeInfo) == HuiFalse) {
	eMsgType = HuiEMsg_SetupAgentInfoInPModelError;
	goto Error_Rtn;
    }

    if (huiDrawCanvas(monCtlInfo) == HuiFalse) {
	eMsgType = HuiEMsg_DrawCanvasError;
	goto Error_Rtn;
    }

    if (huiDrawAgentTree(monCtlInfo, monCtlInfo->treeInfo, 0) == HuiFalse) {
	eMsgType = HuiEMsg_DrawAgentTreeError;
	goto Error_Rtn;
    }

    if (huiDrawAgentInPModel(monCtlInfo, monCtlInfo->treeInfo) == HuiFalse) {
	eMsgType = HuiEMsg_DrawAgentInPModelError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Setup position of agent leaf on Tk-canvas
 */
HuiBoolean huiSetupPosOfAgentLeaf(monCtlInfo, treeInfo)
    HuiMonitorCtlInfo	*monCtlInfo;
    HuiAgentTreeInfo	*treeInfo;
{
    HuiAgentTreeInfo	*childInfo;
    float		level;
    float		minY;
    float		maxY;
    HuiErrorMsgType	eMsgType;

    if (treeInfo->child == NULL) {
	treeInfo->posY = monCtlInfo->canvasHeight;

	monCtlInfo->canvasHeight += HuiAgentLeafSpanHeight;

	if (monCtlInfo->canvasWidth < treeInfo->posX) {
	    monCtlInfo->canvasWidth = treeInfo->posX;
	}
    } else {
	level = treeInfo->posX + 1;

	for (childInfo = treeInfo->child, minY = -1, maxY = 0
	     ; childInfo != NULL
	     ; childInfo = childInfo->next) {

	    childInfo->posX = level;
	    if (huiSetupPosOfAgentLeaf(monCtlInfo, childInfo) == HuiFalse) {
		eMsgType = HuiEMsg_SetupPosOfAgentLeafError;
		goto Error_Rtn;
	    }

	    if (minY < 0)		minY = childInfo->posY;
	    if (maxY < childInfo->posY) maxY = childInfo->posY;
	}

	treeInfo->posY = (maxY + minY)/2;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Draw AgentTree on Tk-canvas
 */
HuiBoolean huiDrawAgentTree(monCtlInfo, treeInfo, level)
    HuiMonitorCtlInfo	*monCtlInfo;
    HuiAgentTreeInfo	*treeInfo;
    int			level;
{
    Tcl_Interp		*interp;
    HuiAgentTreeInfo	*childInfo;
    float		parentX, parentY;
    float		childX, childY;
    HuiErrorMsgType	eMsgType;

    interp = monCtlInfo->interp;

    if (treeInfo->child == NULL) {

	if (huiDrawAgentLeaf(interp, "single", treeInfo, level) == HuiFalse) {
	    eMsgType = HuiEMsg_DrawAgentLeafError;
	    goto Error_Rtn;
	}

    } else {
	parentX = HuiAgentLeafStart_X+ HuiAgentLeafWidth
	  + treeInfo->posX * HuiAgentLeafSpanWidth;
	parentY = treeInfo->posY + (float)HuiAgentLeafHeight / 2;

	if (huiDrawAgentLeaf(interp, "complex", treeInfo, level++) == HuiFalse) {
	    eMsgType = HuiEMsg_DrawAgentLeafError;
	    goto Error_Rtn;
	}

	for (childInfo = treeInfo->child
	     ; childInfo != NULL
	     ; childInfo = childInfo->next) {

	    childX = HuiAgentLeafStart_X + childInfo->posX * HuiAgentLeafSpanWidth;
	    childY = childInfo->posY + (float)HuiAgentLeafHeight / 2;

	    if (huiDrawAgentBranch(interp, parentX, parentY, childX, childY)
		== HuiFalse) {
		eMsgType = HuiEMsg_DrawAgentBranchError;
		goto Error_Rtn;
	    }
	    sscanf(interp->result, "%d", &childInfo->lineConfig);

	    if (huiDrawAgentTree(monCtlInfo, childInfo, level) == HuiFalse) {
		eMsgType = HuiEMsg_DrawAgentTreeError;
		goto Error_Rtn;
	    }
	}
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Draw the canvas on Tk-window
 */
HuiBoolean huiDrawCanvas(monCtlInfo)
    HuiMonitorCtlInfo	*monCtlInfo;
{
    Tcl_Interp		*interp;
    HuiText		script;
    HuiErrorMsgType	eMsgType;

    interp = monCtlInfo->interp;

    sprintf(script, "huiDispMonitor %.1f %.1f"
	    , monCtlInfo->canvasWidth, monCtlInfo->canvasHeight);

    if (Tcl_Eval(interp, script) != TCL_OK) {
	eMsgType = HuiEMsg_DispMonitorError;
	goto TclError_Rtn;
    }

    return HuiTrue;

 TclError_Rtn:
    huiPutsTclResult(interp);
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Draw the Agent leaf on Tk-canvas
 */
HuiBoolean huiDrawAgentLeaf(interp, type, treeInfo, level)
    Tcl_Interp		*interp;
    char		*type;
    HuiAgentTreeInfo	*treeInfo;
    int			level;
{
    HuiText		script;
    HuiErrorMsgType	eMsgType;

    sprintf(script, "mkAgentLeaf %s %s %i %.1f %.1f"
	    , type, treeInfo->agentName, level + treeInfo->groupNum,
	    HuiAgentLeafStart_X + treeInfo->posX * HuiAgentLeafSpanWidth,
	    treeInfo->posY);

    if (Tcl_Eval(interp, script) != TCL_OK) {
	eMsgType = HuiEMsg_mkAgentLeafError;
	goto TclError_Rtn;
    }

    sscanf(interp->result, "%d", &treeInfo->leafConfig);

    return HuiTrue;

 TclError_Rtn:
    huiPutsTclResult(interp);
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Draw the branch between two agent leaves.
 */
HuiBoolean huiDrawAgentBranch(interp, x1, y1, x2, y2)
    Tcl_Interp		*interp;
    float		x1, y1;
    float		x2, y2;
{
    HuiText		script;
    HuiErrorMsgType	eMsgType;

    sprintf(script, "mkAgentBranch %.1f %.1f %.1f %.1f"
	    , x1, y1, x2, y2);

    if (Tcl_Eval(interp, script) != TCL_OK) {
	eMsgType = HuiEMsg_mkAgentBranchError;
	goto TclError_Rtn;
    }

    return HuiTrue;

 TclError_Rtn:
    huiPutsTclResult(interp);
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/*
 *  Set agent leaf size to Tcl global variables
 */
HuiBoolean huiTclSetAgentLeafSize(interp, width, height)
    Tcl_Interp		*interp;
    float		width;
    float		height;
{
    HuiErrorMsgType	eMsgType;

    if (huiTclSetGlobalVarFloat(interp, "leafWidth", width) == HuiFalse) {
	eMsgType = HuiEMsg_TclSetGlobalVarFloatError;
	goto Error_Rtn;
    }

    if (huiTclSetGlobalVarFloat(interp, "leafHeight", height) == HuiFalse) {
	eMsgType = HuiEMsg_TclSetGlobalVarFloatError;
	goto Error_Rtn;
    }

    return HuiTrue;

 Error_Rtn:
    huiPutsErrorMsg(eMsgType);
    return HuiFalse;
}

/* =====================================
 *  get infos from HuiAgentTreeInfo
 * =====================================*/
/*
 * Get AgentTree info for a given string.
 */
HHuiAgentTreeInfo huiSearchTreeInfo(treeInfo, str)
    HuiAgentTreeInfo	*treeInfo;
    char		*str;
{
    HHuiAgentTreeInfo	tempInfo;

    if ((tempInfo = huiSearchTreeInfoForId(treeInfo, str))
	== NULL) {
	if ((tempInfo = huiSearchTreeInfoForName(treeInfo, str))
	    == NULL) {
	    tempInfo = huiSearchTreeInfoForFunc(treeInfo, str);
	}
    }

    return tempInfo;
}

/*
 * Get AgentTree info for a given agent name.
 */
HHuiAgentTreeInfo huiSearchTreeInfoForName(treeInfo, agentName)
    HuiAgentTreeInfo	*treeInfo;
    char		*agentName;
{
    HHuiAgentTreeInfo	tempInfo;

    for (; treeInfo != NULL; treeInfo = treeInfo->next) {
	if (strcmp(agentName, treeInfo->agentName) == 0) {
	    return treeInfo;
	}
	if ((tempInfo = huiSearchTreeInfoForName(treeInfo->child, agentName)) != NULL) {
	    return tempInfo;
	}
    }

    /*
    huiPutsErrorMsg(HuiEMsg_NotExistAgentTreeInfo);
    */
    return NULL;
}

/*
 * get HuiAgentTreeInfo for agent ID
 */
HHuiAgentTreeInfo huiSearchTreeInfoForId(treeInfo, aid)
    HuiAgentTreeInfo	*treeInfo;
    char		*aid;
{
    HuiAgentTreeInfo	*tempInfo;

    for (; treeInfo != NULL; treeInfo = treeInfo->next) {
	if (strcmp(treeInfo->agentId, aid) == 0) {
	    return treeInfo;
	}

	if ((tempInfo = huiSearchTreeInfoForId(treeInfo->child, aid))
	    != NULL) {
	    return tempInfo;
	}
    }

    /*
    huiPutsErrorMsg(HuiEMsg_NotExistAgentTreeInfoForId);
    */
    return NULL;
}

/*
 * get HuiAgentTreeInfo for Function name
 */
HHuiAgentTreeInfo huiSearchTreeInfoForFunc(treeInfo, funcName)
    HuiAgentTreeInfo	*treeInfo;
    char		*funcName;
{
    HuiAgentTreeInfo	*tempInfo;
    FuncDirInfo		*currFunc;

    for (; treeInfo != NULL; treeInfo = treeInfo->next) {

	for (currFunc = treeInfo->funcDir
	     ; currFunc != NULL
	     ; currFunc = currFunc->next) {

	    if (strcmp(currFunc->funcName, funcName) == 0) {
		return treeInfo;
	    }
	}

	if ((tempInfo = huiSearchTreeInfoForFunc(treeInfo->child, funcName))
	    != NULL) {
	    return tempInfo;
	}
    }

    /*
    huiPutsErrorMsg(HuiEMsg_NotExistAgentTreeInfoForFunc);
    */
    return NULL;
}
