/* mkhinit.c:
 *	main program to make the file - helios.init
 *		created by k3sato on February 22nd in '95.
 *
 *	modification history:	
 *	(1) 
 *
 * $Id: mkhinit.c,v 1.0 1995/03/02 02:34:45 k3sato Exp $
 */

#include <stdio.h>
#include <dirent.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <string.h>
#include <fcntl.h>
#include <memory.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <time.h>

#include <helios/heliosd.h>
#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>

#include "hdcommon.h"

extern time_t time();
extern int strftime();
extern int rename();

/* defined in hdHiaInfo.c
 */
extern HdBoolean hdGetHiaInfo();

static char *hdGetHeliosdFullPath();
static HdBoolean heliosdPathSearch();

HdBoolean hdMakeInitFile();
HdBoolean hdRenameInitFile();

static HdBoolean hdSetupInitFile();
static HdBoolean hdPutsInitInfo();
static HdBoolean heliosdPathSearch();

static HdBoolean hdGetMachInfo();
static HMachInfo hdMakeDataOfMachInfo();

static void prnCurrTime();
static void hdErrorExit();

static char *HeaderOfInitFile =
  "# helios.init\n\
#\tmade with mkhinit command\n";

void main(argc, argv)
    int		argc;
    char	*argv[];
{
    HeliosdCtlInfo	hdCtlInfo;	/* Heliosd Control information */
    HeliosdErrorMsgType	eMsgType;

    memset(&hdCtlInfo, 0, sizeof(HeliosdCtlInfo));

    /*
     * Get full path name of heliosd;
     *	<full path> + '/'
     */
    if ((hdCtlInfo.fullPath = hdGetHeliosdFullPath(argv[0])) == NULL) {
	eMsgType = HdEMsg_GetHeliosdFullPathError;
	goto Error_Exit;
    }

    /*
     *	$B%^%7%s>pJs$N3MF@(B
     */
    if (hdGetMachInfo(&hdCtlInfo, &(hdCtlInfo.machInfo)) == HdFalse) {
	eMsgType = HdEMsg_GetMachInfoError;
	goto Error_Exit;
    }

    /*
     *	$B%X%j%*%9!&%G!<%b%s$,N)$A>e$2$i$l$?%^%7%s$G(B
     *	$B07$&$3$H$N=PMh$kA4%(!<%8%'%s%H>pJs$N3MF@(B
     */
    if (hdGetHiaInfo(&hdCtlInfo, &hdCtlInfo.allAgents) == HdFalse) {
	eMsgType = HdEMsg_HiaFileError;
	goto Error_Exit;
    }

    /*
     * write the initial data of agent to the file; helios.init
     */
    if (hdMakeInitFile(&hdCtlInfo) == HdFalse) {
	hdPrintf("Error in making helios.init\n");
	exit(2);
    }

    exit(0);

 Error_Exit:
    hdErrorExit(eMsgType);
}

/*
 * Make the file - helios.init
 */
HdBoolean hdMakeInitFile(hdCtlInfo)
    HeliosdCtlInfo	*hdCtlInfo;
{

    if (hdRenameInitFile(hdCtlInfo) == HdTrue) {

	return hdSetupInitFile(hdCtlInfo);
    }

    hdPrintf("Error in renaming helios.init to helios.init~\n");
    return HdFalse;
}

/*
 * Rename init file if helios.init already exists.
 */
HdBoolean hdRenameInitFile(hdCtlInfo)
    HeliosdCtlInfo	*hdCtlInfo;
{
    DIR		*initDirP;
    int		initFd;
    char	*bufPtr;
    char	initFile[HeliosdFilePathLen_Lim];
    char	initFileTilde[HeliosdFilePathLen_Lim];

    sprintf(initFile, "%s", hdCtlInfo->fullPath);

    if ((bufPtr = strrchr(initFile, '/')) == NULL) {
	hdPrintf("Data error in fullPath(%s)\n", hdCtlInfo->fullPath);
	return HdFalse;
    }

    *bufPtr = HdChar_NULL;

    if ((initDirP = opendir(initFile)) == NULL) {
	hdPrintf("Can't opendir %s\n", initFile);
	return HdFalse;
    }

    sprintf(initFile, "%s%s", hdCtlInfo->fullPath, HeliosInitFileName);
    sprintf(initFileTilde, "%s%s~", hdCtlInfo->fullPath, HeliosInitFileName);

    if ((initFd = open(initFile, O_CREAT|O_RDWR|O_EXCL, 00664)) < 0) {
	if (errno == EEXIST) {
	    if (rename(initFile, initFileTilde) < 0) {
		hdPerror();
		goto Close_Error_Rtn;
	    }
	}
    } else {
	close(initFd);
    }
    closedir(initDirP);

    return HdTrue;

 Close_Error_Rtn:
    closedir(initDirP);
    return HdFalse;
}

/*
 *	Setup the init file
 */
static HdBoolean hdSetupInitFile(hdCtlInfo)
    HeliosdCtlInfo	*hdCtlInfo;
{
    char		initFile[HeliosdFilePathLen_Lim];
    FILE		*fp;
    HeliosdErrorMsgType	eMsgType;

    sprintf(initFile, "%s/%s", hdCtlInfo->fullPath, HeliosInitFileName);

    if ((fp = fopen(initFile, "w")) == (FILE *)NULL) {
	eMsgType = HdEMsg_InitFileOpenError;
	goto Error_Rtn;
    }

    fprintf(fp, HeaderOfInitFile);
    prnCurrTime(fp);

    if (hdPutsInitInfo(hdCtlInfo, fp, hdCtlInfo->allAgents) == HdFalse) {
	eMsgType = NULL;
	hdPrintf("Error in hdPutsInitInfo.\n");
	goto Close_Error_Rtn;
    }

    fclose(fp);

    return HdTrue;

 Close_Error_Rtn:
    fclose(fp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	Write the init data to the init file - helios.init
 */
static HdBoolean hdPutsInitInfo(hdCtlInfo, fp, agentInfo)
    HeliosdCtlInfo	*hdCtlInfo;
    FILE		*fp;
    HAgentInfo		agentInfo;
{
    HAgentInfo		info;

    for (info = agentInfo; info != NULL; info = info->next) {

	if (info->type == HdAT_Complex) {
	    if (fprintf(fp, "#\n# %s\tComplex Agent : %s\n#\n",
			info->agentId, info->agentName) == EOF) {
		goto Error_Rtn;
	    }
	    if (hdPutsInitInfo(hdCtlInfo, fp, info->child) == HdFalse) {
		goto Error_Rtn;
	    }
	} else {
	    if (fprintf(fp, "%s\t%s\t%d",
			info->agentId,
			info->sockInfo.hostName,
			info->groupNum)
		== EOF) {
		goto Error_Rtn;
	    }
	    if (fprintf(fp, (info->procArgs != NULL) ? "\t%s\n" : "\n",
			info->procArgs)
		== HdFalse) {
		goto Error_Rtn;
	    }
	}
    }

    return HdTrue;

 Error_Rtn:
    return HdFalse;
}

/*
 * Get full path name of heliosd;
 *	<full path> + '/'
 */
static char *hdGetHeliosdFullPath(heliosdName)
    char	*heliosdName;
{
    char	*heliosdPath;
    char	*p;
    char	tempPath[MAXPATHLEN];
    char	hdPathName[MAXPATHLEN];

#ifdef _PATH_DEBUG
    fprintf(stderr, "heliosdName : %s\n", heliosdName);
#endif

    strcpy(tempPath, heliosdName);
    if ((p = strrchr(tempPath, '/')) != NULL) {
	*p = NULL;
    } else {
	if (heliosdPathSearch(heliosdName, tempPath) == HdFalse)
	  return NULL;
    }

    if (realpath(tempPath, hdPathName) == NULL) {
	hdPerror();
	fprintf(stderr, "path name of heliosd : %s\n", heliosdName);
	return NULL;
    }

    if ((p = strrchr(hdPathName, '/')) != NULL) {
	*p = NULL;
	sprintf(tempPath, "%s/config/", hdPathName);

	if ((heliosdPath = HdMallocString(tempPath)) != NULL) {

#ifdef _PATH_DEBUG
    fprintf(stderr, "heliosdPath : %s\n", heliosdPath);
#endif
	    return heliosdPath;
	}
    }
    return NULL;
}

/*
 * Get full path name for a program name;
 *	<full path> + '/'
 */
static HdBoolean heliosdPathSearch(hdProgName, hdPathName)
    char		*hdProgName;
    char		*hdPathName;
{
    char		*hdPathEnv;
    char		*p1, *p2;
    char		*pathName = NULL;
    DIR			*dirp;
    DIR			*tempDirp;
    struct dirent	*dp;
    char		fileName[HeliosdFilePathLen_Lim];

#ifdef _PATH_DEBUG
	fprintf(stderr, "hdProgName : %s\n", hdProgName);
#endif

    if ((hdPathEnv = HdMallocString(getenv("PATH"))) == NULL)
      return HdFalse;

    for (p1 = p2 = hdPathEnv; *p1 != NULL; p1 = p2) {
	for (p2 = p1; *p2 != NULL && *p2 != ':'; p2++)
	  ;

	if (*p2 != NULL)
	  *p2++ = '\0';

#ifdef _PATH_DEBUG
	fprintf(stderr, "pathName in PathSearch : %s\n", p1);
#endif

	if ((dirp = opendir(p1)) != NULL) {

	    while ((dp = readdir(dirp)) != NULL) {
		if (strcmp(dp->d_name, hdProgName) == 0) {

		    sprintf(fileName, "%s/%s", p1, hdProgName);
		    if ((tempDirp = opendir(fileName)) != NULL) {
#ifdef _PATH_DEBUG
			fprintf(stderr, "fileName in PathSearch : %s\n", fileName);
#endif
			closedir(tempDirp);
			continue;
		    }
#ifdef _PATH_DEBUG
		    fprintf(stderr, "pathName in PathSearch : %s\n", p1);
#endif
		    pathName = p1;
		    break;
		}
	    }
	    closedir(dirp);

	    if (dp != NULL && pathName != NULL) {
		sprintf(hdPathName, "%s", pathName);
		break;
	    }
	}
    }

    free(hdPathEnv);
    return (pathName != NULL) ? HdTrue : HdFalse;
}

/*
 *	$B%^%7%s>pJs$N3MF@(B
 */
static HdBoolean hdGetMachInfo(hdCtlInfo, hMachInfo)
    HeliosdCtlInfo	*hdCtlInfo;
    HMachInfo		*hMachInfo;
{
    char		hostname[HostNameLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    /* $B%5!<%P$r5/F0$7$?%^%7%s$N%[%9%HL>(B */
    if (gethostname(hostname, HostNameLen_Lim) == ERROR_FLAG) {
	hdPerror();
	eMsgType = HdEMsg_NotFindHostName;
	goto Error_Rtn;
    }

#ifdef _DEBUG_HELIOSD_HOSTNAME
    fprintf(stderr, "host name : %s\n", hostname);
#endif /* _DEBUG_HELIOSD */

    if ((*hMachInfo = hdMakeDataOfMachInfo(hostname)) == NULL) {
	eMsgType = HdEMsg_MakeDataOfMachInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	Make data of MachInfo structure
 */
static HMachInfo hdMakeDataOfMachInfo(hostname)
    char		*hostname;
{
    HMachInfo		machInfo;
    struct hostent	*myhost;
    struct sockaddr_in	*sockAddr;
    HeliosdErrorMsgType	eMsgType;


    if ((myhost = gethostbyname(hostname)) == NULL) {
	eMsgType = HdEMsg_BadHostName;
	goto Error_Rtn;
    }

    if ((machInfo = (HMachInfo)HdMalloc(sizeof(MachInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    if ((machInfo->hostName = HdMallocString(hostname)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    machInfo->addressSize = sizeof(machInfo->sockAddress);
    sockAddr = &(machInfo->sockAddress);

    bzero((char *)sockAddr, machInfo->addressSize);
    sockAddr->sin_family = AF_INET;

    bcopy(myhost->h_addr, (char *)&(sockAddr->sin_addr), myhost->h_length);

#ifdef _Heliosd_Static_Port_Num

    sockAddr->sin_port = HeliosdPortNum;

#else

    sockAddr->sin_port = 0;
    sockAddr->sin_addr.s_addr = INADDR_ANY;

#endif

    return machInfo;

 Error_Rtn:
    hdFreeMachInfo(&machInfo);

    hdPutsErrorMsg(eMsgType);
    return NULL;
}

static void prnCurrTime(fp)
    FILE	*fp;
{
    time_t      clock;
    struct tm   *tm;
    char	buf[AgentNameLen_Lim];

    time(&clock);
    tm = localtime(&clock);

    strftime(buf, AgentNameLen_Lim, "at %T on %A, %B %d in %Y", tm);

    fprintf(fp, "#\t%s.\n\n", buf);
}

/*
 *
 */
static void hdErrorExit(eMsgType)
    HeliosdErrorMsgType	eMsgType;
{
    hdPutsErrorMsg(eMsgType);
    exit(eMsgType);
}
