/* mkhdal.c:
 *	main program to make the file - helios.dal
 *		created by k3sato on March 1st in '95.
 *
 *	modification history:	
 *	(1) 
 *
 * $Id: mkhdal.c,v 1.0 1995/03/02 02:38:46 k3sato Exp $
 */

#include <stdio.h>
#include <dirent.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <string.h>
#include <fcntl.h>
#include <memory.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <time.h>

#include <helios/heliosd.h>
#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>

#include "hdcommon.h"

extern time_t time();
extern int strftime();
extern int rename();

/* defined in hdHiaInfo.c
 */
extern HdBoolean hdGetHiaInfo();

static char *hdGetHeliosdFullPath();
static HdBoolean heliosdPathSearch();

static HdBoolean hdSetupMachInfo();
static HdBoolean hdCheckMachName();

HdBoolean hdMakeDalFile();
HdBoolean hdRenameDalFile();

static HdBoolean hdSetupDalFile();
static HdBoolean hdPutsDalInfo();
static HdBoolean heliosdPathSearch();

static HdBoolean hdGetMachInfo();
static HMachInfo hdMakeDataOfMachInfo();

static void prnCurrTime();
static void hdErrorExit();

static char *HeaderOfDalFile =
  "# helios.dal\n\
#\tmade with mkhdal command\n";


void main(argc, argv)
    int		argc;
    char	*argv[];
{
    HeliosdCtlInfo	hdCtlInfo;	/* Heliosd Control information */
    HeliosdErrorMsgType	eMsgType;

    memset(&hdCtlInfo, 0, sizeof(HeliosdCtlInfo));

    /*
     * Get full path name of heliosd;
     *	<full path> + '/'
     */
    if ((hdCtlInfo.fullPath = hdGetHeliosdFullPath(argv[0])) == NULL) {
	eMsgType = HdEMsg_GetHeliosdFullPathError;
	goto Error_Exit;
    }

    /*
     *	$B%^%7%s>pJs$N3MF@(B
     */
    if (hdGetMachInfo(&hdCtlInfo, &(hdCtlInfo.machInfo)) == HdFalse) {
	eMsgType = HdEMsg_GetMachInfoError;
	goto Error_Exit;
    }

    /*
     *	$B%X%j%*%9!&%G!<%b%s$,N)$A>e$2$i$l$?%^%7%s$G(B
     *	$B07$&$3$H$N=PMh$kA4%(!<%8%'%s%H>pJs$N3MF@(B
     */
    if (hdGetHiaInfo(&hdCtlInfo, &hdCtlInfo.allAgents) == HdFalse) {
	eMsgType = HdEMsg_HiaFileError;
	goto Error_Exit;
    }

    /*
     * Setup machine information
     */
    if (hdSetupMachInfo(hdCtlInfo.machInfo, hdCtlInfo.allAgents) == HdFalse) {
	hdPrintf("Error in setupping machine info.\n");
	exit(2);
    }

    /*
     * write the initial data of agent to the file; helios.dal
     */
    if (hdMakeDalFile(&hdCtlInfo) == HdFalse) {
	hdPrintf("Error in making helios.dal\n");
	exit(2);
    }

    exit(0);

 Error_Exit:
    hdErrorExit(eMsgType);
}

/*
 * Setup machine information
 */
static HdBoolean hdSetupMachInfo(machInfo, agentInfo)
    HMachInfo		machInfo;
    HAgentInfo		agentInfo;
{
    HAgentInfo		info;

    for (info = agentInfo; info != NULL; info = info->next) {

	if (info->type == HdAT_Complex) {
	    if (hdSetupMachInfo(machInfo, info->child) == HdFalse) {
		goto Error_Rtn;
	    }
	} else {
	    if (hdCheckMachName(machInfo, info->sockInfo.hostName) == HdFalse) {
		goto Error_Rtn;
	    }
	}
    }

    return HdTrue;

 Error_Rtn:
    return HdFalse;
}

static HdBoolean hdCheckMachName(machInfo, name)
    HMachInfo		machInfo;
    char		*name;
{
    HMachInfo		prevInfo;
    HeliosdErrorMsgType	eMsgType;

    for (; machInfo != NULL; machInfo = machInfo->next) {
	if (strcmp(machInfo->hostName, name) == 0) {
	    goto Good_Rtn;
	}

	prevInfo = machInfo;
    }

    if ((prevInfo->next = hdMakeDataOfMachInfo(name)) == NULL) {
	eMsgType = HdEMsg_MakeDataOfMachInfoError;
	goto Error_Rtn;
    }

 Good_Rtn:
    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * if helios.dal already exists, rename dal file
 * and then make dal file.
 */
HdBoolean hdMakeDalFile(hdCtlInfo)
    HeliosdCtlInfo	*hdCtlInfo;
{

    if (hdRenameDalFile(hdCtlInfo) == HdTrue) {

	return hdSetupDalFile(hdCtlInfo);
    }

    hdPrintf("Error in renaming helios.dal to helios.dal~\n");

    return HdFalse;
}

/*
 * Rename dal file if helios.dal already exists.
 */
HdBoolean hdRenameDalFile(hdCtlInfo)
    HeliosdCtlInfo	*hdCtlInfo;
{
    DIR		*dalDirP;
    int		dalFd;
    char	*bufPtr;
    char	dalFile[HeliosdFilePathLen_Lim];
    char	dalFileTilde[HeliosdFilePathLen_Lim];

    sprintf(dalFile, "%s", hdCtlInfo->fullPath);

    if ((bufPtr = strrchr(dalFile, '/')) == NULL) {
	hdPrintf("Data error in fullPath(%s)\n", hdCtlInfo->fullPath);
	return HdFalse;
    }

    *bufPtr = HdChar_NULL;

    if ((dalDirP = opendir(dalFile)) == NULL) {
	hdPrintf("Can't opendir %s\n", dalFile);
	return HdFalse;
    }

    sprintf(dalFile, "%s%s", hdCtlInfo->fullPath, HeliosDalFileName);
    sprintf(dalFileTilde, "%s%s~", hdCtlInfo->fullPath, HeliosDalFileName);

    if ((dalFd = open(dalFile, O_CREAT|O_RDWR|O_EXCL, 00664)) < 0) {
	if (errno == EEXIST) {
	    if (rename(dalFile, dalFileTilde) < 0) {
		hdPerror();
		goto Close_Error_Rtn;
	    }
	}
    } else {
	close(dalFd);
    }
    closedir(dalDirP);

    return HdTrue;

 Close_Error_Rtn:
    closedir(dalDirP);
    return HdFalse;
}

/*
 *	Setup the dal file
 */
static HdBoolean hdSetupDalFile(hdCtlInfo)
    HeliosdCtlInfo	*hdCtlInfo;
{
    char		dalFile[HeliosdFilePathLen_Lim];
    FILE		*fp;
    HeliosdErrorMsgType	eMsgType;

    sprintf(dalFile, "%s/%s", hdCtlInfo->fullPath, HeliosDalFileName);

    if ((fp = fopen(dalFile, "w")) == (FILE *)NULL) {
	eMsgType = HdEMsg_DalFileOpenError;
	goto Error_Rtn;
    }

    fprintf(fp, HeaderOfDalFile);
    prnCurrTime(fp);

    if (hdPutsDalInfo(hdCtlInfo, fp, hdCtlInfo->machInfo) == HdFalse) {
	eMsgType = NULL;
	hdPrintf("Error in hdPutsDalInfo.\n");
	goto Close_Error_Rtn;
    }

    fclose(fp);

    return HdTrue;

 Close_Error_Rtn:
    fclose(fp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	Write machine name to the dal file - helios.dal
 */
static HdBoolean hdPutsDalInfo(hdCtlInfo, fp, machInfo)
    HeliosdCtlInfo	*hdCtlInfo;
    FILE		*fp;
    HMachInfo		machInfo;
{
    HMachInfo		info;

    for (info = machInfo; info != NULL; info = info->next) {

	if (fprintf(fp, "%s\n", info->hostName) == EOF) {
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    return HdFalse;
}

/*
 * Get full path name of heliosd;
 *	<full path> + '/'
 */
static char *hdGetHeliosdFullPath(heliosdName)
    char	*heliosdName;
{
    char	*heliosdPath;
    char	*p;
    char	tempPath[MAXPATHLEN];
    char	hdPathName[MAXPATHLEN];

#ifdef _PATH_DEBUG
    fprintf(stderr, "heliosdName : %s\n", heliosdName);
#endif

    strcpy(tempPath, heliosdName);
    if ((p = strrchr(tempPath, '/')) != NULL) {
	*p = NULL;
    } else {
	if (heliosdPathSearch(heliosdName, tempPath) == HdFalse)
	  return NULL;
    }

    if (realpath(tempPath, hdPathName) == NULL) {
	hdPerror();
	fprintf(stderr, "path name of heliosd : %s\n", heliosdName);
	return NULL;
    }

    if ((p = strrchr(hdPathName, '/')) != NULL) {
	*p = NULL;
	sprintf(tempPath, "%s/config/", hdPathName);

	if ((heliosdPath = HdMallocString(tempPath)) != NULL) {

#ifdef _PATH_DEBUG
    fprintf(stderr, "heliosdPath : %s\n", heliosdPath);
#endif
	    return heliosdPath;
	}
    }
    return NULL;
}

/*
 * Get full path name for a program name;
 *	<full path> + '/'
 */
static HdBoolean heliosdPathSearch(hdProgName, hdPathName)
    char		*hdProgName;
    char		*hdPathName;
{
    char		*hdPathEnv;
    char		*p1, *p2;
    char		*pathName = NULL;
    DIR			*dirp;
    DIR			*tempDirp;
    struct dirent	*dp;
    char		fileName[HeliosdFilePathLen_Lim];

#ifdef _PATH_DEBUG
	fprintf(stderr, "hdProgName : %s\n", hdProgName);
#endif

    if ((hdPathEnv = HdMallocString(getenv("PATH"))) == NULL)
      return HdFalse;

    for (p1 = p2 = hdPathEnv; *p1 != NULL; p1 = p2) {
	for (p2 = p1; *p2 != NULL && *p2 != ':'; p2++)
	  ;

	if (*p2 != NULL)
	  *p2++ = '\0';

#ifdef _PATH_DEBUG
	fprintf(stderr, "pathName in PathSearch : %s\n", p1);
#endif

	if ((dirp = opendir(p1)) != NULL) {

	    while ((dp = readdir(dirp)) != NULL) {
		if (strcmp(dp->d_name, hdProgName) == 0) {

		    sprintf(fileName, "%s/%s", p1, hdProgName);
		    if ((tempDirp = opendir(fileName)) != NULL) {
#ifdef _PATH_DEBUG
			fprintf(stderr, "fileName in PathSearch : %s\n", fileName);
#endif
			closedir(tempDirp);
			continue;
		    }
#ifdef _PATH_DEBUG
		    fprintf(stderr, "pathName in PathSearch : %s\n", p1);
#endif
		    pathName = p1;
		    break;
		}
	    }
	    closedir(dirp);

	    if (dp != NULL && pathName != NULL) {
		sprintf(hdPathName, "%s", pathName);
		break;
	    }
	}
    }

    free(hdPathEnv);
    return (pathName != NULL) ? HdTrue : HdFalse;
}

/*
 *	$B%^%7%s>pJs$N3MF@(B
 */
static HdBoolean hdGetMachInfo(hdCtlInfo, hMachInfo)
    HeliosdCtlInfo	*hdCtlInfo;
    HMachInfo		*hMachInfo;
{
    char		hostname[HostNameLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    /* $B%5!<%P$r5/F0$7$?%^%7%s$N%[%9%HL>(B */
    if (gethostname(hostname, HostNameLen_Lim) == ERROR_FLAG) {
	hdPerror();
	eMsgType = HdEMsg_NotFindHostName;
	goto Error_Rtn;
    }

#ifdef _DEBUG_HELIOSD_HOSTNAME
    fprintf(stderr, "host name : %s\n", hostname);
#endif /* _DEBUG_HELIOSD */

    if ((*hMachInfo = hdMakeDataOfMachInfo(hostname)) == NULL) {
	eMsgType = HdEMsg_MakeDataOfMachInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	Make data of MachInfo structure
 */
static HMachInfo hdMakeDataOfMachInfo(hostname)
    char		*hostname;
{
    HMachInfo		machInfo;
    struct hostent	*myhost;
    struct sockaddr_in	*sockAddr;
    HeliosdErrorMsgType	eMsgType;


    if ((myhost = gethostbyname(hostname)) == NULL) {
	eMsgType = HdEMsg_BadHostName;
	goto Error_Rtn;
    }

    if ((machInfo = (HMachInfo)HdMalloc(sizeof(MachInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    if ((machInfo->hostName = HdMallocString(hostname)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    machInfo->addressSize = sizeof(machInfo->sockAddress);
    sockAddr = &(machInfo->sockAddress);

    bzero((char *)sockAddr, machInfo->addressSize);
    sockAddr->sin_family = AF_INET;

    bcopy(myhost->h_addr, (char *)&(sockAddr->sin_addr), myhost->h_length);

#ifdef _Heliosd_Static_Port_Num

    sockAddr->sin_port = HeliosdPortNum;

#else

    sockAddr->sin_port = 0;
    sockAddr->sin_addr.s_addr = INADDR_ANY;

#endif

    return machInfo;

 Error_Rtn:
    hdFreeMachInfo(&machInfo);

    hdPutsErrorMsg(eMsgType);
    return NULL;
}

static void prnCurrTime(fp)
    FILE	*fp;
{
    time_t      clock;
    struct tm   *tm;
    char	buf[AgentNameLen_Lim];

    time(&clock);
    tm = localtime(&clock);

    strftime(buf, AgentNameLen_Lim, "at %T on %A, %B %d in %Y", tm);

    fprintf(fp, "#\t%s.\n\n", buf);
}

/*
 *
 */
static void hdErrorExit(eMsgType)
    HeliosdErrorMsgType	eMsgType;
{
    hdPutsErrorMsg(eMsgType);
    exit(eMsgType);
}
