/* hiagDirInfo.c
 *	module to handle directory informations for agent processes.
 *		created by k3sato on March15th in '95.
 *
 * $Id: hiagDirInfo.c,v 1.0 1995/03/31 11:29:57 k3sato Exp k3sato $
 */

#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <sys/param.h>
#include <string.h>
#include <fcntl.h>
#include <memory.h>

#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>

#include "hiag.h"
#include "hiagAgentInfo.h"
#include "hiagMakefile.h"
#include "hiagCommon.h"
#include "hdcommon.h"

char *getcwd();

static char *hdGetConfigFullPath();
static HdBoolean heliosdPathSearch();;
static char *hdGetCurrentDir();;
static HdBoolean hdCheckAgentDir();;
static HdBoolean heliosdPathSearch();
static HdBoolean hdGetHeliosDir();
static char *hdGetCurrentDir();
static HdBoolean hdCheckAgentDir();
static HdBoolean hdCheckDirPath();
static HdBoolean hdGetAllAppDirName();
static HdBoolean hdMakeCmdDir();

/*
 *  setup directories in HELIOSDIR/app
 */
HdBoolean hdSetupDirInfo(hiagCtlInfo, argc, argv)
    HiagCtlInfo		*hiagCtlInfo;
    int			argc;
    char		*argv[];
{
    HeliosdCtlInfo	*hdCtlInfo;
    HeliosdErrorMsgType	eMsgType;

    hdCtlInfo = hiagCtlInfo->hdCtlInfo;

    if ((hdCtlInfo->fullPath = hdGetConfigFullPath(argv[0])) == NULL) {
	eMsgType = HdEMsg_GetConfigFullPathError;
	goto Error_Rtn;
    }

    if (hdGetHeliosDir(hiagCtlInfo) == HdFalse) {
	eMsgType = HdEMsg_GetHeliosDirError;
	goto Error_Rtn;
    }

    if ((hiagCtlInfo->currDir = hdGetCurrentDir()) == NULL) {
	eMsgType = HdEMsg_GetCurrentDirError;
	goto Error_Rtn;
    }

    if (hdCheckAgentDir(hiagCtlInfo->currDir) == HdTrue) {

	hiagCtlInfo->type = HiagPT_Single;
	if ((hiagCtlInfo->appDirName = HdMalloc(sizeof(HiagNameInfo)))
	    == NULL) {
	    eMsgType = HdEMsg_HdMallocError;
	    goto Error_Rtn;
	}
	hiagCtlInfo->appDirName->name = hiagCtlInfo->currDir;

    } else {
	hiagCtlInfo->type = HiagPT_All;
	if (hdGetAllAppDirName(hiagCtlInfo) == HdFalse) {
	    eMsgType = HdEMsg_GetAllAppDirNameError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Get full path name of heliosd;
 *	<full path> + '/'
 */
static char *hdGetConfigFullPath(heliosdName)
    char	*heliosdName;
{
    char	*heliosdPath;
    char	*p;
    char	tempPath[MAXPATHLEN];
    char	hdPathName[MAXPATHLEN];

#ifdef _PATH_DEBUG
    fprintf(stderr, "heliosdName : %s\n", heliosdName);
#endif

    strcpy(tempPath, heliosdName);
    if ((p = strrchr(tempPath, '/')) != NULL) {
	*p = NULL;
    } else {
	if (heliosdPathSearch(heliosdName, tempPath) == HdFalse)
	  return NULL;
    }

    if (realpath(tempPath, hdPathName) == NULL) {
	hdPerror();
	fprintf(stderr, "path name of heliosd : %s\n", heliosdName);
	return NULL;
    }

    if ((p = strrchr(hdPathName, '/')) != NULL) {
	*p = NULL;
	sprintf(tempPath, "%s/config/", hdPathName);

	if ((heliosdPath = HdMallocString(tempPath)) != NULL) {

#ifdef _PATH_DEBUG
    fprintf(stderr, "heliosdPath : %s\n", heliosdPath);
#endif
	    return heliosdPath;
	}
    }
    return NULL;
}

/*
 * Get full path name for a program name;
 *	<full path> + '/'
 */
static HdBoolean heliosdPathSearch(hdProgName, hdPathName)
    char		*hdProgName;
    char		*hdPathName;
{
    char		*hdPathEnv;
    char		*p1, *p2;
    char		*pathName = NULL;
    DIR			*dirp;
    DIR			*tempDirp;
    struct dirent	*dp;
    char		fileName[HeliosdFilePathLen_Lim];

#ifdef _PATH_DEBUG
	fprintf(stderr, "hdProgName : %s\n", hdProgName);
#endif

    if ((hdPathEnv = HdMallocString(getenv("PATH"))) == NULL)
      return HdFalse;

    for (p1 = p2 = hdPathEnv; *p1 != NULL; p1 = p2) {
	for (p2 = p1; *p2 != NULL && *p2 != ':'; p2++)
	  ;

	if (*p2 != NULL)
	  *p2++ = '\0';

#ifdef _PATH_DEBUG
	fprintf(stderr, "pathName in PathSearch : %s\n", p1);
#endif

	if ((dirp = opendir(p1)) != NULL) {

	    while ((dp = readdir(dirp)) != NULL) {
		if (strcmp(dp->d_name, hdProgName) == 0) {

		    sprintf(fileName, "%s/%s", p1, hdProgName);
		    if ((tempDirp = opendir(fileName)) != NULL) {
#ifdef _PATH_DEBUG
			fprintf(stderr, "fileName in PathSearch : %s\n", fileName);
#endif
			closedir(tempDirp);
			continue;
		    }
#ifdef _PATH_DEBUG
		    fprintf(stderr, "pathName in PathSearch : %s\n", p1);
#endif
		    pathName = p1;
		    break;
		}
	    }
	    closedir(dirp);

	    if (dp != NULL && pathName != NULL) {
		sprintf(hdPathName, "%s", pathName);
		break;
	    }
	}
    }

    free(hdPathEnv);
    return (pathName != NULL) ? HdTrue : HdFalse;
}

static HdBoolean hdGetHeliosDir(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    char		*strPtr;
    char		strBuf[HeliosdFilePathLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if ((strPtr = getenv("HELIOSDIR")) == NULL) {

	strcpy(strBuf, hiagCtlInfo->hdCtlInfo->fullPath);
	if ((strPtr = strrchr(strBuf, HdChar_SlantCode)) == NULL) {
	    eMsgType = HdEMsg_ErrorInFullPath;
	    goto Error_Rtn;
	}

	*strPtr = HdChar_NULL;

	if ((strPtr = strrchr(strBuf, HdChar_SlantCode)) == NULL) {
	    eMsgType = HdEMsg_ErrorInFullPath;
	    goto Error_Rtn;
	}

	*strPtr = HdChar_NULL;

	strPtr = strBuf;
    }

    if ((hiagCtlInfo->heliosDir = HdMallocString(strPtr)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static char *hdGetCurrentDir()
{
    char	*currDir;
    char	*cwd;

    if ((cwd = getcwd(NULL, HeliosdFilePathLen_Lim)) == NULL) {
	hdPerror();
	currDir = NULL;
    } else {
	printf("current directory :%s\n", cwd);
	currDir = HdMallocString(cwd);
    }

    return currDir;
}

static HdBoolean hdCheckAgentDir(dirPath)
    char		*dirPath;
{
    DIR			*dirp;
    struct dirent	*dp;
    HdBoolean		evlFlag = HdFalse;
    HdBoolean		cplFlag = HdFalse;
    char		fileName[HeliosdFilePathLen_Lim];

    if ((dirp = opendir(dirPath)) == NULL) {
	goto Error_Rtn;
    }

    while ((dp = readdir(dirp)) != NULL
	   && (evlFlag == HdFalse || cplFlag == HdFalse)) {

	if (evlFlag == HdFalse 
	    && strcmp(dp->d_name, HiagEvlDirName) == 0) {

	    sprintf(fileName, "%s/%s", dirPath, HiagEvlDirName);
	    if (hdCheckDirPath(fileName) == HdTrue) {
		evlFlag = HdTrue;
	    }
	}

	if (cplFlag == HdFalse 
	    && strcmp(dp->d_name, HiagCplDirName) == 0) {

	    sprintf(fileName, "%s/%s", dirPath, HiagCplDirName);
	    if (hdCheckDirPath(fileName) == HdTrue) {
		cplFlag = HdTrue;
	    }
	}
    }

    closedir(dirp);

    return evlFlag & cplFlag;

 Error_Rtn:
    return HdFalse;
}

static HdBoolean hdCheckDirPath(dirPath)
    char	*dirPath;
{
    DIR		*tempDirp;
    HdBoolean	flag;

    if ((tempDirp = opendir(dirPath)) != NULL) {
	flag = HdTrue;
	closedir(tempDirp);
    } else {
	flag = HdFalse;
    }

    return flag;
}

static HdBoolean hdGetAllAppDirName(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    DIR			*dirp;
    struct dirent	*dp;
    HHiagNameInfo	*currInfoP;
    char		appDir[HeliosdFilePathLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    sprintf(appDir, "%s/%s", hiagCtlInfo->heliosDir, HiagAppTopDirName);

    if ((dirp = opendir(appDir)) == NULL) {
	eMsgType = HdEMsg_DisableOpenAppDir;
	goto Error_Rtn;
    }

    for (currInfoP = &(hiagCtlInfo->appDirName)
	 ; (dp = readdir(dirp)) != NULL
	 ;) {

	if (hdCheckAgentDir(dp->d_name) == HdFalse) {
	    continue;
	}

	if ((*currInfoP = HdMalloc(sizeof(HiagNameInfo))) == NULL) {
	    eMsgType = HdEMsg_HdMallocError;
	    goto Close_Error_Rtn;
	}

	if (((*currInfoP)->name = HdMallocString(dp->d_name)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Close_Error_Rtn;
	}
	currInfoP = &((*currInfoP)->next);
    }

    closedir(dirp);

    return HdTrue;

 Close_Error_Rtn:
    closedir(dirp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}


HdBoolean hdSetupCmdDir(hiagCtlInfo, agentId, cplInfo)
    HiagCtlInfo		*hiagCtlInfo;
    char		*agentId;
    HiagCplInfo		*cplInfo;
{
    HeliosdErrorMsgType	eMsgType;

    if (hdMakeCmdDir(hiagCtlInfo, cplInfo->agentName) == HdFalse) {
	eMsgType = HdEMsg_MakeCmdDirError;
	goto Error_Rtn;
    }

    if (hdSetupAgentTblInCmdDir(hiagCtlInfo, agentId, cplInfo) == HdFalse) {
	eMsgType = HdEMsg_SetupAgentTblInCmdDirError;
	goto Error_Rtn;
    }

    if (hdSetupMakefileInCmdDir(hiagCtlInfo, cplInfo->agentName) == HdFalse) {
	eMsgType = HdEMsg_SetupAgentTblInCmdDirError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdMakeCmdDir(hiagCtlInfo, agentName)
    HiagCtlInfo		*hiagCtlInfo;
    char		*agentName;
{
    char		dirPath[HeliosdFilePathLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    sprintf(dirPath, "%s/cmd_%s", hiagCtlInfo->currDir, agentName);

    if (mkdir(dirPath, HiagDirFileMode) == -1) {
	hdPerror();
	if (errno != EEXIST) {
	    eMsgType = HdEMsg_MkdirError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType, dirPath);
    return HdFalse;
}
