/* hiagAgentInfo.c
 *	module to handle agent information;
 *	especially *.cpl and *.evl files.
 *		created by k3sato on March17th in '95.
 *
 * $Id: hiagAgentInfo.c,v 1.1 1995/03/31 12:11:32 k3sato Exp $
 */

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <dirent.h>

#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>

#include "hiag.h"
#include "hiagAgentInfo.h"
#include "hiagDirInfo.h"
#include "hiagKeyWord.h"
#include "hiagCommon.h"
#include "hdcommon.h"

static HdBoolean hdUpdateHiaForOneApp();
static HdBoolean hdGetMaxAgentId();
static HdBoolean hdGetCplEvlInfo();
static HdBoolean hdGetCplInfoFromFile();
static HdBoolean hdCheckCplFile();
static HdBoolean hdGetOneCplInfo();
static HdBoolean hdGetCplKeyTypeValue();
static HdBoolean hdChgTypeName2AgentType();
static HdBoolean hdGetEvlInfoFromFile();
static HdBoolean hdCheckEvlFile();
static HdBoolean hdGetOneEvlInfo();
static HdBoolean hdGetEvlKeyTypeValue();
static HdBoolean hdChgTypeName2AgentType();
static HdBoolean hiagSetupAgentDir2EvlInfo();
static HdBoolean hdSetupCplEvlInfo();
static HdBoolean hdSetupTopCplInfo();
static HdBoolean hdLinkCplWithEvl();
static HdBoolean hdLinkCplWithInnerEvl();
static HiagCplInfo *hdGetCplInfoForName();
static HdBoolean hdUpdateHiaWithCplEvlInfo();
static HAgentInfo hdSearchAgentInfoForName();
static HdBoolean hdAppendAgentInfoWithCplEvlInfo();
static HdBoolean hdAppendOneAgentInfo();
static HdBoolean hdFwriteAgentInfo2Hia();
static HAgentInfo hdGetLastAgentInfo();
static char *hdGetAgentDirNameList();
static HdBoolean hdGetKeyWordValue();
static HdBoolean hdFileGetString();
static char *hdGetOneWord(); 
static char *hdSkipSpace(); 

/*
 * $B%U%!%$%k(B *.cpl, *.evl $B$+$iF@$?>pJs$r85$K!"(B
 * hia $B%U%!%$%k$r99?7$9$k!#(B
 */
HdBoolean hdUpdateHiaInfo(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    HiagNameInfo	*appDirName;
    HeliosdErrorMsgType	eMsgType;

    if (hdGetMaxAgentId(hiagCtlInfo) == HdFalse) {
	eMsgType = HdEMsg_GetMaxAgentIdError;
	goto Error_Rtn;
    }

    for (appDirName = hiagCtlInfo->appDirName
	 ; appDirName != NULL
	 ; appDirName = appDirName->next) {
	
	hiagCtlInfo->currDir = appDirName->name;

	if (hdUpdateHiaForOneApp(hiagCtlInfo) == HdFalse) {
	    eMsgType = HdEMsg_UpdateHiaForOneAppError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdGetMaxAgentId(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		agentInfo;
    int			maxAgentId;
    char		*strPtr;
    char		agentId[AgentIDLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    hdCtlInfo = hiagCtlInfo->hdCtlInfo;

    if ((agentInfo = hdCtlInfo->allAgents) == NULL) {
	hiagCtlInfo->maxAgentId = 1;
    } else {
	for (; agentInfo != NULL; agentInfo = agentInfo->next) {
	    strcpy(agentId, agentInfo->agentId);

	    if ((strPtr = strchr(agentId, HdChar_Period)) == NULL) {
		eMsgType = HdEMsg_IllegalAgentId;
		goto Error_Rtn;
	    }
	    maxAgentId = atoi(strPtr);

	    if (maxAgentId > hiagCtlInfo->maxAgentId) {
		hiagCtlInfo->maxAgentId = maxAgentId;
	    }
	}
	hiagCtlInfo->maxAgentId++;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * $B#1$D$N%"%W%j%1!<%7%g%s$N%U%!%$%k(B *.cpl, *.evl $B$+$iF@$?>pJs$r85$K!"(B
 * hia $B%U%!%$%k$r99?7$9$k!#(B
 */
static HdBoolean hdUpdateHiaForOneApp(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    HeliosdErrorMsgType	eMsgType;

    if (hdGetCplEvlInfo(hiagCtlInfo) == HdFalse) {
	eMsgType = HdEMsg_GetCplEvlInfoError;
	goto Error_Rtn;
    }

    if (hdUpdateHiaWithCplEvlInfo(hiagCtlInfo) == HdFalse) {
	eMsgType = HdEMsg_UpdateHiaWithCplEvlInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * $B%U%!%$%k(B *.cpl, *.evl $B$+$i(B hia $B$KI,MW$J>pJs$N3MF@(B
 */
static HdBoolean hdGetCplEvlInfo(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    char		*appDir;
    HeliosdErrorMsgType	eMsgType;

    appDir = hiagCtlInfo->currDir;

    if (hdGetCplInfoFromFile(hiagCtlInfo, appDir) == HdFalse) {
	eMsgType = HdEMsg_GetCplInfoFromFileError;
	goto Error_Rtn;
    }

    if (hdGetEvlInfoFromFile(hiagCtlInfo, appDir) == HdFalse) {
	eMsgType = HdEMsg_GetEvlInfoFromFileError;
	goto Error_Rtn;
    }

    if (hdSetupCplEvlInfo(hiagCtlInfo) == HdFalse) {
	eMsgType = HdEMsg_SetupCplEvlInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdGetCplInfoFromFile(hiagCtlInfo, appDir)
    HiagCtlInfo		*hiagCtlInfo;
    char		*appDir;
{
    DIR			*dirp;
    struct dirent	*dp;
    HHiagCplInfo	*currInfoP;
    char		cplDir[HeliosdFilePathLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    sprintf(cplDir, "%s/%s", appDir, HiagCplDirName);

    if ((dirp = opendir(cplDir)) == NULL) {
	eMsgType = HdEMsg_DisableOpenCplDir;
	goto Error_Rtn;
    }

    for (currInfoP = &(hiagCtlInfo->cplInfo) ; (dp = readdir(dirp)) != NULL; ) {

	if (hdCheckCplFile(dp->d_name) == HdFalse) {
	    continue;
	}

	if (hdGetOneCplInfo(hiagCtlInfo, cplDir, dp->d_name, currInfoP)
	    == HdFalse) {
	    eMsgType = HdEMsg_GetOneCplInfoError;
	    goto Close_Error_Rtn;
	}
	currInfoP = &((*currInfoP)->next);
    }

    closedir(dirp);

    return HdTrue;

 Close_Error_Rtn:
    closedir(dirp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdCheckCplFile(fileName)
    char	*fileName;
{
    char	*strPtr;
    HeliosdErrorMsgType	eMsgType;

    if ((strPtr = strrchr(fileName, HdChar_Period)) == NULL) {
	eMsgType = HdEMsg_IsNotCplFile;
	goto Error_Rtn;
    }

    if (*(++strPtr) == HdChar_NULL) {
	eMsgType = HdEMsg_IsNotCplFile;
	goto Error_Rtn;
    }

    if (strcmp(strPtr, HiagCplFileDescriptor) != 0) {
	eMsgType = HdEMsg_IsNotCplFile;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType, fileName);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdGetOneCplInfo(hiagCtlInfo, cplDir, cplName, cplInfoP)
    HiagCtlInfo		*hiagCtlInfo;
    char		*cplDir;
    char		*cplName;
    HHiagCplInfo	*cplInfoP;
{
    unsigned int	flag;
    FILE		*fp;
    char		fileName[HeliosdFilePathLen_Lim];
    HdCplKeyWordType	keyType;
    char		value[HiagValueBufLen_lim];
    char		insideEnv[AgentNameLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    sprintf(fileName, "%s/%s", cplDir, cplName);

    if ((fp = fopen(fileName, "r")) == NULL) {
	eMsgType = HdEMsg_CplFileOpenError;
	goto Error_Rtn;
    }

    if ((*cplInfoP = HdMalloc(sizeof(HiagCplInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Close_Error_Rtn;
    }

    for (flag = 0; flag != HCplInfo_All;) {
	if (hdGetCplKeyTypeValue(fp, &keyType, value) == HdFalse) {

	    if ((flag & HCplInfo_Type)
		&& (flag & HCplInfo_Agent)
		&& (flag & HCplInfo_Env)) {
		break;
	    }
	    eMsgType = HdEMsg_GetCplKeyTypeValueError;
	    goto Close_Error_Rtn;
	}

	switch (keyType) {
	case HdCplKW_Type:
	    if ((flag & HCplInfo_Type) != NULL) {
		eMsgType = HdEMsg_AlreadyGetTypeFromCpl;
		goto Close_Error_Rtn;
	    }

	    flag |= HCplInfo_Type;

	    if (hdChgTypeName2AgentType(value, &((*cplInfoP)->type)) == HdFalse) {
		eMsgType = HdEMsg_ChgTypeName2AgentTypeError;
		goto Close_Error_Rtn;
	    }
	    break;

	case HdCplKW_Agent:
	    if ((flag & HCplInfo_Agent) != NULL) {
		eMsgType = HdEMsg_AlreadyGetAgentFromCpl;
		goto Close_Error_Rtn;
	    }

	    flag |= HCplInfo_Agent;

	    if (((*cplInfoP)->agentName = HdMallocString(value)) == NULL) {
		eMsgType = HdEMsg_HdMallocStringError;
		goto Close_Error_Rtn;
	    }
	    break;

	case HdCplKW_Env:
	    if ((flag & HCplInfo_Env) != NULL) {
		eMsgType = HdEMsg_AlreadyGetEnvFromCpl;
		goto Close_Error_Rtn;
	    }

	    flag |= HCplInfo_Env;

	    if (((*cplInfoP)->env = HdMallocString(value)) == NULL) {
		eMsgType = HdEMsg_HdMallocStringError;
		goto Close_Error_Rtn;
	    }
	    break;

	case HdCplKW_Inside:
	    if ((flag & HCplInfo_Inside) != NULL) {
		eMsgType = HdEMsg_AlreadyGetInsideFromCpl;
		goto Close_Error_Rtn;
	    }

	    flag |= HCplInfo_Inside;

	    if ((*cplInfoP)->type == HdAT_Complex) {

		if (hdGetOneWord(insideEnv, value) == NULL) {
		    eMsgType = HdEMsg_GetOneWordError;
		    goto Error_Rtn;
		}

		if (((*cplInfoP)->inside = HdMallocString(insideEnv)) == NULL) {
		    eMsgType = HdEMsg_HdMallocStringError;
		    goto Close_Error_Rtn;
		}
	    }
	    break;

	case HdCplKW_ImpMethod:
	    if ((flag & HCplInfo_ImpMethod) != NULL) {
		eMsgType = HdEMsg_AlreadyGetImpMethodFromCpl;
		goto Close_Error_Rtn;
	    }

	    flag |= HCplInfo_ImpMethod;
	    break;

	case HdCplKW_ExpMethod:
	    if ((flag & HCplInfo_ExpMethod) != NULL) {
		eMsgType = HdEMsg_AlreadyGetExpMethodFromCpl;
		goto Close_Error_Rtn;
	    }

	    flag |= HCplInfo_ExpMethod;
	    break;

	case HdEvlKW_NULL:
	default:
	    break;
	}
    }

    fclose(fp);

    return HdTrue;

 Close_Error_Rtn:
    fclose(fp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType, fileName);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdGetCplKeyTypeValue(fp, type, value)
    FILE		*fp;
    HdCplKeyWordType	*type;
    char		*value;
{
    char		keyWord[HiagKeyWordBufLen_lim];
    HeliosdErrorMsgType	eMsgType;

    if (hdGetKeyWordValue(fp, keyWord, value) == HdFalse) {
	eMsgType = HdEMsg_GetKeyWordValueError;
	goto False_Rtn;
    }

    if (keyWord[0] != HdChar_Ampersand) {
	eMsgType = HdEMsg_NotCplKeyWord;
	goto Error_Rtn;
    }

    *type = hdCheckKeyWord(HdCplKeyList, keyWord + 1);

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);

 False_Rtn:
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdChgTypeName2AgentType(name, typeP)
    char	*name;
    HdAgentType	*typeP;
{
    switch (hdCheckKeyWord(HdAgentTypeNameList, name)) {
    case HdAgentTypeKW_Complex:
	*typeP = HdAT_Complex;
	break;

    case HdAgentTypeKW_Single:
	*typeP = HdAT_Single;
	break;

    case HdAgentTypeKW_Parametric:
	*typeP = HdAT_Parametric;
	break;

    case HdAgentTypeKW_SinglePara:
	*typeP = HdAT_ParaInstance;
	break;

    case HdAgentTypeKW_NULL:
    default:
	*typeP = HdAT_NotDefined;
	break;
    }

    return HdTrue;
}

/*
 *
 */
static HdBoolean hdGetEvlInfoFromFile(hiagCtlInfo, appDir)
    HiagCtlInfo		*hiagCtlInfo;
    char		*appDir;
{
    DIR			*dirp;
    struct dirent	*dp;
    HHiagEvlInfo	*currInfoP;
    char		evlDir[HeliosdFilePathLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    sprintf(evlDir, "%s/%s", appDir, HiagEvlDirName);

    if ((dirp = opendir(evlDir)) == NULL) {
	eMsgType = HdEMsg_DisableOpenEvlDir;
	goto Error_Rtn;
    }

    for (currInfoP = &(hiagCtlInfo->evlInfo); (dp = readdir(dirp)) != NULL; ) {

	if (hdCheckEvlFile(dp->d_name) == HdFalse) {
	    continue;
	}

	if (hdGetOneEvlInfo(hiagCtlInfo, evlDir, dp->d_name, currInfoP)
	    == HdFalse) {
	    eMsgType = HdEMsg_GetOneEvlInfoError;
	    goto Close_Error_Rtn;
	}
	currInfoP = &((*currInfoP)->next);
    }

    closedir(dirp);

    return HdTrue;

 Close_Error_Rtn:
    closedir(dirp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdCheckEvlFile(fileName)
    char	*fileName;
{
    char	*strPtr;
    HeliosdErrorMsgType	eMsgType;

    if ((strPtr = strrchr(fileName, HdChar_Period)) == NULL) {
	eMsgType = HdEMsg_IsNotEvlFile;
	goto Error_Rtn;
    }

    if (*(++strPtr) == HdChar_NULL) {
	eMsgType = HdEMsg_IsNotEvlFile;
	goto Error_Rtn;
    }

    if (strcmp(strPtr, HiagEvlFileDescriptor) != 0) {
	eMsgType = HdEMsg_IsNotEvlFile;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType, fileName);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdGetOneEvlInfo(hiagCtlInfo, evlDir, evlName, evlInfoP)
    HiagCtlInfo		*hiagCtlInfo;
    char		*evlDir;
    char		*evlName;
    HHiagEvlInfo	*evlInfoP;
{
    unsigned int	flag;
    FILE		*fp;
    char		*strPtr;
    char		fileName[HeliosdFilePathLen_Lim];
    HdEvlKeyWordType	keyType;
    char		value[HiagValueBufLen_lim];
    HeliosdErrorMsgType	eMsgType;

    sprintf(fileName, "%s/%s", evlDir, evlName);

    if ((fp = fopen(fileName, "r")) == NULL) {
	eMsgType = HdEMsg_EvlFileOpenError;
	goto Error_Rtn;
    }

    if ((*evlInfoP = HdMalloc(sizeof(HiagEvlInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Close_Error_Rtn;
    }

    for (flag = 0; flag != HEvlInfo_All; ) {
	if (hdGetEvlKeyTypeValue(fp, &keyType, value) == HdFalse) {

	    if ((flag & HEvlInfo_Environ)
		&& (flag & HEvlInfo_AgentDir)) {
		break;
	    }

	    eMsgType = HdEMsg_GetEvlKeyTypeValueError;
	    goto Close_Error_Rtn;
	}

	switch (keyType) {
	case HdEvlKW_Environment:
	    if ((flag & HEvlInfo_Environ) != NULL) {
		eMsgType = HdEMsg_AlreadyGetEnvironmentFromEvl;
		goto Close_Error_Rtn;
	    }

	    flag |= HEvlInfo_Environ;
	    
	    if (*value == HdChar_NULL) {
		strPtr = HiagTopEvlName;
	    }else {
		for (strPtr = value
		     ; *strPtr != HdChar_NULL
		     && *strPtr != HdChar_Space
		     && *strPtr != HdChar_Tab
		     ; strPtr++)
		  ;
		*strPtr = HdChar_NULL;

		strPtr = value;
	    }

	    if (((*evlInfoP)->environment = HdMallocString(strPtr)) == NULL) {
		eMsgType = HdEMsg_HdMallocStringError;
		goto Close_Error_Rtn;
	    }
	    break;

	case HdEvlKW_AgentDir:
	    if ((flag & HEvlInfo_AgentDir) != NULL) {
		eMsgType = HdEMsg_AlreadyGetAgentDirFromEvl;
		goto Close_Error_Rtn;
	    }

	    flag |= HEvlInfo_AgentDir;

	    if (hiagSetupAgentDir2EvlInfo(&((*evlInfoP)->agentDir), value) == HdFalse) {
		eMsgType = HdEMsg_SetupAgentDir2EvlInfoError;
		goto Close_Error_Rtn;
	    }
	    break;

	case HdEvlKW_FuncDir:
	    if ((flag & HEvlInfo_FuncDir) != NULL) {
		eMsgType = HdEMsg_AlreadyGetFuncDirFromEvl;
		goto Close_Error_Rtn;
	    }

	    flag |= HEvlInfo_FuncDir;

	    if (((*evlInfoP)->funcDir = HdMallocString(value)) == NULL) {
		eMsgType = HdEMsg_HdMallocStringError;
		goto Close_Error_Rtn;
	    }
	    break;

	case HdEvlKW_MethodDir:
	    if ((flag & HEvlInfo_MethodDir) != NULL) {
		eMsgType = HdEMsg_AlreadyGetMethodDirFromEvl;
		goto Close_Error_Rtn;
	    }

	    flag |= HEvlInfo_MethodDir;

	    if (((*evlInfoP)->methodDir = HdMallocString(value)) == NULL) {
		eMsgType = HdEMsg_HdMallocStringError;
		goto Close_Error_Rtn;
	    }
	    break;

	case HdEvlKW_NULL:
	default:
	    break;
	}
    }

    fclose(fp);

    return HdTrue;

 Close_Error_Rtn:
    fclose(fp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdGetEvlKeyTypeValue(fp, type, value)
    FILE		*fp;
    HdEvlKeyWordType	*type;
    char		*value;
{
    char		keyWord[HiagKeyWordBufLen_lim];
    HeliosdErrorMsgType	eMsgType;

    if (hdGetKeyWordValue(fp, keyWord, value) == HdFalse) {
	eMsgType = HdEMsg_GetKeyWordValueError;
	goto False_Rtn;
    }

    if (keyWord[0] != HdChar_Ampersand) {
	eMsgType = HdEMsg_NotEvlKeyWord;
	goto Error_Rtn;
    }

    *type = hdCheckKeyWord(HdEvlKeyList, keyWord + 1);

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);

 False_Rtn:
    return HdFalse;
}

/*
 *
 */
static HdBoolean hiagSetupAgentDir2EvlInfo(agentDirP, value)
    HHiagNameInfo	*agentDirP;
    char		*value;
{
    HHiagNameInfo	*infoP;
    char		*strPtr;
    char		agentName[AgentNameLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    for (infoP = agentDirP, strPtr = value
	 ; *strPtr != HdChar_NULL
	 ; infoP = &((*infoP)->next)) {

	if ((strPtr = hdGetOneWord(agentName, strPtr)) == NULL) {
	    eMsgType = HdEMsg_GetOneWordError;
	    goto Error_Rtn;
	}

	if ((*infoP = HdMalloc(sizeof(HiagNameInfo))) == NULL) {
	    eMsgType = HdEMsg_HdMallocError;
	    goto Error_Rtn;
	}

	if (((*infoP)->name = HdMallocString(agentName)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdSetupCplEvlInfo(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    HeliosdErrorMsgType	eMsgType;

    if (hdSetupTopCplInfo(hiagCtlInfo) == HdFalse) {
	eMsgType = HdEMsg_SetupTopCplInfoError;
	goto Error_Rtn;
    }

    if (hdLinkCplWithEvl(hiagCtlInfo, hiagCtlInfo->topCplInfo) == HdFalse) {
	eMsgType = HdEMsg_LinkCplWithEvlError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdSetupTopCplInfo(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    HiagCplInfo		*cplInfo;
    HdTopEnvKeyWordType	type;

    for (cplInfo = hiagCtlInfo->cplInfo; cplInfo != NULL; cplInfo = cplInfo->next) {
	
	if ((type = hdCheckKeyWord(HdTopEvlKeyList, cplInfo->env)) == HdTEvlKW_User) {
	    hiagCtlInfo->topCplInfo = cplInfo;
	    return HdTrue;
	}
    }

    hdPutsErrorMsg(HdEMsg_NotExistTopCplInfo);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdLinkCplWithEvl(hiagCtlInfo, cplInfo)
    HiagCtlInfo		*hiagCtlInfo;
    HiagCplInfo		*cplInfo;
{
    HiagNameInfo	*agentDir;
    HiagEvlInfo		*innerEvl;
    HiagCplInfo		**cplInfoP;
    HeliosdErrorMsgType	eMsgType;

    if (cplInfo->type != HdAT_Complex) {
	goto Good_Rtn;
    }

    if (hdLinkCplWithInnerEvl(hiagCtlInfo, cplInfo) == HdFalse) {
	eMsgType = HdEMsg_LinkCplWithInnerEvlError;
	goto Error_Rtn;
    }

    innerEvl = cplInfo->innerEvl;
    for (agentDir = innerEvl->agentDir, cplInfoP = &(innerEvl->innerCpl)
	 ; agentDir != NULL
	 ; agentDir = agentDir->next, cplInfoP = &(cplInfo->brother)) {

	if ((cplInfo = hdGetCplInfoForName(hiagCtlInfo, agentDir->name)) == NULL) {
	    eMsgType = HdEMsg_GetCplInfoForNameError;
	    goto Error_Rtn;
	}

	if (strcmp(cplInfo->env, innerEvl->environment) != 0) {
	    eMsgType = HdEMsg_ConflictCplEnvWithEvlInfo;
	    goto Error_Rtn;
	}

	*cplInfoP = cplInfo;
	cplInfo->outerEvl = innerEvl;

	if (hdLinkCplWithEvl(hiagCtlInfo, cplInfo) == HdFalse) {
	    eMsgType = NULL;
	    goto Error_Rtn;
	}
    }

 Good_Rtn:
    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdLinkCplWithInnerEvl(hiagCtlInfo, cplInfo)
    HiagCtlInfo		*hiagCtlInfo;
    HiagCplInfo		*cplInfo;
{
    char		*innerEvl;
    HiagEvlInfo		*evlInfo;
    HeliosdErrorMsgType	eMsgType;

    if ((innerEvl = cplInfo->inside) == NULL) {
	eMsgType = HdEMsg_NotInnerEvlInCplInfo;
	goto Error_Rtn;
    }

    for (evlInfo = hiagCtlInfo->evlInfo; evlInfo != NULL; evlInfo = evlInfo->next) {

	if (strcmp(evlInfo->environment, innerEvl) == 0) {
	    cplInfo->innerEvl = evlInfo;
	    evlInfo->outerCpl = cplInfo;
	    goto Good_Rtn;
	}
    }

    eMsgType = HdEMsg_NotInnerEvlInCplInfo;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;

 Good_Rtn:
    return HdTrue;
}

/*
 *
 */
static HiagCplInfo *hdGetCplInfoForName(hiagCtlInfo, agentName)
    HiagCtlInfo	*hiagCtlInfo;
    char	*agentName;
{
    HiagCplInfo		*cplInfo;

    for (cplInfo = hiagCtlInfo->cplInfo; cplInfo != NULL; cplInfo = cplInfo->next) {
	if (strcmp(cplInfo->agentName, agentName) == 0) {
	    return cplInfo;
	}
    }

    return NULL;
}

/*
 *
 */
static HdBoolean hdUpdateHiaWithCplEvlInfo(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		agentInfo;
    char		hiaFileName[HeliosdFilePathLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    hdCtlInfo = hiagCtlInfo->hdCtlInfo;

    if ((agentInfo =
	 hdSearchAgentInfoForName(hdCtlInfo->allAgents,
				  hiagCtlInfo->topCplInfo->agentName))
	!= NULL) {
	goto Good_Rtn;
    }

    sprintf(hiaFileName, "%s%s", hdCtlInfo->fullPath, HeliosHiaFileName);

    if ((hiagCtlInfo->hiaFp = fopen(hiaFileName, "a+")) == NULL) {
	eMsgType = HdEMsg_HiaFileOpenError;
	goto Error_Rtn;
    }

    if (hdAppendAgentInfoWithCplEvlInfo(hiagCtlInfo) == HdFalse) {
	eMsgType = HdEMsg_AppendAgentInfoWithCplEvlInfoError;
	goto Close_Error_Rtn;
    }

    fclose(hiagCtlInfo->hiaFp);

 Good_Rtn:
    return HdTrue;

 Close_Error_Rtn:
    fclose(hiagCtlInfo->hiaFp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HAgentInfo hdSearchAgentInfoForName(agentInfo, agentName)
    HAgentInfo		agentInfo;
    char		*agentName;
{
    HAgentInfo		currInfo;

    for (currInfo = agentInfo; currInfo != NULL; currInfo = currInfo->next) {

	if (currInfo->agentName != NULL) {
	    if (strcmp(currInfo->agentName, agentName) == 0) {
		return currInfo;
	    }
	}
	if ((agentInfo = hdSearchAgentInfoForName(currInfo->child, agentName))
	    != NULL) {
	    return agentInfo;
	}
    }

    return NULL;
}

/*
 *
 */
static HdBoolean hdAppendAgentInfoWithCplEvlInfo(hiagCtlInfo)
    HiagCtlInfo		*hiagCtlInfo;
{
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		lastInfo;
    HAgentInfo		*infoP;
    char		agentId[AgentIDLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    hdCtlInfo = hiagCtlInfo->hdCtlInfo;

    if ((lastInfo = hdGetLastAgentInfo(hdCtlInfo->allAgents))
	== NULL) {
	infoP = &(hdCtlInfo->allAgents);
    } else {
	infoP = &(lastInfo->next);
    }

    sprintf(agentId, "%d.%d",
	    HiagAgentId_TopLevel, hiagCtlInfo->maxAgentId);

    if (hdAppendOneAgentInfo(hiagCtlInfo, infoP, agentId,
			     hiagCtlInfo->topCplInfo)
	== HdFalse) {
	eMsgType = HdEMsg_AppendOneAgentInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdAppendOneAgentInfo(hiagCtlInfo, agentInfoP, agentId, cplInfo)
    HiagCtlInfo		*hiagCtlInfo;
    HAgentInfo		*agentInfoP;
    char		*agentId;
    HiagCplInfo		*cplInfo;
{
    HAgentInfo		currInfo;
    int			currId;
    char		newId[AgentIDLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if (cplInfo == NULL) {
	goto Good_Rtn;
    }

    if ((currInfo =
	 *agentInfoP = HdMalloc(sizeof(AgentInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    if ((currInfo->agentName = HdMallocString(cplInfo->agentName))
	== NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    if ((currInfo->agentId = HdMallocString(agentId))
	== NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    if (hdFwriteAgentInfo2Hia(hiagCtlInfo, agentId, cplInfo) == HdFalse) {
	eMsgType = HdEMsg_FwriteAgentInfo2HiaError;
	goto Error_Rtn;
    }

    if (hdSetupCmdDir(hiagCtlInfo, agentId, cplInfo) == HdFalse) {
	eMsgType = HdEMsg_SetupCmdDirError;
	goto Error_Rtn;
    }

    if (cplInfo->innerEvl != NULL) {
	for (cplInfo = cplInfo->innerEvl->innerCpl, currId = HiagAgentId_StartNum
	     ; cplInfo != NULL
	     ; cplInfo = cplInfo->brother, currId++) {

	    sprintf(newId, "%s.%d", agentId, currId);

	    if (hdAppendOneAgentInfo(hiagCtlInfo, &(currInfo->child), newId,
				     cplInfo) == HdFalse) {
		eMsgType = HdEMsg_AppendOneAgentInfoError;
		goto Error_Rtn;
	    }
	}
    }

 Good_Rtn:
    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdFwriteAgentInfo2Hia(hiagCtlInfo, agentId, cplInfo)
    HiagCtlInfo		*hiagCtlInfo;
    char		*agentId;
    HiagCplInfo		*cplInfo;
{
    FILE		*fp;
    char		procPath[HeliosdFilePathLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    fp = hiagCtlInfo->hiaFp;

    if (fprintf(fp, "%s\t%s\t%d", agentId, cplInfo->agentName,
		(cplInfo->type == HdAT_Single) ? 0 :
		(cplInfo->type == HdAT_Complex) ? 1 : 2) == EOF) {
	eMsgType = HdEMsg_Fail2FprintfAgentId;
	goto Error_Rtn;
    }

    /* write env name
     */
    if (fprintf(fp, "\t%s \\\n", cplInfo->env) == EOF) {
	eMsgType = HdEMsg_Fail2FprintfEnvName;
	goto Error_Rtn;
    }

    sprintf(procPath, "\t%s/cmd_%s/%s",
	    hiagCtlInfo->currDir, cplInfo->agentName, cplInfo->agentName);

    /* write path of agent tbl file
     */
    if (fprintf(fp, "\t%s.tbl \\\n", procPath) == EOF) {
	eMsgType = HdEMsg_Fail2FprintfTblPath;
	goto Error_Rtn;
    }

    /* write path of exec file of agent process
     */
    if (fprintf(fp, "\t%s\n", procPath) == EOF) {
	eMsgType = HdEMsg_Fail2FprintfProcPath;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HAgentInfo hdGetLastAgentInfo(agentInfo)
    HAgentInfo		agentInfo;
{
    HAgentInfo		lastInfo;

    for (lastInfo = NULL
	 ; agentInfo != NULL
	 ; lastInfo = agentInfo, agentInfo = agentInfo->next)
      ;

    return lastInfo;
}

HdBoolean hdSetupAgentTblInCmdDir(hiagCtlInfo, agentId, cplInfo)
    HiagCtlInfo		*hiagCtlInfo;
    char		*agentId;
    HiagCplInfo		*cplInfo;
{
    FILE		*fp;
    char		tblPath[HeliosdFilePathLen_Lim];
    char		fileBuf[HiagValueBufLen_lim];
    HiagEvlInfo		*evlInfo;
    HeliosdErrorMsgType	eMsgType;

    sprintf(tblPath, "%s/cmd_%s/%s.tbl",
	    hiagCtlInfo->currDir, cplInfo->agentName, cplInfo->agentName);

    if ((fp = fopen(tblPath, "w")) == NULL) {
	eMsgType = HdEMsg_TblFileOpenError;
	goto Error_Rtn;
    }

    if (fprintf(fp, "# Agent Name\n%s\n", cplInfo->agentName) == EOF) {
	eMsgType = HdEMsg_Fail2FprintfAgentNameInTbl;
	goto Close_Error_Rtn;
    }

    if (fprintf(fp, "# Agent ID\n%s\n", agentId) == EOF) {
	eMsgType = HdEMsg_Fail2FprintfAgentIdInTbl;
	goto Close_Error_Rtn;
    }

    if (fprintf(fp, "# Agent Type\n%d\n",
		(cplInfo->type == HdAT_Single) ? 0 :
		(cplInfo->type == HdAT_Complex) ? 1 : 2) == EOF) {
	eMsgType = HdEMsg_Fail2FprintfAgentTypeInTbl;
	goto Close_Error_Rtn;
    }

    if (fprintf(fp, "# Comment\n%s\n", 
		(cplInfo->comment == NULL) ? "{}" : cplInfo->comment)
	== EOF) {
	eMsgType = HdEMsg_Fail2FprintfCommentInTbl;
	goto Close_Error_Rtn;
    }

    if ((evlInfo = cplInfo->innerEvl) == NULL) {
	if (fprintf(fp, "# Method Dir\n{}\n# Agent Dir\n{}\n# Function Dir\n{}\n")
	    == EOF) {
	    eMsgType = HdEMsg_Fail2FprintfNullDirsInTbl;
	    goto Close_Error_Rtn;
	}
    } else {
	if (fprintf(fp, "# Method Dir\n%s\n",
		    (evlInfo->methodDir == NULL) ? "{}" : evlInfo->methodDir)
	    == EOF) {
	    eMsgType = HdEMsg_Fail2FprintfMethodDirInTbl;
	    goto Close_Error_Rtn;
	}

	hdGetAgentDirNameList(fileBuf, evlInfo->agentDir);

	if (fprintf(fp, "# Agent Dir\n%s\n", fileBuf) == EOF) {
	    eMsgType = HdEMsg_Fail2FprintfAgentDirInTbl;
	    goto Close_Error_Rtn;
	}

	if (evlInfo->funcDir != NULL) {
	    sprintf(fileBuf, "%s\n", evlInfo->funcDir);
	} else {
	    *fileBuf = HdChar_NULL;
	}

	strcat(fileBuf, "{}"); 

	if (fprintf(fp, "# Function Dir\n%s\n", fileBuf) == EOF) {
	    eMsgType = HdEMsg_Fail2FprintfFuncDirInTbl;
	    goto Close_Error_Rtn;
	}
    }

    fclose(fp);

    return HdTrue;

 Close_Error_Rtn:
    fclose(fp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static char *hdGetAgentDirNameList(buf, agentDir)
    char		*buf;
    HiagNameInfo	*agentDir;
{
    char		*strPtr;
    char		*namePtr;

    if (agentDir == NULL) {
	strcpy(buf, "{}");
    } else {

	strPtr = buf;

	for ( ;; ) {
	    if ((namePtr = agentDir->name) == NULL) {
		break;
	    }

	    while (*namePtr != HdChar_NULL) {
		*strPtr++ = *namePtr++;
	    }

	    if ((agentDir = agentDir->next)) {
		*strPtr++ = HdChar_Comma;
	    } else {
		break;
	    }
	}
	*strPtr = HdChar_NULL;
    }

    return buf;
}

/*
 *
 */
static HdBoolean hdGetKeyWordValue(fp, keyWord, value)
    FILE	*fp;
    char	*keyWord;
    char	*value;
{
    char		*strPtr;
    char		fileBuf[HdFileBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if (hdFileGetString(fp, fileBuf) == HdFalse) {
	eMsgType = HdEMsg_FileGetStringError;
	goto False_Rtn;
    }

    if ((strPtr = hdSkipSpace(fileBuf)) == NULL) {
	eMsgType = HdEMsg_SkipSpaceError;
	goto False_Rtn;
    }

    if (*strPtr != HdChar_Ampersand) {
	eMsgType = HdEMsg_NotKeyWordInStrLineOfFile;
	goto Error_Rtn;
    }

    if ((strPtr = hdGetOneWord(keyWord, strPtr)) == NULL) {
	eMsgType = HdEMsg_GetOneWordError;
	goto Error_Rtn;
    }
    
    if ((strPtr = hdSkipSpace(strPtr)) == NULL) {
	eMsgType = HdEMsg_SkipSpaceError;
	goto Error_Rtn;
    }

    for (; *strPtr != HdChar_Semicolon; strPtr++) {
	if (*strPtr == HdChar_NULL) {
	    if (hdFileGetString(fp, fileBuf) == HdFalse) {
		eMsgType = HdEMsg_FileGetStringError;
		goto Error_Rtn;
	    }
	    strPtr = fileBuf;
	}

	switch (*strPtr) {
	case HdChar_CR:
	    break;

	default:
	    *value++ = *strPtr;
	    break;
	}
    }

    *value = HdChar_NULL;

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);

 False_Rtn:
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdFileGetString(fp, buf)
    FILE	*fp;
    char	*buf;
{
    return (fgets(buf, HdFileBufferLen_Lim, fp) == NULL) ? HdFalse : HdTrue;
}

/*
 *
 */
static char *hdGetOneWord(word, strPtr) 
    char	*word;
    char	*strPtr;
{
    char	ch;

    if (strPtr == NULL) {
	goto Error_Rtn;
    }

    while ((ch = *strPtr) != NULL
	   && (ch == HdChar_Space
	       || ch == HdChar_Comma
	       || ch == HdChar_Tab)) {
	strPtr++;
    }

    while ((ch = *strPtr) != HdChar_NULL
	   && ch != HdChar_Space
	   && ch != HdChar_Tab
	   && ch != HdChar_Comma
	   && ch != HdChar_Semicolon) {
	*word++ = *strPtr++;
    }

    *word = HdChar_NULL;

    return strPtr;

 Error_Rtn:
    return NULL;
}

/*
 *
 */
static char *hdSkipSpace(strPtr) 
    char	*strPtr;
{
    char	ch;

    if (strPtr == NULL) {
	goto Error_Rtn;
    }

    while ((ch = *strPtr) != NULL
	   && (ch == HdChar_Space
	       || ch == HdChar_Tab)) {
	strPtr++;
    }

    return strPtr;

 Error_Rtn:
    return NULL;
}
