/* hdcommon.c;
 *
 *	Common functions in Helios Daemon program
 *		created by k3sato on June 16th in 1994.
 *
 *	modification history:	
 *	(1) add HdStr2Int()
 *		on July 21st in '94.
 *	(2) add hdFreeFuncDirInfo(), hdFreeMethodDirInfo()
 *		on November 18th in '94.
 *
 * $Id: hdcommon.c,v 2.1 1995/02/27 07:32:01 k3sato Exp $
 */

#include <stdio.h>
#include <malloc.h>
#include <memory.h>

#include <helios/heliosd.h>
#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>

#include "hdcommon.h"

/* for errno */
extern char *sys_errlist[];

HdBoolean hdSetupAgentInfoFromMsg(hAgentInfo, msgType, msgBuf)
    HAgentInfo		*hAgentInfo;
    HeliosdMsgType	*msgType;
    char		*msgBuf;
{
    AgentInfo		agentInfo;
    char		agentId[AgentIDLen_Lim];
    char		agentPath[AgentPathLen_Lim];
    char		agentName[AgentNameLen_Lim];
    char		hostName[HostNameLen_Lim];
    int			sin_family;
    int			sin_port;
    int			status;
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    memset(&agentInfo, 0, sizeof(AgentInfo));

    ret = sscanf(msgBuf, "%d %d %s %s %s %x %x %x %s %d %d %d %d %i %x %x %x %x",
		 msgType,
		 &(agentInfo.type),
		 agentId,
		 agentPath,
		 agentName,
		 &(agentInfo.procArgs),
		 &(agentInfo.funcDir),
		 &(agentInfo.methodDir),
		 hostName,
		 &sin_family,
		 &sin_port,
		 &(agentInfo.pid),
		 &status,
		 &(agentInfo.groupNum),
		 &(agentInfo.comment),
		 &(agentInfo.parent),
		 &(agentInfo.child),
		 &(agentInfo.next)
		 );

    if (ret < 18 || ret == EOF) {
	eMsgType = (*msgType == HdMsg_AgentInfoEnd) ?
	  NULL : HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    if (*msgType != HdMsg_AgentInfo && *msgType != HdMsg_AgentInfoEnd) {
	eMsgType = HdEMsg_MsgTypeMismatch;
	goto Error_Rtn;
    }

    if ((agentInfo.agentId = HdMallocString(agentId)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    if ((agentInfo.agentPath = HdMallocString(agentPath)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    if ((agentInfo.agentName = HdMallocString(agentName)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    if ((agentInfo.sockInfo.hostName = HdMallocString(hostName)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    agentInfo.sockInfo.sin_family = sin_family;
    agentInfo.sockInfo.sin_port = sin_port;

    agentInfo.status = status;

    if ((*hAgentInfo = (HAgentInfo)HdMalloc(sizeof(AgentInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    memcpy(*hAgentInfo, &agentInfo, sizeof(AgentInfo));

    return HdTrue;

 Error_Rtn:

    hdFreeString(&(agentInfo.agentId));
    hdFreeString(&(agentInfo.agentPath));
    hdFreeString(&(agentInfo.agentName));
    hdFreeString(&(agentInfo.sockInfo.hostName));

    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	Malloc of AgentInfo for Parametric instance agent
 */
HAgentInfo hdMallocParaInstance(parentInfo)
    HAgentInfo		parentInfo;
{
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;

    if ((agentInfo = (HAgentInfo)HdMalloc(sizeof(AgentInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    agentInfo->type = HdAT_ParaInstance;
    agentInfo->status = AS_NotActive;
    agentInfo->parent = parentInfo;

    return agentInfo;

 Error_Rtn:
    hdFreeAgentInfo(&agentInfo);
    hdPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 *	Memory manipulation functions
 */
void *HdMalloc(size)
    int	size;
{
    void	*ptr;
    HeliosdErrorMsgType	eMsgType;

    if (size > 0) {
	if ((ptr = malloc(size)) == NULL) {
	    eMsgType = HdEMsg_MallocError;
	    goto Error_Rtn;
	}
    } else {
	eMsgType = HdEMsg_SizeZeroMalloc;
	goto Error_Rtn;
    }

    memset(ptr, 0, size);

    return ptr;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return NULL;
}

char *HdMallocString(string)
    char	*string;
{
    int		size;
    void	*ptr;
    HeliosdErrorMsgType	eMsgType;

    if (string == NULL) {
	eMsgType = HdEMsg_NullPointer;
	goto Error_Rtn;
    }

    size = strlen(string) + 1;

    if (size > HdStringLength_Lim) {
	eMsgType = HdEMsg_StringLenLimitOver;
	goto Error_Rtn;
    }

    if ((ptr = HdMalloc(size)) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    strcpy(ptr, string);

    return ptr;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return NULL;
}


/* =============================
 *  functions to free memory
 * ============================= */
void hdFreeMachInfo(hMachInfoP)
    HMachInfo	*hMachInfoP;
{
    HMachInfo	machInfo, nextMachInfo;

    if (hMachInfoP == NULL) return;

    for (machInfo = *hMachInfoP; machInfo != NULL; machInfo = nextMachInfo) {
	nextMachInfo = machInfo->next;

	hdFreeString(&(machInfo->hostName));

	hdFree(machInfo);
    }
    *hMachInfoP = NULL;
}

void hdFreeAgentInfo(hAgentInfoP)
    HAgentInfo	*hAgentInfoP;
{
    HAgentInfo	agentInfo, nextAgentInfo;

    if (hAgentInfoP == NULL) return;

    for (agentInfo = *hAgentInfoP; agentInfo != NULL; agentInfo = nextAgentInfo) {
	nextAgentInfo = agentInfo->next;

	hdFreeOneAgentInfo(&agentInfo);
    }
    *hAgentInfoP = NULL;
}

void hdFreeOneAgentInfo(hAgentInfoP)
    HAgentInfo	*hAgentInfoP;
{
    HAgentInfo	agentInfo;

    if (hAgentInfoP == NULL ||
	(agentInfo = *hAgentInfoP) == NULL) return;

    hdFreeString(&(agentInfo->agentId));
    hdFreeString(&(agentInfo->agentPath));
    hdFreeString(&(agentInfo->agentName));
    hdFreeString(&(agentInfo->procArgs));
    hdFreeFuncDirInfo(&(agentInfo->funcDir));
    /* hdFreeMethodDirInfo(&(agentInfo->methodDir)); */
    hdFreeString(&(agentInfo->methodDir));
    hdFreeString(&(agentInfo->sockInfo.hostName));
    hdFreeString(&(agentInfo->comment));

    hdFree(agentInfo);

    *hAgentInfoP = NULL;
}

void hdFreeFuncDirInfo(funcDirInfoP)
    HFuncDirInfo	*funcDirInfoP;
{
    HFuncDirInfo	funcDirInfo, nextFuncDirInfo;

    if (funcDirInfoP == NULL) return;

    for (funcDirInfo = *funcDirInfoP; funcDirInfo != NULL; funcDirInfo = nextFuncDirInfo) {
	nextFuncDirInfo = funcDirInfo->next;
	hdFreeString(&(funcDirInfo->funcName));
	hdFreeString(&(funcDirInfo->agentNames));
	hdFree(funcDirInfo);
    }
    *funcDirInfoP = NULL;
}

void hdFreeMethodDirInfo(methodDirInfoP)
    HMethodDirInfo	*methodDirInfoP;
{
    HMethodDirInfo	methodDirInfo, nextMethodDirInfo;

    if (methodDirInfoP == NULL) return;

    for (methodDirInfo = *methodDirInfoP; methodDirInfo != NULL; methodDirInfo = nextMethodDirInfo) {
	nextMethodDirInfo = methodDirInfo->next;
	hdFreeString(&(methodDirInfo->methodName));
	hdFree(methodDirInfo);
    }
    *methodDirInfoP = NULL;
}

void hdFreeString(strP)
    char	**strP;
{
    if (strP != NULL) {
	hdFree(*strP);
	*strP = NULL;
    }
}

void hdFree(ptr)
    void	*ptr;
{
    if (ptr != NULL) {
	free(ptr);
    }
}

/*
 *	system error handle function
 */
void hdPerror()
{
    fprintf(stderr,"system error (%d) : %s\n",
	    errno, sys_errlist[errno]);
}

int HdStr2Int(str)
    char	*str;
{
    if (str != NULL) {
	return atoi(str);
    }

    return NULL;
}
