/* hDaemonLib.c;
 *	Library to access Helios Daemon.
 *		created by k3sato on May 25th in '94.
 *
 *	modification history:	
 *	(1) modified on June 22nd in '94.
 *	(2) add hdSearchAgentSockInfo(), hdAppendAgentSockInfo()
 *		on August 5th in '94.
 *	(3) add delete_agent()
 *		on October 11th in '94.
 *	(4) add register_paraAgent()
 *		on October 12th in '94.
 *	(5) modified hdGetAgentInfoFromDaemon()
 *		on November 9th in '94.
 *	(6) moved hdSetupAgentInfoFromMsg() to hdcommon.c
 *		on February 22nd in '95.
 *
 * $Id: hDaemonLib.c,v 2.5 1995/02/27 07:32:01 k3sato Exp $
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <string.h>
#include <fcntl.h>
#include <sys/time.h>
#include <memory.h>

#include <helios/hDaemonLib.h>
#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>

#include "hdcommon.h"

/* hdSetupAgentInfoFromMsg() is defined in hdcommon.c
 */

static struct sockaddr_in	hdSockaddr;
static int			hdSocket;
static struct timeval		hdTimeout = {(long)25, (long)0};

static HHdAgentSockInfo		agentSockInfoHead = NULL;
static HHdAgentSockInfo		agentSockInfoTail = NULL;

static struct sockaddr_in *hdSearchAgentSockInfo();
static HdBoolean hdAppendAgentSockInfo();
static HdBoolean hdRemoveAgentSockInfo();

static HdBoolean hdGetAgentInfoFromDaemon();
static HdBoolean hdGetAgentInfo_a();
static HdBoolean hdGetOneAgentInfo();
static HdBoolean hdSetupFuncDirInfoFromMsg();
static char *hdGetNextTermPtr();
static HdBoolean hdSetupMethodDirInfoFromMsg();

static HdBoolean hdGetResultFromDaemon();
static HdBoolean hdGetDaemonPortNum();


/*
 *	initialization of heliosd
 */
int heliosdInit()
{
    char		hostname[HostNameLen_Lim];
    struct hostent	*myhost;
    int			hdPortNum;
    HeliosdErrorMsgType	eMsgType;

    /*
     *	get the port number of Helios Daemon
     *		from the file /tmp/.heliosdLOCK
     */
    if (hdGetDaemonPortNum(&hdPortNum) == HdFalse) {
	eMsgType = HdEMsg_GetDaemonPortNum;
	goto Error_Rtn;
    }

    /*
     *	setup sockadder_in to access to heliosd
     */

    /* $B%5!<%P$r5/F0$7$?%^%7%s$N%[%9%HL>(B */
    if (gethostname(hostname, HostNameLen_Lim) == ERROR_FLAG) {
	eMsgType = HdEMsg_NotFindHostName;
	goto Error_Rtn;
    }

    if ((myhost = gethostbyname(hostname)) == NULL) {
	eMsgType = HdEMsg_BadHostName;
	goto Error_Rtn;
    }

    bzero((char *)&hdSockaddr, sizeof(hdSockaddr));
    hdSockaddr.sin_family = AF_INET;
    hdSockaddr.sin_port = hdPortNum;
    
    bcopy(myhost->h_addr, (char *)&hdSockaddr.sin_addr, myhost->h_length);

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ======================================================
 *	Ask heliosd to create a agent process,
 *	and get the agent information from heliosd
 * ====================================================== */
struct sockaddr_in *get_agent(agentId)
    char	*agentId;
{
    struct sockaddr_in	*sockInfo;
    HeliosdErrorMsgType	eMsgType;

    if ((sockInfo = hdSearchAgentSockInfo(agentId)) != NULL) {
	return sockInfo;
    }

    /*	create socket to heliosd
     */
    if (hdCreateSocket() == HdFalse) {
	eMsgType = HdEMsg_MsgTypeMismatch;
	goto Error_Rtn;
    }

    if (hdSendMessage(HdMsg_GetAgent, agentId) == HdFalse) {
	eMsgType = HdEMsg_SendMsg2Daemon;
	goto Error_Rtn_Close;
    }

    if (hdGetSockaddrFromDaemon(&sockInfo) == HdFalse) {
	eMsgType = HdEMsg_GetSockaddrFromDaemonError;
	goto Error_Rtn_Close;
    }

    /*	close socket to heliosd
     */
    hdCloseSocket();

    /*
     * append sockInfo to agentSockInfo structure
     */
    if (hdAppendAgentSockInfo(agentId, sockInfo) == HdFalse) {
	eMsgType = HdEMsg_AppendAgentSockInfoError;
	goto Error_Rtn;
    }

    return sockInfo;

 Error_Rtn_Close:
    /*	close socket to heliosd
     */
    hdCloseSocket();

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return NULL;
}

/* ======================================================
 *	Ask heliosd to wake a agent process,
 *	and get the agent information from heliosd
 * ====================================================== */
struct sockaddr_in *wake_agent(agentId)
    char	*agentId;
{
    struct sockaddr_in	*sockInfo;
    HeliosdErrorMsgType	eMsgType;

    /*	create socket to heliosd
     */
    if (hdCreateSocket() == HdFalse) {
	eMsgType = HdEMsg_MsgTypeMismatch;
	goto Error_Rtn;
    }

    if (hdSendMessage(HdMsg_GetAgent, agentId) == HdFalse) {
	eMsgType = HdEMsg_SendMsg2Daemon;
	goto Error_Rtn_Close;
    }

    if (hdGetSockaddrFromDaemon(&sockInfo) == HdFalse) {
	eMsgType = HdEMsg_GetSockaddrFromDaemonError;
	goto Error_Rtn_Close;
    }

    /*	close socket to heliosd
     */
    hdCloseSocket();

    return sockInfo;

 Error_Rtn_Close:
    /*	close socket to heliosd
     */
    hdCloseSocket();

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 * Search Agent socket information for agentId
 */
static struct sockaddr_in
  *hdSearchAgentSockInfo(agentId)
    char	*agentId;
{
    HHdAgentSockInfo	currSockInfo;

    for (currSockInfo = agentSockInfoHead
	 ; currSockInfo != NULL
	 ; currSockInfo = currSockInfo->next) {
	if (strcmp(agentId, currSockInfo->agentId) == 0) {
	    return currSockInfo->sockInfo;
	}
    }

    return NULL;
}

/*
 * Append Agent socket information to hAgentSockInfo structure
 */
static HdBoolean hdAppendAgentSockInfo(agentId, sockInfo)
    char		*agentId;
    struct sockaddr_in	*sockInfo;
{
    HHdAgentSockInfo	agentSockInfo;
    HeliosdErrorMsgType	eMsgType;

    if ((agentSockInfo = HdMalloc(sizeof(HdAgentSockInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }
    if ((agentSockInfo->agentId = HdMallocString(agentId)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }
    agentSockInfo->sockInfo = sockInfo;

    if (agentSockInfoHead == NULL) {
	agentSockInfoHead =
	  agentSockInfoTail = agentSockInfo;
    } else {
	agentSockInfoTail->next = agentSockInfo;
	agentSockInfoTail = agentSockInfo;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ======================================================
 *	get the agent information activated already
 * ====================================================== */
HAgentInfo get_agentInfo(agentId)
    char	*agentId;
{
    HAgentInfo		agentInfo;
    HeliosdMsgType	hMsgType;
    HeliosdErrorMsgType	eMsgType;


    /*	create socket to heliosd
     */
    if (hdCreateSocket() == HdFalse) {
	eMsgType = HdEMsg_MsgTypeMismatch;
	goto Error_Rtn;
    }

    hMsgType = (agentId == NULL || *agentId == NULL)
      ? HdMsg_GetAllAgentInfo : HdMsg_GetAgentInfo;

    if (hdSendMessage(hMsgType, agentId) == HdFalse) {
	goto Error_Rtn;
    }

    if (hdGetAgentInfoFromDaemon(&agentInfo) == HdFalse) {
	eMsgType = HdEMsg_GetAgentFromDaemonError;
	goto Error_Rtn;
    }

    /*	close socket to heliosd
     */
    hdCloseSocket();

    return agentInfo;

 Error_Rtn:
    /*	close socket to heliosd
     */
    hdCloseSocket();

    hdPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 *	Get SocketInfo from received data from Daemon,
 *	and construct sockadd_in value from SocketInfo.
 *
 *	typedef struct _SocketInfo {
 *	    char	*hostName;
 *	    short	sin_family;
 *	    u_short	sin_port;
 *	} SocketInfo;
 *
 *	struct sockaddr_in {
 *		short	sin_family;
 *		u_short	sin_port;
 *		struct	in_addr sin_addr;
 *		char	sin_zero[8];
 *	};
 */
HdBoolean hdGetSockaddrFromDaemon(sockInfoP)
    struct sockaddr_in	**sockInfoP;
{
    HeliosdMsgType	msgType;
    char		hostname[HostNameLen_Lim];
    int			sin_family;
    int			sin_port;
    struct hostent	*myhost;
    struct sockaddr_in	me;
    char		recvBuf[SocketBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    if (hdReceiveMessage(recvBuf) == HdFalse) {
	eMsgType = HdEMsg_RecvMsgFromDaemon;
	goto Error_Rtn;
    }

    ret = sscanf(recvBuf, "%d %s", &msgType, hostname);
    if (ret == 0 || ret == EOF) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    if (msgType == HdMsg_ErrorDetected) {
	eMsgType = HdEMsg_ErrorInHeliosd;
	fprintf(stderr, "Error Msg from heliosd: %s\n", hostname);
	goto Error_Rtn;
    }
	
    ret = sscanf(recvBuf, "%d %s %d %d", &msgType, hostname, &sin_family, &sin_port);
    if (ret == 0 || ret == EOF) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    if (msgType != HdMsg_SocketInfo) {
	eMsgType = HdEMsg_MsgTypeMismatch;
	goto Error_Rtn;
    }

    if (sin_port == 0) {
	eMsgType = HdEMsg_IllegalPortNumber;
	goto Error_Rtn;
    }

    bzero((char *)&me, sizeof(me));
    me.sin_family = sin_family;
    me.sin_port = sin_port;
    
    if ((myhost = gethostbyname(hostname)) == NULL) {
	eMsgType = HdEMsg_BadHostName;
	goto Error_Rtn;
    }

    bcopy(myhost->h_addr, (char *)&me.sin_addr, myhost->h_length);

    if ((*sockInfoP = (struct sockaddr_in *)HdMalloc(sizeof(me))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    memcpy(*sockInfoP, &me, sizeof(me));

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	Get AgentInfo from received data from Daemon,
 */
static HdBoolean hdGetAgentInfoFromDaemon(agentInfo)
    HAgentInfo		*agentInfo;
{
    HeliosdErrorMsgType	eMsgType;

    *agentInfo = (HAgentInfo)!NULL;

    if (hdGetAgentInfo_a(agentInfo, NULL) == HdFalse) {
	eMsgType = HdEMsg_GetAgentInfo_aError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdFreeAgentInfo(agentInfo);

    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *  Get the structure data of AgentInfo 
 *  from daemon
 */
static HdBoolean hdGetAgentInfo_a(agentInfo, parentInfo)
    HAgentInfo		*agentInfo;
    HAgentInfo		parentInfo;
{
    HAgentInfo		currInfo;
    HeliosdMsgType	msgType = HdMsg_AgentInfo;

    for (; *agentInfo != NULL; agentInfo = &(currInfo->next)) {

	if (msgType == HdMsg_AgentInfoEnd) {
	    goto Error_Rtn;
	}

	if (hdGetOneAgentInfo(&msgType, agentInfo) == HdFalse) {
	    goto Error_Rtn;
	}

	currInfo = *agentInfo;
	currInfo->parent = parentInfo;

	if (msgType != HdMsg_AgentInfoEnd) {
	    if (hdSendMessage(HdMsg_AskForNextInfo, NULL) == HdFalse) {
		goto Error_Rtn;
	    }

	    if (currInfo->child != NULL) {
		if (hdGetAgentInfo_a(&(currInfo->child), currInfo) == HdFalse) {
		    goto Error_Rtn;
		}
	    }
	} else {
	    if (currInfo->child != NULL) {
		goto Error_Rtn;
	    }
	}
    }

    return HdTrue;

 Error_Rtn:
    return HdFalse;
}

/*
 *
 *	hdSetupAgentInfoFromMsg() is defined in hdcommon.c
 */
static HdBoolean hdGetOneAgentInfo(msgType, agentInfo)
    HeliosdMsgType	*msgType;
    HAgentInfo		*agentInfo;
{
    char		recvBuf[SocketBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    *agentInfo = NULL;

    if (hdReceiveMessage(recvBuf) == HdFalse) {
	eMsgType = HdEMsg_RecvMsgFromDaemon;
	goto Error_Rtn;
    }

    if (hdSetupAgentInfoFromMsg(agentInfo, msgType, recvBuf) == HdFalse) {
	eMsgType = (*msgType == HdMsg_AgentInfoEnd) ?
	  NULL : HdEMsg_SetupAgentInfoFromMsgError;
	goto Error_Rtn;
    }

    if ((*agentInfo)->procArgs != NULL) {

	if (hdSendMessage(HdMsg_AskForNextInfo, NULL) == HdFalse) {
	    eMsgType = HdEMsg_SendMsg2Daemon;
	    goto Error_Rtn;
	}

	if (hdReceiveMessage(recvBuf) == HdFalse) {
	    eMsgType = HdEMsg_RecvMsgFromDaemon;
	    goto Error_Rtn;
	}

	if (((*agentInfo)->procArgs = HdMallocString(recvBuf)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Error_Rtn;
	}
    }

    if ((*agentInfo)->funcDir != NULL) {

	if (hdSendMessage(HdMsg_AskForNextInfo, NULL) == HdFalse) {
	    eMsgType = HdEMsg_SendMsg2Daemon;
	    goto Error_Rtn;
	}

	if (hdReceiveMessage(recvBuf) == HdFalse) {
	    eMsgType = HdEMsg_RecvMsgFromDaemon;
	    goto Error_Rtn;
	}

	if (hdSetupFuncDirInfoFromMsg(&((*agentInfo)->funcDir), recvBuf) == HdFalse) {
	    eMsgType = HdEMsg_GetFuncDirInfoError;
	    goto Error_Rtn;
	}
    }

    if ((*agentInfo)->methodDir != NULL) {

	if (hdSendMessage(HdMsg_AskForNextInfo, NULL) == HdFalse) {
	    eMsgType = HdEMsg_SendMsg2Daemon;
	    goto Error_Rtn;
	}

	if (hdReceiveMessage(recvBuf) == HdFalse) {
	    eMsgType = HdEMsg_RecvMsgFromDaemon;
	    goto Error_Rtn;
	}

	if (hdSetupMethodDirInfoFromMsg(&((*agentInfo)->methodDir), recvBuf) == HdFalse) {
	    eMsgType = HdEMsg_GetMethodDirInfoError;
	    goto Error_Rtn;
	}
    }

    if ((*agentInfo)->comment != NULL) {

	if (hdSendMessage(HdMsg_AskForNextInfo, NULL) == HdFalse) {
	    eMsgType = HdEMsg_SendMsg2Daemon;
	    goto Error_Rtn;
	}

	if (hdReceiveMessage(recvBuf) == HdFalse) {
	    eMsgType = HdEMsg_RecvMsgFromDaemon;
	    goto Error_Rtn;
	}

	if (((*agentInfo)->comment = HdMallocString(recvBuf)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdFreeOneAgentInfo(agentInfo);

    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

HdBoolean hdSetupFuncDirInfoFromMsg(funcDirInfo, msgBuf)
    HFuncDirInfo	*funcDirInfo;
    char		*msgBuf;
{
    HeliosdMsgType	msgType;
    char		*bufPtr;
    char		funcName[AgentNameLen_Lim];
    char		*agentNames;
    HFuncDirInfo	*currInfoP;
    HFuncDirInfo	currInfo;
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    ret = sscanf(msgBuf, "%d", &msgType);

    if (ret < 1 || ret == EOF) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    if (msgType != HdMsg_FuncDirInfo) {
	eMsgType = HdEMsg_MsgTypeMismatch;
	goto Error_Rtn;
    }

    if (*(bufPtr = hdGetNextTermPtr(msgBuf)) == HdChar_NULL) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    for (currInfoP = funcDirInfo
	 ; bufPtr != NULL && *bufPtr != NULL
	 ;currInfoP = &(currInfo->next)) {

	ret = sscanf(bufPtr, "%s", funcName);
	if (ret < 1 || ret == EOF) {
	    eMsgType = HdEMsg_ReceivedDataError;
	    goto Error_Rtn;
	}

	if ((*currInfoP =
	     currInfo = (HFuncDirInfo)HdMalloc(sizeof(FuncDirInfo))) == NULL) {
	    eMsgType = HdEMsg_HdMallocError;
	    goto Error_Rtn;
	}

	if ((currInfo->funcName = HdMallocString(funcName)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Error_Rtn;
	}

	if (*(agentNames = hdGetNextTermPtr(bufPtr)) == HdChar_NULL) {
	    eMsgType = HdEMsg_ReceivedDataError;
	    goto Error_Rtn;
	}

	if ((bufPtr = strchr(agentNames, '\n')) == NULL) {
	    eMsgType = HdEMsg_ReceivedDataError;
	    goto Error_Rtn;
	}

	*bufPtr++ = HdChar_NULL;

	if ((currInfo->agentNames = HdMallocString(agentNames)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdFreeFuncDirInfo(funcDirInfo);

    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

static char *hdGetNextTermPtr(bufPtr)
    char	*bufPtr;
{
    char	ch;

    while ((ch = *bufPtr) != HdChar_NULL
	   && (ch == HdChar_Space
	       || ch == HdChar_Tab)) {
	bufPtr++;
    }

    while ((ch = *bufPtr) != HdChar_NULL
	   && ch != HdChar_Space
	   && ch != HdChar_Tab) {
	bufPtr++;
    }

    while ((ch = *bufPtr) != HdChar_NULL
	   && (ch == HdChar_Space
	       || ch == HdChar_Tab)) {
	bufPtr++;
    }

    return bufPtr;
}

HdBoolean hdSetupMethodDirInfoFromMsg(methodDirInfo, msgBuf)
    char		**methodDirInfo;	/* HMethodDirInfo */
    char		*msgBuf;
{
    HeliosdMsgType	msgType;
    /*
    char		methodName[AgentNameLen_Lim];
    HMethodDirInfo	*currInfoP;
    HMethodDirInfo	currInfo;
    */
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    ret = sscanf(msgBuf, "%d", &msgType);

    if (ret < 1 || ret == EOF) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    if (msgType != HdMsg_MethodDirInfo) {
	eMsgType = HdEMsg_MsgTypeMismatch;
	goto Error_Rtn;
    }

    if ((msgBuf = strchr(msgBuf, ' ')) != NULL) {
	while (*++msgBuf == ' ' && *msgBuf != '\0')
	  ;
    }

    if ((*methodDirInfo = HdMallocString(msgBuf)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    /*
    for (currInfoP = methodDirInfo
	 ; msgBuf != NULL && *msgBuf != NULL
	 ; currInfoP = &(currInfo->next)) {

	ret = sscanf(msgBuf, "%s", methodName);
	if (ret < 1 || ret == EOF) {
	    eMsgType = HdEMsg_ReceivedDataError;
	    goto Error_Rtn;
	}

	if ((*currInfoP =
	     currInfo = (HMethodDirInfo)HdMalloc(sizeof(MethodDirInfo))) == NULL) {
	    eMsgType = HdEMsg_HdMallocError;
	    goto Error_Rtn;
	}

	if ((currInfo->methodName = HdMallocString(methodName)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Error_Rtn;
	}

	if ((msgBuf = strchr(msgBuf, ' ')) != NULL) {
	    while (*++msgBuf == ' ' && *msgBuf != '\0')
	      ;
	}
    }
    */

    return HdTrue;

 Error_Rtn:
    hdFreeMethodDirInfo(methodDirInfo);

    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ======================================================
 *	Ask heliosd to delete a agent process,
 * ====================================================== */
HdBoolean delete_agent(agentId)
    char	*agentId;
{
    HeliosdMsgType	msgType;
    char		msg[MsgBufferLen_Lim];
    struct sockaddr_in	*sockInfo;
    HeliosdErrorMsgType	eMsgType;

    /* get_agent $B$7$?(B agentID $B$+$I$&$+3NG'$9$k!#(B
     */
    if ((sockInfo = hdSearchAgentSockInfo(agentId)) != NULL) {

	/* remove sockInfo of a given agentId
	 * from agentSockInfo structure
	 */
	if (hdRemoveAgentSockInfo(agentId) == HdFalse) {
	    eMsgType = HdEMsg_RemoveAgentSockInfoError;
	    goto Error_Rtn;
	}
    }

    /*	create socket to heliosd
     */
    if (hdCreateSocket() == HdFalse) {
	eMsgType = HdEMsg_MsgTypeMismatch;
	goto Error_Rtn;
    }

    if (hdSendMessage(HdMsg_DeleteAgent, agentId) == HdFalse) {
	eMsgType = HdEMsg_SendMsg2Daemon;
	goto Error_Rtn_Close;
    }

   if (hdGetResultFromDaemon(&msgType, msg) == HdFalse) {
	eMsgType = HdEMsg_GetResultFromDaemonError;
	goto Error_Rtn_Close;
    }

    /*	close socket to heliosd
     */
    hdCloseSocket();

    return HdTrue;

 Error_Rtn_Close:
    /*	close socket to heliosd
     */
    hdCloseSocket();

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Remove Agent socket information of given agentId
 *	from hAgentSockInfo structure
 */
static HdBoolean hdRemoveAgentSockInfo(agentId)
    char		*agentId;
{
    HHdAgentSockInfo	currSockInfo, prevSockInfo;
    HeliosdErrorMsgType	eMsgType;

    for (currSockInfo = agentSockInfoHead, prevSockInfo = NULL
	 ; currSockInfo != NULL
	 ; prevSockInfo = currSockInfo, currSockInfo = currSockInfo->next) {

	if (strcmp(agentId, currSockInfo->agentId) == 0) {

	    if (prevSockInfo == NULL) {
		agentSockInfoHead = currSockInfo->next;
	    } else {
		prevSockInfo->next = currSockInfo->next;
	    }

	    if (currSockInfo == agentSockInfoTail) {
		if ((agentSockInfoTail = currSockInfo->next) == NULL) {
		    agentSockInfoTail = prevSockInfo;
		}
	    }

	    hdFreeString(&(currSockInfo->agentId));
	    hdFree(currSockInfo->sockInfo);
	    hdFree(currSockInfo);

	    return HdTrue;
	}
    }

    eMsgType = HdEMsg_NoWakedProccessForAgentId;

 /*
 Error_Rtn:
  */
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ======================================================
 *	Ask heliosd to register a parametric agent
 * ====================================================== */
HdBoolean register_paraAgent(sockNum, agentId, agentName)
    int		sockNum;
    char	*agentId;
    char	*agentName;
{
    HeliosdMsgType	msgType;
    char		msgBuf[MsgBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    /*	create socket to heliosd
     */
    if (hdCreateSocket() == HdFalse) {
	eMsgType = HdEMsg_MsgTypeMismatch;
	goto Error_Rtn;
    }

    sprintf(msgBuf, "%d:%s:%s", sockNum, agentId, agentName);

    if (hdSendMessage(HdMsg_RegisterParaAgent, msgBuf) == HdFalse) {
	eMsgType = HdEMsg_SendMsg2Daemon;
	goto Error_Rtn_Close;
    }

   if (hdGetResultFromDaemon(&msgType, msgBuf) == HdFalse) {
	eMsgType = HdEMsg_GetResultFromDaemonError;
	goto Error_Rtn_Close;
    }

    /*	close socket to heliosd
     */
    hdCloseSocket();

    return HdTrue;

 Error_Rtn_Close:
    /*	close socket to heliosd
     */
    hdCloseSocket();

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ===================================================
 *	Get Result from Daemon,
 * =================================================== */
static HdBoolean hdGetResultFromDaemon(msgType, msg)
    HeliosdMsgType	*msgType;
    char		*msg;
{
    char		*tempStr;
    char		recvBuf[SocketBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    if (hdReceiveMessage(recvBuf) == HdFalse) {
	eMsgType = HdEMsg_RecvMsgFromDaemon;
	goto Error_Rtn;
    }

    ret = sscanf(recvBuf, "%d", msgType);
    if (ret == 0 || ret == EOF) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    if ((tempStr = strchr(recvBuf, ' ')) != NULL) {
	if (strlen(++tempStr) >= MsgBufferLen_Lim) {
	    eMsgType = HdEMsg_StringLenLimitOver;
	    goto Error_Rtn;
	}
	strcpy(msg, tempStr);
    } else {
	strcpy(msg, "Only message Type, no other message from heliosd");
    }

    if (*msgType == HdMsg_ErrorDetected) {
	eMsgType = HdEMsg_ErrorInHeliosd;
	fprintf(stderr, "Error Msg from heliosd:\n %s\n", msg);
	goto Error_Rtn;
    }
	
    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ==================================================
 *	Create/Close a socket to Helios Daemon.
 * ================================================== */
/*
 *	Create a socket
 *	and connect to Helios Daemon
 */
HdBoolean hdCreateSocket()
{
    HeliosdErrorMsgType	eMsgType;


    if ((hdSocket = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	eMsgType = HdEMsg_SocketForServerFailed;
	goto Error_Rtn;
    }

/*
    if (bind(hdSocket, (struct sockaddr *)&hdSockaddr, sizeof(hdSockaddr)) == ERROR_FLAG) {
	eMsgType = HdEMsg_CouldNotBind;
	goto Error_Rtn;
    }

    if (getsockname(hdSocket, (struct sockaddr *)&hdSockaddr, sizeof(hdSockaddr)) == ERROR_FLAG) {
	eMsgType = HdEMsg_GetSockName;
	goto Error_Rtn;
    }
*/

    if (connect(hdSocket, &hdSockaddr, sizeof(hdSockaddr)) == ERROR_FLAG) {
	eMsgType = HdEMsg_CouldNotConnect;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPerror();
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	Close a socket to Helios Daemon
 */
HdBoolean hdCloseSocket()
{
    close(hdSocket);

    return HdTrue;
}

/* ==================================================
 *	send/receive a message to/from Helios Daemon.
 * ================================================== */
/*
 *	Send a message to Helios Daemon
 */
HdBoolean hdSendMessage(msgType, msg)
    HeliosdMsgType	msgType;
    char		*msg;
{
    fd_set		mask;
    fd_set		writeOk;
    int			width;
    char		sendBuf[SocketBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;


    if (msg == NULL)
      msg = "";

    if (strlen(msg) >= SocketBufferLen_Lim) {
	eMsgType = HdEMsg_SocketBufferLenLimitOver;
	goto Error_Rtn;
    }

    sprintf(sendBuf, "%d %s", msgType, msg);

    FD_ZERO(&mask);
    FD_SET(hdSocket, &mask);
    width = hdSocket + 1;

    /* select $B$NA0$KKh2s(B writeOk $B$r=i4|2=$9$k!#(B */
    writeOk = mask;
    select(width, NULL, (fd_set *)&writeOk, NULL, &hdTimeout);

    /* $B%=%1%C%H$K=q$-9~$_2D$+!)(B */
    if (FD_ISSET(hdSocket, &writeOk)) {
	if (write(hdSocket, sendBuf, strlen(sendBuf)) == ERROR_FLAG) {
	    hdPerror();
	    eMsgType = HdEMsg_SocketWriteError;
	    goto Error_Close;
	}
    }

    return HdTrue;

 Error_Close:
    close(hdSocket);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	Receive a message to Helios Daemon
 */
HdBoolean hdReceiveMessage(recvBuf)
    char		*recvBuf;
{
    fd_set		mask;
    fd_set		readOk;
    int			width;
    /*
    int			retry;
    */
    int			length;
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    FD_ZERO(&mask);
    FD_SET(hdSocket, &mask);
    width = hdSocket + 1;

    /*
      for (retry = HdLibMaxNumOfRetry; retry > 0; retry--) {
      */
    for (;;){
	/* select $B$NA0$K(B readOk $B$r=i4|2=$9$k!#(B
	 */
	readOk = mask;
	ret = select(width, (fd_set *)&readOk, NULL, NULL, NULL); /* &hdTimeout */
	if (ret <= 0) {
	    if (ret == -1) {
		hdPerror();
	    }
	    continue;
	}

	/* $B%=%1%C%H$+$iFI$_9~$_2D$+!)(B
	 */
	if (FD_ISSET(hdSocket, &readOk)) {
	    if ((length = read(hdSocket, recvBuf, SocketBufferLen_Lim)) == ERROR_FLAG) {
		hdPerror();
		eMsgType = HdEMsg_SocketReadError;
		goto Error_Rtn;
	    }
	    if (length >= SocketBufferLen_Lim) {
		eMsgType = HdEMsg_SocketBufferLenLimitOver;
		goto Error_Rtn;
	    }

	    recvBuf[length] = '\0';
	    return HdTrue;
	}
    }

    eMsgType = HdEMsg_OverMaxNumOfRetryToReadSoc;

 Error_Rtn:
    close(hdSocket);

    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* =============================
 *	input functions
 * ============================= */
/*
 *	get Port Number from /tmp/.heliosdLOCK
 */
static HdBoolean hdGetDaemonPortNum(portNum)
    int	*portNum;
{
    char	lockFile[HeliosdFilePathLen_Lim];
    FILE	*lockfp;
    int		pid;
    HeliosdErrorMsgType	eMsgType;

    sprintf(lockFile, "%s/%s", HeliosdLockFileDir, HeliosdLockFileName);

    if ((lockfp = fopen(lockFile, "r")) == NULL) {
	eMsgType = HdEMsg_LockFileOpenError;
	goto Error_Rtn;
    }

    if (fscanf(lockfp, "%d", &pid) == 0) {
	eMsgType = HdEMsg_LockFileReadError;
	goto Error_Close;
    }

    if (fscanf(lockfp, "%d", portNum) == 0) {
	eMsgType = HdEMsg_LockFileReadError;
	goto Error_Close;
    }

    fclose(lockfp);
    return HdTrue;

 Error_Close:
    fclose(lockfp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}
