/* heliosd.c;
 *	Helios Daemon main program
 *		created by k3sato on May 18th in 1994.
 *
 *	modification history:	
 *	(1) add hdSetupUserInfo() in main()
 *		on July 21st in '94.
 *	(2) add case HdMsg_GetParaAgent,
 *		case HdMsg_RegisterParaAgent in main()
 *		on November 14th in '94.
 *
 * $Id: heliosd.c,v 2.4 1995/03/14 06:13:07 k3sato Exp $
 */

#include <stdio.h>
#include <dirent.h>
#include <sys/socket.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <memory.h>

#include <helios/heliosd.h>
#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>

#include "hdVersion.h"
#include "hdDAMessage.h"
#include "hdDDMessage.h"
#include "hdcommon.h"
#include "hdSocket.h"

/* Heliosd Control information
 */
HeliosdCtlInfo	hdCtlInfo;


/*	defined in hdHiaInfo.c
 */
extern HdBoolean hdGetHiaInfo();

/*	defined in hdUserInfo.c
 */
extern HdBoolean hdSetupUserInfo();
extern HdBoolean hdGetUserInfo();

/*	defined in hdAppGuiInfo.c
 */
extern HdBoolean hdGetAppGuiInfo();

/*	functions defined here
 */
static void hdSetupCtlInfo();
static HdBoolean hdCreateTmpFile();
static HdBoolean hdSetupTmpFile();

static char *hdGetHeliosdFullPath();
static HdBoolean heliosdPathSearch();
static HdBoolean hdGetMachInfo();
static HdBoolean hdGetOtherMachInfo();
static HdBoolean hdCheckDalComment();

static HMachInfo hdMakeDataOfMachInfo();
static void hdCatchupSigCld();
static HAgentInfo hdSearchAgentInfoForPid();

static void hdCatchupSigQuit();
static void hdCloseAgentProcess();
static void hdCloseGuiProcess();
static void hdRemoveLockFile();
static void hdErrorExit();
void hdPutsRusage();

int main(argc, argv)
    int		argc;
    char	*argv[];
{
    int			s_listen, s;
    int			msgType;
    char		agentId[MsgBufferLen_Lim];
    char		msgBuf[MsgBufferLen_Lim];
    int			namelen;
    HdBoolean		ret;

    /*
     *	/tmp $B$K(B heliosd $B$N%m%C%/MQ%U%!%$%k$N:n@.$9$k!#(B
     *	$B0J2<$N>pJs$r%;%C%H(B
     *		(1) heliosd $B$N%W%m%;%9#I#D(B
     *		(2) heliosd $B$N%=%1%C%H$N%]!<%HHV9f(B
     */
    if (hdCreateTmpFile() == HdFalse) {
	exit(HdEMsg_CreateTmpFile);
    }

    /* Setup (HeliosdCtlInfo) hdCtlInfo
     */
    hdSetupCtlInfo(argc, argv);

    if ((s_listen = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	hdErrorExit(HdEMsg_SocketForServerFailed);
    }

    hdCtlInfo.s_listen = s_listen;

    if (bind(s_listen, (struct sockaddr *)&(hdCtlInfo.machInfo->sockAddress),
	     sizeof(struct sockaddr_in)) == ERROR_FLAG) {
	hdErrorExit(HdEMsg_CouldNotBind);
    }

    namelen = 0;
    if (getsockname(s_listen, (struct sockaddr *)&(hdCtlInfo.machInfo->sockAddress),
		    &namelen) == ERROR_FLAG) {
	hdErrorExit(HdEMsg_GetSockName);
    }
#ifdef _DEBUG_HELIOSD_PORT
    printf("namelen = %d\n", namelen);
    printf("sin_port = %d\n", ntohs(hdCtlInfo.machInfo->sockAddress.sin_port));
#endif

    /*
     *	/tmp $B$N(B heliosd $B$N%m%C%/MQ%U%!%$%k$K0J2<$N>pJs$r%;%C%H(B
     *		(1) heliosd $B$N%W%m%;%9#I#D(B
     *		(2) heliosd $B$N%=%1%C%H$N%]!<%HHV9f(B
     */
    if (hdSetupTmpFile(hdCtlInfo.machInfo->sockAddress.sin_port) == HdFalse) {
	hdErrorExit(HdEMsg_SetupTmpFile);
    }

    /* opening message
     */
    printf("\n\t%s \t%s \t%s\n\t\t\t%s\n",
	   HELIOSD_VER_HEADER, HELIOSD_VER_NUM, HELIOSD_VER_DATE, HELIOSD_COPYRIGHT);

    if (listen(s_listen, HdSocketBackLog_Lim) == ERROR_FLAG) {
	hdErrorExit(HdEMsg_CanNotListen);
    }

    hdPutsGoodMsg(HdGMsg_NowListens);

    for (;;) {
	if ((s = hdSelectSocket(&hdCtlInfo)) == -1) {
	    continue;
	}

	/*
	 * $B%a%C%;!<%8$N2r@O(B
	 */
	bzero(msgBuf, MsgBufferLen_Lim);

	if (hdSockRead(&hdCtlInfo, s, msgBuf, MsgBufferLen_Lim) == HdFalse) {
	    continue;
	}

	sscanf(msgBuf, "%d %s", &msgType, agentId);

	hdCtlInfo.msgType = msgType;

	switch(msgType) {
	/*
	 *	Messages between daemon and agents
	 */
	case HdMsg_GetAgent:
	    ret = hdGetAgent(&hdCtlInfo, agentId);
	    break;
	case HdMsg_GetAgentInfo:
	    ret = hdGetAgentInfo(&hdCtlInfo, agentId);
	    break;
	case HdMsg_GetAllAgentInfo:
	    ret = hdGetAllAgentInfo(&hdCtlInfo, agentId);
	    break;
	case HdMsg_DeleteAgent:
	    ret = hdDeleteAgent(&hdCtlInfo, agentId);
	    break;
	case HdMsg_SetupUserInfo:
	    ret = hdSetupUserInfo(&hdCtlInfo, agentId);
	    break;
	case HdMsg_GetUserInfo:
	    ret = hdGetUserInfo(&hdCtlInfo, agentId);
	    break;

	case HdMsg_StartAppGui:
	    ret = hdStartAppGui(&hdCtlInfo, agentId);
	    break;

	case HdMsg_ResetAgent:
	    ret = hdResetAgent(&hdCtlInfo, agentId);
	    break;

	case HdMsg_GetParaAgent:
	    ret = hdGetParaAgent(&hdCtlInfo, agentId);
	    break;

	case HdMsg_RegisterParaAgent:
	    ret = hdRegisterParaAgent(&hdCtlInfo, agentId);
	    break;

	/*
	 *	Messages between daemons
	 */
	case HdMsg_AskAgentInfo:
	    ret = hdAskAgentInfo(&hdCtlInfo, agentId);
	    break;
	case HdMsg_AskForCreateAgent:
	    ret = hdAskForCreateAgent(&hdCtlInfo, agentId);
	    break;
	default:
	    hdSendErrorMsg(&hdCtlInfo, HdEMsg_NotExistMsgType);
	    ret = HdTrue;
	    break;
	}

	if (ret == HdTrue) {
	    close(s);
	}
    }
}

/*
 * Setup (HeliosdCtlInfo) hdCtlInfo
 */
static void hdSetupCtlInfo(argc, argv)
    int		argc;
    char	*argv[];
{
    HeliosdErrorMsgType	eMsgType;

    memset(&hdCtlInfo, 0, sizeof(HeliosdCtlInfo));

    hdCtlInfo.nextPortNum = HdAgentPortNumBase;

    /*
     * Get full path name of heliosd;
     *	<full path> + '/'
     */
    if ((hdCtlInfo.fullPath = hdGetHeliosdFullPath(argv[0])) == NULL) {
	eMsgType = HdEMsg_GetHeliosdFullPathError;
	goto Error_Exit;
    }

    /*
     *	$B;R%W%m%;%9$+$iAw$i$l$F$/$k%7%0%J%k$r=hM}$9$k(B
     *	$B4X?t$N@_Dj(B
     */
    signal(SIGCLD, hdCatchupSigCld);

    /*
     *	QUIT $B%7%0%J%k(B (= SIGQUIT) $B$r=hM}$9$k4X?t$N@_Dj(B
     */
    signal(SIGQUIT, hdCatchupSigQuit);

    /* $B%5!<%P$r5/F0$7$?%^%7%s>pJs(B
     * $B5Z$S(B heliosd $B$N$"$k%^%7%s>pJs$N3MF@(B
     */
    if (hdGetMachInfo(&hdCtlInfo, &(hdCtlInfo.machInfo)) == HdFalse) {
	eMsgType = HdEMsg_GetMachInfoError;
	goto Error_Exit;
    }

    /*
     *	$B%X%j%*%9!&%G!<%b%s$,N)$A>e$2$i$l$?%^%7%s$G(B
     *	$B07$&$3$H$N=PMh$kA4%(!<%8%'%s%H>pJs$N3MF@(B
     */
    if (hdGetHiaInfo(&hdCtlInfo, &hdCtlInfo.allAgents) == HdFalse) {
	eMsgType = HdEMsg_HiaFileError;
	goto Error_Exit;
    }

    /*
     *	$B%X%j%*%9!&%G!<%b%s$,N)$A>e$2$i$l$?%^%7%s$G(B
     *	$B07$&$3$H$N=PMh$k%(!<%8%'%s%H8GM-$N(B GUI $B>pJs$N3MF@(B
     */
    if (hdGetAppGuiInfo(&hdCtlInfo, &hdCtlInfo.appGuiInfo) == HdFalse) {
	eMsgType = HdEMsg_AppGuiFileError;
	goto Error_Exit;
    }

    return;

 Error_Exit:
    hdRemoveLockFile();
    hdErrorExit(eMsgType);
}

static char HeliosdAlreadyRun[] = "\
'heliosd' may be already running.\n\
\tIf 'heliosd' is not running, lock-file may remain accidentally.\n\
\tAfter making sure that 'heliosd' is not running,\n\
\tPlease execute the following command.\n\n\
\trm %s\n";

static char hdNotCreateLockFile[] = "Can't create lock file(%s)\n";

/*
 * Create the lock file for heliosd
 *	$BF10l%^%7%s>e$GJ#?t$N%5!<%P$,5/F0$7$J$$$h$&$K(B
 *	$B%m%C%/%U%!%$%k$r:n$k(B
 */
static HdBoolean hdCreateTmpFile()
{
    DIR		*lockDirP;
    int		lockfd;
    char	lockFile[HeliosdFilePathLen_Lim];

    if ((lockDirP = opendir(HeliosdLockFileDir)) == NULL) {
	fprintf(stderr, "Can't opendir %s\n", HeliosdLockFileDir);
	return HdFalse;
    }

    sprintf(lockFile, "%s/%s", HeliosdLockFileDir, HeliosdLockFileName);
    if ((lockfd = open(lockFile, O_CREAT|O_RDWR|O_EXCL, 00644)) < 0) {
	fprintf(stderr, (errno == EEXIST) ? HeliosdAlreadyRun : hdNotCreateLockFile,
		lockFile);
	return HdFalse;
    }
    close(lockfd);
    closedir(lockDirP);

    return HdTrue;
}

/*
 *	Setup the lock file for heliosd
 */
static HdBoolean hdSetupTmpFile(portNum)
    short	portNum;
{
    char	lockFile[HeliosdFilePathLen_Lim];
    FILE	*lockfp;
    HeliosdErrorMsgType	eMsgType;

    sprintf(lockFile, "%s/%s", HeliosdLockFileDir, HeliosdLockFileName);

    if ((lockfp = fopen(lockFile, "w")) == (FILE *)NULL) {
	eMsgType = HdEMsg_LockFileOpenError;
	goto Error_Rtn;
    }

    if (fprintf(lockfp, "%d\n", getpid()) == EOF) {
	eMsgType = HdEMsg_LockFileWriteError;
	goto Error_Close;
    }
    if (fprintf(lockfp, "%d\n", portNum) == EOF) {
	eMsgType = HdEMsg_LockFileWriteError;
	goto Error_Close;
    }

    fflush(lockfp);
    fclose(lockfp);

    return HdTrue;

 Error_Close:
    fclose(lockfp);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Get full path name of heliosd;
 *	<full path> + '/'
 */
static char *hdGetHeliosdFullPath(heliosdName)
    char	*heliosdName;
{
    char	*heliosdPath;
    char	*p;
    char	tempPath[MAXPATHLEN];
    char	hdPathName[MAXPATHLEN];

#ifdef _PATH_DEBUG
    fprintf(stderr, "heliosdName : %s\n", heliosdName);
#endif

    strcpy(tempPath, heliosdName);
    if ((p = strrchr(tempPath, '/')) != NULL) {
	*p = NULL;
    } else {
	if (heliosdPathSearch(heliosdName, tempPath) == HdFalse)
	  return NULL;
    }

    if (realpath(tempPath, hdPathName) == NULL) {
	hdPerror();
	fprintf(stderr, "path name of heliosd : %s\n", heliosdName);
	return NULL;
    }

    if ((p = strrchr(hdPathName, '/')) != NULL) {
	*p = NULL;
	sprintf(tempPath, "%s/config/", hdPathName);

	if ((heliosdPath = HdMallocString(tempPath)) != NULL) {

#ifdef _PATH_DEBUG
    fprintf(stderr, "heliosdPath : %s\n", heliosdPath);
#endif
	    return heliosdPath;
	}
    }
    return NULL;
}

/*
 * Get full path name for a program name;
 *	<full path> + '/'
 */
static HdBoolean heliosdPathSearch(hdProgName, hdPathName)
    char		*hdProgName;
    char		*hdPathName;
{
    char		*hdPathEnv;
    char		*p1, *p2;
    char		*pathName = NULL;
    DIR			*dirp;
    DIR			*tempDirp;
    struct dirent	*dp;
    char		fileName[HeliosdFilePathLen_Lim];

#ifdef _PATH_DEBUG
	fprintf(stderr, "hdProgName : %s\n", hdProgName);
#endif

    if ((hdPathEnv = HdMallocString(getenv("PATH"))) == NULL)
      return HdFalse;

    for (p1 = p2 = hdPathEnv; *p1 != NULL; p1 = p2) {
	for (p2 = p1; *p2 != NULL && *p2 != ':'; p2++)
	  ;

	if (*p2 != NULL)
	  *p2++ = '\0';

#ifdef _PATH_DEBUG
	fprintf(stderr, "pathName in PathSearch : %s\n", p1);
#endif

	if ((dirp = opendir(p1)) != NULL) {

	    while ((dp = readdir(dirp)) != NULL) {
		if (strcmp(dp->d_name, hdProgName) == 0) {

		    sprintf(fileName, "%s/%s", p1, hdProgName);
		    if ((tempDirp = opendir(fileName)) != NULL) {
#ifdef _PATH_DEBUG
			fprintf(stderr, "fileName in PathSearch : %s\n", fileName);
#endif
			closedir(tempDirp);
			continue;
		    }
#ifdef _PATH_DEBUG
		    fprintf(stderr, "pathName in PathSearch : %s\n", p1);
#endif
		    pathName = p1;
		    break;
		}
	    }
	    closedir(dirp);

	    if (dp != NULL && pathName != NULL) {
		sprintf(hdPathName, "%s", pathName);
		break;
	    }
	}
    }

    free(hdPathEnv);
    return (pathName != NULL) ? HdTrue : HdFalse;
}

/*
 *	$B%^%7%s>pJs$N3MF@(B
 */
static HdBoolean hdGetMachInfo(hdCtlInfo, hMachInfo)
    HeliosdCtlInfo	*hdCtlInfo;
    HMachInfo		*hMachInfo;
{
    char		hostname[HostNameLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    /* $B%5!<%P$r5/F0$7$?%^%7%s$N%[%9%HL>(B */
    if (gethostname(hostname, HostNameLen_Lim) == ERROR_FLAG) {
	hdPerror();
	eMsgType = HdEMsg_NotFindHostName;
	goto Error_Rtn;
    }

#ifdef _DEBUG_HELIOSD_HOSTNAME
    fprintf(stderr, "host name : %s\n", hostname);
#endif /* _DEBUG_HELIOSD */

    if ((*hMachInfo = hdMakeDataOfMachInfo(hostname)) == NULL) {
	eMsgType = HdEMsg_MakeDataOfMachInfoError;
	goto Error_Rtn;
    }

    if (hdGetOtherMachInfo(hdCtlInfo, *hMachInfo) == HdFalse) {
	eMsgType = HdEMsg_GetOtherMachInfo;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Get machine information of other host machines
 */
static HdBoolean hdGetOtherMachInfo(hdCtlInfo, machInfo)
    HeliosdCtlInfo	*hdCtlInfo;
    HMachInfo		machInfo;
{
    char		dalFileName[HeliosdFilePathLen_Lim];
    char		hostName[HdFileBufferLen_Lim];
    char		*selfHostName;
    FILE		*fd;
    HMachInfo		currInfo;
    HeliosdErrorMsgType	eMsgType;

    sprintf(dalFileName, "%s%s", hdCtlInfo->fullPath, HeliosDalFileName);

    if ((fd = fopen(dalFileName, "r")) == NULL) {
	eMsgType = HdEMsg_DalFileOpenError;
	goto Error_Rtn;
    }

    selfHostName = machInfo->hostName;

    for (currInfo = machInfo
	 ; fgets(hostName, HdFileBufferLen_Lim, fd) != NULL
	 ; ) {
	if (*hostName == NULL)
	  break;

	if (hdCheckDalComment(hostName) == HdTrue)
	  continue;

	if (strcmp(hostName, selfHostName) == 0)
	  continue;

	if ((currInfo->next = hdMakeDataOfMachInfo(hostName)) == NULL) {
	    eMsgType = HdEMsg_MakeDataOfMachInfoError;
	    goto Error_Rtn;
	}
	currInfo = currInfo->next;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Check comment line in *.dal file
 *	return HdTrue, if comment line.
 *	return HdFalse, if not comment line.
 */
static HdBoolean hdCheckDalComment(str)
    char	*str;
{
    for (; *str != NULL && *str == ' '; str++)
      ;

    if (*str == NULL || *str == '#' || *str == '\n')
      goto Good_Rtn;

    /*
     * delete the comment part of non-comment line
     */
    for (; *str != NULL && *str != '#'&& *str != '\n'; str++)
      ;

    if (*str == '#' || *str == '\n')
      *str = '\0';

    return HdFalse;

 Good_Rtn:
    return HdTrue;
}

/*
 *	Make data of MachInfo structure
 */
static HMachInfo hdMakeDataOfMachInfo(hostname)
    char		*hostname;
{
    HMachInfo		machInfo;
    struct hostent	*myhost;
    struct sockaddr_in	*sockAddr;
    HeliosdErrorMsgType	eMsgType;


    if ((myhost = gethostbyname(hostname)) == NULL) {
	eMsgType = HdEMsg_BadHostName;
	goto Error_Rtn;
    }

    if ((machInfo = (HMachInfo)HdMalloc(sizeof(MachInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    if ((machInfo->hostName = HdMallocString(hostname)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    machInfo->addressSize = sizeof(machInfo->sockAddress);
    sockAddr = &(machInfo->sockAddress);

    bzero((char *)sockAddr, machInfo->addressSize);
    sockAddr->sin_family = AF_INET;

    bcopy(myhost->h_addr, (char *)&(sockAddr->sin_addr), myhost->h_length);

#ifdef _Heliosd_Static_Port_Num

    sockAddr->sin_port = HeliosdPortNum;

#else

    sockAddr->sin_port = 0;
    sockAddr->sin_addr.s_addr = INADDR_ANY;

#endif

    return machInfo;

 Error_Rtn:
    hdFreeMachInfo(&machInfo);

    hdPutsErrorMsg(eMsgType);
    return NULL;
}

/* ============================================================================
 *	signal handler
 *
 *	argument: meaning
 *	sig	$B%7%0%J%kHV9f(B
 *	code	$BDI2C$N>\:Y$rDs6!$9$kFCDj$N%7%0%J%k$N%Q%i%a!<%?(B
 *	scp	sigcontext $B9=B$BN!J(Bsignal.h $B$KDj5A$5$l$k!K$r;X$9%]%$%s%?$G$"$j!"(B
 *		$B%7%0%J%kA0$+$i$N%3%s%F%-%9%H$r%j%9%H%"$9$k$?$a$K;HMQ$7$^$9!#(B
 *	addr	$BDI2C$N%"%I%l%9>pJs(B
 * ============================================================================ */
/*
 *	$B;R%W%m%;%9$,(B stop $B0?$O!"(Bexit $B$7$?;~$K?F%W%m%;%9$K(B
 *	$BAw$i$l$k%7%0%J%k$r=hM}$9$k4X?t(B
 */
static void hdCatchupSigCld(sig, code, scp, addr)
    int			sig, code;
    struct sigcontext	*scp;
    char		*addr;
{
    int			pid;
    int			status;
    HAgentInfo		agentInfo;
    int			mask;

    mask = sigblock(sigmask(SIGCLD));

    fprintf(stderr, "stop or exit a child process of heliosd.\n");

    hdPutsRusage();

    if ((pid = wait(&status)) != -1) {
	fprintf(stderr, "exited process ID = %d.\n", pid);

	if ((agentInfo = hdSearchAgentInfoForPid(hdCtlInfo.allAgents, pid)) != NULL) {
	    fprintf(stderr, "killed agent name(full path): %s(%s)\n",
		    agentInfo->agentName, agentInfo->agentPath);
	    agentInfo->pid = 0;
	    agentInfo->status = AS_Sleep;
	} else {
	    fprintf(stderr, "no agent process.\n");
	}
    }

    sigsetmask(mask);
    return;
}

static HAgentInfo hdSearchAgentInfoForPid(agentInfo, pid)
    HAgentInfo		agentInfo;
    int			pid;
{
    HAgentInfo		tempInfo;

    for (; agentInfo != NULL
	 ; agentInfo = agentInfo->next) {
	if (agentInfo->pid == pid && agentInfo->agentPath != NULL) {
	    return agentInfo;
	}
	if ((tempInfo = hdSearchAgentInfoForPid(agentInfo->child, pid)) != NULL) {
	    return tempInfo;
	}
    }

    return NULL;
}

/*
 *	$B?F%W%m%;%9$KBP$7$FAw$i$l$?(B QUIT $B%7%0%J%k(B
 *	(= SIGQUIT)$B$r=hM}$9$k4X?t!#(B
 */
static void hdCatchupSigQuit(sig, code, scp, addr)
    int			sig, code;
    struct sigcontext	*scp;
    char		*addr;
{
    char		lockFile[HeliosdFilePathLen_Lim];
    char		msgBuf[MsgBufferLen_Lim];
    FILE		*fd;
    int			sock;

    /* close agent processes under heliosd
     */
    hdCloseAgentProcess(hdCtlInfo.allAgents, hdCtlInfo.machInfo->hostName);

    hdCloseGuiProcess(hdCtlInfo.userInfo);

    /* close socket.
     */
    sprintf(lockFile, "%s/%s", HeliosdLockFileDir, HeliosdLockFileName);
    fd = fopen(lockFile, "r");
    fgets(msgBuf, MsgBufferLen_Lim, fd);
    fgets(msgBuf, MsgBufferLen_Lim, fd);
    sscanf(msgBuf,"%d", &sock);
    close(fd);

    close(sock);

    hdRemoveLockFile();

    exit(sig);
}

/*
 * Close agent processes under heliosd
 */
static void hdCloseAgentProcess(allAgents, selfMachName)
    HAgentInfo	allAgents;
    char	*selfMachName;
{
    HAgentInfo	currAgent;
    int		pid;
    char	*pathName;
    int		mask;

    mask = sigblock(sigmask(SIGCLD));

    for (currAgent = allAgents
	 ; currAgent != NULL
	 ; currAgent = currAgent->next) {

	hdCloseAgentProcess(currAgent->child, selfMachName);

	if (currAgent->status == AS_Active
	    && (currAgent->type == HdAT_ParaInstance ||
		(currAgent->type != HdAT_Parametric &&
		 currAgent->agentPath != NULL))
	    && (strcmp(selfMachName, currAgent->sockInfo.hostName) == 0)) {

	    if (kill(currAgent->pid, SIGQUIT) != -1) {

		pathName = (currAgent->type == HdAT_ParaInstance) ?
		  currAgent->parent->agentPath : currAgent->agentPath;

		pid = wait(NULL);
		fprintf(stderr,
			"killed agent info:\n\tpid = %d\n\tpath name = %s\n",
			pid, pathName);
		hdPutsRusage();
	    }
	}
    }
    sigsetmask(mask);
}

/*
 * Close Gui processes under heliosd
 */
static void hdCloseGuiProcess(userInfo)
    HAgentInfo	userInfo;
{
    HAgentInfo	currInfo;
    int		mask;

    mask = sigblock(sigmask(SIGCLD));

    for (currInfo = userInfo
	 ; currInfo != NULL
	 ; currInfo = currInfo->next) {

	if (currInfo->agentPath != NULL &&
	    currInfo->status == AS_Active) {
	    if (kill(currInfo->pid, SIGQUIT) != -1) {
		wait(NULL);
		fprintf(stderr, "killed gui name: %s\n",
			currInfo->agentPath);
		hdPutsRusage();
	    }
	}
    }
    sigsetmask(mask);
}

/*
 *	remove lock file 
 */
static void hdRemoveLockFile()
{
    char		lockFile[HeliosdFilePathLen_Lim];

    sprintf(lockFile, "%s/%s", HeliosdLockFileDir, HeliosdLockFileName);

    if (unlink(lockFile) < 0) {
	hdPerror();
	hdPutsErrorMsg(HdEMsg_LockFileRemoveError);
    }
}

/*
 *
 */
static void hdErrorExit(eMsgType)
    HeliosdErrorMsgType	eMsgType;
{
    hdPutsErrorMsg(eMsgType);
    exit(eMsgType);
}

void hdPutsRusage()
{
    struct rusage rusageBuf;

    if (getrusage(RUSAGE_CHILDREN, &rusageBuf) == -1) {
	hdPerror();
    } else {
	fprintf(stderr, "  user time :%ld $BIC(B %ld\n",
		rusageBuf.ru_utime.tv_sec, rusageBuf.ru_utime.tv_usec);
	fprintf(stderr, "system time :%ld $BIC(B %ld\n",
		rusageBuf.ru_stime.tv_sec, rusageBuf.ru_stime.tv_usec);
	fprintf(stderr, "%ld messages are sent through sockets.\n",
		rusageBuf.ru_msgsnd);
	fprintf(stderr, "%ld messages are received through sockets.\n",
		rusageBuf.ru_msgrcv);
    }
}
