/* hdSocket.c;
 *	module to handle socket between Helios Daemon and Agent.
 *		created by k3sato on June 21st in 1994.
 *
 *	modification history:	
 *	(1) add hdSetPortNum2AgentInfo()
 *		on October 4th in '94.
 *	(2) add hdSendErrorMsg()
 *		on October 14th in '94.
 *	(3) add hdSendGoodMsg()
 *		on October 17th in '94.
 *	(4) add hdSelectSocket()
 *		on December 8th in '94.
 *
 * $Id: hdSocket.c,v 2.1 1994/12/10 09:26:10 k3sato Exp $
 */

#include <stdio.h>
#include <sys/time.h>

#include <helios/heliosd.h>
#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>

#include <sys/socket.h>

#include "hdRetryMsgHandle.h"
#include "hdSocket.h"
#include "hdcommon.h"

static struct timeval hdSelectTimeout = {(long)1, (long)0};
static struct timeval hdSockRWTimeout = {(long)10, (long)0};

/* =================================================
 *  select event handling for helios daemon socket
 * ================================================= */
int hdSelectSocket(hHdCtlInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
{
    int			sock;
    int			s;
    fd_set		mask;
    fd_set		readOk;
    int			width;
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    hHdCtlInfo->waitFlag = HdAMS_NoWait;

    sock = hHdCtlInfo->s_listen;

    FD_ZERO(&mask);
    FD_SET(sock, &mask);
    width = sock + 1;

    for (;;) {
	/* select $B$NA0$K(B readOk $B$r=i4|2=$9$k!#(B
	 */
	readOk = mask;

	ret = select(width, (fd_set *)&readOk, NULL, NULL, &hdSelectTimeout);
	if (ret <= 0) {
	    if (ret == -1) {
		hdPerror();
	    }
	    hdRetryWaitMsg(hHdCtlInfo);
	    continue;
	}

	/* $B%=%1%C%H$+$iFI$_9~$_2D$+!)(B
	 */
	if (FD_ISSET(sock, &readOk)) {
	    if ((s = accept(sock, NULL, NULL)) == -1) {
		hdPerror();
		continue;
	    }

	    hHdCtlInfo->s = s;
	    break;
	}
    }

    return s;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return -1;
}

/* =================================================
 *  retry to write waiting messages to Daemon
 *     or to read reply messages from Daemon
 * ================================================= */
void hdRetryWaitMsg(hHdCtlInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
{
    HHdAcceptMsgInfo	currInfo;
    HeliosdErrorMsgType	eMsgType;

    for (currInfo = hHdCtlInfo->acceptMsg
	 ; currInfo != NULL
	 ; currInfo = currInfo->next) {

	eMsgType = NULL;

	switch (currInfo->type) {
	case HdMsg_GetAgent:
	    if (hdRetry2GetAgent(hHdCtlInfo, currInfo) == HdFalse) {
		eMsgType = HdEMsg_Retry2GetAgentError;
	    }
	    break;

	default:
	    break;
	}

	if (eMsgType != NULL) {
	    hdPutsErrorMsg(eMsgType);
	}
    }

    return;
}

/* ==================================================
 *	Functions to Write a message to Agent
 * ================================================== */
/* ====================================================
 *  Send a HeliosdMsgType message to a accepted socket
 * ==================================================== */
HdBoolean hdSendMsgType(hHdCtlInfo, msgType)
    HHeliosdCtlInfo	hHdCtlInfo;
    HeliosdMsgType	msgType;
{
    char		sendBuf[MsgBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    sprintf(sendBuf, "%d", msgType);

    if (hdSockWrite(hHdCtlInfo, hHdCtlInfo->s, sendBuf, strlen(sendBuf) + 1)
	== HdFalse) {
	eMsgType = HdEMsg_SocketWriteError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ======================================
 *	Send error message
 * ======================================*/
void hdSendErrorMsg(hHdCtlInfo, eMsgType)
    HHeliosdCtlInfo	hHdCtlInfo;
    HeliosdErrorMsgType	eMsgType;
{
    char	*msg;
    char	sendBuf[MsgBufferLen_Lim];

    if ((msg = hdGetErrorMsg(eMsgType)) == NULL) {
	msg = "No error message";
    }

    if (sprintf(sendBuf, "%d %s", HdMsg_ErrorDetected, msg)
	!= NULL) {
	hdSend2Socket(hHdCtlInfo, hHdCtlInfo->s, sendBuf);
    }
}

/* ======================================
 *	Send good message
 * ======================================*/
void hdSendGoodMsg(hHdCtlInfo, gMsgType)
    HHeliosdCtlInfo	hHdCtlInfo;
    HeliosdGoodMsgType	gMsgType;
{
    char	*msg;
    char	sendBuf[MsgBufferLen_Lim];

    if ((msg = hdGetGoodMsg(gMsgType)) == NULL) {
	msg = "No good message";
    }

    if (sprintf(sendBuf, "%d %s", HdMsg_Success, msg)
	!= NULL) {
	hdSend2Socket(hHdCtlInfo, hHdCtlInfo->s, sendBuf);
    }
}

/* ==================================================
 *	Send a message to socket
 * ================================================== */
HdBoolean hdSend2Socket(hHdCtlInfo, sock, sendBuf)
    HHeliosdCtlInfo	hHdCtlInfo;
    int			sock;
    char		*sendBuf;
{
    HeliosdErrorMsgType	eMsgType;

    if (hdSockWrite(hHdCtlInfo, sock, sendBuf, strlen(sendBuf)) == HdFalse) {
	eMsgType = HdEMsg_SocketWriteError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ===============================
 *	send a message to Daemon
 * =============================== */
HdBoolean hdSend2Daemon(hHdCtlInfo, toMachInfo, sendBuf)
    HHeliosdCtlInfo	hHdCtlInfo;
    HMachInfo		toMachInfo;
    char		*sendBuf;
{
    HeliosdErrorMsgType	eMsgType;

    if (hdSockWrite(hHdCtlInfo, toMachInfo->sock, sendBuf, strlen(sendBuf))
	== HdFalse) {
	eMsgType = (hHdCtlInfo->waitFlag == HdTrue) ? NULL : HdEMsg_SocketWriteError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ==================================================
 *	Write a message to Socket
 * ================================================== */
HdBoolean hdSockWrite(hHdCtlInfo, sock, sendBuf, sendBufLen)
    HHeliosdCtlInfo	hHdCtlInfo;
    int			sock;
    char		*sendBuf;
    int			sendBufLen;
{
    fd_set		writeOk;
    int			width;
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    /* select $B$NA0$KKh2s(B writeOk $B$r=i4|2=$9$k!#(B
     */
    FD_ZERO(&writeOk);
    FD_SET(sock, &writeOk);
    width = sock + 1;

    ret = select(width, NULL, (fd_set *)&writeOk, NULL, &hdSockRWTimeout);
    if (ret > 0) {
	/* $B%=%1%C%H$K=q$-9~$_2D$+!)(B
	 */
	if (FD_ISSET(sock, &writeOk)) {
	    if (write(sock, sendBuf, sendBufLen) == ERROR_FLAG) {
		hdPerror();
		eMsgType = HdEMsg_SocketWriteError;
		goto Error_Rtn;
	    }
	} else {
	    hHdCtlInfo->waitFlag = HdTrue;
	    eMsgType = HdEMsg_NotSelect2Write2Sock;
	    goto Error_Rtn;
	}
    } else {
	if (ret == 0) {
	    hHdCtlInfo->waitFlag = HdTrue;
	    eMsgType = HdEMsg_TimeoutOnSelect2WriteSock;
	} else {
	    hdPerror();
	    eMsgType = HdEMsg_SelectError;
	}
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ==================================================
 *	Functions to receive a message from a socket
 * ================================================== */
/*
 *  Get the message type of HdMsg_AskforNextInfo
 */
HdBoolean hdGetAskForNextInfo(hHdCtlInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
{
    char		msgBuf[MsgBufferLen_Lim];
    HeliosdMsgType	msgType;    
    HeliosdErrorMsgType	eMsgType;

    if (hdSockRead(hHdCtlInfo, hHdCtlInfo->s, msgBuf, MsgBufferLen_Lim)
	== HdFalse) {
	eMsgType = HdEMsg_SocketReadError;
	goto Error_Rtn;
    }

    sscanf(msgBuf, "%d", &msgType);

    if (msgType != HdMsg_AskForNextInfo) {
	eMsgType = HdEMsg_NotAskForNextInfo;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	Read from socket
 */
HdBoolean hdSockRead(hHdCtlInfo, sock, readBuf, bufMaxLen)
    HHeliosdCtlInfo	hHdCtlInfo;
    int			sock;
    char		*readBuf;
    int			bufMaxLen;
{
    fd_set		mask;
    fd_set		readOk;
    int			width;
    int			retry;
    int			length;
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    FD_ZERO(&mask);
    FD_SET(sock, &mask);
    width = sock + 1;

    for (retry = HdMaxNumOfRetry; retry > 0; retry--) {
	/* select $B$NA0$K(B readOk $B$r=i4|2=$9$k!#(B
	 */
	readOk = mask;

	ret = select(width, (fd_set *)&readOk, NULL, NULL, &hdSockRWTimeout);
	if (ret == -1) {
	    hdPerror();
	    eMsgType = HdEMsg_SelectError;
	    goto Error_Rtn;
	}
	if (ret == 0) {
	    continue;
	}

	/* $B%=%1%C%H$+$iFI$_9~$_2D$+!)(B
	 */
	if (FD_ISSET(sock, &readOk)) {
	    if ((length = read(sock, readBuf, bufMaxLen)) == ERROR_FLAG) {
		hdPerror();
		eMsgType = HdEMsg_SocketReadError;
		goto Error_Rtn;
	    }
	    readBuf[length] = '\0';
	    if (length >= bufMaxLen) {
		eMsgType = HdEMsg_SocketBufferLenLimitOver;
		goto Error_Rtn;
	    }
	    return HdTrue;
	}
    }

    if (ret == 0) {
	hHdCtlInfo->waitFlag = HdTrue;
    }

    eMsgType = HdEMsg_OverMaxNumOfRetryToReadSoc;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ==================================================
 *	functions to handle SocketInfo in AgentInfo
 * ================================================== */
/*
 *	set sin_port in AgentInfo
 */
HdBoolean hdSetPortNum2AgentInfo(hHdCtlInfo, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HAgentInfo		agentInfo;
{
    HeliosdErrorMsgType	eMsgType;

    if (hHdCtlInfo->nextPortNum > HeliosdPortNum_Lim) {
	eMsgType = HdEMsg_OverMaxNumOfHeliosdPortNum;
	goto Error_Rtn;
    }

    agentInfo->sockInfo.sin_port = hHdCtlInfo->nextPortNum++;

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	$BL$;HMQ$N%]!<%HHV9f$N3MF@(B
 */
HdBoolean hdGetPortNumber(hHdCtlInfo, portNum)
    HHeliosdCtlInfo	hHdCtlInfo;
    u_short		*portNum;

#ifdef _Heliosd_Static_Port_Num

{
    HeliosdErrorMsgType	eMsgType;

    if (hHdCtlInfo->nextPortNum > HeliosdPortNum_Lim) {
	eMsgType = HdEMsg_OverMaxNumOfHeliosdPortNum;
	goto Error_Rtn;
    }

   *portNum = hHdCtlInfo->nextPortNum++;

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

#else

{
    int			s_listen;
    struct hostent	*myhost;
    struct sockaddr_in	me;
    int			namelen;
    HeliosdErrorMsgType	eMsgType;

    bzero((char *)&me, sizeof(me));
    me.sin_family = AF_INET;
    me.sin_addr.s_addr = INADDR_ANY;
    me.sin_port = 0;
    
    bcopy(hHdCtlInfo->machInfo->hostAddress, (char *)&me.sin_addr, hHdCtlInfo->machInfo->h_length);

    if ((s_listen = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	eMsgType = HdEMsg_SocketCreateFailed;
	goto Error_Rtn;
    }

    if (bind(s_listen, (struct sockaddr *)&me, sizeof(me)) == ERROR_FLAG) {
	eMsgType = HdEMsg_CouldNotBind;
	goto Error_Rtn;
    }

    namelen = 0;

    if (getsockname(s_listen, (struct sockaddr *)&me, &namelen) == ERROR_FLAG) {
	eMsgType = HdEMsg_GetSockName;
	goto Error_Rtn;
    }

    /*	bind $B$5$l$?%]!<%HHV9f$rJV$9!#(B
     */
    *portNum = me.sin_port;

    close(s_listen);

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

#endif	/* _Heliosd_Static_Port_Num */
