/* hdRetryMsgHandle.c;
 *	module to retry to handle messages between Helios Daemon and Agent
 *	and between Daemons;
 *		created by k3sato on December 8th in 1994.
 *
 *	modification history:
 *
 * $Id: hdRetryMsgHandle.c,v 2.3 1994/12/10 10:50:05 k3sato Exp $
 */

#include <stdio.h>
#include <string.h>
#include <memory.h>

#include <helios/heliosd.h>
#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>
#include "hdAgentInfo.h"
#include "hdDDMessage.h"
#include "hdRetryMsgHandle.h"
#include "hdSocket.h"
#include "hdcommon.h"


/* ===================================================
 *  retry functions for the message of HdMsg_GetAgent
 * =================================================== */
/*
 *  retry to get a agent socket info
 */
HdBoolean hdRetry2GetAgent(hHdCtlInfo, msgInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HHdAcceptMsgInfo	msgInfo;
{
    HHdMsgHandleArgs	hArgs;
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;
    HdBoolean		ret;

    hArgs = &(msgInfo->args);

    switch (msgInfo->status) {
    case HdAMS_ReceiveAgentInfoWait:

	ret = hdRetry2ReceiveAgentInfoFromDaemon(hHdCtlInfo,
						 msgInfo->machInfo, &agentInfo);

	if (ret == HdTrue) {
	    if (hdCloseSocket2OtherDaemon(msgInfo->machInfo) == HdFalse) {
		eMsgType = HdEMsg_CloseSocket2OtherDaemon;
		goto Error_Rtn;
	    }
	    break;
	}
	msgInfo->machInfo = msgInfo->machInfo->next;

    case HdAMS_SendAskAgentInfoWait:
	ret = hdRetry2GetAgentInfoFromOthers(hHdCtlInfo, msgInfo, &agentInfo);

	break;

    case HdAMS_SendAskForCreatingWait:

	break;

    case HdAMS_SendSockInfoWait:
	ret = HdTrue;
	break;
    case HdAMS_ReceiveSockWait:

	switch (hArgs->type) {
	case HdFT_ReceiveSockInfo:
	    ret = hdRetry2ReceiveSockInfoFromDaemon(hHdCtlInfo, msgInfo->machInfo,
						    msgInfo->args.arg1.agentInfo);

	    break;
	default:
	    break;
	}

	if (ret == HdTrue) {
	    if (hdCloseSocket2OtherDaemon(msgInfo->machInfo) == HdFalse) {
		eMsgType = HdEMsg_CloseSocket2OtherDaemon;
		goto Error_Rtn;
	    }
	    agentInfo = msgInfo->args.arg1.agentInfo;
	}

	break;
    case HdAMS_NoWait:
    default:
	eMsgType = HdEMsg_AgentStatusTypeError;
	goto Error_Rtn;
    }

    /*  send SocketInfo to agent
     */
    if (ret == HdTrue) {
	if (hdSendSocketInfo(hHdCtlInfo, msgInfo->acceptFd, agentInfo) == HdFalse) {
	    msgInfo->status = HdAMS_SendSockInfoWait;
	    eMsgType = HdEMsg_SendSocketInfoError;
	    goto Error_Rtn;
	}

	if (hdRemoveAcceptMsgInfo(hHdCtlInfo, msgInfo) == HdFalse) {
	    eMsgType = HdEMsg_RemoveAcceptMsgInfoError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Retry to get agentInfo from other Daemons
 */
HdBoolean hdRetry2GetAgentInfoFromOthers(hHdCtlInfo, msgInfo, hAgentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HHdAcceptMsgInfo	msgInfo;
    HAgentInfo		*hAgentInfo;    
{
    HMachInfo		currMachInfo;
    char		sendBuf[MsgBufferLen_Lim];
    int			ret;
    HeliosdErrorMsgType	eMsgType;

    sprintf(sendBuf, "%d %s",
	    HdMsg_AskAgentInfo, msgInfo->args.arg1.agentId);

    for (currMachInfo = msgInfo->machInfo, ret = HdFalse
	 ; currMachInfo != NULL && ret == HdFalse
	 ; currMachInfo = currMachInfo->next) {

	if (hdCreateSocket2OtherDaemon(currMachInfo) == HdFalse) {
	    eMsgType = HdEMsg_CreateSocket2OtherDaemon;
	    goto Error_Rtn;
	}

	if (hdSend2Daemon(hHdCtlInfo, currMachInfo, sendBuf) == HdFalse) {
	    eMsgType = HdEMsg_Send2Daemon;
	    goto Error_Rtn_Close;
	}

	ret = hdReceiveAgentInfoFromDaemon(hHdCtlInfo, currMachInfo, hAgentInfo);

	if (hdCloseSocket2OtherDaemon(currMachInfo) == HdFalse) {
	    eMsgType = HdEMsg_CloseSocket2OtherDaemon;
	    goto Error_Rtn;
	}
    }

    if (ret == HdFalse) {
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn_Close:
    hdCloseSocket2OtherDaemon(currMachInfo);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * retry to ask for creating to other daemon
 */
HdBoolean hdRetry2AskForCreating2Other(hHdCtlInfo, msgInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HHdAcceptMsgInfo	msgInfo;
{
    HMachInfo		machInfo;
    char		sendBuf[MsgBufferLen_Lim];
    int			ret;
    HeliosdErrorMsgType	eMsgType;

    sprintf(sendBuf, "%d %s", HdMsg_AskForCreateAgent, msgInfo->agentId);

    if (hdSend2Daemon(hHdCtlInfo, msgInfo->machInfo, sendBuf) == HdFalse) {
	if (hHdCtlInfo->waitFlag == HdTrue) {
	    eMsgType = NULL;
	} else {
	    eMsgType = HdEMsg_Send2Daemon;
	}
	goto Error_Rtn;
    }

    ret = hdRetry2ReceiveSockInfoFromDaemon(hHdCtlInfo, machInfo,
					    msgInfo->args.arg1.agentInfo);

    if (ret == HdFalse && hHdCtlInfo->waitFlag == HdTrue) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    if (hdCloseSocket2OtherDaemon(machInfo) == HdFalse) {
	eMsgType = HdEMsg_CloseSocket2OtherDaemon;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Retry to receive agentInfo from Other Daemons
 */
HdBoolean hdRetry2ReceiveAgentInfoFromDaemon(hHdCtlInfo, machInfo, hAgentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HMachInfo		machInfo;
    HAgentInfo		*hAgentInfo;
{
    HeliosdMsgType	msgType;
    char		recvBuf[MsgBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if (hdSockRead(hHdCtlInfo, machInfo->sock, recvBuf, MsgBufferLen_Lim)
	== HdFalse) {
	if (hHdCtlInfo->waitFlag == HdTrue) {
	    eMsgType = NULL;
	} else {
	    eMsgType = HdEMsg_SocketReadError;
	}
	goto Error_Rtn;
    }

    if (hdSetupAgentInfoFromMsg(hAgentInfo, &msgType, recvBuf) == HdFalse) {
	eMsgType = HdEMsg_SetupAgentInfoFromMsgError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Retry to Receive sockInfo from Other Daemons
 */
HdBoolean hdRetry2ReceiveSockInfoFromDaemon(hHdCtlInfo, machInfo, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HMachInfo		machInfo;
    HAgentInfo		agentInfo;
{
    HeliosdMsgType	msgType;
    char		hostName[HostNameLen_Lim];
    int			sin_family;
    int			sin_port;
    char		recvBuf[MsgBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    if (hdSockRead(hHdCtlInfo, machInfo->sock, recvBuf, MsgBufferLen_Lim)
	== HdFalse) {
	if (hHdCtlInfo->waitFlag == HdTrue) {
	    eMsgType = NULL;
	} else {
	    eMsgType = HdEMsg_SocketReadError;
	}
	goto Error_Rtn;
    }

    ret = sscanf(recvBuf, "%d", &msgType);
    if (ret == 0 || ret == EOF) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    if (msgType != HdMsg_SocketInfo) {
	eMsgType = HdEMsg_NotDefinedAgentInfoOnOthers;
	goto Error_Rtn;
    }

    ret = sscanf(recvBuf, "%d %s %d %d",
		 &msgType, hostName, &sin_family, &sin_port);

    if (ret == 0 || ret == EOF) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    agentInfo->sockInfo.sin_family = sin_family;
    agentInfo->sockInfo.sin_port = sin_port;
    agentInfo->status = AS_Active;

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* 
 *
 */
HHdAcceptMsgInfo hdCreateAcceptMsgInfo(hHdCtlInfo, machInfo, status)
    HHeliosdCtlInfo		hHdCtlInfo;
    HMachInfo			machInfo;
    HdAcceptMsgStatusType	status;
{
    HHdAcceptMsgInfo	msgInfo;
    HeliosdErrorMsgType	eMsgType;

    if ((msgInfo = (HHdAcceptMsgInfo)HdMalloc(sizeof(HdAcceptMsgInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    msgInfo->type = hHdCtlInfo->msgType;
    msgInfo->acceptFd = hHdCtlInfo->s;
    msgInfo->status = status;

    if ((msgInfo->machInfo = (HMachInfo)HdMalloc(sizeof(MachInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    memcpy(msgInfo->machInfo, machInfo, sizeof(MachInfo));

    hdAppendAcceptMsgInfo(hHdCtlInfo, msgInfo);

    return msgInfo;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 *
 */
HdBoolean hdAppendAcceptMsgInfo(hHdCtlInfo, msgInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HHdAcceptMsgInfo	msgInfo;
{
    HHdAcceptMsgInfo	*currInfo;

    for (currInfo = &(hHdCtlInfo->acceptMsg)
	 ; *currInfo != NULL
	 ; currInfo = &((*currInfo)->next))
      ;

    *currInfo = msgInfo;

    return HdTrue;
}

/*
 *
 */
HdBoolean hdRemoveAcceptMsgInfo(hHdCtlInfo, msgInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HHdAcceptMsgInfo	msgInfo;
{
    HHdAcceptMsgInfo	*currInfo;

    for (currInfo = &(hHdCtlInfo->acceptMsg)
	 ; *currInfo != NULL
	 ; currInfo = &((*currInfo)->next)) {

	if (*currInfo == msgInfo) {
	    *currInfo = msgInfo->next;

	    hdFreeAcceptMsgInfo(&msgInfo);

	    return HdTrue;
	}
    }
    return HdFalse;
}

void hdFreeAcceptMsgInfo(msgInfoP)
    HHdAcceptMsgInfo	*msgInfoP;
{
    HHdAcceptMsgInfo	currInfo;

    if (msgInfoP == NULL || (currInfo = *msgInfoP) == NULL)
      return;

    hdFreeString(&(currInfo->agentId));
    hdFree(currInfo->machInfo);
    hdFree(currInfo);
    *msgInfoP = NULL;
}
