/* hdDDMessage.c;
 *	module to handle messages between Helios Daemons.
 *		created by k3sato on Aug. 8th in 1994.
 *
 *	modification history:	
 *	(1) add hdAskForCreateAgent();
 *		on December 7th in '94.
 *
 * $Id: hdDDMessage.c,v 2.5 1994/12/20 02:26:50 k3sato Exp $
 */

#include <stdio.h>

#include <helios/heliosd.h>
#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>

#include "hdDDMessage.h"
#include "hdAgentInfo.h"
#include "hdRetryMsgHandle.h"
#include "hdSocket.h"
#include "hdcommon.h"

#include <sys/socket.h>

/*  defined in hdUserInfo.c
 */
extern HAgentInfo hdSearchUserInfo();

/* defined in hdDAMessage.c
 */
extern HdBoolean hdCreateAgentProcess();


/* ======================================================
 *  functions to get AgentInfo from other daemones
 * ====================================================== */
/*
 * Search AgentInfo on HeliosdCtlInfo->otherDaemonsInfo,
 * if not on otherDaemonsInfo,
 *	get agentInfo from Other Daemons.
 */
HAgentInfo hdSearchOtherDaemonsInfo(hHdCtlInfo, agentId)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
{
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;

    for (agentInfo = hHdCtlInfo->otherDaemonsInfo
	 ; agentInfo != NULL
	 ; agentInfo = agentInfo->next) {
	if (strcmp(agentInfo->agentId, agentId) == 0) {
	    return agentInfo;
	}
    }

    if (hdGetAgentInfoFromOthers(hHdCtlInfo, agentId, &agentInfo) == HdFalse) {
	eMsgType = HdEMsg_NotDefinedAgentInfoOnOthers;
	goto Error_Rtn;
    }

    if (hdAppendOtherDaemonsInfo(hHdCtlInfo, agentInfo) == HdFalse) {
	eMsgType = HdEMsg_AppendOtherDaemonsInfoError;
	goto Error_Rtn_Free;
    }

    return agentInfo;

 Error_Rtn_Free:
    hdFreeOneAgentInfo(&agentInfo);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 * Get agentInfo from other Daemons
 */
HdBoolean hdGetAgentInfoFromOthers(hHdCtlInfo, agentId, hAgentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
    HAgentInfo		*hAgentInfo;
{
    HMachInfo		currMachInfo;
    HHdAcceptMsgInfo	msgInfo;
    char		sendBuf[MsgBufferLen_Lim];
    int			ret;
    HeliosdErrorMsgType	eMsgType;

    sprintf(sendBuf, "%d %s", HdMsg_AskAgentInfo, agentId);

    for (currMachInfo = hHdCtlInfo->machInfo->next, ret = HdFalse
	 ; currMachInfo != NULL && ret == HdFalse
	 ; currMachInfo = currMachInfo->next) {

	if (hdCreateSocket2OtherDaemon(currMachInfo) == HdFalse) {
	    eMsgType = HdEMsg_CreateSocket2OtherDaemon;
	    goto Error_Rtn;
	}

	if (hdSend2Daemon(hHdCtlInfo, currMachInfo, sendBuf) == HdFalse) {
	    if (hHdCtlInfo->waitFlag == HdTrue) {
		if ((msgInfo = hdCreateAcceptMsgInfo(hHdCtlInfo, currMachInfo,
						     HdAMS_SendAskAgentInfoWait))
		    == NULL) {
		    hHdCtlInfo->waitFlag = HdFalse;
		    eMsgType = HdEMsg_CreateAcceptMsgInfoError;
		    goto Error_Rtn;
		}
		msgInfo->agentId = HdMallocString(agentId);
		eMsgType = NULL;
	    } else {
		eMsgType = HdEMsg_Send2Daemon;
	    }
	    goto Error_Rtn_Close;
	}

	ret = hdReceiveAgentInfoFromDaemon(hHdCtlInfo, currMachInfo, hAgentInfo);

	if (hdCloseSocket2OtherDaemon(currMachInfo) == HdFalse) {
	    eMsgType = HdEMsg_CloseSocket2OtherDaemon;
	    goto Error_Rtn;
	}
    }

    if (ret == HdFalse) {
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn_Close:
    hdCloseSocket2OtherDaemon(currMachInfo);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Receive agentInfo from Other Daemons
 */
HdBoolean hdReceiveAgentInfoFromDaemon(hHdCtlInfo, machInfo, hAgentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HMachInfo		machInfo;
    HAgentInfo		*hAgentInfo;
{
    HeliosdMsgType	msgType;
    HHdAcceptMsgInfo	msgInfo;
    char		recvBuf[MsgBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if (hdSockRead(hHdCtlInfo, machInfo->sock, recvBuf, MsgBufferLen_Lim)
	== HdFalse) {
	if (hHdCtlInfo->waitFlag == HdTrue) {
	    if ((msgInfo = hdCreateAcceptMsgInfo(hHdCtlInfo, machInfo,
						 HdAMS_ReceiveAgentInfoWait))
		== NULL) {
		hHdCtlInfo->waitFlag = HdFalse;
		eMsgType = HdEMsg_CreateAcceptMsgInfoError;
		goto Error_Rtn;
	    }
	    eMsgType = NULL;
	} else {
	    eMsgType = HdEMsg_SocketReadError;
	}
	goto Error_Rtn;
    }

    if (hdSetupAgentInfoFromMsg(hAgentInfo, &msgType, recvBuf) == HdFalse) {
	eMsgType = HdEMsg_SetupAgentInfoFromMsgError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Append agentInfo to OtherDaemonsInfo of HeliosdCtlInfo.
 */
HdBoolean hdAppendOtherDaemonsInfo(hHdCtlInfo, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HAgentInfo		agentInfo;
{
    HAgentInfo		*currInfo;
    HeliosdErrorMsgType	eMsgType;

    if (agentInfo == NULL) {
	eMsgType = HdEMsg_NullPointer;
	goto Error_Rtn;
    }

    for (currInfo = &(hHdCtlInfo->otherDaemonsInfo)
	 ; *currInfo != NULL
	 ; currInfo = &((*currInfo)->next))
      ;

    *currInfo = agentInfo;

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ======================================================
 *	functions for the message of HdMsg_AskAgentInfo
 * ====================================================== */
/*
 * AgentId $B$KBP1~$7$?(B AgentInfo $B$r(B accecpt $B$7$?(B socket $B$KJV$9!#(B
 * $BBP1~$7$?(B AgentInfo $B$,L5$1$l$P!"(BHdMsg_NotExistAgent $B$rJV$9!#(B
 */
HdBoolean hdAskAgentInfo(hHdCtlInfo, agentId)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
{
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;

    if ((agentInfo = hdSearchAgentInfo(hHdCtlInfo->allAgents, agentId)) == NULL) {
	/* check parametric agent
	 */



	/* check in userInfo
	 */
	if ((agentInfo = hdSearchUserInfo(hHdCtlInfo->userInfo, agentId)) == NULL) {
	    /* $B%a%C%;!<%8!&%?%$%W(B HdMsg_NotExistAgent $B$r%G!<%b%s$KJV$9!#(B
	     */
	    if (hdSendMsgType(hHdCtlInfo, HdMsg_NotExistAgent) == HdFalse) {
		hdPutsErrorMsg(HdEMsg_SendMsgTypeError);
	    }
	    eMsgType = HdEMsg_NotDefinedAgentInfo;
	    goto Error_Rtn;
	}
    }

    switch(agentInfo->status) {
    case AS_NotActive:
	/* $B5/F0$5$l$F$$$J$$(B agent process $B$N5/F0(B
	 */
	if (hdCreateAgentProcess(hHdCtlInfo, agentInfo) == HdFalse) {
	    eMsgType = HdEMsg_CreateAgentProcessError;
	    goto Error_Rtn;
	}
	break;
    case AS_Active:
	break;
    case AS_NotExist:
	eMsgType = HdEMsg_NotAgentProcessFile;
	goto Error_Rtn;
    default:
	eMsgType = HdEMsg_AgentStatusTypeError;
	goto Error_Rtn;
    }

    /*
     *	send AgentInfo to daemon
     */
    if (hdSendOneAgentInfo(hHdCtlInfo, HdMsg_AgentInfoEnd, agentInfo) == HdFalse) {
	eMsgType = HdEMsg_SendOneAgentInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ========================================================
 *	functions for asking for creating to other daemon
 * ======================================================== */
/*
 * 
 */
HdBoolean hdAskForCreating2OtherDaemon(hHdCtlInfo, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HAgentInfo		agentInfo;
{
    HMachInfo		machInfo;
    HAgentInfo		repAgentInfo;
    HHdAcceptMsgInfo	msgInfo;
    char		sendBuf[MsgBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if ((repAgentInfo = hdGetRepAgentInfo(agentInfo)) == NULL) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    if ((machInfo = hdSearchMachInfo(hHdCtlInfo, repAgentInfo->sockInfo.hostName))
	== NULL) {
	eMsgType = NULL;
	goto Error_Rtn;
    }

    sprintf(sendBuf, "%d %s", HdMsg_AskForCreateAgent, repAgentInfo->agentId);

    if (hdCreateSocket2OtherDaemon(machInfo) == HdFalse) {
	eMsgType = HdEMsg_CreateSocket2OtherDaemon;
	goto Error_Rtn;
    }

    if (hdSend2Daemon(hHdCtlInfo, machInfo, sendBuf) == HdFalse) {
	if (hHdCtlInfo->waitFlag == HdTrue) {
	    if ((msgInfo = hdCreateAcceptMsgInfo(hHdCtlInfo, machInfo,
						 HdAMS_SendAskForCreatingWait))
		== NULL) {
		hHdCtlInfo->waitFlag = HdFalse;
		eMsgType = HdEMsg_CreateAcceptMsgInfoError;
		goto Error_Rtn;
	    }
	    msgInfo->args.arg1.agentInfo = repAgentInfo;
	    eMsgType = NULL;
	    goto Error_Rtn;
	} else {
	    eMsgType = HdEMsg_Send2Daemon;
	    goto Error_Rtn_Close;
	}
    }

    if (hdReceiveSockInfoFromDaemon(hHdCtlInfo, machInfo, repAgentInfo)
	== HdFalse) {

	if (hHdCtlInfo->waitFlag == HdFalse) {
	    if (hdCloseSocket2OtherDaemon(machInfo) == HdFalse) {
		eMsgType = HdEMsg_CloseSocket2OtherDaemon;
		goto Error_Rtn;
	    }
	    eMsgType = HdEMsg_ReceiveSockInfoFromDaemonError;
	} else {
	    eMsgType = NULL;
	}
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn_Close:
    hdCloseSocket2OtherDaemon(machInfo);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Receive sockInfo from Other Daemons
 */
HdBoolean hdReceiveSockInfoFromDaemon(hHdCtlInfo, machInfo, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HMachInfo		machInfo;
    HAgentInfo		agentInfo;
{
    HeliosdMsgType	msgType;
    HHdAcceptMsgInfo	msgInfo;
    char		hostName[HostNameLen_Lim];
    int			sin_family;
    int			sin_port;
    char		recvBuf[MsgBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;
    int			ret;

    if (hdSockRead(hHdCtlInfo, machInfo->sock, recvBuf, MsgBufferLen_Lim)
	== HdFalse) {
	if (hHdCtlInfo->waitFlag == HdTrue) {
	    if ((msgInfo = hdCreateAcceptMsgInfo(hHdCtlInfo, machInfo,
						 HdAMS_ReceiveSockWait))
		== HdFalse) {
		hHdCtlInfo->waitFlag = HdFalse;
		eMsgType = HdEMsg_CreateAcceptMsgInfoError;
		goto Error_Rtn;
	    }
	    msgInfo->args.arg1.agentInfo = agentInfo;
	    eMsgType = NULL;
	} else {
	    eMsgType = HdEMsg_SocketReadError;
	}
	goto Error_Rtn;
    }

    ret = sscanf(recvBuf, "%d", &msgType);
    if (ret == 0 || ret == EOF) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    if (msgType != HdMsg_SocketInfo) {
	eMsgType = HdEMsg_NotDefinedAgentInfoOnOthers;
	goto Error_Rtn;
    }

    ret = sscanf(recvBuf, "%d %s %d %d",
		 &msgType, hostName, &sin_family, &sin_port);

    if (ret == 0 || ret == EOF) {
	eMsgType = HdEMsg_ReceivedDataError;
	goto Error_Rtn;
    }

    agentInfo->sockInfo.sin_family = sin_family;
    agentInfo->sockInfo.sin_port = sin_port;
    agentInfo->status = AS_Active;

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ======================================================
 *  functions for the message of HdMsg_AskForCreateAgent
 * ====================================================== */
/*
 * Execute agent process requested from other daemons
 */
HdBoolean hdAskForCreateAgent(hHdCtlInfo, agentId)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
{
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;

#ifdef _DEBUG_HELIOSD
    fprintf(stderr, "Agent Id in hdAskForCreateAgent : %s\n", agentId);
#endif

    if ((agentInfo = hdSearchAgentInfo(hHdCtlInfo->allAgents, agentId)) == NULL) {
	/* check parametric agent
	 */


	eMsgType = HdEMsg_NotDefinedAgentInfo;
	goto Error_Rtn;
    }

    switch(agentInfo->status) {
    case AS_NotActive:
	/* $B5/F0$5$l$F$$$J$$(B agent process $B$N5/F0(B
	 */
	if (hdCreateAgentProcess(hHdCtlInfo, agentInfo) == HdFalse) {
	    eMsgType = HdEMsg_CreateAgentProcessError;
	    goto Error_Rtn;
	}
	break;
    case AS_Active:
	break;
    case AS_NotExist:
	eMsgType = HdEMsg_NotAgentProcessFile;
	goto Error_Rtn;
    default:
	eMsgType = HdEMsg_AgentStatusTypeError;
	goto Error_Rtn;
    }

    /*
     *	send port info of socket to daemon
     */
    if (hdSendSocketInfo(hHdCtlInfo, hHdCtlInfo->s, agentInfo) == HdFalse) {
	eMsgType = HdEMsg_SendSocketInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ====================================
 *  functins to connect other daemons
 *	with socket.
 * ==================================== */
HdBoolean hdCreateSocket2OtherDaemon(machInfo)
    HMachInfo		machInfo;
{
    int			sock;
    HeliosdErrorMsgType	eMsgType;

    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	eMsgType = HdEMsg_SocketForServerFailed;
	goto Error_Rtn;
    }

    if (connect(sock, &(machInfo->sockAddress), machInfo->addressSize) == ERROR_FLAG) {
	eMsgType = HdEMsg_CouldNotConnect;
	goto Error_Rtn;
    }

    machInfo->sock = sock;

    return HdTrue;

 Error_Rtn:
    hdPerror();
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

HdBoolean hdCloseSocket2OtherDaemon(machInfo)
    HMachInfo		machInfo;
{
    HeliosdErrorMsgType	eMsgType;

    if (close(machInfo->sock) == -1) {
	eMsgType = HdEMsg_CloseError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPerror();
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}
