/* hdDAMessage.c;
 *	module to handle messages between Helios Daemon and Agent.
 *		created by k3sato on May 31st in 1994.
 *
 *	modification history:	
 *	(1) add hdSearchUserInfo() in hdGetHAgentInfo(),
 *		on July 21st in '94.
 *	(2) modify hdCreateAgentProcess so as to create a agent
 *		with the representative child agent
 *		on July 25th in '94.
 *	(3) add hdDeleteAgentProcess so as to delete a agent
 *		on October 11th in '94.
 *	(4) add hdRegisterParaAgent() to register parametric agents
 *		on November 11th in '94.
 *	(5) add hdGetParaAgent() to get a instance of a parametric agent
 *		on November 14th in '94.
 *
 * $Id: hdDAMessage.c,v 2.2 1995/03/02 09:26:27 k3sato Exp $
 */

#include <stdio.h>
#include <signal.h>
#include <sys/wait.h>
#include <string.h>
#include <memory.h>

#include <helios/heliosd.h>
#include <helios/hdMessage.h>
#include <helios/hExternStdFunc.h>
#include "hdDAMessage.h"
#include "hdDDMessage.h"
#include "hdAgentInfo.h"
#include "hdSocket.h"
#include "hdcommon.h"

#include <sys/socket.h>

/*  defined in hdUserInfo.c
 */
extern HdBoolean hdAppendUserInfo();

/*  defined in hdAppGuiInfo.c
 */
extern HdBoolean hdSearchAppGuiInfo();

HdBoolean hdCreateAgentProcess();
HdBoolean hdDeleteAgentProcess();

/* ===================================================
 *	functions for the message of HdMsg_GetAgent
 * =================================================== */
/*
 *	get a agent process
 */
HdBoolean hdGetAgent(hHdCtlInfo, agentId)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
{
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;

    /*
     * agent process $B$N3NG'(B
     */
    if (hdGetHAgentInfo(hHdCtlInfo, agentId, &agentInfo) == HdFalse) {
	if (hdCheckParaInstance(hHdCtlInfo, agentId, &agentInfo) == HdFalse) {
	    eMsgType = HdEMsg_GetAgentPathError;
	    goto Error_Rtn;
	}
    }

    switch(agentInfo->status) {
    case AS_NotActive:
    case AS_Sleep:
	/* $B5/F0$5$l$F$$$J$$(B agent process $B$N5/F0(B
	 */
	if (hdCheckHomeMachine(hHdCtlInfo, agentInfo) == HdTrue) {
	    if (hdCreateAgentProcess(hHdCtlInfo, agentInfo) == HdFalse) {
		eMsgType = HdEMsg_CreateAgentProcessError;
		goto Error_Rtn;
	    }
	} else {
	    if (hdAskForCreating2OtherDaemon(hHdCtlInfo, agentInfo)
		== HdFalse) {
		eMsgType = HdEMsg_AskForCreating2OtherDaemonError;
		goto Error_Rtn;
	    }
	}
	break;
    case AS_Active:
	break;
    case AS_NotExist:
	eMsgType = HdEMsg_NotAgentProcessFile;
	goto Error_Rtn;
    default:
	eMsgType = HdEMsg_AgentStatusTypeError;
	goto Error_Rtn;
    }

    /*
     *	send SocketInfo to agent
     */
    if (hdSendSocketInfo(hHdCtlInfo, hHdCtlInfo->s, agentInfo) == HdFalse) {
	eMsgType = HdEMsg_SendSocketInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    if (hHdCtlInfo->waitFlag == HdFalse) {
	hdSendErrorMsg(hHdCtlInfo, eMsgType);
    }
    return HdFalse;
}

/*
 *	$B5/F0$5$l$F$$$J$$(B agent process $B$N5/F0(B
 */
HdBoolean hdCreateAgentProcess(hHdCtlInfo, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HAgentInfo		agentInfo;
{
    int			pid;
    u_short		hdSocketNum;
    char		*agentPath;
    HAgentInfo		repAgentInfo;
    HeliosdErrorMsgType	eMsgType;
    int			mask;

    switch (agentInfo->type) {
    case HdAT_Parametric:
	if ((agentInfo->child =
	     repAgentInfo = hdMallocParaInstance(agentInfo)) == NULL) {
	    eMsgType = HdEMsg_MallocParaInstanceError;
	    goto Error_Rtn;
	}

	agentPath = agentInfo->agentPath;
	break;

    case HdAT_ParaInstance:
	repAgentInfo = agentInfo;
	agentInfo = repAgentInfo->parent;
	agentPath = agentInfo->agentPath;

	break;

    default:
	if ((repAgentInfo = hdGetRepAgentInfo(agentInfo)) == NULL) {
	    eMsgType = HdEMsg_NotGetRepAgentInfo;
	    goto Error_Rtn;
	}
	agentPath = repAgentInfo->agentPath;

	if (repAgentInfo->type == HdAT_Parametric) {
	    if (repAgentInfo->child != NULL) {
		eMsgType = HdEMsg_AlreadyParametricInstance;
		goto Error_Rtn;
	    }
	    if ((repAgentInfo->child = hdMallocParaInstance(repAgentInfo)) == NULL) {
		eMsgType = HdEMsg_MallocParaInstanceError;
		goto Error_Rtn;
	    }
	    repAgentInfo = repAgentInfo->child;
	}
    }

    if (agentPath == NULL) {
	eMsgType = HdEMsg_NotDefinedAgentPath;
	goto Error_Rtn;
    }

#ifdef _Heliosd_Static_Port_Num

    if (repAgentInfo->status == AS_Sleep) {
	hdSocketNum = repAgentInfo->sockInfo.sin_port;
    } else {
	/* $B;HMQ$5$l$F$$$J$$(B port $BHV9f$N3MF@(B
	 */
	if (hdGetPortNumber(hHdCtlInfo, &hdSocketNum) == HdFalse) {
	    return HdFalse;
	}
    }

#else

    hdSocketNum = repAgentInfo->sockInfo.sin_port;

#endif

    mask = sigblock(sigmask(SIGCLD));

    /* $B;R%W%m%;%9$N5/F0(B
     */
    if ((pid = fork()) == 0) {	/* $B$3$3$+$i;R%W%m%;%9(B */
	char	*argv[3];
	char	socketNum[HdIntString_Lim];

	/* $B?F$N(B socket $B$N(B close
	 */
	close(hHdCtlInfo->s_listen);

	sprintf(socketNum, "%u", hdSocketNum);

#ifdef _DEBUG_HELIOSD
	fprintf(stderr, "debug: socketNum = %s\n", socketNum);
#endif

	argv[0] = agentPath;
	argv[1] = socketNum;
	argv[2] = NULL;

	execv(agentPath, argv);

	exit(0);

	/* $B;R%W%m%;%9=*$j(B
	 */
    } else if (pid == ERROR_FLAG) {
	eMsgType = HdEMsg_CreateChildProcessError;
	hdPerror();
	goto Error_Rtn;
    }

#ifdef _Heliosd_Static_Port_Num

    if (repAgentInfo->status != AS_Sleep) {
	repAgentInfo->sockInfo.sin_port = hdSocketNum;
    }

#endif

    repAgentInfo->sockInfo.sin_family = AF_INET;

    repAgentInfo->pid = pid;
    repAgentInfo->status = AS_Active;

    if (agentInfo != repAgentInfo) {
	if ((hdSetupStatus2ParentAgent(agentInfo, repAgentInfo)) == HdFalse) {
	    eMsgType = HdEMsg_SetupStatus2ParentAgentError;
	    goto Error_Rtn;
	}
    }

    sigsetmask(mask);

    /*
     *	sleep time for creating the above subprocess
     *	and for preparing socket in the subprocess.
     */
    sleep(HELIOSD_SLEEP_SPAN);

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ======================================================
 *	functions for the message of HdMsg_GetAgentInfo
 * ====================================================== */
/*
 *	$B%(!<%8%'%s%H>pJs$N3MF@(B
 */
HdBoolean hdGetAgentInfo(hHdCtlInfo, agentId)
    HHeliosdCtlInfo	hHdCtlInfo;
    char 	       *agentId;
{
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;

    /*
     * agent process $B$N3NG'(B
     */
    if (hdGetHAgentInfo(hHdCtlInfo, agentId, &agentInfo) == HdFalse) {
	eMsgType = HdEMsg_GetAgentPathError;
	goto Error_Rtn;
    }

    switch(agentInfo->status) {
    case AS_NotActive:
    case AS_Active:
    case AS_Sleep:
	break;
    case AS_NotExist:
	eMsgType = HdEMsg_NotAgentProcessFile;
	goto Error_Rtn;
    default:
	eMsgType = HdEMsg_AgentStatusTypeError;
	goto Error_Rtn;
    }

    /*	send AgentInfo to agent
     */
    if (hdSendOneAgentInfo(hHdCtlInfo, HdMsg_AgentInfoEnd, agentInfo) == HdFalse) {
	eMsgType = HdEMsg_SendOneAgentInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ==========================================================
 *	functions for the message of HdMsg_GetAllAgentInfo
 * ========================================================== */
/*
 *	$B%H%C%W!&%l%Y%k$+$i8+$($k%(!<%8%'%s%H$NA4>pJs$N3MF@(B
 */
HdBoolean hdGetAllAgentInfo(hHdCtlInfo, agentId)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
{
    HeliosdErrorMsgType	eMsgType;

    if (hdSendAgentInfo(hHdCtlInfo, hHdCtlInfo->allAgents) == HdFalse) {
	eMsgType = HdEMsg_SendAgentInfoError;
	goto Error_Rtn;
    }

    if (hdSendMsgType(hHdCtlInfo, HdMsg_AgentInfoEnd) == HdFalse) {
	eMsgType = HdEMsg_SendMsgTypeError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ======================================================
 *	functions for the message of HdMsg_DeleteAgent
 * ====================================================== */
/*
 *	$B%(!<%8%'%s%H!&%W%m%;%9$N:o=|(B
 */
HdBoolean hdDeleteAgent(hHdCtlInfo, agentId)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
{
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;

    /*
     * agent process $B$N3NG'(B
     */
    if (hdGetHAgentInfo(hHdCtlInfo, agentId, &agentInfo) == HdFalse) {
	eMsgType = HdEMsg_GetAgentPathError;
	goto Error_Rtn;
    }

    switch(agentInfo->status) {
    case AS_NotActive:
	eMsgType = HdEMsg_NotActiveAgent;
	goto Error_Rtn;
    case AS_Sleep:
	eMsgType = HdEMsg_AlreadySleepAgent;
	break;
    case AS_Active:
	/* $B5/F0$5$l$F$$$k(B agent process $B$N:o=|(B
	 */
	if (hdDeleteAgentProcess(hHdCtlInfo, agentInfo) == HdFalse) {
	    eMsgType = HdEMsg_DeleteAgentProcessError;
	    goto Error_Rtn;
	}
	break;
    case AS_NotExist:
	eMsgType = HdEMsg_NotAgentProcessFile;
	goto Error_Rtn;
    default:
	eMsgType = HdEMsg_AgentStatusTypeError;
	goto Error_Rtn;
    }

    hdSendGoodMsg(hHdCtlInfo, HdGMsg_DeleteAgent);

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    hdSendErrorMsg(hHdCtlInfo, eMsgType);
    return HdFalse;
}

/*
 *	$B5/F0$5$l$F$$$k(B agent process $B$N:o=|(B
 */
HdBoolean hdDeleteAgentProcess(hHdCtlInfo, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HAgentInfo		agentInfo;
{
    HAgentInfo		repAgentInfo;
    HeliosdErrorMsgType	eMsgType;


    if ((repAgentInfo = hdGetRepAgentInfo(agentInfo)) == NULL) {
	eMsgType = HdEMsg_NotGetRepAgentInfo;
	goto Error_Rtn;
    }

    if (repAgentInfo->type == HdAT_Parametric) {
	if ((repAgentInfo = repAgentInfo->child) == NULL) {
	    eMsgType = HdEMsg_NotGetRepAgentInfo;
	    goto Error_Rtn;
	}
    }

    if (repAgentInfo->status == AS_Active) {

	/* check self machine,
	 * if other machine then
	 * ask other machine to delete agent process.
	 */

	if (kill(repAgentInfo->pid, SIGQUIT) == -1) {
	    hdPerror();
	    eMsgType = NULL;
	    goto Error_Rtn;
	}
	fprintf(stderr, "Sleep agent name: %s\n", repAgentInfo->agentPath);

	/* repAgentInfo->pid = 0; */
	repAgentInfo->status = AS_Sleep;
    }

    if (agentInfo != repAgentInfo) {
	if ((hdSetupStatus2ParentAgent(agentInfo, repAgentInfo)) == HdFalse) {
	    eMsgType = HdEMsg_SetupStatus2ParentAgentError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ===================================================
 *   functions for the message of HdMsg_StartAppGui
 * =================================================== */
/*
 *	start the application Gui for a agen name
 */
HdBoolean hdStartAppGui(hHdCtlInfo, agentName)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentName;
{
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;

    /*
     * Application Gui process $B$N3NG'(B
     */
    if (hdSearchAppGuiInfo(hHdCtlInfo, agentName, &agentInfo) == HdFalse) {
	eMsgType = HdEMsg_SearchAppGuiInfoError;
	goto Error_Rtn;
    }

    switch(agentInfo->status) {
    case AS_NotActive:
	/* $B5/F0$5$l$F$$$J$$(B agent process $B$N5/F0(B
	 */
	if (hdCreateAgentProcess(hHdCtlInfo, agentInfo) == HdFalse) {
	    eMsgType = HdEMsg_CreateAppGuiProcessError;
	    goto Error_Rtn;
	}

	/* Append the application Gui info to UserInfo
	 */
	if ((hdAppendUserInfo(hHdCtlInfo, agentInfo)) == HdFalse) {
	    eMsgType = HdEMsg_AppendUserInfoError;
	    goto Error_Rtn;
	}

	break;
    case AS_Active:
	break;
    case AS_NotExist:
	eMsgType = HdEMsg_NotAppGuiProcessFile;
	goto Error_Rtn;
    default:
	eMsgType = HdEMsg_AgentStatusTypeError;
	goto Error_Rtn;
    }

    /*
     *	send SocketInfo to agent

    if (hdSendSocketInfo(hHdCtlInfo, hHdCtlInfo->s, agentInfo) == HdFalse) {
	eMsgType = HdEMsg_SendSocketInfoError;
	goto Error_Rtn;
    }
     */

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ===================================================
 *	functions for the message of HdMsg_GetParaAgent
 * =================================================== */
/*
 *	get a parametric agent process
 */
HdBoolean hdGetParaAgent(hHdCtlInfo, agentId)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
{
    HAgentInfo		agentInfo;
    HeliosdErrorMsgType	eMsgType;

    /*
     * agent process $B$N3NG'(B
     */
    if (hdGetHAgentInfo(hHdCtlInfo, agentId, &agentInfo) == HdFalse) {
	eMsgType = HdEMsg_GetAgentPathError;
	goto Error_Rtn;
    }

    if (agentInfo->type != HdAT_Parametric) {
	eMsgType = HdEMsg_NotParametricAgent;
	goto Error_Rtn;
    }

    switch(agentInfo->status) {
    case AS_NotActive:
    case AS_Sleep:
	/* $B5/F0$5$l$F$$$J$$(B agent process $B$N5/F0(B
	 */
	if (hdCreateAgentProcess(hHdCtlInfo, agentInfo) == HdFalse) {
	    eMsgType = HdEMsg_CreateAgentProcessError;
	    goto Error_Rtn;
	}
	break;
    case AS_Active:
	break;
    case AS_NotExist:
	eMsgType = HdEMsg_NotAgentProcessFile;
	goto Error_Rtn;
    default:
	eMsgType = HdEMsg_AgentStatusTypeError;
	goto Error_Rtn;
    }

    /*
     *	send SocketInfo to agent
     */
    if (hdSendSocketInfo(hHdCtlInfo, hHdCtlInfo->s, agentInfo) == HdFalse) {
	eMsgType = HdEMsg_SendSocketInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    hdSendErrorMsg(hHdCtlInfo, eMsgType);
    return HdFalse;
}

/* =======================================================
 *   functions for the message of HdMsg_RegisterParaAgent
 * ======================================================= */
/*
 *	Register parametric agents in AgentInfo
 */
HdBoolean hdRegisterParaAgent(hHdCtlInfo, paraInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*paraInfo;
{
    AgentInfo		tempInfo;
    char		parentAgentId[AgentNameLen_Lim];
    HAgentInfo		parentInfo;
    HAgentInfo		agentInfo;
    char		*tempStr;
    HeliosdErrorMsgType	eMsgType;

    memset(&tempInfo, 0, sizeof(AgentInfo));

    if (hdSetupParaInfo2AgentInfo(paraInfo, &tempInfo) == HdFalse) {
	eMsgType = HdEMsg_SetupParaInfo2AgentInfoError;
	goto Error_Rtn;
    }

    strcpy(parentAgentId, tempInfo.agentId);

    if ((tempStr = strrchr(parentAgentId, '.')) == NULL) {
	eMsgType = HdEMsg_NotProperAgentId;
	goto Error_Rtn;
    }

    *tempStr = '\0';

    if ((parentInfo = hdSearchAgentInfo(hHdCtlInfo->allAgents, parentAgentId))
	== NULL) {
	eMsgType = HdEMsg_NotExistParametricTemplateAgent;
	goto Error_Rtn;
    }

    if ((agentInfo = hdSearchAgentInfo(parentInfo, tempInfo.agentId))
	== NULL) {
	if ((agentInfo = hdSearchAgentInfoForPort(parentInfo,
						   tempInfo.sockInfo.sin_port))
	    == NULL) {
	    eMsgType = HdEMsg_NotExistParametricInstanceAgent;
	    goto Error_Rtn;
	}
    }

    if (agentInfo->agentId == NULL) {
	agentInfo->agentId = tempInfo.agentId;
    } else {
	hdFreeString(&(tempInfo.agentId));
    }

    if (agentInfo->agentName != NULL) {
	hdFreeString(&(agentInfo->agentName));
    }

    agentInfo->agentName = tempInfo.agentName;

    if (hdSendMsgType(hHdCtlInfo, HdMsg_Success) == HdFalse) {
	eMsgType = HdEMsg_SendMsgTypeError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdFreeString(&(tempInfo.agentId));
    hdFreeString(&(tempInfo.agentName));

    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ===================================================
 *   functions for the message of HdMsg_ResetAgent
 * =================================================== */
/*
 *	reset agent for a given agentId
 */
HdBoolean hdResetAgent(hHdCtlInfo, agentId)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
{
    HeliosdErrorMsgType	eMsgType;

    eMsgType = HdEMsg_NotYetImplemented;
    goto Error_Rtn;

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}
