/* Asynchronous RPC Module */
/* Copyright (C) 1994 ICOT */
/* Written by gniibe       */
/* $Id: comm.c,v 1.23 1995/03/03 08:36:51 m-gniibe Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <memory.h>
#include <string.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <netdb.h>
#include <errno.h>
#include <helios/comm.h>
#include <helios/event.h>
#include "mem.h"

/* structure of P-message:
  +---------+
  |         |
  | SIZE    | network order int
  |         |
  +---------+
  | RPC_ID  | network order short
  +---------+
  |contents |
  =         =
  |         |
  +---------+
 */
fd_set sock_r_fds, sock_w_fds, sock_e_fds;

int listen_sock;
int udp_sock;
int userif_sock;

int
write_to_socket (struct agent *a)
{
  struct q *q;
  int len, ready;

  ready = 0;
  FD_CLR (a->sock, &io_w_fds);
  while (a->dirty_tx < a->cur_tx)
    {
      q = &a->tx_queue[a->dirty_tx & TX_QUEUE_MASK];
      if (q->tx_status == QS_SENDING)
	{
	  len = write (a->sock, q->rest, q->byte);
	  if (len == q->byte)
	    {
	      q->tx_status = QS_SENT;
	      free (q->p);
	      q->p = NULL;
	    }
	  else if (len > 0)
	    {
	      q->rest += len;
	      q->byte -= len;
	      FD_SET (a->sock, &io_w_fds);
	      break;
	    }
	  else
	    {
	      FD_SET (a->sock, &io_w_fds);
	      break;
	    }
	}
      else if (q->tx_status == QS_WAIT)
	{
	  q->tx_status = QS_SENDING;
	  len = write (a->sock, q->rest, q->byte);
	  if (len == q->byte)
	    {
	      q->tx_status = QS_SENT;
	      free (q->p);
	      q->p = NULL;
	    }
	  else if (len > 0)
	    {
	      q->rest += len;
	      q->byte -= len;
	      FD_SET (a->sock, &io_w_fds);
	      break;
	    }
	  else
	    {
	      FD_SET (a->sock, &io_w_fds);
	      break;
	    }
	}
      a->dirty_tx++;
      ready = 1;
    }

  return ready;
}

#if 0
#define RBUF_SIZE 256
static char read_buf[RBUF_SIZE];
#endif /* now, read_from_socket invokes three read.
	  you can reduce number of times to call read
	  preparing read_buf and work... */

int
read_from_socket (struct agent *a)
{
  struct q *q;
  int r, len, ready;

  ready = 0;
  if (a->status == AGENT_CONNECTING)
    {
      r = read (a->sock, &a->addr, sizeof (struct sockaddr_in));
      a->status = AGENT_ACTIVE;
    }
  while (a->dirty_rx > a->cur_rx - RX_QUEUE_SIZE)
    {
      q = &a->rx_queue[a->cur_rx & RX_QUEUE_MASK];
      if (q->rx_status == QS_HEAD)
	{
	  len = read (a->sock, q->rest, q->byte);
	  if (len == q->byte)
	    q->rx_status = QS_RECEIVING;
	  else if (len > 0)
	    {
	      q->rest += len;
	      q->byte -= len;
	      break;
	    }
	  else if (len == 0)
	    { /* connection closed by foreign agent */
	      invalidate_agent_internal (a);
	      break;
	    }
	  else
	    break;
	  q->byte = (q->head[0] << 24) + (q->head[1] << 16)
	    + (q->head[2] << 8) + q->head[3];
	  q->rpc_id = (q->head[4] << 8) + q->head[5] ;
	  q->p = q->rest = malloc (q->byte);
	  if (q->p == NULL)
	    {
	      fprintf (stderr, "oom...\n");
	      exit (10);
	    }
	  continue;
	}
      else if (q->rx_status == QS_RECEIVING)
	{
	  len = read (a->sock, q->rest, q->byte);
	  if (len == q->byte)
	    q->rx_status = QS_RECEIVED;
	  else if (len > 0)
	    {
	      q->rest += len;
	      q->byte -= len;
	      break;
	    }
	  else
	    break;
	}
      else if (q->rx_status == QS_WAIT)
	{
	  q->rx_status = QS_HEAD;
	  q->rest = q->head;
	  q->byte = sizeof (int) + sizeof (short);
	  continue;
	}
      a->cur_rx++;
      ready = 1;
    }

  return ready;
}

static int rpc_id_serial = 0;

void
accept_socket (void)
{
  int new;
  struct sockaddr_in client_addr;
  int client_addrlen = sizeof (struct sockaddr_in);

  new = accept (listen_sock, &client_addr, &client_addrlen);
  accept_agent (new, &client_addr);
}

int
send_a_message (char *ag, void *m, unsigned int len)
{
  struct q *q;
  unsigned short rpc_id = rpc_id_serial++;
  unsigned char *p;
  struct agent *a;

  if ((a = comm_get_agent (ag)) == NULL)
    return -ENOENT;

  if (a->cur_tx - a->dirty_tx >= TX_QUEUE_SIZE)
    return -EAGAIN;

  if ((p = malloc (len + sizeof (int) + sizeof (short))) == NULL)
    return -ENOMEM;

  if (a->status != AGENT_ACTIVE)
    connect_agent (a);

  q = &a->tx_queue[a->cur_tx++ & TX_QUEUE_MASK];
  q->tx_status = QS_WAIT;
  q->rpc_id = rpc_id;
  q->rest = p;
  q->p = p;
  q->byte = len + sizeof (int) + sizeof (short);

  *p++ = len >> 24;
  *p++ = (len & 0xff0000) >> 16;
  *p++ = (len & 0x00ff00) >> 8;
  *p++ =  len & 0x0000ff;
  *p++ = (rpc_id & 0x00ff00) >> 8;
  *p++ =  rpc_id & 0x0000ff;
  memcpy (p, m, len);

  if (a->dirty_tx == a->cur_tx -1)
    write_to_socket (a);

  return (int)rpc_id;
}

int
receive_a_message (int *rpc_idP, void *m, int *lenP)
{
  int i;
  struct agent *a;
  struct q *q;

  for (i=0; i < MAX_AGENT; i++)
    {
      a = &agents[i];
      if (a->status != AGENT_ACTIVE)
	continue;

      if (a->cur_rx > a->dirty_rx)
	{ /* there are packets */
	  q = &a->rx_queue[a->dirty_rx & RX_QUEUE_MASK];
	  if (*lenP < q->byte)
	    return -EINVAL;

	  memcpy (m, q->p, q->byte);
	  free (q->p);
	  q->rx_status = QS_WAIT;
	  a->dirty_rx++;
	  *rpc_idP = q->rpc_id;
	  *lenP    = q->byte;
	  return 0;
	}
    }
  return -EAGAIN;
}

struct sockaddr_in my_addr;

int
init_comm (int port, int user_p)
{
  int s, r, u;
  int len;
  char hn[256];
  struct hostent *he;

  if (gethostname (hn, 256) < 0)
    return -errno;
  he = gethostbyname(hn);
  if (he == NULL)
    return -ENOENT;

  memcpy (&my_addr.sin_addr, he->h_addr_list[0], sizeof (struct in_addr));
  my_addr.sin_family = AF_INET;
  my_addr.sin_port   = htons(port);

  /* Setup TCP port */
  s = socket (PF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (s < 0)
    {
      perror ("socket");
      return s;
    }

  if ((r =fcntl (s, F_SETFD, FD_CLOEXEC)) < 0)
    return r;

  if ((r = bind (s, (struct sockaddr *)&my_addr,
		 sizeof (struct sockaddr_in))) < 0)
    {
      close (s);
      return r;
    }
  if ((r = listen (s, 5)) < 0)
    {
      close (s);
      return r;
    }
  listen_sock = s;

  /* Setup UDP port */
  if (user_p)
    {
      u = socket (PF_INET, SOCK_DGRAM, IPPROTO_UDP);
      if (u < 0)
	return u;

      if ((r =fcntl (u, F_SETFD, FD_CLOEXEC)) < 0)
	return r;

      if ((r = bind (u, (struct sockaddr *)&my_addr,
		     sizeof (struct sockaddr_in))) < 0) /* same port # as TCP */
	{
	  close (u);
	  return r;
	}

      len = 1500;
      if ((r = setsockopt(u, SOL_SOCKET, SO_RCVBUF, &len, sizeof (int))) < 0)
	{
	  perror ("comm:setsockopt");
	  close (u);
	  return r;
	}
      len = 43;
      if ((r = setsockopt(u, SOL_SOCKET, SO_SNDBUF, &len, sizeof (int))) < 0)
	{
	  perror ("comm:setsockopt");
	  close (u);
	  return r;
	}

      udp_sock = u;
    }
  else
    udp_sock = -1;

  u = socket (PF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if (u < 0)
    return u;

  if ((r =fcntl (u, F_SETFD, FD_CLOEXEC)) < 0)
    return r;
  len = 1500;
  if ((r = setsockopt(u, SOL_SOCKET, SO_SNDBUF, &len, sizeof (int))) < 0)
    {
      perror ("comm:setsockopt");
      close (u);
      return r;
    }
  len = 43;
  if ((r = setsockopt(u, SOL_SOCKET, SO_RCVBUF, &len, sizeof (int))) < 0)
    {
      perror ("comm:setsockopt");
      close (u);
      return r;
    }
  userif_sock = u;

  init_agent ();
  init_event ();

  FD_SET (listen_sock, &io_r_fds);
  if (udp_sock > 0)
    FD_SET (udp_sock, &io_r_fds);

  heliosdInit();  /* added by k3sato */

  return 0;
}

void
rudp_send_ack (struct sockaddr_in *to, unsigned short seq)
{
  unsigned char head[3];
  head[0] = RUDP_ACK;
  head[1] = (seq >> 8);
  head[2] = seq & 0xff;
  if (sendto (udp_sock, head, 3, 0, to, sizeof (struct sockaddr_in)) < 0)
    {
      perror ("sendto");
    }
}

static int rudp_len = 0;
static struct rudp rudp_table[MAX_AGENT];

struct rudp *
get_rudp (struct sockaddr_in *sock_addr)
{
  int i;

  for (i=0; i<rudp_len; i++)
    if (memcmp (&rudp_table[i].sock_addr, sock_addr,
		sizeof (struct sockaddr_in))
	== 0)
      return &rudp_table[i];

  if (i >= MAX_AGENT)
    abort ("too many agent\n");

  memcpy (&rudp_table[i].sock_addr, sock_addr, sizeof (struct sockaddr_in));
  rudp_len++;
  return &rudp_table[i];
}

#define UDP_MAX 1500
unsigned char udp_buffer[UDP_MAX];

int
receive_datagram (void)
{
  int len;
  unsigned short seq;
  struct rudp *rudp;
  struct sockaddr_in from;
  int fromlen = sizeof (struct sockaddr_in);

  len = recvfrom (udp_sock, udp_buffer, UDP_MAX, 0,
		  (struct sockaddr *)&from, &fromlen);
  if (len < 0)
    return 0;

  udp_buffer[len] = 0;
  seq = udp_buffer[1]*256 + udp_buffer[2];
  rudp = get_rudp (&from);
  rudp_send_ack (&from, seq);

  switch (*udp_buffer)
    {
    case RUDP_SYNC:
      rudp->seq = seq;
      return 0;
      break;

    case RUDP_DATA:
      if (seq == rudp->seq)
	{
	  /* new message */
	  rudp->seq++;
	  return 1;
	}
      else
	return 0;
      break;

    case RUDP_ACK:
    default:
      /* What's happen? */
      return 0;
      break;
    }
}

char *
dreceive (void)
{
  return udp_buffer+3;
}

int
send_datagram (struct sockaddr_in *addr, unsigned char *data)
{
  static unsigned short seq = 0;

  int r;
  struct msghdr msg;
  struct iovec iov[2];
  char header[3];
  unsigned short r_seq;

  header[0] = RUDP_DATA;
  header[1] = (seq >> 8);
  header[2] = (seq & 0xff);
  iov[0].iov_base = header;
  iov[0].iov_len = 3;
  iov[1].iov_base = data;
  iov[1].iov_len = strlen (data);
  msg.msg_name = addr;
  msg.msg_namelen = sizeof (struct sockaddr_in);
  msg.msg_iov = iov;
  msg.msg_iovlen = 2;
  msg.msg_accrights = NULL;
  msg.msg_accrightslen = 0;

  while (1)
    {
      struct timeval timeout;
      fd_set fds;
      int read_all;

      if ((r = sendmsg (userif_sock, &msg, 0)) < 0)
	{
	  /* failure */
	  perror ("comm:sendmsg");
	  continue;
	}

      timeout.tv_sec  = 0;
      timeout.tv_usec = 500000; /* 500msec */
      FD_ZERO (&fds);
      FD_SET (userif_sock, &fds);
      r = select (FD_SETSIZE, &fds, 0, 0, &timeout);
      if (r < 0)
	{
	  /* failure */
	  perror ("comm:select");
	  continue;
	}
      if (r == 0) /* should send SYNC on many failure... but it is not yet */
	continue;

      read_all = 0;
      do
	{
	  r = recv (userif_sock, udp_buffer, UDP_MAX, 0);
	  if (r < 0)
	    {
	      /* failure */
	      perror ("comm:recv");
	      read_all = 1;
	    }

	  r_seq = udp_buffer[1]*256 + udp_buffer[2];
	  if (r_seq == seq)
	    { /* success */
	      seq++;
	      return 0;
	    }

	  timeout.tv_sec  = 0;
	  timeout.tv_usec = 0;

	  FD_ZERO (&fds);
	  FD_SET (userif_sock, &fds);
	  r = select (FD_SETSIZE, &fds, 0, 0, &timeout);
	  if (r < 0)
	    read_all = 1;
	  if (r == 0)
	    read_all = 1;
	}
      while (!read_all);
    }
}
