#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <memory.h>
#include <string.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <errno.h>
#include <helios/comm.h>
#include <helios/hDaemonLib.h>
#include "mem.h"

struct agent agents[MAX_AGENT];
struct agent *fd_to_agent[FD_SETSIZE];

static struct agent *
alloc_agent (void)
{
  int i;
  struct agent *a;

  for (i=0; i < MAX_AGENT; i++)
    {
      a = &agents[i];
      if (a->status == AGENT_FREE)
        return a;
    }
  return NULL;
}

struct agent *
comm_get_agent (char *name)
{
  int i;
  struct agent *a;
  struct sockaddr_in *addr;

  addr = get_agent (name);
  if (addr == NULL)
    return NULL;

  /* search in the cache for addr */
  for (i=0; i < MAX_AGENT; i++)
    {
      a = &agents[i];
      if (a->status != AGENT_ACTIVE)
	continue;
      if (memcmp (&a->addr, addr, sizeof (struct sockaddr_in)) == 0)
	return a;
    }

  a = alloc_agent ();
  if (a == NULL)
    return NULL;
  a->status = AGENT_NO_CONNECTION;
  a->sock = 0;
  memcpy (&a->addr, addr, sizeof (struct sockaddr_in));
  a->cur_tx = a->cur_rx = 0;
  a->dirty_tx = a->dirty_rx = 0;
  return a;
}

int
accept_agent (int s, struct sockaddr_in *peer)
{
  struct agent *a;
  int flag;

  a = alloc_agent ();
  a->status = AGENT_CONNECTING;
  a->sock = s;
  fd_to_agent[s] = a;
  a->cur_tx = a->cur_rx = a->dirty_tx = a->dirty_rx = 0;

  fcntl (s, F_SETFD, FD_CLOEXEC);
  fcntl (s, F_SETOWN, getpid ());
  flag = fcntl (s, F_GETFL);
  fcntl (s, F_SETFL, FNDELAY | flag);
  FD_SET (s, &io_r_fds);
  return 0;
}

int
connect_agent (struct agent *a)
{
  int s;
  int flag;
  int r;

  s = socket (PF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (s < 0)
    return s;
  fcntl (s, F_SETFD, FD_CLOEXEC);

  a->sock = s;

  if ((r = connect (s, &a->addr, sizeof (struct sockaddr_in))) < 0)
    {
      close (s);
      return r;
    }
  else
    {
      write (s, &my_addr, sizeof (struct sockaddr_in));
      fcntl (s, F_SETOWN, getpid ());
      flag = fcntl (s, F_GETFL);
      fcntl (s, F_SETFL, FNDELAY | flag);
      FD_SET (s, &io_r_fds);
      fd_to_agent[s] = a;
      a->status = AGENT_ACTIVE;
      return 0;
    }
}

void
init_agent (void)
{
  int i;
  struct agent *a;

  for (i=0; i < MAX_AGENT; i++)
    {
      a = &agents[i];
      a->status = AGENT_FREE;
    }
}

void
invalidate_agent_internal (struct agent *a)
{
  struct q *q;

  switch (a->status)
    {
    case AGENT_ACTIVE:
      FD_CLR (a->sock, &io_r_fds);
      FD_CLR (a->sock, &io_w_fds);
      fd_to_agent[a->sock] = NULL;
      close (a->sock);

      while (a->dirty_tx < a->cur_tx)
	{
	  q = &a->tx_queue[a->dirty_tx & TX_QUEUE_MASK];
	  if (q->tx_status == QS_SENDING)
	    {
	      q->tx_status = QS_SENT;
	      free (q->p);
	      q->p = NULL;
	    }
	  else if (q->tx_status == QS_WAIT)
	    {
	      q->tx_status = QS_SENT;
	      free (q->p);
	      q->p = NULL;
	    }
	  a->dirty_tx++;
	}

      while (a->cur_rx > a->dirty_rx)
	{ /* there are packets */
	  q = &a->rx_queue[a->dirty_rx & RX_QUEUE_MASK];
	  free (q->p);
	  q->rx_status = QS_RECEIVED;
	  a->dirty_rx++;
	}
      break;

    case AGENT_CONNECTING:
      fprintf (stderr, "Bad agent state to close connection.\n");
      fprintf (stderr, "close anyway.\n");
      FD_CLR (a->sock, &io_r_fds);
      FD_CLR (a->sock, &io_w_fds);
      fd_to_agent[a->sock] = NULL;
      close (a->sock);
      break;

    case AGENT_FREE:
    case AGENT_NO_CONNECTION:
    default:
      fprintf (stderr, "Bad agent state to close connection.\n");
      break;
    }

  a->status = AGENT_FREE;
}

void
invalidate_agent (char *agent_name)
{
  int i;
  struct agent *a;
  struct sockaddr_in *addr;

  addr = get_agent (agent_name);
  if (addr == NULL)
    {
      fprintf (stderr, "invalidate_agent:no such agent.\n");
      return;
    }

  /* search in the cache for addr */
  for (i=0; i < MAX_AGENT; i++)
    {
      a = &agents[i];
      if (a->status != AGENT_ACTIVE)
	continue;
      if (memcmp (&a->addr, addr, sizeof (struct sockaddr_in)) == 0)
	{
	  invalidate_agent_internal (a);
	  return;
	}
    }

  fprintf (stderr, "invalidate_agent:cannot find agent.\n");
  return;
}
