/* Common Data Handling Library */

/* Data Structure */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define Cstruct 0
#define Cinteger 1
#define Cstring 2
#define Cbool 3
#define Cnone 0
#define Cvector 1
#define Clist 2
#define Ctupple 3
#define Cset 4

struct ComData ComData;

typedef struct ComData *CDATA;

union CElem {
  long integer;
  int bool;
  char *string;
  CDATA element;
};

struct ComData {
  int base; /* base = 0: structure (Use type) 1:int  2:String 3:bool */ 
  int type; /* (if base = 0) type 0:none 1:vector 2:list 3:tupple 4:set 5:functor*/
  union CElem next;
  union CElem data;
} ComData;


/* Handling Function */

CDATA CNew(int base,int type);
CDATA CDelete(CDATA val);
CDATA CDuplicate(CDATA val);
CDATA CPutElement(CDATA vec,int position,CDATA element);
CDATA CGetElement(CDATA vec,int position);
int CLength(CDATA vec);
char *CStructToCommon(CDATA data,char *buf);

/* for BASIC */

CDATA CBPutInt(CDATA basic,long integer);
long CBGetInt(CDATA basic);

CDATA CBPutBool(CDATA basic,int bool);
int CBGetBool(CDATA basic);

CDATA CBPutString(CDATA basic,char* string);
char *CBGetString(CDATA basic);

/* for Vector */

CDATA CVPutElement(CDATA vec,int position,CDATA data);
CDATA CVGetElement(CDATA vec,int position);
int CVLength(CDATA vec);

/* for List */

CDATA CLCar(CDATA list);
CDATA CLCdr(CDATA list);
CDATA CLPutElement(CDATA list,int position,CDATA data);
CDATA CLGetElement(CDATA list,int position);
int CLLength(CDATA list);

/* for tupple */

CDATA CTPutElement(CDATA tup,char *tag,CDATA data);
CDATA CTGetElement(CDATA tup,char *tag);
char *CTGetTag(CDATA tup,int position);
int CTLength(CDATA tup);

/* for set */
CDATA CSPutData(CDATA set,int position,CDATA data);
CDATA CSGetData(CDATA set,int position);
int CSLength(CDATA set);

/* for Functor */

CDATA CFPutElement(CDATA fun,int position,CDATA data);
CDATA CFGetElement(CDATA fun,int position);
int CFLength(CDATA vec);


CDATA CNew(int base,int type){
  
  CDATA val;
  val =  malloc(sizeof(ComData));
  val->base = base;
  val->type = type;
  val->next.element = NULL;
  val->data.element = NULL;
  
  return val;
}


CDATA CDelete(CDATA val){

  char *p;

  if(val == NULL) return NULL;
  
  switch (val->base) {
  case 0:
    /* for structure */
    if(val->data.element != NULL) {
      if(CDelete(val->data.element)) return NULL;
      val->data.element = NULL;
    }
    if(val->next.element != NULL) {
      if(CDelete(val->next.element)) return NULL;
      val->next.element = NULL;
    }
    break;
  case 2:
    /* for String */
    if(val->data.string != NULL) {
      for(p = val->data.string;*p != NULL;p++) *p = NULL;
      free(val->data.string);
    }

    val->data.string = NULL;
    break;
  default:
    /* nothing to do for int nand bool */
    val->data.element = NULL;
  }
  free(val);
  return NULL;
}


CDATA CDuplicate(CDATA val){
  
  CDATA dup;
  if(val == NULL) return NULL;
  
  dup = CNew(val->base,val->type);
  
  switch (val->base) {
  case 0:
    /* for structure */
    dup->data.element = CDuplicate(val->data.element);
    dup->next.element = CDuplicate(val->next.element);
    break;
  case 1:
    /* for integer */
    dup->data.integer = val->data.integer;
    break;
  case 2:
    /* for String */
    if(val->data.string != NULL) {
      dup->data.string = strdup(val->data.string);
    }
    break;
  case 3:
    /* for bool */
    dup->data.bool = val->data.bool;
    break;
  }

 return dup;
}

CDATA CPutElement(CDATA vec,int position,CDATA element){
  
  int p;
  CDATA target,newElement;
  
  if(vec == NULL) return NULL;
  
  newElement = CDuplicate(element);
  
  target = vec;
  
  for(p = 0; p < position;p++){
    if(target->next.element == NULL){
      target->next.element = CNew(target->base,target->type);
    }
    target = target->next.element;
  }
  
  
  if(target->data.element != NULL) CDelete(target->data.element);
  target->data.element = newElement;
  
  return vec;
}

CDATA CGetElement(CDATA vec,int position){
  
  int p;
  CDATA target;
  
  if(vec == NULL) return NULL;
  
  target = vec;
  
  for(p = 0; p < position;p++){
    if(target->next.element == NULL){
      target->next.element = CNew(target->base,target->type);
    }
    target = target->next.element;
  }
  return target->data.element;
}

int CLength(CDATA vec){
  
  int len;
  CDATA target;
  
  if(vec == NULL) return NULL;
  
  target = vec;
  
  for(len = 0; target != NULL; len++){
    target = target->next.element;
  }
  if(len == 1 && vec->data.element == NULL) len = 0;

  return len;
}

/* for BASIC */

CDATA CBPutInt(CDATA basic,long integer){
  
  if(basic == NULL) return NULL;
  if(basic->base != 1) return NULL;
  
  basic->data.integer = integer;
  return basic;
}

long CBGetInt(CDATA basic){
  if(basic == NULL) return 0;
  if(basic->base != 1) return 0;
  
  return basic->data.integer;
}


CDATA CBPutBool(CDATA basic,int bool){
  
  if(basic == NULL) return NULL;
  if(basic->base != 3) return NULL;
  
  basic->data.bool = bool;
  return basic;
}


int CBGetBool(CDATA basic){
  
  if(basic == NULL) return 0;
  if(basic->base != 3 && basic->base != 1) return 0;
  
  return basic->data.bool;
}

CDATA CBPutString(CDATA basic,char* string){
  if(basic == NULL) return NULL;
  if(basic->base != 2) return NULL;
  
  if(basic->data.string != NULL) free(basic->data.string);
  basic->data.string = strdup(string);
  
  return basic;
}
char *CBGetString(CDATA basic){
  if(basic == NULL) return NULL;
  if(basic->base != 2) return NULL;
  
  return   basic->data.string;
}



/* for vector */

CDATA CVPutElement(CDATA vec0,int position,CDATA data){


  CDATA element,vec;

  if(vec0 == NULL) return NULL;
  if(vec0->base != 0) return NULL;
  if(vec0->type != 1) return NULL;
  
  if(position < 0) {
    position = CVLength(vec0);
    if(position ==  1 && vec0->data.element == NULL)
      position = 0;
  }

  vec = CDuplicate(vec0);

  element =  CDuplicate(data);
  vec = CPutElement(vec,position,element);
  
  return vec;
}

CDATA CVGetElement(CDATA vec0,int position){
  
  CDATA element;
  
  if(vec0 == NULL) return NULL;
  if(vec0->base != 0) return NULL;
  if(vec0->type != 1) return NULL;
  
  element = CGetElement(vec0,position);
  
  CDuplicate(element);

  return CDuplicate(element);
}

int CVLength(CDATA vec){
  
  int length;
  
  if(vec == NULL) return NULL;
  if(vec->base != 0) return NULL;
  if(vec->type != 1) return NULL;
  
  length = CLength(vec);
  
  return length;
}


/* for List */

CDATA CLCar(CDATA list0){
  
  CDATA list,car;
  
  if(list0 == NULL) return NULL;
  if(list0->base != 0) return NULL;
  if(list0->type != 2) return NULL;
  
  list = CDuplicate(list0); 
  
  car = list->data.element;
  list->data.element = NULL;
  
  CDelete(list);
  
  return car;
}

CDATA CLCdr(CDATA list0){

 CDATA list,cdr;

 if(list0 == NULL) return NULL;
 if(list0->base != 0) return NULL;
 if(list0->type != 2) return NULL;
 
 list = CDuplicate(list0); 

 cdr = list->next.element;
 list->next.element = NULL;

 CDelete(list);

 if(cdr == NULL) {
   cdr = CNew(0,2);
 }

 return cdr;
}


CDATA CLPutElement(CDATA list0,int position,CDATA data){

  CDATA element,list;

  if(list0 == NULL) return NULL;
  if(list0->base != 0) return NULL;
  if(list0->type != 2) return NULL;
  
  list = CDuplicate(list0);

  if(position < 0) {
    position = CLLength(list);
    if(position ==  1 && list0->data.element == NULL)
      position = 0;
  }

  element =  CDuplicate(data);
  list = CPutElement(list,position,element);
  
  return list;
}


CDATA CLGetElement(CDATA list,int position){

  CDATA element;

  if(list == NULL) return NULL;
  if(list->base != 0) return NULL;
  if(list->type != 2) return NULL;
  
 if(position < 0) position = 0;

  element = CGetElement(list,position);

  return   CDuplicate(element);  
}

int CLLength(CDATA list){
  int length;
  
  if(list == NULL) return NULL;
  if(list->base != 0) return NULL;
  if(list->type != 2) return NULL;
  
  length = CLength(list);
  
  return length;
}



/* for tupple */

CDATA CTPutElement(CDATA tup,char *tag,CDATA data){

  int p,len;
  CDATA target,targetTmp,element,tagElem;

  if(tup == NULL) return NULL;
  if(tup->base != 0) return NULL;
  if(tup->type != 3) return NULL;

  target = CDuplicate(tup);
  element = CDuplicate(data);
  targetTmp = target;

  len = CLength(tup);
  if(len == 1) len = 0;

/* check tag */

  for(p = 0; p < len;p += 2){
    if(strcmp(tag,targetTmp->data.element->data.string) == 0){
      break;
    }
    targetTmp = targetTmp->next.element;
    targetTmp = targetTmp->next.element;
  }

  if(p == len) {
    tagElem = CNew(2,0);
    tagElem = CBPutString(tagElem,tag);
    target = CPutElement(target,p,tagElem);
    target = CPutElement(target,p+1,element);
  } else {
    target = CPutElement(target,p+1,element);
  }

 return target;
}

CDATA CTGetElement(CDATA tup,char *tag){

 int p,len;

  if(tup == NULL) return NULL;
  if(tup->base != 0) return NULL;
  if(tup->type != 3) return NULL;


 len = CLength(tup);

/* check tag */

  for(p = 0; p < len;p += 2){
    if(strcmp(tag,tup->data.element->data.string)== 0){
      break;
    }
    tup = tup->next.element;
    tup = tup->next.element;
  }

 if(p == len) {
   return NULL;
 }else {
    return CDuplicate(tup->next.element->data.element);
  }
}

char *CTGetTag(CDATA tup,int position){

 int len,p;

  if(tup == NULL) return NULL;
  if(tup->base != 0) return NULL;
  if(tup->type != 3) return NULL;

 len = CTLength(tup);

 if(position >= len) return NULL;

  for(p = 0; p < position;p++){
    tup = tup->next.element;
    tup = tup->next.element;
  }
 return strdup(tup->data.element->data.string);

}

int CTLength(CDATA tup){

  int length;
  
  if(tup == NULL) return NULL;
  if(tup->base != 0) return NULL;
  if(tup->type != 3) return NULL;
  
  length = CLength(tup);
  
  return length/2;
}

/* for set */
CDATA CSPutData(CDATA set0,int position,CDATA data){

  CDATA element,set;

  if(set0 == NULL) return NULL;
  if(set0->base != 0) return NULL;
  if(set0->type != 4) return NULL;
  
  set = CDuplicate(set0);

  if(position < 0) {
    position = CSLength(set);
    if(position ==  1 && set->data.element == NULL)
      position = 0;
  }

  element =  CDuplicate(data);
  set = CPutElement(set,position,element);
  
  return set;
}

CDATA CSGetData(CDATA set0,int position){

  CDATA element;
  
  if(set0 == NULL) return NULL;
  if(set0->base != 0) return NULL;
  if(set0->type != 4) return NULL;
  
  element = CGetElement(set0,position);
  
  CDuplicate(element);

  return CDuplicate(element);
}

int CSLength(CDATA set){

  int length;
  
  if(set == NULL) return NULL;
  if(set->base != 0) return NULL;
  if(set->type != 4) return NULL;
  
  length = CLength(set);
  return length;
}

/* for functor */

CDATA CFPutElement(CDATA fun0,int position,CDATA data){


  CDATA element,fun;

  if(fun0 == NULL) return NULL;
  if(fun0->base != 0) return NULL;
  if(fun0->type != 5) return NULL;
  
  if(position < 0) {
    position = CFLength(fun0);
    if(position ==  1 && fun0->data.element == NULL)
      position = 0;
  }

  fun = CDuplicate(fun0);

  element =  CDuplicate(data);
  fun = CPutElement(fun,position,element);
  
  return fun;
}

CDATA CFGetElement(CDATA fun0,int position){
  
  CDATA element;
  
  if(fun0 == NULL) return NULL;
  if(fun0->base != 0) return NULL;
  if(fun0->type != 5) return NULL;
  
  element = CGetElement(fun0,position);
  
  return CDuplicate(element);
}

int CFLength(CDATA fun){
  
  int length;
  
  if(fun == NULL) return NULL;
  if(fun->base != 0) return NULL;
  if(fun->type != 5) return NULL;
  
  length = CLength(fun);
  
  return length;
}

