/* hdHiaInfo.c;
 *	module to manipulate files named *.hia in Helios Daemon.
 *		created by k3sato on May 31st in 1994.
 *
 *	modification history:	
 *	(1) modified hdSetupHostName2HiaInfo()
 *		to set machine name for not defined machine name,
 *		on December 7th in '94.
 *	(2) add functions to read initial data from two files;
 *	    'helios.hia' and 'helios.init',
 *		on January 20th in '95.
 *
 * $Id: hdHiaInfo.c,v 1.8 1995/03/02 09:26:27 k3sato Exp $
 */

#include <stdio.h>
#include <memory.h>
#include <string.h>

#include <heliosd.h>
#include <hdMessage.h>
#include <hExternStdFunc.h>
#include <hdAgentInfo.h>
#include <hdHiaInfo.h>
#include <hdcommon.h>

#ifdef _DEBUG_HELIOSD_HIA

/*	defined in hdTestHiaInfo.c
 */
extern AgentInfo	hdTestHiaInfo;

#endif /* _DEBUG_HELIOSD_HIA */

HdBoolean hdGetHiaInfoFromFiles();

static HdBoolean hdGetDataFromHiaFile();
static HdBoolean hdGetAgentInfoFromHia();
static char *hdGetAgentIdFromHiaInfo();
static HdBoolean hdGetOneHiaInfoFromFile();
static HdBoolean hdSetHiaInfo2AgentInfo();
static HdBoolean hdGetAgentTblData();
static HdBoolean hdCheckNameIdInTbl();
static HdBoolean hdGetCommentFromFile();
static HdBoolean hdGetMethodDirFromFile();
static HdBoolean hdSetupMethodDir2AgentInfo();
static HdBoolean hdGetAgentDirFromFile();
static HdBoolean hdGetFuncDirFromFile();
static HdBoolean hdSetupFuncDir2AgentInfo();
static HdBoolean hdGetDataFromInitFile();
static HdBoolean hdGetAgentInfoFromInit();
static HdBoolean hdSetupAgentInfoFromInit();
static char *hdGetOneTermFromBuf();
static HdBoolean hdGetLineFromFile();
static HdBoolean hdCheckHiaComment();
static char *hdSkipSpace();
static HdBoolean hdCheckEmptyDataString();
static HdBoolean hdSetupHostName2HiaInfo();

void hdFreeHiaInfo();

/*
 *	get informations from files named *.hia
 */
HdBoolean hdGetHiaInfo(hdCtlInfo, allAgents)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		*allAgents;
{
    HeliosdErrorMsgType	eMsgType;

#ifdef _DEBUG_HELIOSD_HIA
    *allAgents = &hdTestHiaInfo;
#else
    if (hdGetHiaInfoFromFiles(hdCtlInfo, allAgents) == HdFalse) {
	eMsgType = HdEMsg_GetHiaInfoFromFilesError;
	goto Error_Rtn;
    }
#endif /* _DEBUG_HELIOSD_HIA */

    if (hdSetupHostName2HiaInfo(*allAgents, hdCtlInfo->machInfo->hostName) == HdFalse) {
	eMsgType = HdEMsg_SetupHostName2HiaInfoError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Get Hia information from 2 files; helios.hia, helios.init
 */
HdBoolean hdGetHiaInfoFromFiles(hdCtlInfo, allAgents)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		*allAgents;
{
    HeliosdErrorMsgType	eMsgType;

    if (hdGetDataFromHiaFile(hdCtlInfo, allAgents) == HdFalse) {
	eMsgType = HdEMsg_GetDataFromHiaFileError;
	goto Error_Rtn;
    }

    if (hdGetDataFromInitFile(hdCtlInfo, allAgents) == HdFalse) {
	eMsgType = HdEMsg_GetDataFromInitFileError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Get Data from a file named helios.hia
 */
static HdBoolean hdGetDataFromHiaFile(hdCtlInfo, allAgents)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		*allAgents;
{
    char		hiaFileName[HeliosdFilePathLen_Lim];
    FILE		*fd;
    HAgentInfo		*currInfo;
    HeliosdErrorMsgType	eMsgType;

    sprintf(hiaFileName, "%s%s", hdCtlInfo->fullPath, HeliosHiaFileName);

    if ((fd = fopen(hiaFileName, "r")) == NULL) {
	eMsgType = HdEMsg_HiaFileOpenError;
	goto Error_Rtn;
    }

    for (currInfo = allAgents; ; currInfo = &((*currInfo)->next)) {
	if (hdGetAgentInfoFromHia(hdCtlInfo, fd, NULL, currInfo) == HdFalse) {
	    eMsgType = HdEMsg_hdGetAgentInfoFromHiaError;
	    goto Close_Error_Rtn;
	}

	if (*currInfo == NULL) {
	    break;
	}
    }

    fclose(fd);

    return HdTrue;

 Close_Error_Rtn:
    fclose(fd);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Get AgentInfo from a file named helios.hia
 */
static HdBoolean hdGetAgentInfoFromHia(hdCtlInfo, fd, parentInfo, agentInfoP)
    HeliosdCtlInfo	*hdCtlInfo;
    FILE		*fd;
    HAgentInfo		parentInfo;
    HAgentInfo		*agentInfoP;
{
    HdHiaInfo		hiaInfo;
    HAgentInfo		agentInfo;
    HAgentInfo		*currInfoP;
    char		*bufPtr;
    char		agentId[AgentIDLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if (hdGetOneHiaInfoFromFile(hdCtlInfo, fd, &hiaInfo) == HdFalse) {
	eMsgType = HdEMsg_GetOneHiaInfoFromFileError;
	goto Error_Rtn;
    }

    if (hiaInfo.agentId == NULL) {
	agentInfo = NULL;
	goto Good_Rtn;
    }

    if (hdSetHiaInfo2AgentInfo(hdCtlInfo, &hiaInfo, &agentInfo) == HdFalse) {
	eMsgType = HdEMsg_SetHiaInfo2AgentInfoError;
	hdFreeHiaInfo(&hiaInfo);
	goto Error_Rtn;
    }

    agentInfo->parent = parentInfo;

    if (agentInfo->type == HdAT_Complex) {

	bufPtr = hiaInfo.processName;

	for (currInfoP = &(agentInfo->child)
	     ;; currInfoP = &((*currInfoP)->next)) {

	    bufPtr = hdGetAgentIdFromHiaInfo(agentId, bufPtr);

	    if (*bufPtr == '\0') {
		break;
	    }

	    if (hdGetAgentInfoFromHia(hdCtlInfo, fd, agentInfo, currInfoP)
		== HdFalse) {
		eMsgType = NULL;
		goto Error_Rtn;
	    }

	    if (*currInfoP == NULL) {
		break;
	    }

	    if (strcmp((*currInfoP)->agentId, agentId) != 0) {
		eMsgType = HdEMsg_ConflictAgentIdInHiaFile;
		goto Error_Rtn;
	    }
	}

	hdFreeString(&(hiaInfo.processName));
    }

    hdFreeString(&(hiaInfo.agentType));
    hdFreeString(&(hiaInfo.envName));
    hdFreeString(&(hiaInfo.tblName));

 Good_Rtn:
    *agentInfoP = agentInfo;
    return HdTrue;

 Error_Rtn:
    *agentInfoP = NULL;
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Get AgentId from HiaInfo->processName
 */
static char *hdGetAgentIdFromHiaInfo(agentId, hiaInfo)
    char	*agentId;
    char	*hiaInfo;
{
    char	ch;

    while ((ch = *hiaInfo) != HdChar_NULL
	   && (ch == HdChar_LeftBracket
	       || ch == HdChar_Space
	       || ch == HdChar_Tab
	       || ch == HdChar_Comma
	       || ch == HdChar_RightBracket)) {
	hiaInfo++;
    }

    while ((ch = *hiaInfo) != HdChar_NULL
	   && ch != HdChar_Space
	   && ch != HdChar_Tab
	   && ch != HdChar_Comma
	   && ch != HdChar_RightBracket) {
	*(agentId++) = ch;
	hiaInfo++;
    }
    *agentId = '\0';

    return hiaInfo;
}

/*
 * Get Hia Info from a file named helios.hia
 */
static HdBoolean hdGetOneHiaInfoFromFile(hdCtlInfo, fd, hiaInfo)
    HeliosdCtlInfo	*hdCtlInfo;
    FILE		*fd;
    HdHiaInfo		*hiaInfo;
{
    int			i;
    char		**strPtr;
    char		*bufPtr, *nextPtr;
    char		fileBuf[HdFileBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    memset(hiaInfo, 0, sizeof(HdHiaInfo));

    if (hdGetLineFromFile(fileBuf, fd) == HdFalse) {
	goto Good_Rtn;
    }

    for (i = 0, strPtr = (char **)hiaInfo, bufPtr = fileBuf
	 ; i < sizeof(HdHiaInfo)/sizeof(char *)
	 ; bufPtr = nextPtr, strPtr++, i++) {

	if ((nextPtr = hdGetOneTermFromBuf(&bufPtr)) == NULL) {
	    eMsgType = HdEMsg_EncounterEndOfLineInHia;
	    goto Error_Rtn;
	}

	*strPtr = HdMallocString(bufPtr);
    }

 Good_Rtn:
    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdSetHiaInfo2AgentInfo(hdCtlInfo, hiaInfo, agentInfo)
    HeliosdCtlInfo	*hdCtlInfo;
    HdHiaInfo		*hiaInfo;
    HAgentInfo		*agentInfo;
{
    HAgentInfo		info;
    int			agentType;
    HeliosdErrorMsgType	eMsgType;

    agentType = (int) hiaInfo->agentType[0] - (int)'0';

    if (agentType < 0 || agentType > 2) {
	eMsgType = HdEMsg_AgentTypeErrorInHiaInfo;
	goto Error_Rtn;
    }

    if ((info = *agentInfo = HdMalloc(sizeof(AgentInfo))) == NULL) {
	eMsgType = HdEMsg_HdMallocError;
	goto Error_Rtn;
    }

    info->type = (agentType == 0) ? HdAT_Single :
      (agentType == 1) ? HdAT_Complex : HdAT_Parametric;

    info->agentId = hiaInfo->agentId;
    info->agentName = hiaInfo->agentName;

    if (info->type != HdAT_Complex) {
	info->agentPath = hiaInfo->processName;
    }

    info->status = AS_NotActive;
#if 0
    if (hdGetAgentTblData(hdCtlInfo, info, hiaInfo->tblName) == HdFalse) {
	eMsgType = HdEMsg_GetAgentTblDataError;
	goto Error_Rtn;
    }
#endif
    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    fprintf(stderr, "Agent ID : %s\nAgent Type : %s\n Table Name : %s\nProcess Name : %s\n",
	    hiaInfo->agentId, hiaInfo->agentType, hiaInfo->tblName, hiaInfo->processName);
    return HdFalse;
}

/*
 *
 */
static HdBoolean hdGetAgentTblData(hdCtlInfo, agentInfo, tblFileName)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		agentInfo;
    char		*tblFileName;
{
    FILE		*fd;
    HeliosdErrorMsgType	eMsgType;

    if ((fd = fopen(tblFileName, "r")) == NULL) {
	eMsgType = HdEMsg_TblFileOpenError;
	goto Error_Rtn;
    }

    /* check Agent name and agent ID
     */
    if (hdCheckNameIdInTbl(hdCtlInfo, agentInfo, fd) == HdFalse) {
	eMsgType = HdEMsg_Fail2CheckAgentNameID;
	goto Close_Error_Rtn;
    }
    
    /* get Comment for agent
     */
    if (hdGetCommentFromFile(hdCtlInfo, agentInfo, fd) == HdFalse) {
	eMsgType = HdEMsg_CommentFromFileError;
	goto Close_Error_Rtn;
    }

    /* get Method directory
     */
    if (hdGetMethodDirFromFile(hdCtlInfo, agentInfo, fd) == HdFalse) {
	eMsgType = HdEMsg_GetMethodDirFromFileError;
	goto Close_Error_Rtn;
    }

    /* get agent directory
     * but skip the data.
     */
    if (hdGetAgentDirFromFile(hdCtlInfo, agentInfo, fd) == HdFalse) {
	eMsgType = HdEMsg_GetAgentDirFromFileError;
	goto Close_Error_Rtn;
    }

    /* get function directory
     */
    if (hdGetFuncDirFromFile(hdCtlInfo, agentInfo, fd) == HdFalse) {
	eMsgType = HdEMsg_GetFuncDirFromFileError;
	goto Close_Error_Rtn;
    }

    fclose(fd);

    return HdTrue;

 Close_Error_Rtn:
    fclose(fd);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Check Agent Name and Agent ID in tbl file.
 */
static HdBoolean hdCheckNameIdInTbl(hdCtlInfo, agentInfo, fd)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		agentInfo;
    FILE		*fd;
{
    char		fileBuf[HdFileBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    /* get Agent name
     */
    if (hdGetLineFromFile(fileBuf, fd) == HdFalse) {
	eMsgType = HdEMsg_NotAgentNameInTblFile;
	goto Error_Rtn;
    }
    
    if (strcmp(agentInfo->agentName, fileBuf) != 0) {
	eMsgType = HdEMsg_AgentNameUnmatchToHiaInfo;
	goto Error_Rtn;
    }

    /* get Agent ID
     * but skip the data.
     */
    if (hdGetLineFromFile(fileBuf, fd) == HdFalse) {
	eMsgType = HdEMsg_NotAgentIDInTblFile;
	goto Error_Rtn;
    }

    /* get Agent Type
     * but skip the data.
     */
    if (hdGetLineFromFile(fileBuf, fd) == HdFalse) {
	eMsgType = HdEMsg_NotAgentTypeInTblFile;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * get Comment for agent
 */
static HdBoolean hdGetCommentFromFile(hdCtlInfo, agentInfo, fd)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		agentInfo;
    FILE		*fd;
{
    char		fileBuf[HdFileBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if (hdGetLineFromFile(fileBuf, fd) == HdFalse) {
	eMsgType = HdEMsg_NotAgentCommentInTblFile;
	goto Error_Rtn;
    }

    if (hdCheckEmptyDataString(fileBuf) == HdFalse) {
	if ((agentInfo->comment = HdMallocString(fileBuf)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * get Method directory
 *
 */
static HdBoolean hdGetMethodDirFromFile(hdCtlInfo, agentInfo, fd)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		agentInfo;
    FILE		*fd;
{
    char		fileBuf[HdFileBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if (hdGetLineFromFile(fileBuf, fd) == HdFalse) {
	eMsgType = HdEMsg_NotMethodDirInTblFile;
	goto Error_Rtn;
    }

    if (hdCheckEmptyDataString(fileBuf) == HdFalse) {
	if (hdSetupMethodDir2AgentInfo(&(agentInfo->methodDir), fileBuf) == HdFalse) {
	    eMsgType = HdEMsg_SetupMethodDir2AgentInfoError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Setup Method directory to AgentInfo
 *
 *	Import\n {<method name> | none} [, {<method name> | none}]\n \
 *	Export\n {<method name> | none} [, {<method name> | none}]
 */
static HdBoolean hdSetupMethodDir2AgentInfo(methodDir, fileBuf)
    char		**methodDir;
    char		*fileBuf;
{
    char		*bufPtr;
    char		*newPtr;
    HeliosdErrorMsgType	eMsgType;

    /*
     * transform control string to control code.
     */
    for (bufPtr = newPtr = fileBuf; *bufPtr != HdChar_NULL; bufPtr++) {
	if (*bufPtr == HdChar_EscCode) {
	    switch(*++bufPtr) {
	    case 'n':
		*newPtr++ = HdChar_CR;
		break;
	    case 't':
		*newPtr++ = HdChar_Tab;
		break;
	    case 0:
		eMsgType = NULL;
		goto Error_Rtn;
	    default:
		*newPtr++ = *bufPtr;
	    }
	} else {
	    *newPtr++ = *bufPtr;
	}
    }

    *newPtr = HdChar_NULL;

    if ((*methodDir = HdMallocString(fileBuf)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * get agent directory
 * but skip the data.
 */
static HdBoolean hdGetAgentDirFromFile(hdCtlInfo, agentInfo, fd)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		agentInfo;
    FILE		*fd;
{
    char		fileBuf[HdFileBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

     if (hdGetLineFromFile(fileBuf, fd) == HdFalse) {
	eMsgType = HdEMsg_NotAgentDirInTblFile;
	goto Error_Rtn;
    }

    if (hdCheckEmptyDataString(fileBuf) == HdFalse) {
	if (agentInfo->type != HdAT_Complex) {
	    eMsgType = HdEMsg_AgentDirForNotComplexAgent;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* Get function directory
 *
 *  {[$B!c5!G=L>!d(B, $B!c%(!<%8%'%s%HL>!d(B [,$B!c%(!<%8%'%s%HL>!d(B]]}
 * [{[$B!c5!G=L>!d(B, $B!c%(!<%8%'%s%HL>!d(B [,$B!c%(!<%8%'%s%HL>!d(B]]}]
 *
 *  $B!JCm!((BFunction Directory $B$N:G8e$O!"(B"{}"$B!K(B
 */
static HdBoolean hdGetFuncDirFromFile(hdCtlInfo, agentInfo, fd)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		agentInfo;
    FILE		*fd;
{
    HFuncDirInfo	*funcDir;
    char		fileBuf[HdFileBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    for (funcDir = &(agentInfo->funcDir);;) {

	if (hdGetLineFromFile(fileBuf, fd) == HdFalse) {
	    eMsgType = HdEMsg_NotFunctionDirInTblFile;
	    goto Error_Rtn;
	}

	if (hdCheckEmptyDataString(fileBuf) == HdTrue) {
	    break;
	}

	if (hdSetupFuncDir2AgentInfo(funcDir, fileBuf) == HdFalse) {
	    eMsgType = HdEMsg_SetupFuncDir2AgentInfoError;
	    goto Error_Rtn;
	}

	for (; *funcDir != NULL; funcDir = &((*funcDir)->next))
	  ;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* Setup FuncDir of AgentInfo
 */
static HdBoolean hdSetupFuncDir2AgentInfo(funcDir, fileBuf)
    HFuncDirInfo	*funcDir;
    char		*fileBuf;
{
    HFuncDirInfo	*currDir;
    char		*bufPtr;
    char		*endPtr;
    HeliosdErrorMsgType	eMsgType;

    for (bufPtr = fileBuf, currDir = funcDir
	 ; *bufPtr != HdChar_NULL
	 ; bufPtr = ++endPtr, currDir = &((*currDir)->next)) {

	if ((bufPtr = strchr(bufPtr, HdChar_LeftBrace)) == NULL) {
	    if (currDir != funcDir) {
		break;
	    } else {
		eMsgType = HdEMsg_NotExistLeftBraceInData;
		goto Error_Rtn;
	    }
	}

	bufPtr = hdSkipSpace(++bufPtr);

	if ((endPtr = strchr(bufPtr, HdChar_Comma)) == NULL) {
	    eMsgType = HdEMsg_NotExistCommaInData;
	    goto Error_Rtn;
	}

	*endPtr = HdChar_NULL;

	if ((*currDir = HdMalloc(sizeof(FuncDirInfo))) == NULL) {
	    eMsgType = HdEMsg_HdMallocError;
	    goto Error_Rtn;
	}

	if (((*currDir)->funcName = HdMallocString(bufPtr)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Free_Error_Rtn;
	}

	bufPtr = ++endPtr;

	if ((endPtr = strchr(bufPtr, HdChar_RightBrace)) == NULL) {
	    eMsgType = HdEMsg_NotExistRightBraceInData;
	    goto Error_Rtn;
	}

	*endPtr = HdChar_NULL;

	if (((*currDir)->agentNames = HdMallocString(bufPtr)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Free_Error_Rtn;
	}
    }

    return HdTrue;

 Free_Error_Rtn:
	hdFreeFuncDirInfo(funcDir);

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Get Data from a file named helios.init
 */
static HdBoolean hdGetDataFromInitFile(hdCtlInfo, allAgents)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		*allAgents;
{
    char		initFileName[HeliosdFilePathLen_Lim];
    FILE		*fd;
    HeliosdErrorMsgType	eMsgType = NULL;
    HdBoolean		ret = HdTrue;

    sprintf(initFileName, "%s%s", hdCtlInfo->fullPath, HeliosInitFileName);

    if ((fd = fopen(initFileName, "r")) != NULL) {

	if (hdGetAgentInfoFromInit(hdCtlInfo, *allAgents, fd) == HdFalse) {
	    eMsgType = HdEMsg_GetAgentInfoFromInitError;
	    ret = HdFalse;
	}
	fclose(fd);

    } else {
	eMsgType = HdEMsg_InitFileOpenError;
    }

    hdPutsErrorMsg(eMsgType);
    return ret;
}

/*
 * Get AgentInfo from helios.init;
 *
 * $B!cC1=c%(!<%8%'%s%H(BID$B!d(B $B!c%^%7%sL>!d(B $B!c(Bgroup Id$B!d(B $B!c%(!<%8%'%s%H!&%W%m%;%9$KEO$90z?t!d(B
 */
static HdBoolean hdGetAgentInfoFromInit(hdCtlInfo, agentInfo, fd)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		agentInfo;
    FILE		*fd;
{
    char		fileBuf[HdFileBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    for (; hdGetLineFromFile(fileBuf, fd) == HdTrue; ) {
	if (hdSetupAgentInfoFromInit(hdCtlInfo, agentInfo, fileBuf) == HdFalse) {
	    eMsgType = HdEMsg_SetupAgentInfoFromInitError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * setup AgentInfo with the data of helios.init
 */
static HdBoolean hdSetupAgentInfoFromInit(hdCtlInfo, allAgents, bufPtr)
    HeliosdCtlInfo	*hdCtlInfo;
    HAgentInfo		allAgents;
    char		*bufPtr;
{
    HAgentInfo		currInfo;
    char		*nextPtr;
    HeliosdErrorMsgType	eMsgType;

    if ((nextPtr = hdGetOneTermFromBuf(&bufPtr)) == NULL) {
	eMsgType = HdEMsg_NotExistAgentIdInData;
	goto Error_Rtn;
    }

    if ((currInfo = hdSearchAgentInfo(allAgents, bufPtr)) == NULL) {
	hdPutsErrorMsg(HdEMsg_NotExistAgentInfoForId, bufPtr);
	goto Good_Rtn;
    }

    bufPtr = nextPtr;

    if ((nextPtr = hdGetOneTermFromBuf(&bufPtr)) == NULL) {
	eMsgType = HdEMsg_NotExistHostNameInData;
	goto Error_Rtn;
    }

    if ((currInfo->sockInfo.hostName = HdMallocString(bufPtr)) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    bufPtr = nextPtr;

    if ((nextPtr = hdGetOneTermFromBuf(&bufPtr)) == NULL) {
	eMsgType = HdEMsg_NotExistGroupNumInData;
	goto Error_Rtn;
    }

    currInfo->groupNum = atoi(bufPtr);

    if (nextPtr != NULL
	&& *nextPtr != NULL
	&& *(bufPtr = hdSkipSpace(nextPtr)) != NULL) {
	if ((currInfo->procArgs = HdMallocString(bufPtr)) == NULL) {
	    eMsgType = HdEMsg_HdMallocStringError;
	    goto Error_Rtn;
	}
    }

 Good_Rtn:
    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Get one term which is terminated with tab and space
 *	from string buffer.
 * skip spaces in head of buffer.
 *
 * return string pointer of next term in buffer.
 */
static char *hdGetOneTermFromBuf(startPtr)
    char	**startPtr;
{
    char		ch;
    char		*endPtr;
    HeliosdErrorMsgType	eMsgType;

    if ((endPtr = *startPtr = hdSkipSpace(*startPtr)) == NULL) {
	eMsgType = HdEMsg_NoMoreTermInBuffer;
	goto Error_Rtn;
    }

    if (*endPtr == HdChar_LeftBracket) {
	while ((ch = *endPtr) != HdChar_NULL
	       && ch != HdChar_RightBracket) {
	    endPtr++;
	}
	if (ch == HdChar_RightBracket) {
	    ch = *++endPtr;
	}
    } else {
	while ((ch = *endPtr) != HdChar_NULL
	       && ch != HdChar_Space
	       && ch != HdChar_Tab) {
	    endPtr++;
	}
    }

    if (ch != HdChar_NULL) {
	*endPtr++ = HdChar_NULL;
    } else {
	if (*startPtr == endPtr) {
	    endPtr = NULL;
	}
    }

    return endPtr;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 * Get one line from file buffer,
 *	but skipping comment lines.
 */
static HdBoolean hdGetLineFromFile(fileBuf, fd)
    char	*fileBuf;
    FILE	*fd;
{
    int		bufLength;
    char	*bufPtr;

    for (;;) {
	for (bufPtr = fileBuf, bufLength = HdFileBufferLen_Lim;;) {
	    if (fgets(bufPtr, bufLength, fd) == NULL) {
		goto Error_Rtn;
	    }

	    if ((bufLength -= strlen(bufPtr)) <= 0) {
		goto Error_Rtn;
	    }

	    if ((bufPtr = strstr(bufPtr, "\\\n")) == NULL)
	      break;
	}

	if (hdCheckHiaComment(fileBuf) == HdFalse)
	  break;
    }

    return HdTrue;

 Error_Rtn:
    return HdFalse;
}

/*
 * Check comment line in *.hia and *.init file
 *	return HdTrue, if comment line.
 *	return HdFalse, if not comment line.
 */
static HdBoolean hdCheckHiaComment(str)
    char	*str;
{
    for (; *str != NULL && *str == HdChar_Space; str++)
      ;

    if (*str == HdChar_NULL || *str == '#' || *str == HdChar_CR)
      goto Good_Rtn;

    /*
     * delete the comment part of non-comment line
     */
    for (; *str != HdChar_NULL && *str != '#'&& *str != HdChar_CR; str++)
      ;

    if (*str == '#' || *str == HdChar_CR)
      *str = '\0';

    return HdFalse;

 Good_Rtn:
    return HdTrue;
}

/*
 * Skip space or tab code in string
 */
static char *hdSkipSpace(str)
    char	*str;
{
    for (; *str != HdChar_NULL
	 && (*str == HdChar_Space
	     || *str == HdChar_Tab); str++)
      ;

    return str;
}

static HdBoolean hdCheckEmptyDataString(fileBuf)
    char	*fileBuf;
{

    if (strcmp(fileBuf, "{}") == 0) {
	goto Good_Rtn;
    }

    return HdFalse;

 Good_Rtn:
    return HdTrue;
}

/*
 * Setup host name and port number in agentInfo
 */
static HdBoolean hdSetupHostName2HiaInfo(agentInfo, hostName)
    HAgentInfo	agentInfo;
    char	*hostName;
{
    HeliosdErrorMsgType	eMsgType;

    for (; agentInfo != NULL; agentInfo = agentInfo->next) {

	if (agentInfo->agentPath != NULL
	    && agentInfo->sockInfo.hostName == NULL) {
	    if ((agentInfo->sockInfo.hostName = HdMallocString(hostName))
		== NULL) {
		eMsgType = HdEMsg_HdMallocStringError;
		goto Error_Rtn;
	    }
	}

	if (hdSetupHostName2HiaInfo(agentInfo->child, hostName) == HdFalse) {
	    eMsgType = HdEMsg_SetupHostName2HiaInfoError;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

void hdFreeHiaInfo(hiaInfo)
    HdHiaInfo	*hiaInfo;
{
    hdFreeString(&(hiaInfo->agentId));
    hdFreeString(&(hiaInfo->agentName));
    hdFreeString(&(hiaInfo->agentType));
    hdFreeString(&(hiaInfo->envName));
    hdFreeString(&(hiaInfo->tblName));
    hdFreeString(&(hiaInfo->processName));
}
