/* hdAgentInfo.c
 *	module to handle Agent information for Helios Daemon
 *		created by k3sato on November 10th in 1994.
 *
 *	modification history:	
 *
 * $Id: hdAgentInfo.c,v 2.3 1995/03/02 09:26:27 k3sato Exp $
 */

#include <stdio.h>
#include <string.h>
#include <memory.h>

#include <heliosd.h>
#include <hdMessage.h>
#include <hExternStdFunc.h>
#include <hdAgentInfo.h>
#include <hdSocket.h>
#include <hdcommon.h>

/* defined in hdcommon.c
 *
 * HdBoolean hdSetupAgentInfoFromMsg();
 */

/*  defined in hdDDMessage.c
 */
extern HAgentInfo hdSearchOtherDaemonsInfo();

/*  defined in hdUserInfo.c
 */
extern HAgentInfo hdSearchUserInfo();
extern char *hdGetNextComponent();

HdBoolean hdAppendParaInstance();

/*
 * Get the AgentInfo of the representative agent.
 *
 *	If parametric agent,
 *	return parametric template agentInfo.
 */
HAgentInfo hdGetRepAgentInfo(agentInfo)
    HAgentInfo		agentInfo;
{
    HeliosdErrorMsgType	eMsgType;

    switch (agentInfo->type) {
    case HdAT_Parametric:
	break;

    case HdAT_ParaInstance:
	if ((agentInfo = agentInfo->parent) == NULL) {
	    eMsgType = HdEMsg_NoExistRepAgentForParametric;
	    goto Error_Rtn;
	}
	break;
    case HdAT_Single:
	break;

    case HdAT_Complex:
	for (; agentInfo->agentPath == NULL; agentInfo = agentInfo->child) {
	    if (agentInfo->child == NULL) {
		eMsgType = HdEMsg_NoExistRepAgent;
		goto Error_Rtn;
	    }
	}
	break;

    case HdAT_NotDefined:
    default:
	eMsgType = HdEMsg_NoExistRepAgent;
	goto Error_Rtn;
    }

    return agentInfo;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return NULL;
}

/*
 *	Get the pointer of AgentInfo for agentId
 *	from HIA data.
 */
HdBoolean hdGetHAgentInfo(hHdCtlInfo, agentId, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
    HAgentInfo		*agentInfo;
{
    HeliosdErrorMsgType	eMsgType;

    if (hHdCtlInfo == NULL) {
	eMsgType = HdEMsg_NotDefinedHdCtlInfo;
	goto Error_Rtn;
    }

    if (agentId == NULL) {
	eMsgType = HdEMsg_NotDefinedAgentId;
	goto Error_Rtn;
    }

    if ((*agentInfo = hdSearchAgentInfo(hHdCtlInfo->allAgents, agentId)) == 0) {

	if ((*agentInfo = hdSearchUserInfo(hHdCtlInfo->userInfo, agentId)) == 0) {

	    if ((*agentInfo = hdSearchOtherDaemonsInfo(hHdCtlInfo, agentId)) == 0) {
		eMsgType = HdEMsg_NotDefinedAgentInfo;
		goto Error_Rtn;
	    }
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 * Set the agent status to the parent agents
 */
HdBoolean hdSetupStatus2ParentAgent(agentInfo, repAgentInfo)
    HAgentInfo		agentInfo;
    HAgentInfo		repAgentInfo;
{
    HAgentInfo		parentAgent;
    AgentStatusType	status;
    int			pid;

    status = repAgentInfo->status;
    pid = repAgentInfo->pid;

    for (parentAgent = repAgentInfo->parent
	 ; parentAgent != NULL && agentInfo != repAgentInfo
	 ; parentAgent = parentAgent->parent) {

	if (parentAgent->agentPath != NULL) {
	    if (parentAgent->type == HdAT_Parametric) {

		if (parentAgent->status == AS_NotActive) {
		    parentAgent->pid = pid;
		}
		parentAgent->status = status;
	    } else {
		break;
	    }
	}

	parentAgent->status = status;
	parentAgent->pid = pid;
    }

    return HdTrue;
}

/*
 *	Search AgentInfo for agentId
 *	from HIA data.
 */
HAgentInfo hdSearchAgentInfo(allAgents, agentId)
    HAgentInfo		allAgents;
    char		*agentId;
{
    HAgentInfo		agentInfo;

    for (; allAgents != NULL
	 ; allAgents = allAgents->next) {

	if (allAgents->agentId != NULL) {
	    if (strcmp(allAgents->agentId, agentId) == 0) {
		return allAgents;
	    }
	}
	if ((agentInfo = hdSearchAgentInfo(allAgents->child, agentId))
	    != NULL) {
	    return agentInfo;
	}
    }

    return NULL;
}

/*
 *	Search AgentInfo of agent with port number
 */
HAgentInfo hdSearchAgentInfoForPort(allAgents, sin_port)
    HAgentInfo		allAgents;
    u_short		sin_port;
{
    HAgentInfo		agentInfo;

    for (; allAgents != NULL
	 ; allAgents = allAgents->next) {
	if (allAgents->sockInfo.sin_port == sin_port) {
	    return allAgents;
	}
	if ((agentInfo = hdSearchAgentInfoForPort(allAgents->child, sin_port))
	    != NULL) {
	    return agentInfo;
	}
    }

    return NULL;
}

/* ==========================================================
 *  Functions to send AgentInfo to agent or user inteface
 * ========================================================== */
/*
 *  Send AgentInfo to Agent
 */
HdBoolean hdSendAgentInfo(hHdCtlInfo, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HAgentInfo		agentInfo;
{
    HeliosdErrorMsgType	eMsgType;

    for (; agentInfo != NULL; agentInfo = agentInfo->next) {

	if (hdSendOneAgentInfo(hHdCtlInfo, HdMsg_AgentInfo, agentInfo) == HdFalse) {
	    eMsgType = HdEMsg_SendOneAgentInfoError;
	    goto Error_Rtn;
	}

	if (hdGetAskForNextInfo(hHdCtlInfo) == HdFalse) {
	    eMsgType = HdEMsg_GetAskForNextInfoError;
	    goto Error_Rtn;
	}

	if (hdSendAgentInfo(hHdCtlInfo, agentInfo->child) == HdFalse) {
	    eMsgType = NULL;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	send AgenttInfo to agent
 */
HdBoolean hdSendOneAgentInfo(hHdCtlInfo, msgType, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HeliosdMsgType	msgType;
    HAgentInfo		agentInfo;
{
    char		sendBuf[SocketBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if (hdSetupAgentInfo2Msg(sendBuf, msgType, agentInfo) == HdFalse) {
	eMsgType = HdEMsg_SetupAgentInfo2Msg;
	goto Error_Rtn;
    }

    if (hdSend2Socket(hHdCtlInfo, hHdCtlInfo->s, sendBuf) == HdFalse) {
	eMsgType = HdEMsg_SendMsg2Agent;
	goto Error_Rtn;
    }

    if (agentInfo->procArgs != NULL) {
	if (hdGetAskForNextInfo(hHdCtlInfo) == HdFalse) {
	    eMsgType = HdEMsg_GetAskForNextInfoError;
	    goto Error_Rtn;
	}

	if (hdSend2Socket(hHdCtlInfo, hHdCtlInfo->s, agentInfo->procArgs) == HdFalse) {
	    eMsgType = HdEMsg_SendMsg2Agent;
	    goto Error_Rtn;
	}
    }

    if (agentInfo->funcDir != NULL) {
	if (hdSetupFuncDirInfo2Msg(sendBuf, HdMsg_FuncDirInfo, agentInfo->funcDir) == HdFalse) {
	    eMsgType = HdEMsg_SetupFuncDirInfoError;
	    goto Error_Rtn;
	}

	if (hdGetAskForNextInfo(hHdCtlInfo) == HdFalse) {
	    eMsgType = HdEMsg_GetAskForNextInfoError;
	    goto Error_Rtn;
	}

	if (hdSend2Socket(hHdCtlInfo, hHdCtlInfo->s, sendBuf) == HdFalse) {
	    eMsgType = HdEMsg_SendMsg2Agent;
	    goto Error_Rtn;
	}
    }

    if (agentInfo->methodDir != NULL) {
	if (hdSetupMethodDirInfo2Msg(sendBuf, HdMsg_MethodDirInfo, agentInfo->methodDir) == HdFalse) {
	    eMsgType = HdEMsg_SetupMethodDirInfoError;
	    goto Error_Rtn;
	}

	if (hdGetAskForNextInfo(hHdCtlInfo) == HdFalse) {
	    eMsgType = HdEMsg_GetAskForNextInfoError;
	    goto Error_Rtn;
	}

	if (hdSend2Socket(hHdCtlInfo, hHdCtlInfo->s, sendBuf) == HdFalse) {
	    eMsgType = HdEMsg_SendMsg2Agent;
	    goto Error_Rtn;
	}
    }

    if (agentInfo->comment != NULL) {
	if (hdGetAskForNextInfo(hHdCtlInfo) == HdFalse) {
	    eMsgType = HdEMsg_GetAskForNextInfoError;
	    goto Error_Rtn;
	}

	if (hdSend2Socket(hHdCtlInfo, hHdCtlInfo->s, agentInfo->comment) == HdFalse) {
	    eMsgType = HdEMsg_SendMsg2Agent;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	setup AgentInfo to message buffer
 */
HdBoolean hdSetupAgentInfo2Msg(msgBuf, msgType, agentInfo)
    char		*msgBuf;
    HeliosdMsgType	msgType;
    HAgentInfo		agentInfo;
{
    HSocketInfo		sockInfo;
    HeliosdErrorMsgType	eMsgType;

    sockInfo = &(agentInfo->sockInfo);

    sprintf(msgBuf, "%d %d %s %s %s %x %x %x %s %d %d %d %d %d %x %x %x %x",
	    msgType,
	    agentInfo->type,
	    agentInfo->agentId,
	    agentInfo->agentPath,
	    agentInfo->agentName,
	    agentInfo->procArgs,
	    agentInfo->funcDir,
	    agentInfo->methodDir,
	    sockInfo->hostName,
	    sockInfo->sin_family,
	    sockInfo->sin_port,
	    agentInfo->pid,
	    agentInfo->status,
	    agentInfo->groupNum,
	    agentInfo->comment,
	    agentInfo->parent,
	    agentInfo->child,
	    agentInfo->next
	    );

    if (strlen(msgBuf) >= SocketBufferLen_Lim) {
	eMsgType = HdEMsg_SocketBufferLenLimitOver;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	setup FuncDirInfo of AgentInfo to message buffer
 */
HdBoolean hdSetupFuncDirInfo2Msg(msgBuf, msgType, funcDirInfo)
    char		*msgBuf;
    HeliosdMsgType	msgType;
    HFuncDirInfo	funcDirInfo;
{
    int			length;
    int			totalLen;
    char		*currBufPtr;
    HeliosdErrorMsgType	eMsgType;

    sprintf(msgBuf, "%d ", msgType);

    totalLen = strlen(msgBuf);
    for (currBufPtr = msgBuf + totalLen
	 ; funcDirInfo != NULL
	 ; funcDirInfo = funcDirInfo->next, currBufPtr += length) {

	sprintf(currBufPtr, "%s %s\n",
		funcDirInfo->funcName, funcDirInfo->agentNames);

	length = strlen(currBufPtr);

	if ((totalLen += length) >= SocketBufferLen_Lim) {
	    eMsgType = HdEMsg_SocketBufferLenLimitOver;
	    goto Error_Rtn;
	}
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/*
 *	setup methodDirInfo of AgentInfo to message buffer
 */
HdBoolean hdSetupMethodDirInfo2Msg(msgBuf, msgType, methodDirInfo)
    char		*msgBuf;
    HeliosdMsgType	msgType;
    char		*methodDirInfo;	/* HMethodDirInfo */
{
    /*
    int			length;
    int			totalLen;
    char		*currBufPtr;
    HeliosdErrorMsgType	eMsgType;

    sprintf(msgBuf, "%d ", msgType);

    totalLen = strlen(msgBuf);
    for (currBufPtr = msgBuf + totalLen
	 ; methodDirInfo != NULL
	 ; methodDirInfo = methodDirInfo->next, currBufPtr += length) {

	sprintf(currBufPtr, "%s ", methodDirInfo->methodName);

	length = strlen(currBufPtr);

	if ((totalLen += length) >= SocketBufferLen_Lim) {
	    eMsgType = HdEMsg_SocketBufferLenLimitOver;
	    goto Error_Rtn;
	}
    }
    */
    sprintf(msgBuf, "%d %s", msgType, methodDirInfo);

    return HdTrue;

    /*
 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
    */
}

/*
 *	send SocketInfo to agent or daemon
 */
HdBoolean hdSendSocketInfo(hHdCtlInfo, sockFd, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    int			sockFd;
    HAgentInfo		agentInfo;
{
    HAgentInfo		repAgentInfo;
    HSocketInfo		sockInfo;
    char		*hostName;
    char		sendBuf[SocketBufferLen_Lim];
    HeliosdErrorMsgType	eMsgType;

    if (agentInfo->sockInfo.sin_port != NULL) {
	repAgentInfo = agentInfo;
    } else {
	if ((repAgentInfo = hdGetRepAgentInfo(agentInfo)) == NULL) {
	    eMsgType = HdEMsg_NotGetRepAgentInfo;
	    goto Error_Rtn;
	}
	if (repAgentInfo->type == HdAT_Parametric) {
	    if ((repAgentInfo = repAgentInfo->child) == NULL) {
		eMsgType = HdEMsg_NotGetRepAgentInfo;
		goto Error_Rtn;
	    }
	}
    }

    sockInfo = &(repAgentInfo->sockInfo);

    hostName = (repAgentInfo->type == HdAT_ParaInstance) ?
      repAgentInfo->parent->sockInfo.hostName : sockInfo->hostName;

    sprintf(sendBuf, "%d %s %d %d",
	    HdMsg_SocketInfo, hostName, sockInfo->sin_family, sockInfo->sin_port);

    if (strlen(sendBuf) >= SocketBufferLen_Lim) {
	eMsgType = HdEMsg_SocketBufferLenLimitOver;
	goto Error_Rtn;
    }

    if (hdSend2Socket(hHdCtlInfo, sockFd, sendBuf) == HdFalse) {
	eMsgType = HdEMsg_SendMsg2Agent;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ========================================================
 *   functions for manipulating parametric agents
 * ======================================================== */
/*
 * Check parametric instance agent
 * and Create structure AgentInfo for a instance of the parametric agent
 */
HdBoolean hdCheckParaInstance(hHdCtlInfo, agentId, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*agentId;
    HAgentInfo		*agentInfo;
{
    char		*tempStr;
    char		parentId[AgentIDLen_Lim];
    HAgentInfo		parentInfo;
    HAgentInfo		tempInfo;
    HeliosdErrorMsgType	eMsgType;

    strcpy(parentId, agentId);

    if ((tempStr = strrchr(parentId, '.')) == NULL) {
	eMsgType = HdEMsg_IllegalAgentId;
	goto Error_Rtn;
    }

    *tempStr = '\0';

    if ((parentInfo = hdSearchAgentInfo(hHdCtlInfo->allAgents, parentId)) == 0) {
	eMsgType = HdEMsg_NotDefinedAsParametricAgent;
	goto Error_Rtn;
    }

    if (parentInfo->type != HdAT_Parametric) {
	eMsgType = HdEMsg_NotParametricAgent;
	goto Error_Rtn;
    }

    if ((*agentInfo =
	 tempInfo = hdMallocParaInstance(parentInfo)) == NULL) {
	eMsgType = HdEMsg_MallocParaInstanceError;
	goto Error_Rtn;
    }

    if ((tempInfo->agentId = HdMallocString(agentId)) == NULL) {
	hdFreeString(agentInfo);
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    if (hdAppendParaInstance(parentInfo, tempInfo) == HdFalse) {
	eMsgType = HdEMsg_AppendParaInstanceError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

HdBoolean hdAppendParaInstance(parentInfo, agentInfo)
    HAgentInfo		parentInfo;
    HAgentInfo		agentInfo;
{
    HAgentInfo		*childInfo;
    HeliosdErrorMsgType	eMsgType;

    if (parentInfo == NULL || parentInfo->type != HdAT_Parametric) {
	eMsgType = HdEMsg_NotParametricAgent;
	goto Error_Rtn;
    }

    if (agentInfo == NULL || agentInfo->type != HdAT_ParaInstance) {
	eMsgType = HdEMsg_NotParaInstanceAgent;
	goto Error_Rtn;
    }

    for (childInfo = &(parentInfo->child)
	 ; *childInfo != NULL
	 ; childInfo = &((*childInfo)->next))
      ;

    *childInfo = agentInfo;

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

#define HdParaInfoComponentMaxNum	3

/*
 * Setup parametric agent data into a AgentInfo structure.
 *
 *  component of paraInfo;
 *	<sin_port>:	# strPtr[0]
 *	<agentId>:	# strPtr[1]
 *	<agent name>:	# strPtr[2]
 */
HdBoolean hdSetupParaInfo2AgentInfo(paraInfo, agentInfo)
    char		*paraInfo;
    HAgentInfo		agentInfo;
{
    char		*strPtr[HdParaInfoComponentMaxNum];
    char		**currStrPtr;
    char		*nextStr;
    int			cnt;
    HeliosdErrorMsgType	eMsgType;

    for (currStrPtr = strPtr, *currStrPtr = paraInfo, cnt = 1
	 ; cnt < HdParaInfoComponentMaxNum
	 ; *(++currStrPtr) = nextStr, cnt++) {

	if ((nextStr = hdGetNextComponent(*currStrPtr)) == NULL) {
	    eMsgType = HdEMsg_GetNextComponentError;
	    goto Error_Rtn;
	}
    }

    currStrPtr = strPtr;

    agentInfo->sockInfo.sin_port = HdStr2Int(*(currStrPtr++));

    if ((agentInfo->agentId = HdMallocString(*(currStrPtr++))) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    if ((agentInfo->agentName = HdMallocString(*(currStrPtr++))) == NULL) {
	eMsgType = HdEMsg_HdMallocStringError;
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    hdPutsErrorMsg(eMsgType);
    return HdFalse;
}

/* ===========================
 *  For machine information
 * =========================== */
/*
 * Check agent process is in home machine
 */
HdBoolean hdCheckHomeMachine(hHdCtlInfo, agentInfo)
    HHeliosdCtlInfo	hHdCtlInfo;
    HAgentInfo		*agentInfo;
{
    HAgentInfo	repAgentInfo;
    char	*hostName;

    if ((hostName = hHdCtlInfo->machInfo->hostName) == NULL) {
	goto Error_Rtn;
    }

    if ((repAgentInfo = hdGetRepAgentInfo(agentInfo)) == NULL) {
	goto Error_Rtn;
    }

    if (strcmp(hostName, repAgentInfo->sockInfo.hostName) != 0) {
	goto Error_Rtn;
    }

    return HdTrue;

 Error_Rtn:
    return HdFalse;
}

HMachInfo hdSearchMachInfo(hHdCtlInfo, hostName)
    HHeliosdCtlInfo	hHdCtlInfo;
    char		*hostName;
{
    HMachInfo		machInfo;

    for (machInfo = hHdCtlInfo->machInfo; machInfo != NULL; machInfo = machInfo->next) {
	if (strcmp(machInfo->hostName, hostName) == 0) {
	    return machInfo;
	}
    }

    return NULL;
}
