/* Copyright	Massachusetts Institute of Technology	1987, 1988
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * M.I.T. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL M.I.T.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifdef DEBUG
#include <stdio.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include <X11/CompositeP.h>
#include <X11/ConstrainP.h>
#include "ModuleP.h"

#define MAX(a, b)       (((a) > (b)) ? (a): (b))
#define IsVertical(lw) ((lw)->module.gravity == NorthGravity || \
			  (lw)->module.gravity == SouthGravity)
#define IsMirror(lw)   ((lw)->module.gravity == SouthGravity || \
			  (lw)->module.gravity == EastGravity)

					/* widget class method */
static void		ClassInitialize();
static void             Initialize();
static void             ConstraintInitialize();
static Boolean          ConstraintSetValues();
static void             Destroy();
static Boolean          SetValues();
static XtGeometryResult GeometryManager();
static void             ChangeManaged();
static void             Redisplay();

					/* utility routines */
static void             insert_node();
static void             delete_node();
static void             layout_module();

static ModuleOffsetPtr	create_offset();
static void		reset_positions();
static void		reset_offset();
static Position		current_position();
static void		set_current_position();
static Position		sum_of_positions();
static int		sort_list();
static int		compute_positions();
static void		set_positions();
static void		mirror_positions();
static void		set_point();
static int		get_depth();

/*
 * resources of the module itself
 */
static XtResource resources[] = {
    { XtNhPad, XtCHPad, XtRDimension, sizeof (Dimension),
	XtOffsetOf(QxtModuleRec, module.hpad), XtRImmediate, (XtPointer) 0 },
    { XtNvPad, XtCVPad, XtRDimension, sizeof (Dimension),
	XtOffsetOf(QxtModuleRec, module.vpad), XtRImmediate, (XtPointer) 0 },
    { XtNlineColor, XtCLineColor, XtRPixel, sizeof (Pixel),
	XtOffsetOf(QxtModuleRec, module.line_pixel), XtRString,
	XtDefaultForeground},
    { XtNlineWidth, XtCLineWidth, XtRDimension, sizeof (Dimension),
	XtOffsetOf(QxtModuleRec,module.line_width),XtRImmediate,(XtPointer)1},
    { XtNgravity, XtCGravity, XtRGravity, sizeof (XtGravity),
	XtOffsetOf(QxtModuleRec, module.gravity), XtRImmediate,
	(XtPointer) NorthGravity },
};

/*
 * resources that are attached to all children of the module
 */
static XtResource moduleConstraintResources[] = {
  { XtNparents, XtCParents, XtRWidgetList, sizeof(XtPointer),
	XtOffsetOf(ModuleConstraintsRec,module.parents),XtRWidgetList,NULL },
  { XtNnumParents, XtCNumParents, XtRInt, sizeof(int),
	XtOffsetOf(ModuleConstraintsRec,module.n_parents), XtRImmediate, 0 },
  { XtNmoduleGC, XtCModuleGC, XtRGC, sizeof(GC),
	XtOffsetOf(ModuleConstraintsRec, module.gc), XtRImmediate, NULL },
};

QxtModuleClassRec QxtmoduleClassRec = {
  {
					/* core_class fields  */
    (WidgetClass) &constraintClassRec,	/* superclass         */
    "Module",				/* class_name         */
    sizeof(QxtModuleRec),		/* widget_size        */
    ClassInitialize,			/* class_init         */
    NULL,				/* class_part_init    */
    FALSE,				/* class_inited       */	
    Initialize,				/* initialize         */
    NULL,				/* initialize_hook    */	
    XtInheritRealize,			/* realize            */
    NULL,				/* actions            */
    0,					/* num_actions        */	
    resources,				/* resources          */
    XtNumber(resources),		/* num_resources      */
    NULLQUARK,				/* xrm_class          */
    TRUE,				/* compress_motion    */	
    TRUE,				/* compress_exposure  */	
    TRUE,				/* compress_enterleave*/	
    TRUE,				/* visible_interest   */
    Destroy,				/* destroy            */
    NULL,				/* resize             */
    Redisplay,				/* expose             */
    SetValues,				/* set_values         */
    NULL,				/* set_values_hook    */	
    XtInheritSetValuesAlmost,		/* set_values_almost  */
    NULL,				/* get_values_hook    */	
    NULL,				/* accept_focus       */
    XtVersion,				/* version            */	
    NULL,				/* callback_private   */
    NULL,				/* tm_table           */
    NULL,				/* query_geometry     */	
    NULL,				/* display_accelerator*/
    NULL,				/* extension          */
  },
  {
					/* composite_class fields */
    GeometryManager,			/* geometry_manager    */
    ChangeManaged,			/* change_managed      */
    XtInheritInsertChild,		/* insert_child        */	
    XtInheritDeleteChild,		/* delete_child        */	
    NULL,				/* extension           */
  },
  { 
					/* constraint_class fields */
   moduleConstraintResources,		/* subresources        */
   XtNumber(moduleConstraintResources),/* subresource_count   */
   sizeof(ModuleConstraintsRec),	/* constraint_size     */
   ConstraintInitialize,		/* initialize          */
   NULL,				/* destroy             */
   ConstraintSetValues,			/* set_values          */
   NULL,				/* extension           */
   },
  {
					/* Module class fields */
    0,					/* ignore              */	
  }
};

WidgetClass QxtmoduleWidgetClass = (WidgetClass)&QxtmoduleClassRec;

/*----------------------------------------------------------------
 * module utility routines 
 *----------------------------------------------------------------*/

static GC get_gc(w)
QxtModuleWidget w;
{
    XtGCMask valuemask = GCBackground | GCForeground | GCCapStyle;
    XGCValues values;

    values.background = w->core.background_pixel;
    values.foreground = w->module.line_pixel;
    values.cap_style  = CapProjecting;
    if (w->module.line_width != 0) {
	valuemask |= GCLineWidth;
	values.line_width = w->module.line_width;
    }

    return XtGetGC ((Widget) w, valuemask, &values);    
}

static void insert_node (pindex, parent, node)
     Widget parent, node;
{
    ModuleConstraints pc;
    ModuleConstraints lc = MODULE_CONSTRAINT(node);
    int nindex;
  
    if (pindex >= lc->module.n_parents) {
	lc->module.parents = (WidgetList)
	    XtRealloc(lc->module.parents, sizeof(Widget) * (pindex+1));
    }
    lc->module.parents[pindex] = parent;

    if (parent == NULL) return;

    /*
     * If there isn't more room in the children array, 
     * allocate additional space.
     */  
    pc = MODULE_CONSTRAINT(parent);
    nindex = pc->module.n_children;
  
    if (pc->module.n_children == pc->module.max_children) {
	pc->module.max_children += (pc->module.max_children / 2) + 2;
	pc->module.children = 
	    (WidgetList) XtRealloc ((char *)pc->module.children, 
				    (unsigned int)
				    ((pc->module.max_children) *
				     sizeof(Widget)));
    } 

    /*
     * Add the sub_node in the next available slot and 
     * increment the counter.
     */
    pc->module.children[nindex] = node;
    pc->module.n_children++;
}

static void delete_node (parent, node)
    Widget parent, node;
{
    ModuleConstraints pc;
    int pos, i;

    /*
     * Make sure the parent exists.
     */
    if (!parent) return;  
  
    pc = MODULE_CONSTRAINT(parent);

    /*
     * Find the sub_node on its parent's list.
     */
    for (pos = 0; pos < pc->module.n_children; pos++)
      if (pc->module.children[pos] == node) break;

    if (pos == pc->module.n_children) return;

    /*
     * Decrement the number of children
     */  
    pc->module.n_children--;

    /*
     * Fill in the gap left by the sub_node.
     * Zero the last slot for good luck.
     */
    for (i = pos; i < pc->module.n_children; i++) 
      pc->module.children[i] = pc->module.children[i+1];

    pc->module.children[pc->module.n_children]=0;
}

static void check_gravity (lw, grav)
QxtModuleWidget lw;
XtGravity grav;
{
    switch (lw->module.gravity) {
      case WestGravity: case NorthGravity: case EastGravity: case SouthGravity:
	break;
      default:
	lw->module.gravity = grav;
	break;
    }
}

/*----------------------------------------------------------------
 * module class methods 
 *----------------------------------------------------------------*/

static void ClassInitialize ()
{
    XawInitializeWidgetSet();
    XtAddConverter (XtRString, XtRGravity, XmuCvtStringToGravity,
		    (XtConvertArgList) NULL, (Cardinal) 0);
}

static void Initialize(grequest, gnew)
Widget grequest, gnew;
{
    QxtModuleWidget request = (QxtModuleWidget) grequest, 
    		  new = (QxtModuleWidget) gnew;

    /*
     * Make sure the widget's width and height are 
     * greater than zero.
     */
    if (request->core.width <= 0) new->core.width = 5;
    if (request->core.height <= 0) new->core.height = 5;

    /*
     * Set the padding according to the orientation
     */
    if (request->module.hpad == 0 && request->module.vpad == 0) {
	new->module.hpad = MODULE_HORIZONTAL_DEFAULT_SPACING;
	new->module.vpad = MODULE_VERTICAL_DEFAULT_SPACING;
    }

    /*
     * Create a graphics context for the connecting lines.
     */
    new->module.gc = get_gc (new);

    /*
     * Create the hidden top and bottom widget.
     */
    new->module.module_root = (Widget) NULL;
    new->module.module_root = XtVaCreateWidget ("root", widgetClass, gnew, 
					    XtNwidth, 1, 
					    XtNheight, 1, 
					    XtNlabel, "root",
					    NULL);
    /*
     * Allocate the tables used by the layout algorithm
     */
    new->module.horizontal = create_offset(10);
    new->module.vertical = create_offset(10);  
  
    /*
     * make sure that our gravity is one of the acceptable values
     */
    check_gravity (new, NorthGravity);
}

static void ConstraintInitialize (request, new)
Widget request, new;
{
    ModuleConstraints lc = MODULE_CONSTRAINT(new);
    QxtModuleWidget lw = (QxtModuleWidget)new->core.parent;
    register int i;

    /* 
     * Initialize the widget to have no subnodes.
     */
    lc->module.n_parents = 0;
    lc->module.parents = NULL;
    lc->module.n_children = lc->module.max_children = 0;
    lc->module.children = (WidgetList)NULL;
    lc->module.x = lc->module.y = 0;

    /*
     * If this widget has a parent_list, add it to that 
     * widget' child list. Otherwise make it a child of 
     * the module_root widget.
     */
    for (i = 0; i < lc->module.n_parents; i++) { 
	insert_node (i, lc->module.parents[i], new);
    }
    if (i == 0 && lw->module.module_root) {
	insert_node (i, lw->module.module_root, new);
	lc->module.n_parents = 1;
    }
}

static Boolean SetValues (gcurrent, grequest, gnew)
Widget gcurrent, grequest, gnew;
{
    QxtModuleWidget current = (QxtModuleWidget)gcurrent, 
    		  new = (QxtModuleWidget)gnew;
    Boolean redraw = FALSE;

    /*
     * If the foreground color has changed, redo the GC's
     * and indicate a redraw.
     */
    if (new->module.line_pixel != current->module.line_pixel ||
	new->core.background_pixel != current->core.background_pixel ||
	new->module.line_width != current->module.line_width) {
	XtReleaseGC (gnew, new->module.gc);
	new->module.gc = get_gc (new);
	redraw = TRUE;     
    }
    /*
     * If the minimum spacing has changed, recalculate the
     * module layout. layout_module() does a redraw, so we don't
     * need SetValues to do another one.
     */
    if (new->module.gravity != current->module.gravity) {
	check_gravity (new, current->module.gravity);
    }

    if (IsVertical(new) != IsVertical(current)) {
	if (new->module.vpad == current->module.vpad &&
	    new->module.hpad == current->module.hpad) {
	    new->module.vpad = current->module.hpad;
	    new->module.hpad = current->module.vpad;
	}
    }

    if (new->module.vpad != current->module.vpad ||
	new->module.hpad != current->module.hpad) {
	layout_module (new);
	redraw = FALSE;
    }
    return redraw;
}

static Boolean ConstraintSetValues (current, request, new, args, num_args)
Widget current, request, new;
ArgList args;
Cardinal *num_args;
{
    ModuleConstraints newc = MODULE_CONSTRAINT(new);
    ModuleConstraints curc = MODULE_CONSTRAINT(current);
    QxtModuleWidget lw = (QxtModuleWidget) new->core.parent;
    Boolean layout = FALSE;
    register int i;

    if (curc->module.n_parents != newc->module.n_parents ||
	curc->module.parents != newc->module.parents) {
	if (curc->module.parents) {
	    for (i = 0; i < curc->module.n_parents; i++) {
		delete_node(curc->module.parents[i], new);
	    }
	}
	for (i = 0; i < newc->module.n_parents; i++) {
	    insert_node(i, newc->module.parents[i], new);
	}
	layout = TRUE;
    }
    /*
     * If the Module widget has been realized, 
     * compute new layout.
     */
    if (layout && XtIsRealized((Widget)lw))
	layout_module(lw);
    return False;
}

/* ARGSUSED */
static XtGeometryResult GeometryManager (w, request, reply)
Widget w;
XtWidgetGeometry *request;
XtWidgetGeometry *reply;
{

    QxtModuleWidget lw = (QxtModuleWidget) w->core.parent;

    /*
     * No position changes allowed!.
     */
    if ((request->request_mode & CWX && request->x!=w->core.x)
	||(request->request_mode & CWY && request->y!=w->core.y))
      return (XtGeometryNo);

    /*
     * Allow all resize requests.
     */

    if (request->request_mode & CWWidth)
      w->core.width = request->width;
    if (request->request_mode & CWHeight)
      w->core.height = request->height;
    if (request->request_mode & CWBorderWidth)
      w->core.border_width = request->border_width;

    /*
     * Compute the new layout based on the new widget sizes.
     */
    layout_module (lw);
    return (XtGeometryYes);
}

static void ChangeManaged (gw)
Widget gw;
{
    layout_module ((QxtModuleWidget) gw);
}

static void Destroy (gw)
Widget gw;
{
    QxtModuleWidget w = (QxtModuleWidget) gw;
    XtReleaseGC (gw, w->module.gc);
}

static void Redisplay (lw, event, region)
QxtModuleWidget lw;
XEvent *event;
Region region;
{
    register int i, j, srcx, srcy;

    /*
     * If the Module widget is visible, visit each managed child.
     */
    if (lw->core.visible) {
	for (i = 0; i < lw->composite.num_children; i++) {
	    Widget child = lw->composite.children[i];
	    ModuleConstraints lc = MODULE_CONSTRAINT(child);

	    if (child != lw->module.module_root && 
		XtIsManaged(child) && lc->module.n_children)
	    {
		switch (lw->module.gravity) {
		  case NorthGravity:
		    srcx = child->core.x + child->core.width/2 
			+ child->core.border_width;
		    srcy = child->core.y + child->core.height
			+ child->core.border_width*2;
		    break;
		  case SouthGravity:
		    srcx = child->core.x + child->core.width/2 
			+ child->core.border_width;
		    srcy = child->core.y;
		    break;
		  case WestGravity:
		    srcx = child->core.x + child->core.width
			+ child->core.border_width*2;
		    srcy = child->core.y + child->core.height/2
			+ child->core.border_width;
		    break;
		  case EastGravity:
		    srcx = child->core.x;
		    srcy = child->core.y + child->core.height/2
			+ child->core.border_width;
		    break;
		}

		for (j = 0; j < lc->module.n_children; j++) {
		    register Widget k = lc->module.children[j];
		    ModuleConstraints klc = MODULE_CONSTRAINT(k);
		    XPoint points[3];
		    register int npoint = 0;

		    if (!XtIsManaged(k))
			continue;
		    set_point(&points[npoint], srcx, srcy); npoint++;

		    switch (lw->module.gravity) {
		      case NorthGravity:
			set_point(&points[npoint], 
				  k->core.x + k->core.border_width
				     + k->core.width/2,
				  k->core.y); 
			npoint++;
			break;
		      case SouthGravity:
			set_point(&points[npoint], 
				  k->core.x + k->core.border_width
				     + k->core.width/2,
				  k->core.y + k->core.height 
				     + k->core.border_width*2); 
			npoint++;
			break;
		      case WestGravity:
			set_point(&points[npoint], 
				  k->core.x,
				  k->core.y + k->core.border_width
				     + k->core.height/2);
			npoint++;
			break;
		      case EastGravity:
			set_point(&points[npoint], 
				  k->core.x + k->core.width 
				     + k->core.border_width*2,
				  k->core.y + k->core.border_width 
				     + k->core.height / 2);
			npoint++;
			break;
		    }
		    XDrawLines(XtDisplay(lw), XtWindow(lw), lw->module.gc,
			       points, npoint, CoordModeOrigin);
		}
	    }
	}
    }
}

/* module layout algorithm */

static void layout_module(lw)
QxtModuleWidget lw;
{
    sort_list(lw);

    /*
     * reset the positions;
     */
    reset_positions(lw);
    reset_offset(lw->module.vertical);
    reset_offset(lw->module.horizontal);

    /*
     * compute each widget's x, y position.
     */
    compute_positions(lw, lw->module.module_root, 0);

    /*
     * Move each widget into place.
     */
    set_positions(lw, lw->module.module_root, 0);
    if (IsMirror(lw))
	mirror_positions(lw, lw->module.module_root, 0);

    /*
     * Redisplay the lines connecting nodes.
     */
    if (XtIsRealized(lw)) {
	XClearWindow(XtDisplay(lw), XtWindow(lw));
	Redisplay(lw, NULL, NULL);
    }
}

static int compare(w1, w2)
Widget *w1, *w2;
{
    return((int)*w1 - (int)*w2);
}

static int sort_list(lw)
QxtModuleWidget lw;
{
    int i;

    for (i = 0; i < lw->composite.num_children; i++) {
	Widget w = lw->composite.children[i];
	ModuleConstraints lc = MODULE_CONSTRAINT(w);

	qsort((char *)lc->module.parents, lc->module.n_parents,
	      sizeof(Widget), compare);
	qsort((char *)lc->module.children, lc->module.n_children,
	      sizeof(Widget), compare);
    }
}

static int compute_positions(lw, w, level)
QxtModuleWidget lw;
Widget w;
int level;
{
    Position cur_hpos, cur_vpos;
    int i, depth = 0;
    ModuleConstraints lc = MODULE_CONSTRAINT(w);
    Boolean isvert = IsVertical(lw);


    if ((lc->module.x >= 0 || lc->module.y >= 0) &&
	level == lc->module.level)
	return;

    /*
     * set level
     */
    lc->module.level = MAX(lc->module.level, level);

    /*
     * get the current positions for this level.
     */
    cur_hpos = current_position(lw->module.horizontal, level);
    cur_vpos = current_position(lw->module.vertical, level);

    /* 
     * set the current vertical height to the max heights of all widgets
     * at this level.
     */
    if (isvert) 
	set_current_position(lw->module.vertical, level,
			     MAX(cur_vpos, w->core.height));
    else {
	if (XtIsManaged(w))
	    set_current_position(lw->module.horizontal, level,
				 MAX(cur_hpos, w->core.width));
	else
	    set_current_position(lw->module.horizontal, level, cur_hpos);
    }

    if (isvert)
	lc->module.x = cur_hpos;
    else
	lc->module.y = cur_vpos;

    for (i = 0; i < lc->module.n_children; i++) {
	depth = compute_positions(lw, lc->module.children[i],level+1);
    }

    /*
     * record the current horizontal position at this level.
     */
    if (isvert)
	set_current_position(lw->module.horizontal, level, 
		lw->module.hpad + lc->module.x + w->core.width);
    else
	set_current_position(lw->module.vertical, level, 
		lw->module.vpad + lc->module.y + w->core.height);
    return(MAX(depth, level));
}

static void set_positions(lw, w, level)
QxtModuleWidget lw;
Widget w;
int level;
{
    register int i;
    Dimension	rw = 0, rh = 0;
    Position	pos = 0;
    XtGeometryResult result;
    ModuleConstraints lc = MODULE_CONSTRAINT(w);
    Boolean isvert = IsVertical(lw);
    static int top;

    lw->module.width = lw->core.width;
    lw->module.height = lw->core.height;

    if (w) {
	if (isvert) {
	    if (lc->module.level == 1) 		/* TOP */
		top = lc->module.x + w->core.width/2;
	    /* 
	     * Add up the sum of the height's of all nodes to this depth,
	     * and use it as the y position.
	     */
	    pos = (level * lw->module.vpad) 
	    	+ sum_of_positions(lw->module.vertical, level);
	    lc->module.y = MAX(pos, lc->module.y);
	} else {
	    if (lc->module.level == 1) 		/* TOP */
		top = lc->module.y + w->core.height/2;
	    /* 
	     * Add up the sum of the height's of all nodes to this depth,
	     * and use it as the y position.
	     */
	    pos = (level * lw->module.hpad) 
	    	+ sum_of_positions(lw->module.horizontal, level);
	    lc->module.x = MAX(pos, lc->module.x);
	}

	/* 
	 * move the widget into position.
	 */
	XtMoveWidget(w, lc->module.x, lc->module.y);

	/*
	 * if the widget position plus its width or height doesn't fit in
	 * the module, ask if the module can be resized.
	 */
	if (lw->core.width < lc->module.x+w->core.width+lw->module.hpad ||
	    lw->core.height < lc->module.y+w->core.height+lw->module.vpad) {
	    result = XtMakeResizeRequest(lw, 
		MAX(lw->core.width, 
		    lc->module.x + w->core.width + lw->module.hpad),
		MAX(lw->core.height,
		    lc->module.y + w->core.height + lw->module.hpad),
		&rw, &rh);
	    /*
	     * accept any comprmise.
	     */
	    if (result == XtGeometryAlmost)
		XtMakeResizeRequest(lw, rw, rh, NULL, NULL);

	    lw->module.width = rw;
	    lw->module.width = rh;
	}
	/* 
	 * set the positions of all children.
	 */
	for (i = 0; i < lc->module.n_children; i++) {
	    set_positions(lw, lc->module.children[i], level+1);
	}
    }
}

static void mirror_positions(lw)
QxtModuleWidget lw;
{
    register int i;

    for (i = 0; i < lw->composite.num_children; i++) {
	Widget w = lw->composite.children[i];
	ModuleConstraints lc = MODULE_CONSTRAINT(w);

	if (w != lw->module.module_root && XtIsManaged(w) &&
	    lc->module.level > 0) {
	    switch (lw->module.gravity) {
	      case EastGravity:
		lc->module.x = (((Position) lw->module.width) -
			      ((Position) w->core.width) - lc->module.x);
		break;

	      case SouthGravity:
		lc->module.y = (((Position) lw->module.height) -
			      ((Position) w->core.height) - lc->module.y);
		break;
	    }
	    /*
	     * Move the widget into position.
	     */
	    XtMoveWidget (w, lc->module.x, lc->module.y);
	}
    }
}

static ModuleOffsetPtr create_offset(size)
int size;
{
    ModuleOffsetPtr offset=(ModuleOffsetPtr)XtMalloc(sizeof(ModuleOffset));
    offset->size = size;
    offset->array = (Dimension *)XtMalloc(size * sizeof(Dimension));
    return(offset);
}

static void
reset_positions(lw)
QxtModuleWidget lw;
{
    register int i;

    for (i = 0; i < lw->composite.num_children; i++) {
	Widget w = lw->composite.children[i];
	ModuleConstraints lc = MODULE_CONSTRAINT(w);

	lc->module.x = lc->module.y = -1;
	lc->module.level = 0;
    }
}

static void reset_offset(offset)
ModuleOffsetPtr offset;
{
    register int i;
    for (i = 0; i < offset->size; i++) {
	offset->array[i] = 0;
    }
}

static Position current_position(offset, position)
ModuleOffsetPtr offset;
int position;
{
    if (position >= offset->size)
	return(0);
    return(offset->array[position]);
}

static void set_current_position(offset, index, value)
ModuleOffsetPtr offset;
int index;
Dimension value;
{
    if (index >= offset->size) {
	offset->size = index + index/2;
	offset->array = (Dimension *)XtRealloc(offset->array, 
					       offset->size*sizeof(Dimension));
    }
    offset->array[index] = value;
}

static Position sum_of_positions(offset, index)
ModuleOffsetPtr offset;
int index;
{
    register int i;
    Position sum = 0;
    int stop = index;

    if (index > offset->size)
	stop = offset->size;
    for (i = 0; i < stop; i++) {
	sum += offset->array[i];
    }
    return(sum);
}

IsModule(lw, w)
QxtModuleWidget lw;
Widget w;
{
    ModuleConstraints lc = MODULE_CONSTRAINT(w);
    Boolean isvert = IsVertical(lw);

    if (lc->module.n_children == 1) {
	ModuleConstraints clc = MODULE_CONSTRAINT(lc->module.children[0]);
	if (isvert) {
	    if (clc->module.n_parents > 1 && clc->module.x >= 0)
		return(1);
	} else {
	    if (clc->module.n_parents > 1 && clc->module.y >= 0)
		return(1);
	}
    }
    return(0);
}

static void
set_point(p, x, y)
XPoint *p;
short x, y;
{
    p->x = x;
    p->y = y;
}

static int
get_depth(w, level)
Widget w;
int level;
{
    int i, depth = 0;
    ModuleConstraints lc = MODULE_CONSTRAINT(w);

    for (i = 0; i < lc->module.n_children; i++) {
	depth = get_depth(lc->module.children[i],level+1);
    }
    return(MAX(level,depth));
}
