#ifdef sequent
#include <strings.h>
#else
#include <string.h>
#endif

#include <math.h>

#include "dotsrc.h"

/* for debugging memory allocation and freeing */

#include "hash.h"

#define HASH_TABLE_SIZE 255
HashTable* memory_usage;

/* not portable macro */
#ifdef sun
#define POINTER_STRING_SIZE 10
#else
#ifdef sequent
#define POINTER_STRING_SIZE 7
#else
#define POINTER_STRING_SIZE 8
#endif
#endif

/*
  type_name(TypeDescriptor td)
  return a type's type name string
*/

Public char* type_name(td)
     TypeDescriptor td;
{
    switch (td) {
      case PROGRAM: return "PROGRAM";
      case ENV_DEF: return "ENV_DEF";
      case DEF_LIB: return "DEF_LIB";
      case STRING: return "STRING";
      case EXP_DEF: return "EXP_DEF";
      case EXP: return "EXP";
      case EXP_NAME: return "EXP_NAME";
      case OBJ_DEF: return "OBJ_DEF";
      case OBJ_SUB: return "OBJ_SUB";
      case MOD_DEF: return "MOD_DEF";
      case M_SUB: return "M_SUB";
      case M_DESC: return "M_DESC";
      case M2_DESC: return "M2_DESC";
      case LINK_DEF: return "LINK_DEF";
      case LINK: return "LINK";
      case M_ID_PAIR: return "M_ID_PAIR";
      case O_TERM_PAIR: return "O_TERM_PAIR";
      case RULE_DEF: return "RULE_DEF";
      case RULE: return "RULE";
      case RULE_ID: return "RULE_ID";
      case NORMAL: return "NORMAL";
      case REL: return "REL";
      case UPDATE: return "UPDATE";
      case TRANSACTION: return "TRANSACTION";
      case PROP: return "PROP";
      case I_CHECK: return "I_CHECK";
      case CLUSTER: return "CLUSTER";
      case A_TERM: return "A_TERM";
      case O_TERM: return "O_TERM";
      case LABEL: return "LABEL";
      case C_O_TERM: return "C_O_TERM";
      case VAR: return "VAR";
      case DOT: return "DOT";
      case SUB_LIST: return "SUB_LIST";
      case LIST: return "LIST";
      case REMAIN: return "REMAIN";
      case INTEGER: return "INTEGER";
      case NON_STRUCT: return "NON_STRUCT";
      case ATTR: return "ATTR";
      case PROLOG: return "PROLOG";
      case SET: return "SET";
      case SORT: return "SORT";
      case VALUE: return "VALUE";
      case QUERY_CNSTR: return "QUERY_CNSTR";
      case CNSTR: return "CNSTR";
      case CNSTR_DATA: return "CNSTR_DATA";
      case QUERY: return "QUERY";
      case P_MODE: return "P_MODE";
      case A_MODE: return "A_MODE";
      case I_MODE: return "I_MODE";
      case M_MODE: return "M_MODE";
      case E_MODE: return "E_MODE";
      case Q_MODE: return "Q_MODE";
      case ANSWER: return "ANSWER";
      case ANSWER_ELEMENT: return "ANSWER_ELEMENT";
      case ANSE_EXPLANATION: return "ANSE_EXPLANATION";
      case RIR_PAIR: return "RIR_PAIR";
      case DOT_CNSTR: return "DOT_CNSTR";
      case VAR_CNSTR: return "VAR_CNSTR";
      case INHERIT: return "INHERIT";
      case REDUCE: return "REDUCE";
      case SUB_GOAL: return "SUB_GOAL";
      case VC_PAIR: return "VC_PAIR";
      case ONE_RULE: return "ONE_RULE";
      case FACT: return "FACT";
      case UNIT_EXPLANATION: return "UNIT_EXPLANATION";
      case MERGE_EXPLANATION: return "MERGE_EXPLANATION";
      case LOOKUP_EXPLANATION: return "LOOKUP_EXPLANATION";
      case EXPLANATION: return "EXPLANATION";
      case CONSTRAINT: return "CONSTRAINT";
      case CON: return "CON";
      case ASSUMP: return "ASSUMP";
      case QUERY_EXPLANATION: return "QUERY_EXPLANATION";
      case PSEUDO_OBJECT: return "PSEUDO_OBJECT";
      case OBJ_LIST: return "OBJ_LIST";
      case OBJ_ARRAY: return "OBJ_ARRAY";
      default:
	illegal_type_descriptor("type_name", td);
	break;
    }
}

/*
  is_type(td1, td2):
  check if td1 is one of sub-types of td2
*/

Public Bool is_type(td1, td2)
     TypeDescriptor td1, td2;
{
    if (td1 == td2)
      return TRUE;
    else
      switch (td2) {
	case PROGRAM:
	  return td1 == PROGRAM;
	case ENV_DEF:
	  return td1 == ENV_DEF;
	case DEF_LIB:
	  return td1 == DEF_LIB;
	case STRING:
	  return td1 == STRING;
	case EXP_DEF:
	  return td1 == EXP_DEF;
	case EXP:
	  return td1 == EXP;
	case EXP_NAME:
	  return td1 == EXP_NAME;
	case OBJ_DEF:
	  return td1 == OBJ_DEF;
	case OBJ_SUB:
	  return td1 == OBJ_SUB;
	case MOD_DEF:
	  return td1 == MOD_DEF;
	case M_SUB:
	  return td1 == M_SUB;
	case M_DESC:
	  return is_type(td1, M_ID) || is_type(td1, M2_DESC);
	case M2_DESC:
	  return td1 == M2_DESC;
	case LINK_DEF:
	  return td1 == LINK_DEF;
	case LINK:
	  return td1 == LINK;
	case M_ID_PAIR:
	  return td1 == M_ID_PAIR;
	case O_TERM_PAIR:
	  return td1 == O_TERM_PAIR;
	case RULE_DEF:
	  return td1 == RULE_DEF;
	case RULE:
	  return td1 == RULE;
	case RULE_ID:
	  return td1 == RULE_ID;
	case NORMAL:
	  return td1 == NORMAL;
	case REL:
	  return td1 == REL;
	case UPDATE:
	  return td1 == UPDATE;
	case TRANSACTION:
	  return td1 == TRANSACTION;
	case PROP:
	  return td1 == PROP;
	case I_CHECK:
	  return is_type(td1, PROP) || is_type(td1, OBJ_LIST);
	case CONSIS:
	  return is_type(td1, I_CHECK);
	case INCONSIS:
	  return is_type(td1, I_CHECK);
	case CLUSTER:
	  return is_type(td1, NORMAL) || is_type(td1, REL)
	         || is_type(td1, UPDATE) || is_type(td1, TRANSACTION)
		 || is_type(td1, CONSIS) || is_type(td1, INCONSIS);
	case A_TERM:
	  return td1 == A_TERM;
	case O_TERM:
	  return is_type(td1, PROLOG) || is_type(td1, C_O_TERM)
	         || is_type(td1, VAR) || is_type(td1, DOT)
		 || is_type(td1, LIST) || is_type(td1, NON_STRUCT);
	case LABEL:
	  return is_type(td1, IND_LAB) || is_type(td1, SET_LAB)
	         || is_type(td1, CURL_IND) || is_type(td1, CURL_SET);
	case IND_LAB:
	  return is_type(td1, LABEL);
	case SET_LAB:
	  return is_type(td1, LABEL);
	case CURL_IND:
	  return is_type(td1, LABEL);
	case CURL_SET:
	  return is_type(td1, LABEL);
	case C_O_TERM:
	  return td1 == C_O_TERM;
	case VAR:
	  return td1 == VAR;
	case DOT:
	  return td1 == DOT;
	case SUB_LIST:
	  return td1 == SUB_LIST;
	case LIST:
	  return is_type(td1, OBJ_LIST) || is_type(SUB_LIST);
	case REMAIN:
	  return is_type(td1, LIST) || is_type(td1, VAR);
	case INTEGER:
	  return td1 == INTEGER;
	case NON_STRUCT:
	  return is_type(td1, EXP_NAME) || is_type(td1, STRING)
	         || is_type(td1, INTEGER);
	case ATTR:
	  return td1 == ATTR;
	case PROLOG:
	  return td1 == PROLOG;
	case SET:
	  return td1 == SET;
	case SORT:
	  return td1 == SORT;
	case VALUE:
	  return is_type(td1, SET) || is_type(td1, SORT)
	         || is_type(td1, O_TERM);
	case QUERY_CNSTR:
	  return td1 == QUERY_CNSTR;
	case CNSTR:
	  return td1 == CNSTR || is_type(td1, QUERY_CNSTR) 
	         || is_type(td1, CNSTR_DATA);
	case CNSTR_DATA: 
	  return td1 == CNSTR_DATA;
	  /* temporary */
	case QUERY:
	  return td1 == QUERY;
	case P_MODE:
	  return td1 == P_MODE;
	case A_MODE:
	  return td1 == A_MODE;
	case I_MODE:
	  return td1 == I_MODE;
	case M_MODE:
	  return td1 == M_MODE;
	case E_MODE:
	  return td1 == E_MODE;
	case Q_MODE:
	  return is_type(td1, P_MODE) || is_type(td1, A_MODE)
	         || is_type(td1, I_MODE) || is_type(td1, M_MODE)
		 || is_type(td1, E_MODE);
	case ANSWER:
	  return td1 == ANSWER;
	case ANSWER_ELEMENT:
	  return td1 == ANSWER_ELEMENT;
	case ANSE_EXPLANATION: 
	  return td1 == ANSE_EXPLANATION;
	case RIR_PAIR: 
	  return td1 == RIR_PAIR;
	case DOT_CNSTR:
	  return td1 == DOT_CNSTR;
	case VAR_CNSTR:
	  return td1 == VAR_CNSTR;
	case INHERIT:
	  return td1 == INHERIT;
	case REDUCE:
	  return td1 == REDUCE;
	case SUB_GOAL:
	  return td1 == SUB_GOAL;
	case VC_PAIR:
	  return td1 == VC_PAIR;
	case ONE_RULE:
	  return is_type(td1, REDUCE) || is_type(td1, FACT)
	         || is_type(td1, QUERY_EXPLANATION);
	case FACT:
	  return td1 == FACT;
	case UNIT_EXPLANATION:
	  return is_type(td1, ONE_RULE) || is_type(td1, INHERIT);
	case MERGE_EXPLANATION:
	  return td1 == MERGE_EXPLANATION;
	case LOOKUP_EXPLANATION:
	  return td1 == LOOKUP_EXPLANATION;
	case EXPLANATION:
	  return is_type(td1, UNIT_EXPLANATION)
	         || is_type(td1, MERGE_EXPLANATION)
		 || is_type(td1, LOOKUP_EXPLANATION);
	case CONSTRAINT:
	  return td1 == CONSTRAINT;
	case CON:
	  return td1 == CON;
	case ASSUMP:
	  return td1 == ASSUMP;
	case QUERY_EXPLANATION: 
	  return td1 == QUERY_EXPLANATION;
	case PSEUDO_OBJECT:
	  return TRUE;
	  /* for speed */
	case OBJ_LIST:
	  return td1 == OBJ_LIST;
	case OBJ_ARRAY:
	  return td1 == OBJ_ARRAY;
	default:
	  illegal_type_descriptor("is_type", td2);
	  break;
      }
}

Local void dotsrc_fprintf(fp, string)
     FILE* fp;
     char* string;
{
     if(string != NULL)
       fprintf(fp, string);
}

/*
  print_pseudo_object:
  prints contents of pseudo_object 'object' to file 'fp' indenting
'indent' length.
  CAUTION: current version doesn't check cyclic data structure.
*/

#define MEMBER_INDENT 2

Local void put_indent(indent, fp)
     int indent;
     FILE* fp;
{
    Local /* const */ char *longBlank = "                                                                                                                                                                                                        ";
    Local int len = 0;

    if (len == 0)
      len = strlen(longBlank);
    fprintf(fp, "\n%s", longBlank+len-indent);
}

Local int print_member_name(member_name, fp, indent)
     char* member_name;
     FILE* fp;
     int indent;
{
    put_indent(indent + MEMBER_INDENT, fp);
    fprintf(fp, member_name);
    return indent + MEMBER_INDENT + strlen(member_name);
}

Local void union_tag(type_name, fp)
     char* type_name;
     FILE* fp;
{
    fprintf(fp, "?union tag %s is used for entity?", type_name);
}

Local void print_type_name(td, fp)
     TypeDescriptor td;
     FILE* fp;
{
    fprintf(fp, type_name(td));
}

Public void print_program (program, fp, indent)
     Program* program;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: PROGRAM");
    new_indent = print_member_name("env_def: ", fp, indent);
    print_pseudo_object(program->env_def, fp, new_indent);
    new_indent = print_member_name("exp_def: ", fp, indent);
    print_pseudo_object(program->exp_def, fp, new_indent);
    new_indent = print_member_name("obj_def: ", fp, indent);
    print_pseudo_object(program->obj_def, fp, new_indent);
    new_indent = print_member_name("mod_def: ", fp, indent);
    print_pseudo_object(program->mod_def, fp, new_indent);
    new_indent = print_member_name("link_def:", fp, indent);
    print_pseudo_object(program->link_def, fp, new_indent);
    new_indent = print_member_name("rule_def:", fp, indent);
    print_pseudo_object(program->rule_def, fp, new_indent);
}

Public void print_env_def(env_def, fp, indent)
     EnvDef* env_def;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: ENV_DEF");
    print_member_name("name:  ", fp, indent);
    dotsrc_fprintf(fp, env_def->name);
    print_member_name("author:", fp, indent);
    dotsrc_fprintf(fp, env_def->author);
    print_member_name("date:  ", fp, indent);
    dotsrc_fprintf(fp, env_def->date);
    new_indent = print_member_name("def_libs:", fp, indent);
    print_pseudo_object(env_def->def_libs, fp, new_indent);
}

Local void print_lib_lab(lib_lab, fp)
     LibLab lib_lab;
     FILE* fp;
{
    switch (lib_lab) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case EXPLIB:
	fprintf(fp, "&exp_lib");
	break;
      case PGMLIB:
	fprintf(fp, "&pgm_lib");
	break;
      case SORTLIB:
	fprintf(fp, "&sort_lib");
	break;
      default:
	fprintf(fp, "?unknown lib_lab value %d?", lib_lab);
	break;
    }
}

Public void print_def_lib(def_lib, fp, indent)
     DefLib* def_lib;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: DEF_LIB");
    print_member_name("library type:", fp, indent);
    print_lib_lab(def_lib->lib_lab, fp);
    new_indent = print_member_name("lib_names:", fp, indent);
    print_pseudo_object(def_lib->lib_names, fp, new_indent);
}

Public void print_string(string, fp, indent)
     String* string;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: STRING");
    print_member_name("string data: ", fp, indent);
    dotsrc_fprintf(fp, string->str_data);
}

Public void print_exp_def(exp_def, fp, indent)
     ExpDef* exp_def;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: EXP_DEF");
    new_indent = print_member_name("exps: ", fp, indent);
    print_pseudo_object(exp_def->exps, fp, new_indent);
}

Public void print_exp(exp, fp, indent)
     Exp* exp;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: EXP");
    new_indent = print_member_name("exp_name:", fp, indent);
    print_pseudo_object(exp->exp_name, fp, new_indent);
    new_indent = print_member_name("o_term:  ", fp, indent);
    print_pseudo_object(exp->o_term, fp, new_indent);
}

Public void print_obj_def(obj_def, fp, indent)
     ObjDef* obj_def;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: OBJ_DEF");
    new_indent = print_member_name("obj_subs: ", fp, indent);
    print_pseudo_object(obj_def->obj_subs, fp, new_indent);
}

Public void print_obj_sub(obj_sub, fp, indent)
     ObjSub* obj_sub;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: OBJ_SUB");
    print_member_name("bobj1: ", fp, indent);
    dotsrc_fprintf(fp, obj_sub->bobj1);
    print_member_name("bobj2: ", fp, indent);
    dotsrc_fprintf(fp, obj_sub->bobj2);
}

Public void print_mod_def(mod_def, fp, indent)
     ModDef* mod_def;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: MOD_DEF");
    new_indent = print_member_name("m_subs: ", fp, indent);
    print_pseudo_object(mod_def->m_subs, fp, new_indent);
}

Public void print_m_sub(m_sub, fp, indent)
     MSub* m_sub;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: M_SUB");
    new_indent = print_member_name("m_id:  ", fp, indent);
    print_pseudo_object(m_sub->m_id, fp, indent);
    new_indent = print_member_name("m_desc:", fp, indent);
    print_pseudo_object(m_sub->m_desc, fp, indent);
}

Public void print_m_desc(m_desc, fp, indent)
     MDesc* m_desc;
     FILE* fp;
     int indent;
{
    if (m_desc->tag == M_DESC)
      union_tag(fp, "O_TERM");
    else
      print_pseudo_object(m_desc, fp, indent);
}

Public void print_m2_desc(m2_desc, fp, indent)
     M2Desc* m2_desc;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: M2_DESC");
    new_indent = print_member_name("m_desc1:", fp, indent);
    print_pseudo_object(m2_desc->m_desc1, fp, indent);
    print_member_name("operator:", fp, indent);
    putc(m2_desc->op, fp);
    new_indent = print_member_name("m_desc2:", fp, indent);
    print_pseudo_object(m2_desc->m_desc2, fp, new_indent);
}

Public void print_link_def(link_def, fp, indent)
     LinkDef* link_def;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: LINK_DEF");
    new_indent = print_member_name("links:", fp, indent);
    print_pseudo_object(link_def->links, fp, new_indent);
}

Public void print_link(link, fp, indent)
     Link* link;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: LINK");
    print_member_name("link name:", fp, indent);
    dotsrc_fprintf(fp, link->link_name);
    new_indent = print_member_name("m_ids:  ", fp, indent);
    print_pseudo_object(link->m_ids, fp, new_indent);
    new_indent = print_member_name("o_terms:", fp, indent);
    print_pseudo_object(link->o_terms, fp, new_indent);
}

Public void print_m_id_pair(m_id_pair, fp, indent)
     MIdPair* m_id_pair;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: M_ID_PAIR");
    new_indent = print_member_name("m_id1:", fp, indent);
    print_pseudo_object(m_id_pair->m_id1, fp, new_indent);
    new_indent = print_member_name("m_id2:", fp, indent);
    print_pseudo_object(m_id_pair->m_id2, fp, new_indent);
}

Public void print_o_term_pair(o_term_pair, fp, indent)
     OTermPair* o_term_pair;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: O_TERM_PAIR");
    new_indent = print_member_name("o_term1:", fp, indent);
    print_pseudo_object(o_term_pair->o_term1, fp, new_indent);
    new_indent = print_member_name("o_term2:", fp, indent);
    print_pseudo_object(o_term_pair->o_term2, fp, new_indent);
}

Public void print_rule_def(rule_def, fp, indent)
     RuleDef* rule_def;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: RULE_DEF");
    new_indent = print_member_name("rules:", fp, indent);
    print_pseudo_object(rule_def->rules, fp, new_indent);
}

Local void print_rule_class(rule_class, fp)
     RuleClass rule_class;
     FILE* fp;
{
    switch (rule_class) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case RCNOUPDATE:
	fprintf(fp, "&noupdate");
	break;
      case RCUPDATE:
	fprintf(fp, "&update");
	break;
      default:
	fprintf(fp, "?unknown rule class %d?", rule_class);
	break;
    }
}

Local void print_inheritance_mode(inheritance_mode, fp)
     InheritanceMode inheritance_mode;
     FILE* fp;
{
    switch (inheritance_mode) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case IML:
	fprintf(fp, "&l");
	break;
      case IMO:
	fprintf(fp, "&o");
	break;
      case IMLO:
	fprintf(fp, "&lo");
	break;
      case IMOL:
	fprintf(fp, "&ol");
	break;
      default:
	fprintf(fp, "?unknown rule inheritance mode %d?", inheritance_mode);
	break;
    }
}

Local void print_no_assume(no_assume, fp)
     NoAssume no_assume;
     FILE* fp;
{
    switch (no_assume) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case NOASSUME:
	fprintf(fp, "yes");
	break;
      default:
	fprintf(fp, "?unknown no-assumption mode %d?", no_assume);
	break;
    }
}

Public void print_rule(rule, fp, indent)
     Rule* rule;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: RULE");
    print_member_name("rule class:", fp, indent);
    print_rule_class(rule->rule_class, fp);
    new_indent = print_member_name("modules:  ", fp, indent);
    print_pseudo_object(rule->m_ids, fp, new_indent);
    new_indent = print_member_name("rule id:", fp, indent);
    print_pseudo_object(rule->rule_id, fp, new_indent);
    new_indent = print_member_name("inheritance mode:   ", fp,
				   indent);
    print_inheritance_mode(rule->inheritance_mode, fp);
    print_member_name("no assumption:", fp, indent);
    print_no_assume(rule->no_assume, fp);
    new_indent = print_member_name("head:", fp, indent);
    print_pseudo_object(rule->a_term, fp, new_indent);
    new_indent = print_member_name("clusters:", fp, indent);
    print_pseudo_object(rule->clusters, fp, new_indent);
    new_indent = print_member_name("constraints:",fp, indent);
    print_pseudo_object(rule->cnstrs, fp, new_indent);
}

Public void print_rule_id(rule_id, fp, indent)
     RuleId* rule_id;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: RULE_ID");
    print_member_name("rule id:", fp, indent);
    dotsrc_fprintf(fp, rule_id->rule_id_string);
}

Public void print_normal(normal, fp, indent)
     Normal *normal;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: NORMAL");
    new_indent = print_member_name("module:", fp, indent);
    print_pseudo_object(normal->m_id, fp, new_indent);
    new_indent = print_member_name("a_term:", fp, indent);
    print_pseudo_object(normal->a_term, fp, new_indent);
}

Local void print_sub_rel(sub_rel, fp)
     CnRel sub_rel;
     FILE* fp;
{
    switch (sub_rel) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case CNLE:
	fprintf(fp, "=<");
	break;
      case CNGE:
	fprintf(fp, ">=");
	break;
      case CNEQ:
	fprintf(fp, "==");
	break;
      case CNPL:
	fprintf(fp, "+<");
	break;
      case CNPG:
	fprintf(fp, ">+");
	break;
      case CNEPE:
	fprintf(fp, "=+=");
	break;
      case CNAL:
	fprintf(fp, "*<");
	break;
      case CNAG:
	fprintf(fp, ">*");
	break;
      case CNEAE:
	fprintf(fp, "=*=");
	break;
      case CNIN:
	fprintf(fp, "&in");
	break;
      case CNNI:
	fprintf(fp, "&ni");
	break;
      default:
	fprintf(fp, "?unknown constraint relation %d?", sub_rel);
	break;
    }
}

Public void print_rel(rel, fp, indent)
     Rel* rel;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: REL");
    new_indent = print_member_name("o_term1:", fp, indent);
    print_pseudo_object(rel->o_term1, fp, new_indent);
    print_member_name("sub_rel:", fp, indent);
    print_sub_rel(rel->sub_rel, fp);
    print_member_name("o_term2:", fp, indent);
    print_pseudo_object(rel->o_term2, fp, new_indent);
}

Public void print_update(update, fp, indent)
     Update* update;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: UPDATE");
    print_member_name("update flag:", fp, indent);
    dotsrc_fprintf("%c", update->u_flag);
    new_indent = print_member_name("module:", fp, indent);
    print_pseudo_object(update->m_id, fp, new_indent);
    new_indent = print_member_name("a_term:", fp, indent);
    print_pseudo_object(update->a_term, fp, new_indent);
}

Local void print_transaction_controller(trn_data, fp)
     TrnData trn_data;
     FILE* fp;
{
    switch (trn_data) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case TDBT:
	fprintf(fp, "&bt");
	break;
      case TDET:
	fprintf(fp, "&et");
	break;
      case TDAT:
	fprintf(fp, "&at");
	break;
      defaut:
	fprintf(fp, "?unknwon transaction controller %d?", trn_data);
	break;
    }
}

Public void print_transaction(transaction, fp, indent)
     Transaction* transaction;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: TRANSACTION");
    print_member_name("transaction controller:", fp, indent);
    print_transaction_controller(transaction->trn_data, fp);
}

Public void print_prop(prop, fp, indent)
     Prop* prop;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: PROP");
    new_indent = print_member_name("module:", fp, indent);
    print_pseudo_object(prop->m_id, fp, new_indent);
    new_indent = print_member_name("a_term:", fp, indent);
    print_pseudo_object(prop->a_term, fp, new_indent);
}

Public void print_i_check(i_check , fp, indent)
     ICheck* i_check;
     FILE* fp;
     int indent;
{
    if (i_check->tag == I_CHECK)
      union_tag("I_CHECK", fp);
    else
      print_pseudo_object(i_check, fp, indent);
}

Public void print_cluster(cluster, fp, indent)
     Cluster* cluster;
     FILE* fp;
     int indent;
{
    if (cluster->tag == CLUSTER)
      union_tag("CLUSTER", fp);
    else
      print_pseudo_object(cluster, fp, indent);
}

Public void print_a_term(a_term, fp, indent)
     ATerm* a_term;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: A_TERM");
    new_indent = print_member_name("o_terms:", fp, indent);
    print_pseudo_object(a_term->o_term, fp, new_indent);
    new_indent = print_member_name("attrs:", fp, indent);
    print_pseudo_object(a_term->attrs, fp, new_indent);
    new_indent = print_member_name("cnstrs:", fp, indent);
    print_pseudo_object(a_term->cnstrs, fp, new_indent);
}

Public void print_o_term(o_term, fp, indent)
     OTerm* o_term;
     FILE* fp;
     int indent;
{
    if (o_term->tag == O_TERM)
      union_tag("O_TERM", fp);
    else
      print_pseudo_object(o_term, fp, indent);
}

Public void print_c_o_term(c_o_term, fp, indent)
     COTerm* c_o_term;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: C_O_TERM");
    print_member_name("head:", fp, indent);
    dotsrc_fprintf(fp, c_o_term->head);
    new_indent = print_member_name("attrs:", fp, indent);
    print_pseudo_object(c_o_term->attrs, fp, new_indent);
    new_indent = print_member_name("cnstrs:", fp, indent);
    print_pseudo_object(c_o_term->cnstrs, fp, new_indent);
}

Public void print_dot(dot, fp, indent)
     Dot* dot;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: DOT");
    new_indent = print_member_name("o_term:", fp, indent);
    print_pseudo_object(dot->o_term, fp, new_indent);
    new_indent = print_member_name("label: ", fp, indent);
    print_pseudo_object(dot->label, fp, new_indent);
}

Public void print_label(label, fp, indent)
     Label* label;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: LABEL");
    new_indent = print_member_name("o_term", fp, indent);
    print_pseudo_object(label->o_term, fp, new_indent);
}

Public void print_list(list, fp, indent)
     List* list;
     FILE* fp;
     int indent;
{
    if (list->tag == LIST)
      union_tag("LIST", fp);
    else
      print_pseudo_object(list, fp, indent);
}

Public void print_sub_list(sub_list, fp, indent)
     SubList* sub_list;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: SUB_LIST");
    new_indent = print_member_name("o_terms:", fp, indent);
    print_pseudo_object(sub_list->o_terms, fp, new_indent);
    new_indent = print_member_name("remain: ", fp, indent);
    print_pseudo_object(sub_list->remain, fp, new_indent);
}

Public void print_remain(remain, fp, indent)
     Remain* remain;
     FILE* fp;
     int indent;
{
    if (remain->tag == REMAIN)
      union_tag("REMAIN", fp);
    else
      print_pseudo_object(remain, fp, indent);
}

Public void print_non_struct(non_struct, fp, indent)
     NonStruct* non_struct;
     FILE* fp;
     int indent;
{
    if (non_struct->tag == NON_STRUCT)
      union_tag("NON_STRUCT", fp);
    else
      print_pseudo_object(non_struct, fp, indent);
}

Public void print_exp_name(exp_name, fp, indent)
     ExpName* exp_name;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: EXP_NAME");
    print_member_name("name:", fp, indent);
    dotsrc_fprintf(fp, exp_name->name);
}

Public void print_integer(integer, fp, indent)
     Integer* integer;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: INTEGER");
    print_member_name("value:", fp, indent);
    if(integer->value != NULL)
      fprintf(fp, "%d", integer->value);
}

Local void print_attr_op(attr_op, fp)
     AttrOp attr_op;
     FILE* fp;
{
    switch (attr_op) {
      case NULL:
	fprintf(fp, "&void");
      case ATRARROW:
	fprintf(fp, " ->");
	break;
      case ATLARROW:
	fprintf(fp, " <-");
	break;
      case ATEQ:
	fprintf(fp, " =");
	break;
      default:
	fprintf(fp, "?unknwon attribute operator %d?", attr_op);
	break;
    }
}

Public void print_attr(attr, fp, indent)
     Attr* attr;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: ATTR");
    new_indent = print_member_name("label:", fp, indent);
    print_pseudo_object(attr->label, fp, new_indent);
    print_member_name("attr op:", fp, indent);
    print_attr_op(attr->attr_op, fp);
    new_indent = print_member_name("value:", fp, indent);
    print_pseudo_object(attr->value, fp, new_indent);
}

Public void print_prolog(prolog, fp, indent)
     Prolog* prolog;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: PROLOG");
    print_member_name("head:", fp, indent);
    dotsrc_fprintf(fp, prolog->head);
    new_indent = print_member_name("o_terms:", fp, indent);
    print_pseudo_object(prolog->o_terms, fp, new_indent);
}

Public void print_value(value, fp, indent)
     Value* value;
     FILE* fp;
     int indent;
{
    if (value->tag == VALUE)
      union_tag("VALUE", fp);
    else
      print_pseudo_object(value, fp, indent);
}

Public void print_set(set, fp, indent)
     Set* set;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: SET");
    new_indent = print_member_name("oterms: ", fp, indent);
    print_pseudo_object(set->o_terms, fp, new_indent);
}

Public void print_cnstr(cnstr, fp, indent)
     Cnstr* cnstr;
     FILE* fp;
     int indent;
{
    if (cnstr->tag == CNSTR)
      union_tag("CNSTR", fp);
    else
      print_pseudo_object(cnstr, fp, indent);
}

Local void print_rel_op(rel, fp)
     CnRel rel;
     FILE* fp;
{
    switch (rel) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case CNLE:
	fprintf(fp, " =<");
	break;
      case CNGE:
	fprintf(fp, " >=");
	break;
      case CNEQ:
	fprintf(fp, " ==");
	break;
      case CNPL:
	fprintf(fp, " +<");
	break;
      case CNPG:
	fprintf(fp, " >+");
	break;
      case CNEPE:
	fprintf(fp, " =+=");
	break;
      case CNAL:
	fprintf(fp, "*<");
	break;
      case CNAG:
	fprintf(fp, ">*");
	break;
      case CNEAE:
	fprintf(fp, "=*=");
	break;
      case CNIN:
	fprintf(fp, " &in");
	break;
      case CNNI:
	fprintf(fp, " &ni");
	break;
      default:
	fprintf(fp, "?unknwon constraint relation %d?", rel);
	break;
    }
}

Public void print_cnstr_data(cnstr_data, fp, indent)
     CnstrData* cnstr_data;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: CNSTR_DATA");
    new_indent = print_member_name("module 1:", fp, indent);
    print_pseudo_object(cnstr_data->m_id1, fp, new_indent);
    new_indent = print_member_name("value 1: ", fp, indent);
    print_pseudo_object(cnstr_data->value1, fp, new_indent);
    new_indent = print_member_name("relation:", fp, indent);
    print_rel_op(cnstr_data->rel, fp);
    new_indent = print_member_name("module 2:", fp, indent);
    print_pseudo_object(cnstr_data->m_id2, fp, new_indent);
    new_indent = print_member_name("value 2: ", fp, indent);
    print_pseudo_object(cnstr_data->value2, fp, new_indent);
}

Public void print_query_cnstr(query_cnstr, fp, indent)
     QueryCnstr* query_cnstr;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: QUERY_CNSTR");
    new_indent = print_member_name("module 1:", fp, indent);
    print_pseudo_object(query_cnstr->m_id1, fp, new_indent);
    new_indent = print_member_name("module 2:", fp, indent);
    print_pseudo_object(query_cnstr->m_id2, fp, new_indent);
}

Local void print_var_type(type, fp)
     VarType type;
     FILE* fp;
{
    switch (type) {
      case NULL:
	fprintf(fp, "NULL");
	break;
      case VARIND:
	fprintf(fp, "individual var");
	break;
      case VARSET:
	fprintf(fp, "set var");
	break;
      default:
	fprintf(fp, "?unknwon variable type %d?", type);
	break;
    }
}

Public void print_var(var, fp, indent)
     Var* var;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: VAR");
    print_member_name("var-type: ", fp, indent);
    print_var_type(var->type, fp);
    print_member_name("var name: ", fp, indent);
    dotsrc_fprintf(fp, var->var_data);
}

Public void print_sort(sort, fp, indent)
     Sort* sort;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: SORT");
    new_indent = print_member_name("prolog: ", fp, indent);
    print_pseudo_object(sort->prolog, fp, new_indent);
}

Public void print_query(query, fp, indent)
     Query* query;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: QUERY");
    print_member_name("query_class: ", fp, indent);
    print_rule_class(query->query_class, fp);
    print_member_name("q_head: ", fp, indent);
    print_rule_class(query->q_head, fp);
    new_indent = print_member_name("clusters: ", fp, indent);
    print_pseudo_object(query->clusters, fp, new_indent);
    new_indent = print_member_name("cnstrs:   ", fp, indent);
    print_pseudo_object(query->cnstrs, fp, new_indent);
    new_indent = print_member_name("q_modes:  ", fp, indent);
    print_pseudo_object(query->q_modes, fp, new_indent);
    new_indent = print_member_name("program:  ", fp, indent);
    print_pseudo_object(query->program, fp, new_indent);
}

Public void print_q_mode(q_mode, fp, indent)
     QMode* q_mode;
     FILE* fp;
     int indent;
{
    if (q_mode->tag == Q_MODE)
      union_tag("Q_MODE", fp);
    else
      print_pseudo_object(q_mode, fp, indent);
}

Local print_p_mode_data(p_mode_data, fp)
     PModeData p_mode_data;
     FILE* fp;
{
    switch (p_mode_data) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case PSINGLE:
	fprintf(fp, "&single");
	break;
      case PMULTI:
	fprintf(fp, "&multi");
	break;
      default:
	fprintf(fp, "?unknown process mode value %d?", p_mode_data);
	break;
    }
}

Public void print_p_mode(p_mode, fp, indent)
     PMode* p_mode;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: P_MODE");
    print_member_name("p_mode_data: ", fp, indent);
    print_p_mode_data(p_mode->p_mode_data, fp);
}

Local void print_a_mode_data(a_mode_data, fp)
     AModeData a_mode_data;
     FILE* fp;
{
    switch (a_mode_data) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case ANORMAL:
	fprintf(fp, "&normal");
	break;
      case AMINIMAL:
	fprintf(fp, "&minimal");
	break;
      default:
	fprintf(fp, "?unknown answer mode value %d?", a_mode_data);
	break;
    }
}

Public void print_a_mode(a_mode, fp, indent)
     AMode* a_mode;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: A_MODE");
    print_member_name("a_mode_data: ", fp, indent);
    print_a_mode_data(a_mode->a_mode_data, fp);
}

Local void print_i_mode_data(i_mode_data, fp)
     IModeData i_mode_data;
     FILE* fp;
{
    switch (i_mode_data) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case IALL:
	fprintf(fp, "&all");
	break;
      case IDOWN:
	fprintf(fp, "&down");
	break;
      case IUP:
	fprintf(fp, "&up");
	break;
      case INO:
	fprintf(fp, "&no");
	break;
      default:
	fprintf(fp, "?unknown inheritance mode value %d?", i_mode_data);
	break;
    }
}

Public void print_i_mode(i_mode, fp, indent)
     IMode *i_mode;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: I_MODE");
    print_member_name("i_mode_data: ", fp, indent);
    print_i_mode_data(i_mode->i_mode_data, fp);
}

Local void print_m_mode_data(m_mode_data, fp)
     MModeData m_mode_data;
     FILE* fp;
{
    switch (m_mode_data) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case MYES:
	fprintf(fp, "&yes");
	break;
      case MNO:
	fprintf(fp, "&no");
	break;
      default:
	fprintf(fp, "?unknown merge mode value %d?", m_mode_data);
	break;
    }
}

Public void print_m_mode(m_mode, fp, indent)
     MMode* m_mode;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: M_MODE");
    print_member_name("m_mode_data: ", fp, indent);
    print_m_mode_data(m_mode->m_mode_data, fp);
}

Local void print_e_mode_data(e_mode_data, fp)
     EModeData e_mode_data;
     FILE* fp;
{
    switch (e_mode_data) {
      case NULL:
	fprintf(fp, "&void");
	break;
      case EON:
	fprintf(fp, "&on");
	break;
      case EOFF:
	fprintf(fp, "&off");
	break;
      default:
	fprintf(fp, "?unknown explanation mode value %d?", e_mode_data);
	break;
    }
}

Public void print_e_mode(e_mode, fp, indent)
     EMode* e_mode;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: E_MODE");
    print_member_name("e_mode_data: ", fp, indent);
    print_e_mode_data(e_mode->e_mode_data, fp);
}

Public void print_answer(answer, fp, indent)
     Answer* answer;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: ANSWER");
    new_indent = print_member_name("answer_elements: ", fp, indent);
    print_pseudo_object(answer->answer_elements, fp, new_indent);
}

Public void print_answer_element(answer_element, fp, indent)
     AnswerElement* answer_element;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: ANSWER_ELEMENT");
    new_indent = print_member_name("dot_cnstrs: ", fp, indent);
    print_pseudo_object(answer_element->dot_cnstrs, fp, new_indent);
    new_indent = print_member_name("var_cnstrs: ", fp, indent);
    print_pseudo_object(answer_element->var_cnstrs, fp, new_indent);
    new_indent = print_member_name("anse_explanation:", fp, indent);
    print_pseudo_object(answer_element->anse_explanation, fp, new_indent);
}

Public void print_anse_explanation(anse_explanation, fp, indent)
     AnseExplanation* anse_explanation;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: ANSE_EXPLANATION");
    new_indent = print_member_name("explanation:", fp, indent);
    print_pseudo_object(anse_explanation->explanation, fp, new_indent);
    new_indent = print_member_name("rir_pairs:     ", fp, indent);
    print_pseudo_object(anse_explanation->rir_pairs, fp, new_indent);
}

Public void print_rir_pair(rir_pair, fp, indent)
     RirPair* rir_pair;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: RIR_PAIR");
    new_indent = print_member_name("rule_id:", fp, indent);
    print_pseudo_object(rir_pair->rule_id, fp, new_indent);
    new_indent = print_member_name("rule:     ", fp, indent);
    print_pseudo_object(rir_pair->rule, fp, new_indent);
}

Public void print_dot_cnstr(dot_cnstr, fp, indent)
     DotCnstr* dot_cnstr;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: DOT_CNSTR");
    new_indent = print_member_name("m_id:  ", fp, indent);
    print_pseudo_object(dot_cnstr->m_id, fp, new_indent);
    new_indent = print_member_name("dot:   ", fp, indent);
    print_pseudo_object(dot_cnstr->dot, fp, new_indent);
    new_indent = print_member_name("rel:   ", fp, indent);
    print_pseudo_object(dot_cnstr->rel, fp, new_indent);
    new_indent = print_member_name("value:", fp, indent);
    print_pseudo_object(dot_cnstr->value, fp, new_indent);
}

Public void print_var_cnstr(var_cnstr, fp, indent)
     VarCnstr* var_cnstr;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: VAR_CNSTR");
    new_indent = print_member_name("var:   ", fp, indent);
    print_pseudo_object(var_cnstr->var, fp, new_indent);
    new_indent = print_member_name("rel:   ", fp, indent);
    print_pseudo_object(var_cnstr->rel, fp, new_indent);
    new_indent = print_member_name("value:", fp, indent);
    print_pseudo_object(var_cnstr->value, fp, new_indent);
}

Public void print_explanation(explanation, fp, indent)
     Explanation* explanation;
     FILE* fp;
     int indent;
{
    if (explanation->tag == EXPLANATION)
      union_tag("EXPLANATION", fp);
    else
      print_pseudo_object(explanation, fp, indent);
}

Public void print_unit_explanation(unit_explanation, fp, indent)
     UnitExplanation* unit_explanation;
     FILE* fp;
     int indent;
{
    if (unit_explanation->tag == UNIT_EXPLANATION)
      union_tag("UNIT_EXPLANATION", fp);
    else
      print_pseudo_object(unit_explanation, fp, indent);
}

Public void print_inherit(inherit, fp, indent)
     Inherit* inherit;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: INHERIT");
    new_indent = print_member_name("one_rule:", fp, indent);
    print_pseudo_object(inherit->one_rule, fp, new_indent);
    new_indent = print_member_name("ups:     ", fp, indent);
    print_pseudo_object(inherit->ups, fp, new_indent);
    new_indent = print_member_name("downs:   ", fp, indent);
    print_pseudo_object(inherit->downs, fp, new_indent);
}

Public void print_merge_explanation(merge_explanation, fp, indent)
     MergeExplanation* merge_explanation;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: MERGE_EXPLANATION");
    new_indent = print_member_name("unit explanations: ", fp, indent);
    print_pseudo_object(merge_explanation->unit_explanations, fp,
			new_indent);
}

Public void print_lookup_explanation(lookup_explanation, fp, indent)
     LookupExplanation* lookup_explanation;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: LOOKUP_EXPLANATION");
    new_indent = print_member_name("subgoal:        ", fp, indent);
    print_pseudo_object(lookup_explanation->sub_goal, fp, new_indent);
    new_indent = print_member_name("looked subgoal: ", fp, indent);
    print_pseudo_object(lookup_explanation->looked_s, fp, new_indent);
    new_indent = print_member_name("looking subgoal:", fp, indent);
    print_pseudo_object(lookup_explanation->looking_s, fp,
			new_indent);
    new_indent = print_member_name("explanation    :", fp, indent);
    print_pseudo_object(lookup_explanation->explanation, fp, new_indent);
}

Public void print_one_rule(one_rule, fp, indent)
     OneRule* one_rule;
     FILE* fp;
     int indent;
{
    if (one_rule->tag == ONE_RULE)
      union_tag("ONE_RULE", fp);
    else
      print_pseudo_object(one_rule, fp, indent);
}

Public void print_fact(fact, fp, indent)
     Fact* fact;
     FILE* fp;
     int indent;
{
    fprintf(fp, "type: FACT");
    print_member_name("fact:", fp, indent);
    dotsrc_fprintf(fp, fact->fact_data);
}


Public void print_reduce(reduce, fp, indent)
     Reduce* reduce;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: REDUCE");
    new_indent = print_member_name("subgoal:     ", fp, indent);
    print_pseudo_object(reduce->sub_goal, fp, new_indent);
    new_indent = print_member_name("rule ID:     ", fp, indent);
    print_pseudo_object(reduce->rule_id, fp, new_indent);
    new_indent = print_member_name("explanations:", fp, indent);
    print_pseudo_object(reduce->explanations, fp, new_indent);
    new_indent = print_member_name("assumptions:  ", fp, indent);
    print_pseudo_object(reduce->assumps, fp, new_indent);
}

Public void print_sub_goal(subgoal, fp, indent)
     SubGoal* subgoal;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: SUB_GOAL");
    new_indent = print_member_name("m_id:    ", fp, indent);
    print_pseudo_object(subgoal->m_id, fp, new_indent);
    new_indent = print_member_name("o_term:  ", fp, indent);
    print_pseudo_object(subgoal->o_term, fp, new_indent);
    new_indent = print_member_name("vc_pairs:", fp, indent);
    print_pseudo_object(subgoal->vc_pairs, fp, new_indent);
}

Public void print_vc_pair(vc_pair, fp, indent)
     VcPair* vc_pair;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: VC_PAIR");
    new_indent = print_member_name("var:       ", fp, indent);
    print_pseudo_object(vc_pair->var, fp, new_indent);
    new_indent = print_member_name("constraint:", fp, indent);
    print_pseudo_object(vc_pair->constraint, fp, new_indent);
}

Public void print_constraint(constraint, fp, indent)
     Constraint* constraint;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: ");
    new_indent = print_member_name("constraint data: ", fp, indent);
    print_pseudo_object(constraint->const_data, fp, new_indent);
}

Public void print_con(con, fp, indent)
     Con* con;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: CON");
    new_indent = print_member_name("o_terms1: ", fp, indent);
    print_pseudo_object(con->o_terms1, fp, new_indent);
    new_indent = print_member_name("o_terms2: ", fp, indent);
    print_pseudo_object(con->o_terms2, fp, new_indent);
}

Public void print_assump(assump, fp, indent)
     Assump* assump;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: ASSUMP");
    new_indent = print_member_name("m_id:    ", fp, indent);
    print_pseudo_object(assump->m_id, fp, new_indent);
    new_indent = print_member_name("dot:     ", fp, indent);
    print_pseudo_object(assump->dot, fp, new_indent);
    new_indent = print_member_name("vc_pairs:", fp, indent);
    print_pseudo_object(assump->vc_pairs, fp, new_indent);
}

Public void print_query_explanation(query_explanation, fp, indent)
     QueryExplanation* query_explanation;
     FILE* fp;
     int indent;
{
    int new_indent;

    fprintf(fp, "type: QUERY_EXPLANATION");
    new_indent = print_member_name("str_data:    ", fp, indent);
    dotsrc_fprintf(fp, query_explanation->str_data);
}

Public void print_obj_list(obj_list, fp, indent)
     ObjList* obj_list;
     FILE* fp;
     int indent;
{
    int new_indent;
    ObjElement* p = obj_list->first;

    fprintf(fp, "type: OBJ_LIST");
    print_member_name("element_tag: ", fp, indent);
    print_type_name(obj_list->element_tag, fp);
    while (p != NULL) {
	new_indent = print_member_name("element: ", fp, indent);
	print_pseudo_object(p->element, fp, new_indent);
	p = p->next;
    }
}

Public void print_obj_array(obj_array, fp, indent)
     ObjArray* obj_array;
     FILE* fp;
     int indent;
{
    int new_indent;
    int i;
    char *string;

    fprintf(fp, "type: OBJ_ARRAY");
    print_member_name("element_tag: ", fp, indent);
    print_type_name(obj_array->element_tag, fp, new_indent);
    print_member_name("size:        ", fp, indent);
    fprintf(fp, "%d", obj_array->size);
    string = dotsrc_malloc(strlen("elements") + 2 +
			   (int)log10(obj_array->size) + 1 + 1);
    for (i = 0; i < obj_array->size; ++i) {
	sprintf(string, "elements[%d]", i);
	new_indent = print_member_name(string, fp, indent);
	print_pseudo_object(obj_array->elements[i], fp, new_indent);
    }
}

Public void print_pseudo_object(object, fp, indent)
     PseudoObject* object;
     FILE* fp;
     int indent;
{
    if (object != NULL)
      switch (object->tag) {
      case PROGRAM:
	print_program((Program*)object, fp, indent);
	break;
      case ENV_DEF:
	print_env_def((EnvDef*)object, fp, indent);
	break;
      case DEF_LIB:
	print_def_lib((DefLib*)object, fp, indent);
	break;
      case STRING:
	print_string((String*)object, fp, indent);
	break;
      case EXP_DEF:
	print_exp_def((ExpDef*)object, fp, indent);
	break;
      case EXP:
	print_exp((Exp*)object, fp, indent);
	break;
      case EXP_NAME:
	print_exp_name((ExpName*)object, fp, indent);
	break;
      case OBJ_DEF:
	print_obj_def((ObjDef*)object, fp, indent);
	break;
      case OBJ_SUB:
	print_obj_sub((ObjSub*)object, fp, indent);
	break;
      case MOD_DEF:
	print_mod_def((ModDef*)object, fp, indent);
	break;
      case M_SUB:
	print_m_sub((MSub*)object, fp, indent);
	break;
      case M_DESC:
	print_m_desc((MDesc*)object, fp, indent);
	break;
      case M2_DESC:
	print_m2_desc((M2Desc*)object, fp, indent);
	break;
      case LINK_DEF:
	print_link_def((LinkDef*)object, fp, indent);
	break;
      case LINK:
	print_link((Link*)object, fp, indent);
	break;
      case M_ID_PAIR:
	print_m_id_pair((MIdPair*)object, fp, indent);
	break;
      case O_TERM_PAIR:
	print_o_term_pair((OTermPair*)object, fp, indent);
	break;
      case RULE_DEF:
	print_rule_def((RuleDef*)object, fp, indent);
	break;
      case RULE:
	print_rule((Rule*)object, fp, indent);
	break;
      case RULE_ID:
	print_rule_id((RuleId*)object, fp, indent);
	break;
      case NORMAL:
	print_normal((Normal*)object, fp, indent);
	break;
      case REL:
	print_rel((Rel*)object, fp, indent);
	break;
      case UPDATE:
	print_update((Update*)object, fp, indent);
	break;
      case TRANSACTION:
	print_transaction((Transaction*)object, fp, indent);
	break;
      case PROP:
	print_prop((Prop*)object, fp, indent);
	break;
      case I_CHECK:
	print_i_check((ICheck*)object, fp, indent);
	break;
      case CLUSTER:
	print_cluster((Cluster*)object, fp, indent);
	break;
      case A_TERM:
	print_a_term((ATerm*)object, fp, indent);
	break;
      case O_TERM:
	print_o_term((OTerm*)object, fp, indent);
	break;
      case LABEL:
	print_label((Label*)object, fp, indent);
	break;
      case C_O_TERM:
	print_c_o_term((COTerm*)object, fp, indent);
	break;
      case VAR:
	print_var((Var*)object, fp, indent);
	break;
      case DOT:
	print_dot((Dot*)object, fp, indent);
	break;
      case SUB_LIST:
	print_sub_list((SubList*)object, fp, indent);
	break;
      case LIST:
	print_list((List*)object, fp, indent);
	break;
      case REMAIN:
	print_remain((Remain*)object, fp, indent);
	break;
      case INTEGER:
	print_integer((Integer*)object, fp, indent);
	break;
      case NON_STRUCT:
	print_non_struct((NonStruct*)object, fp, indent);
	break;
      case ATTR:
	print_attr((Attr*)object, fp, indent);
	break;
      case PROLOG:
	print_prolog((Prolog*)object, fp, indent);
	break;
      case SET:
	print_set((Set*)object, fp, indent);
	break;
      case SORT:
	print_sort((Sort*)object, fp, indent);
	break;
      case VALUE:
	print_value((Value*)object, fp, indent);
	break;
      case QUERY_CNSTR:
	print_query_cnstr((QueryCnstr*)object, fp, indent);
	break;
      case CNSTR:
	print_cnstr((Cnstr*)object, fp, indent);
	break;
      case CNSTR_DATA: 
	print_cnstr_data((CnstrData*)object, fp, indent);
	break;
      case QUERY:
	print_query((Query*)object, fp, indent);
	break;
      case P_MODE:
	print_p_mode((PMode*)object, fp, indent);
	break;
      case A_MODE:
	print_a_mode((AMode*)object, fp, indent);
	break;
      case I_MODE:
	print_i_mode((IMode*)object, fp, indent);
	break;
      case M_MODE:
	print_m_mode((MMode*)object, fp, indent);
	break;
      case E_MODE:
	print_e_mode((EMode*)object, fp, indent);
	break;
      case Q_MODE:
	print_q_mode((QMode*)object, fp, indent);
	break;
      case ANSWER:
	print_answer((Answer*)object, fp, indent);
	break;
      case ANSWER_ELEMENT:
	print_answer_element((AnswerElement*)object, fp, indent);
	break;
      case ANSE_EXPLANATION: 
	print_anse_explanation((AnseExplanation*)object, fp, indent);
	break;
      case RIR_PAIR: 
	print_rir_pair((RirPair*)object, fp, indent);
	break;
      case DOT_CNSTR:
	print_dot_cnstr((DotCnstr*)object, fp, indent);
	break;
      case VAR_CNSTR:
	print_var_cnstr((VarCnstr*)object, fp, indent);
	break;
      case INHERIT:
	print_inherit((Inherit*)object, fp, indent);
	break;
      case REDUCE:
	print_reduce((Reduce*)object, fp, indent);
	break;
      case SUB_GOAL:
	print_sub_goal((SubGoal*)object, fp, indent);
	break;
      case VC_PAIR:
	print_vc_pair((VcPair*)object, fp, indent);
	break;
      case ONE_RULE:
	print_one_rule((OneRule*)object, fp, indent);
	break;
      case FACT:
	print_fact((Fact*)object, fp, indent);
	break;
      case UNIT_EXPLANATION:
	print_unit_explanation((UnitExplanation*)object, fp, indent);
	break;
      case MERGE_EXPLANATION:
	print_merge_explanation((MergeExplanation*)object, fp, indent);
	break;
      case LOOKUP_EXPLANATION:
	print_lookup_explanation((LookupExplanation*)object, fp, indent);
	break;
      case EXPLANATION:
	print_explanation((Explanation*)object, fp, indent);
	break;
      case CONSTRAINT:
	print_constraint((Constraint*)object, fp, indent);
	break;
      case CON:
	print_con((Con*)object, fp, indent);
	break;
      case ASSUMP:
	print_assump((Assump*)object, fp, indent);
	break;
      case QUERY_EXPLANATION: 
	print_query_explanation((QueryExplanation*)object, fp, indent);
	break;
      case PSEUDO_OBJECT:
	union_tag("PSEUDO_OBJECT", fp);
	break;
      case OBJ_LIST:
	print_obj_list((ObjList*)object, fp, indent);
	break;
      case OBJ_ARRAY:
	print_obj_array((ObjArray*)object, fp, indent);
	break;
      default:
	illegal_type_descriptor("print_pseudo_object", object->tag);
	break;
    }
}

/*
  delete_pseudo_object:
  eliminate an any type of pseudo object.
*/

Public void delete_pseudo_object(object)
     PseudoObject* object;
{
    switch (object->tag) {
      case PROGRAM:
	delete_program((Program*)object);
	break;
      case ENV_DEF:
	delete_env_def((EnvDef*)object);
	break;
      case DEF_LIB:
	delete_def_lib((DefLib*)object);
	break;
      case STRING:
	delete_string((String*)object);
	break;
      case EXP_DEF:
	delete_exp_def((ExpDef*)object);
	break;
      case EXP:
	delete_exp((Exp*)object);
	break;
      case EXP_NAME:
	delete_exp_name((ExpName*)object);
	break;
      case OBJ_DEF:
	delete_obj_def((ObjDef*)object);
	break;
      case OBJ_SUB:
	delete_obj_sub((ObjSub*)object);
	break;
      case MOD_DEF:
	delete_mod_def((ModDef*)object);
	break;
      case M_SUB:
	delete_m_sub((MSub*)object);
	break;
      case M_DESC:
	delete_m_desc((MDesc*)object);
	break;
      case M2_DESC:
	delete_m2_desc((M2Desc*)object);
	break;
      case LINK_DEF:
	delete_link_def((LinkDef*)object);
	break;
      case LINK:
	delete_link((Link*)object);
	break;
      case M_ID_PAIR:
	delete_m_id_pair((MIdPair*)object);
	break;
      case O_TERM_PAIR:
	delete_o_term_pair((OTermPair*)object);
	break;
      case RULE_DEF:
	delete_rule_def((RuleDef*)object);
	break;
      case RULE:
	delete_rule((Rule*)object);
	break;
      case RULE_ID:
	delete_rule_id((RuleId*)object);
	break;
      case NORMAL:
	delete_normal((Normal*)object);
	break;
      case REL:
	delete_rel((Rel*)object);
	break;
      case UPDATE:
	delete_update((Update*)object);
	break;
      case TRANSACTION:
	delete_transaction((Transaction*)object);
	break;
      case PROP:
	delete_prop((Prop*)object);
	break;
      case I_CHECK:
	delete_i_check((ICheck*)object);
	break;
      case CLUSTER:
	delete_cluster((Cluster*)object);
	break;
      case A_TERM:
	delete_a_term((ATerm*)object);
	break;
      case O_TERM:
	delete_o_term((OTerm*)object);
	break;
      case LABEL:
	delete_label((Label*)object);
	break;
      case C_O_TERM:
	delete_c_o_term((COTerm*)object);
	break;
      case VAR:
	delete_var((Var*)object);
	break;
      case DOT:
	delete_dot((Dot*)object);
	break;
      case SUB_LIST:
	delete_sub_list((SubList*)object);
	break;
      case LIST:
	delete_list((List*)object);
	break;
      case REMAIN:
	delete_remain((Remain*)object);
	break;
      case INTEGER:
	delete_integer((Integer*)object);
	break;
      case NON_STRUCT:
	delete_non_struct((NonStruct*)object);
	break;
      case ATTR:
	delete_attr((Attr*)object);
	break;
      case PROLOG:
	delete_prolog((Prolog*)object);
	break;
      case SET:
	delete_set((Set*)object);
	break;
      case SORT:
	delete_sort((Sort*)object);
	break;
      case VALUE:
	delete_value((Value*)object);
	break;
      case QUERY_CNSTR:
	delete_query_cnstr((QueryCnstr*)object);
	break;
      case CNSTR:
	delete_cnstr((Cnstr*)object);
	break;
      case CNSTR_DATA: 
	delete_cnstr_data((CnstrData*)object);
	break;
      case QUERY:
	delete_query((Query*)object);
	break;
      case P_MODE:
	delete_p_mode((PMode*)object);
	break;
      case A_MODE:
	delete_a_mode((AMode*)object);
	break;
      case I_MODE:
	delete_i_mode((IMode*)object);
	break;
      case M_MODE:
	delete_m_mode((MMode*)object);
	break;
      case E_MODE:
	delete_e_mode((EMode*)object);
	break;
      case Q_MODE:
	delete_q_mode((QMode*)object);
	break;
      case ANSWER:
	delete_answer((Answer*)object);
	break;
      case ANSWER_ELEMENT:
	delete_answer_element((AnswerElement*)object);
	break;
      case ANSE_EXPLANATION: 
	delete_anse_explanation((AnseExplanation*)object);
	break;
      case RIR_PAIR: 
	delete_rir_pair((RirPair*)object);
	break;
      case DOT_CNSTR:
	delete_dot_cnstr((DotCnstr*)object);
	break;
      case VAR_CNSTR:
	delete_var_cnstr((VarCnstr*)object);
	break;
      case INHERIT:
	delete_inherit((Inherit*)object);
	break;
      case REDUCE:
	delete_reduce((Reduce*)object);
	break;
      case SUB_GOAL:
	delete_sub_goal((SubGoal*)object);
	break;
      case VC_PAIR:
	delete_vc_pair((VcPair*)object);
	break;
      case ONE_RULE:
	delete_one_rule((OneRule*)object);
	break;
      case FACT:
	delete_fact((Fact*)object);
	break;
      case UNIT_EXPLANATION:
	delete_unit_explanation((UnitExplanation*)object);
	break;
      case MERGE_EXPLANATION:
	delete_merge_explanation((MergeExplanation*)object);
	break;
      case LOOKUP_EXPLANATION:
	delete_lookup_explanation((LookupExplanation*)object);
	break;
      case EXPLANATION:
	delete_explanation((Explanation*)object);
	break;
      case CONSTRAINT:
	delete_constraint((Constraint*)object);
	break;
      case CON:
	delete_con((Con*)object);
	break;
      case ASSUMP:
	delete_assump((Assump*)object);
	break;
      case QUERY_EXPLANATION: 
	delete_query_explanation((QueryExplanation*)object);
	break;
      case PSEUDO_OBJECT:
	union_tag("PSEUDO_OBJECT", stderr);
	exit(2);
	break;
      case OBJ_LIST:
	delete_obj_list((ObjList*)object);
	break;
      case OBJ_ARRAY:
	delete_obj_array((ObjArray*)object);
	break;
      default:
	illegal_type_descriptor("delete_pseudo_object", object->tag);
	break;
    }
}

/*
  copy_pseudo_object:
  eliminate an any type of pseudo object.
*/

Public PseudoObject* copy_pseudo_object(object)
     PseudoObject* object;
{
    switch (object->tag) {
      case PROGRAM:
	return (PseudoObject*)copy_program((Program*)object);
      case ENV_DEF:
	return (PseudoObject*)copy_env_def((EnvDef*)object);
      case DEF_LIB:
	return (PseudoObject*)copy_def_lib((DefLib*)object);
      case STRING:
	return (PseudoObject*)copy_string((String*)object);
      case EXP_DEF:
	return (PseudoObject*)copy_exp_def((ExpDef*)object);
      case EXP:
	return (PseudoObject*)copy_exp((Exp*)object);
      case EXP_NAME:
	return (PseudoObject*)copy_exp_name((ExpName*)object);
      case OBJ_DEF:
	return (PseudoObject*)copy_obj_def((ObjDef*)object);
      case OBJ_SUB:
	return (PseudoObject*)copy_obj_sub((ObjSub*)object);
      case MOD_DEF:
	return (PseudoObject*)copy_mod_def((ModDef*)object);
      case M_SUB:
	return (PseudoObject*)copy_m_sub((MSub*)object);
      case M_DESC:
	return (PseudoObject*)copy_m_desc((MDesc*)object);
      case M2_DESC:
	return (PseudoObject*)copy_m2_desc((M2Desc*)object);
      case LINK_DEF:
	return (PseudoObject*)copy_link_def((LinkDef*)object);
      case LINK:
	return (PseudoObject*)copy_link((Link*)object);
      case M_ID_PAIR:
	return (PseudoObject*)copy_m_id_pair((MIdPair*)object);
      case O_TERM_PAIR:
	return (PseudoObject*)copy_o_term_pair((OTermPair*)object);
      case RULE_DEF:
	return (PseudoObject*)copy_rule_def((RuleDef*)object);
      case RULE:
	return (PseudoObject*)copy_rule((Rule*)object);
      case RULE_ID:
	return (PseudoObject*)copy_rule_id((RuleId*)object);
      case NORMAL:
	return (PseudoObject*)copy_normal((Normal*)object);
      case REL:
	return (PseudoObject*)copy_rel((Rel*)object);
      case UPDATE:
	return (PseudoObject*)copy_update((Update*)object);
      case TRANSACTION:
	return (PseudoObject*)copy_transaction((Transaction*)object);
      case PROP:
	return (PseudoObject*)copy_prop((Prop*)object);
      case I_CHECK:
	return (PseudoObject*)copy_i_check((ICheck*)object);
      case CLUSTER:
	return (PseudoObject*)copy_cluster((Cluster*)object);
      case A_TERM:
	return (PseudoObject*)copy_a_term((ATerm*)object);
      case O_TERM:
	return (PseudoObject*)copy_o_term((OTerm*)object);
      case LABEL:
	return (PseudoObject*)copy_label((Label*)object);
      case C_O_TERM:
	return (PseudoObject*)copy_c_o_term((COTerm*)object);
      case VAR:
	return (PseudoObject*)copy_var((Var*)object);
      case DOT:
	return (PseudoObject*)copy_dot((Dot*)object);
      case SUB_LIST:
	return (PseudoObject*)copy_sub_list((SubList*)object);
      case LIST:
	return (PseudoObject*)copy_list((List*)object);
      case REMAIN:
	return (PseudoObject*)copy_remain((Remain*)object);
      case INTEGER:
	return (PseudoObject*)copy_integer((Integer*)object);
      case NON_STRUCT:
	return (PseudoObject*)copy_non_struct((NonStruct*)object);
      case ATTR:
	return (PseudoObject*)copy_attr((Attr*)object);
      case PROLOG:
	return (PseudoObject*)copy_prolog((Prolog*)object);
      case SET:
	return (PseudoObject*)copy_set((Set*)object);
      case SORT:
	return (PseudoObject*)copy_sort((Sort*)object);
      case VALUE:
	return (PseudoObject*)copy_value((Value*)object);
      case QUERY_CNSTR:
	return (PseudoObject*)copy_query_cnstr((QueryCnstr*)object);
      case CNSTR:
	return (PseudoObject*)copy_cnstr((Cnstr*)object);
      case CNSTR_DATA: 
	return (PseudoObject*)copy_cnstr_data((CnstrData*)object);
      case QUERY:
	return (PseudoObject*)copy_query((Query*)object);
      case P_MODE:
	return (PseudoObject*)copy_p_mode((PMode*)object);
      case A_MODE:
	return (PseudoObject*)copy_a_mode((AMode*)object);
      case I_MODE:
	return (PseudoObject*)copy_i_mode((IMode*)object);
      case M_MODE:
	return (PseudoObject*)copy_m_mode((MMode*)object);
      case E_MODE:
	return (PseudoObject*)copy_e_mode((EMode*)object);
      case Q_MODE:
	return (PseudoObject*)copy_q_mode((QMode*)object);
      case ANSWER:
	return (PseudoObject*)copy_answer((Answer*)object);
      case ANSWER_ELEMENT:
	return (PseudoObject*)copy_answer_element((AnswerElement*)object);
      case ANSE_EXPLANATION: 
	return (PseudoObject*)copy_anse_explanation((AnseExplanation*)object);
      case RIR_PAIR: 
	return (PseudoObject*)copy_rir_pair((RirPair*)object);
      case DOT_CNSTR:
	return (PseudoObject*)copy_dot_cnstr((DotCnstr*)object);
      case VAR_CNSTR:
	return (PseudoObject*)copy_var_cnstr((VarCnstr*)object);
      case INHERIT:
	return (PseudoObject*)copy_inherit((Inherit*)object);
      case REDUCE:
	return (PseudoObject*)copy_reduce((Reduce*)object);
      case SUB_GOAL:
	return (PseudoObject*)copy_sub_goal((SubGoal*)object);
      case VC_PAIR:
	return (PseudoObject*)copy_vc_pair((VcPair*)object);
      case ONE_RULE:
	return (PseudoObject*)copy_one_rule((OneRule*)object);
      case FACT:
	return (PseudoObject*)copy_fact((Fact*)object);
      case UNIT_EXPLANATION:
	return (PseudoObject*)
	  copy_unit_explanation((UnitExplanation*)object);
      case MERGE_EXPLANATION:
	return (PseudoObject*)
	  copy_merge_explanation((MergeExplanation*)object);
      case LOOKUP_EXPLANATION:
	return (PseudoObject*)
	  copy_lookup_explanation((LookupExplanation*)object);
      case EXPLANATION:
	return (PseudoObject*)copy_explanation((Explanation*)object);
      case CONSTRAINT:
	return (PseudoObject*)copy_constraint((Constraint*)object);
      case CON:
	return (PseudoObject*)copy_con((Con*)object);
      case ASSUMP:
	return (PseudoObject*)copy_assump((Assump*)object);
      case QUERY_EXPLANATION: 
	return (PseudoObject*)
	         copy_query_explanation((QueryExplanation*)object);
      case PSEUDO_OBJECT:
	union_tag("PSEUDO_OBJECT", stderr);
	exit(2);
	break;
      case OBJ_LIST:
	return (PseudoObject*)copy_obj_list((ObjList*)object);
      case OBJ_ARRAY:
	return (PseudoObject*)copy_obj_array((ObjArray*)object);
      default:
	illegal_type_descriptor("copy_pseudo_object", object->tag);
	break;
    }
}

/*
  type_error:
  output type error message to stderr and exit with code 2.
*/

Public void type_error(object, type_name, func_name)
     PseudoObject* object;
     char *type_name, *func_name;
{
    fprintf(stderr, "type error in function %s\n", func_name);
#ifdef TYPE_DEBUG
    fprintf(stderr, "  expected type: %s\n", type_name);
    fprintf(stderr, "  incoming data:\n");
    print_pseudo_object(object, stderr, 4);
#endif
    exit(2);
}

Public void type_conflict(td1, td2, func_name)
     TypeDescriptor td1, td2;
     char *func_name;
{
    fprintf(stderr, "type confliction %s with %s in %s\n",
	    type_name(td1), type_name(td2), func_name);
    exit(2);
}

/*
  write_through_NULL:
  output error message to stderr and exit with code 2.
*/

Public void write_through_NULL(func_name)
     char* func_name;
{
    fprintf(stderr, "write through NULL pointer in function %s\n",
	    func_name);
    exit(2);
}

/*
  illegal_type_descriptor:
  output error message to stderr and exit with code 2.
*/
Public void illegal_type_descriptor(func_name, td)
     char* func_name;
     TypeDescriptor td;
{
    fprintf(stderr, "illegal type descriptor is found in function %s\n",
	    func_name);
#ifdef TYPE_DEBUG
    fprintf(stderr, "descriptor value: %d", object->tag);
#endif
    exit(2);
}

/*
  array_type_check:
  checks all elements of a pseudo object array are of given type
*/

Public void array_type_check(td, n, obj_array)
     TypeDescriptor td;
     int n;
     PseudoObject** obj_array;
{
    int i;

    for (i = 0; i < n; ++i) {
	if (! is_type(obj_array[i]->tag, td))
	  type_error(obj_array[i], type_name(td), "array_type_check");
    }
}

/*
  memory allocation functions
*/

Public char* dotsrc_cmalloc(size)
     int size;
{
    char* p = malloc(size);

    if (p == NULL) {
	fprintf(stderr, "memory allocation failed.\n");
	exit(2);
    }
    else
      return p;
}

Public char* dotsrc_ccalloc(nelem, elsize)
     int nelem, elsize;
{
    char* p = calloc(nelem, elsize);

    if (p == NULL) {
	fprintf(stderr, "memory allocation failed.\n");
	exit(2);
    }
    else
      return p;
}

Public void dotsrc_cfree(ptr)
     char* ptr;
{
    free(ptr);
}

Public char* dotsrc_strdup(from)
     char* from;
{
    if (from) {
	char* p = dotsrc_malloc(strlen(from)+1);
	return strcpy(p, from);
    }
    else
      return NULL;
}

/* for debugging memory allocation and freeing */

Public void init_memory_usage()
{
    memory_usage = new_hash_table(HASH_TABLE_SIZE);
}

Local char* make_key(ptr)
     char* ptr;
{
    char* p = malloc(POINTER_STRING_SIZE + 1);
    sprintf(p, "%x", (int)ptr);
    return p;
}

Local int cmp_keys(key1, key2)
     char *key1, *key2;
{
    return strcmp(key1, key2) == 0;
}

freed_unused(ptr)
     char *ptr;
{
    printf("freed unused area at %x\n", (int)ptr);
}

Local void print_log(log)
     AllocLog *log;
{
    if (log->tag == Malloc)
      printf("at %x, size %d\n", (int)((MallocLog*)log)->ptr,
	     ((MallocLog*)log)->size);
    else
      printf("at %x, size %d x %d\n", (int)((CallocLog*)log)->ptr,
	     ((CallocLog*)log)->nelem, ((CallocLog*)log)->elsize);
}

Local void duplicate_allocation(dup, log)
     AllocLog *dup, *log;
{
    printf("duplicate allocation\n");
    printf("Old: ");
    print_log(dup);
    printf("New: ");
    print_log(log);
}

Public char* dotsrc_dmalloc(size)
     int size;
{
    char* p = malloc(size);

    if (p == NULL) {
	fprintf(stderr, "memory allocation failed.\n");
	exit(2);
    }
    else {
	char* str = make_key(p);
	MallocLog* log = (MallocLog*)malloc(sizeof(MallocLog));
	AllocLog* dup;

	log->tag = Malloc;
	log->ptr = p;
	log->size = size;
	if (((char*)dup = add_item(memory_usage, str, (char*)log,
				   cmp_keys)) != NULL)
	  duplicate_allocation(dup, log);
	return p;
    }
}

Public char* dotsrc_dcalloc(nelem, elsize)
     int nelem, elsize;
{
    char* p = calloc(nelem, elsize);

    if (p == NULL) {
	fprintf(stderr, "memory allocation failed.\n");
	exit(2);
    }
    else {
	char* str = malloc(POINTER_STRING_SIZE);
	CallocLog* log = (CallocLog*)malloc(sizeof(CallocLog));
	AllocLog* dup;

	sprintf(str, "%x", (int)p);
	log->tag = Calloc;
	log->ptr = p;
	log->nelem = nelem;
	log->elsize = elsize;
	if (((char*)dup = add_item(memory_usage, str, (char*)log,
				   cmp_keys)) != NULL)
	  duplicate_allocation(dup, log);
	return p;
    }
}

Public void dotsrc_dfree(ptr)
     char* ptr;
{
    char* key = make_key(ptr);
    char* data;

    if ((data = remove_item(memory_usage, key, cmp_keys)) == NULL)
      freed_unused(ptr);
    else {
	free(data);
	free(key);
	free(ptr);
    }
}

Local void check_memory_usage_in_entry_list(list)
     HashEntry* list;
{
    if (list != NULL) {
	AllocLog* log;

	printf("    ");
	(char*)log = find_item(memory_usage, list->key, cmp_keys);
	print_log(log);
	check_memory_usage_in_entry_list(list->next);
	remove_item(memory_usage, list->key, cmp_keys);
    }
}

Public void check_memory_usage(location)
	 char* location;
{
    int i;

    if (memory_usage->count == 1)
      printf("An allocated area is not freed in %s.\n", location);
    else if (memory_usage->count != 0)
      printf("%d allocated areas are not freed in %s.\n",
	     memory_usage->count, location);
    for (i = 0; i < HASH_TABLE_SIZE && memory_usage->count > 0; ++i)
      check_memory_usage_in_entry_list(memory_usage->elements[i].next);
}

