/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
#include <ctype.h>
#include <stdio.h>
#include <stream.h>
#include <GetOpt.h>
#include <String.h>
#include "eigenvalue.h"
#include "multivar.h"

multivar::multivar
(
 matrix* DataMatrix,
 char*   DFN,
 int     N,
 int*    FieldID,
 int     PFlag,
 int     OFlag
)
{
  DataFileName  = new char[strlen(DFN)+1];
  strcpy(DataFileName,DFN);

  NumElem   = N;
  NumData   = DataMatrix->sizeI();
  S         = matrix(NumElem,NumElem);
  MV        = matrix(NumElem,      1);
  MH        = matrix(      1,NumElem);
  Statistic = new statistic[NumElem];
  TotalVar  = 0.0;
  Principal = new principal[NumElem];
  for(int I=0;I<NumElem;++I)
    {
      Principal[I].axisInit(NumElem);
    }
  for(    I=0;I<NumElem;++I)
    {
      Statistic[I].id(FieldID[I]);
    }
  for(int K=0;K<NumData;++K)
    {
      for(I = 0;I<NumElem;++I)
	{
	  Statistic[I].val((*DataMatrix)(K,I));
	  MV(I,0) += (*DataMatrix)(K,I);
	  MH(0,I) += (*DataMatrix)(K,I);
	  for(int J =0;J<NumElem;++J)
	    {
	      S(I,J) += (*DataMatrix)(K,I)*(*DataMatrix)(K,J);
	    }
	}
    }
 end(PFlag,OFlag);
}

multivar::multivar(char* DFN,int N,int* FieldID)
{
  DataFileName  = new char[strlen(DFN)+1];
  strcpy(DataFileName,DFN);

  NumElem   = N;
  NumData   = 0;
  S         = matrix(NumElem,NumElem);
  MV        = matrix(NumElem,      1);
  MH        = matrix(      1,NumElem);
  Statistic = new statistic[NumElem];
  TotalVar  = 0.0;
  Principal = new principal[NumElem];
  for(int I=0;I<NumElem;++I)
    {
      Principal[I].axisInit(NumElem);
    }
  for(    I=0;I<NumElem;++I)
    {
      Statistic[I].id(FieldID[I]);
    }
}

void multivar::fprint(FILE* MVFile)
{
  fprintf(MVFile,"Number of Data    : %10d\n",NumData);
  fprintf(MVFile,"Number of Elements: %10d\n",NumElem);
  fprintf(MVFile,"Field Number      :");
  for(int I=0;I<NumElem;++I)
    {
      fprintf(MVFile," %10d",Statistic[I].id());
    }
  fprintf(MVFile,"\n");
  fprintf(MVFile,"Mean Value        :");
  for(I=0;I<NumElem;++I)
    {
      fprintf(MVFile," %10.5lf",Statistic[I].mean());
    }
  fprintf(MVFile,"\n");
  fprintf(MVFile,"Maximum Value     :");
  for(I=0;I<NumElem;++I)
    {
      fprintf(MVFile," %10.5lf",Statistic[I].max());
    }
  fprintf(MVFile,"\n");
  fprintf(MVFile,"Minimum Value     :");
  for(I=0;I<NumElem;++I)
    {
      fprintf(MVFile," %10.5lf",Statistic[I].min());
    }
  fprintf(MVFile,"\n");
  fprintf(MVFile,"Standard Deviation:");
  for(I=0;I<NumElem;++I)
    {
      fprintf(MVFile," %10.5lf",Statistic[I].stdD());
    }
  fprintf(MVFile,"\n");
  fprintf(MVFile,"Total Variance    : %10.5lf\n",TotalVar);
  fprintf(MVFile,"Axis Number       :");
  for(I=0;I<NumElem;++I)
    {
      fprintf(MVFile," %10d",I);
    }
  fprintf(MVFile,"\n");
  fprintf(MVFile,"Contribution Ratio:");
  for(I=0;I<NumElem;++I)
    {
      fprintf(MVFile," %10.5lf",Principal[I].contRatio());
    }
  fprintf(MVFile,"\n");
  fprintf(MVFile,"Axis Vectors      :\n");
  for(int P=0;P<NumElem;++P)
    {
      fprintf(MVFile,"Vector Element %3d:",P);
      for(int I=0;I<NumElem;++I)
	{
	  fprintf(MVFile," %10.5lf",Principal[I].axis(P));
	}
      fprintf(MVFile,"\n");
    }
}

void multivar::print(void)
{
  char MVFileName[256];
  strcpy(MVFileName,DataFileName);
  strcat(MVFileName,".MV");
  FILE* MVFile;
  if(NULL==(MVFile=fopen(MVFileName,"w")))
    {
      cerr << form("Can't open multivar file %s\n",MVFileName);
      exit(1);
    }

  fprintf(MVFile,"Data File Name    : %s\n",  DataFileName);
  fprint(MVFile);
  fclose(MVFile);
}


void multivar::get(char* FileName)
{

  FILE* StatFile;
  if(NULL==(StatFile=fopen(FileName,"r")))
    {
      cerr << form("Can't open Stat File %s\n",FileName);
      exit(1);
    }
  char Buff[1024];
  while(fgets(Buff,1023,StatFile))
    {
      if(NumElem!=0 && Statistic==NULL && Principal==NULL)
	{
	  Statistic = new statistic[NumElem];
	  Principal = new principal[NumElem];
	  for(int I=0;I<NumElem;++I)
	    {
	      Principal[I].axisInit(NumElem);
	    }
	}
      if(0==strncmp(Buff,"Data File Name    :",19))
	{
	  char FileNameBuff[256];
	  sscanf(Buff+19,"%s",FileNameBuff);
	  DataFileName = new char[strlen(FileNameBuff)+1];
	  strcpy(DataFileName,FileNameBuff);
	}
      if(0==strncmp(Buff,"Number of Data    :",19))
	{
	  sscanf(Buff+19,"%d",&NumData);
	}
      if(0==strncmp(Buff,"Number of Elements:",19))
	{
	  sscanf(Buff+19,"%d",&NumElem);
	}
      if(0==strncmp(Buff,"Field Number      :",19))
	{
	  String Field[32];
	  int NW = getField(Buff+19,Field);
	  for(int I=0;I<NumElem;++I)
	    {
	      Statistic[I].id(atoi(Field[I]));
	    }
	}
      if(0==strncmp(Buff,"Mean Value        :",19))
	{
	  String Field[32];
	  int NW = getField(Buff+19,Field);
	  for(int I=0;I<NumElem;++I)
	    {
	      Statistic[I].mean(atof(Field[I]));
	    }
	}
      if(0==strncmp(Buff,"Maximum Value     :",19))
	{
	  String Field[32];
	  int NW = getField(Buff+19,Field);
	  for(int I=0;I<NumElem;++I)
	    {
	      Statistic[I].max(atof(Field[I]));
	    }
	}
      if(0==strncmp(Buff,"Minimum Value     :",19))
	{
	  String Field[32];
	  int NW = getField(Buff+19,Field);
	  for(int I=0;I<NumElem;++I)
	    {
	      Statistic[I].max(atof(Field[I]));
	    }
	}
      if(0==strncmp(Buff,"Standard Deviation:",19))
	{
	  String Field[32];
	  int NW = getField(Buff+19,Field);
	  for(int I=0;I<NumElem;++I)
	    {
	      Statistic[I].stdD(atof(Field[I]));
	    }
	}
      if(0==strncmp(Buff,"Total Variance    :",19))
	{
	  sscanf(Buff+19,"%lf",&TotalVar);
	}
      if(0==strncmp(Buff,"Contribution Ratio:",19))
	{
	  String Field[32];
	  int NW = getField(Buff+19,Field);
	  for(int I=0;I<NumElem;++I)
	    {
	      Principal[I].contRatio(atof(Field[I]));
	    }
	}
      if(0==strncmp(Buff,"Vector Element",14))
	{
	  int P;
	  sscanf(Buff+14,"%d",&P);
	  String Field[32];
	  int NW = getField(Buff+19,Field);
	  for(int I=0;I<NumElem;++I)
	    {
	      Principal[I].axis(P,atof(Field[I]));
	    }
	}
    }
  fclose(StatFile);
}

void multivar::push(char* Buff)
{
  String  Field[32];
  int NW = getField(Buff,Field);
  double  FieldVal[NumElem];
  for(int I = 0;I<NumElem;++I)
    {
      if(NW>Statistic[I].id())
	{
	  FieldVal[I] = atof( Field[Statistic[I].id()]);
	}
    }
  for(I = 0;I<NumElem;++I)
    {
      Statistic[I].val(FieldVal[I]);
      MV(I,0) += FieldVal[I];
      MH(0,I) += FieldVal[I];
      for(int J =0;J<NumElem;++J)
	{
	  S(I,J) += FieldVal[I]*FieldVal[J];
	}
    }
  NumData++;
}

void multivar::end(int PFlag,int OFlag)
{
  MV /= (double)NumData;
  MH /= (double)NumData;
  S  /= (double)NumData;
  matrix SS = S - MV*MH;
  eigenvalue E;
  int I;
  if(PFlag)
    {
      E = SS.eigen();
    }
  if(OFlag)
    {
      for(I=0;I<NumElem;++I)
	{
	  TotalVar += SS(I,I);
	  Statistic[I].mean(0.0);
	  Statistic[I].stdD(sqrt(SS(I,I)));
	}
    }
  else
    {
      for(I=0;I<NumElem;++I)
	{
	  TotalVar += SS(I,I);
	  Statistic[I].mean(MH(0,I));
	  Statistic[I].stdD(sqrt(SS(I,I)));
	}
    }
  if(PFlag)
    {
      for(int I=0;I<NumElem;++I)
	{
	  Principal[I].contRatio(E.value(I)/TotalVar);
	  double Parity = (E(I,I)>0? 1.0: -1.0);
	  for(int J=0;J<NumElem;++J)
	    {
	      Principal[I].axis(J,Parity*E(I,J));
	    }
	}
    }
  else
    {
      for(int I=0;I<NumElem;++I)
	{
	  Principal[I].contRatio(SS(I,I)/TotalVar);
	  for(int J=0;J<NumElem;++J)
	    {
	      Principal[I].axis(J,(I==J? 1.0 :0.0));
	    }
	}
    }
}


multiData::multiData
(
 char* FileName,
 int   NA,
 int*  FieldID,
 int   PFlag,
 int   OFlag
)
{
  NumAxis         = NA;
  First = Last    = NULL;
  Histogram       = NULL;
  HistogramSmooth = NULL;
  Next            = NULL;

  FILE* DataFile;
  if(NULL==(DataFile=fopen(FileName,"r")))
    {
      cerr << form("Can't open Data File %s\n",FileName);
      exit(1);
    }
  NumData = 0;
  char   Buff[1024];
  while(fgets(Buff,1023,DataFile))
    {
      if(Buff[0]=='#')
	{
	  continue;
	}
      String Field[32];
      int NW = getField(Buff,Field);
      for(int J=0;J<NumAxis;++J)
	{
	  push(NumData,J,atof(Field[FieldID[J]]));
	}
      NumData++;
    }
  end(FileName,FieldID,PFlag,OFlag);
  fclose(DataFile);
}

multiData::multiData(multivar* MV)
{
  Multivar        = MV;
  First = Last    = NULL;
  Histogram       = NULL;
  HistogramSmooth = NULL;
  Next            = NULL;

  NumAxis = Multivar->numElem();
  NumData = Multivar->numData();

  DataMatrix = new matrix(NumData,NumAxis);
  FILE* DataFile;
  if(NULL==(DataFile=fopen(Multivar->dataFileName(),"r")))
    {
      cerr << form("Can't open Data File %s\n",Multivar->dataFileName());
      exit(1);
    }
  int    I=0;
  char   Buff[1024];
  while(fgets(Buff,1023,DataFile))
    {
      if(Buff[0]=='#')
	{
	  continue;
	}
      String Field[32];
      int NW = getField(Buff,Field);
      for(int J=0;J<NumAxis;++J)
	{
	  (*DataMatrix)(I,J) = atof(Field[Multivar->statistic(J).id()]);
	}
      I++;
    }
  fclose(DataFile);
  calcRange();
}

multiData::multiData
(
 matrix* DM,
 char*   FileName,
 int*    FieldID,
 int     PFlag,
 int     OFlag
)
{
  First=Last      = NULL;
  DataMatrix      = DM;
  Histogram       = NULL;
  HistogramSmooth = NULL;
  Next            = NULL;

  NumAxis    = DataMatrix->sizeJ();
  NumData    = DataMatrix->sizeI();
  calcRange();
  Multivar = new multivar(DataMatrix,FileName,NumAxis,FieldID,PFlag,OFlag);
}

void multiData::end(char* FileName,int* FieldID,int PFlag,int OFlag)
{
  if(DataMatrix != NULL)
    delete DataMatrix;
  DataMatrix = new matrix(NumData,NumAxis);
  for(
      pushData* PushDataNow = First;
      PushDataNow!=NULL;
      PushDataNow=PushDataNow->next()
      )
    {
      (*DataMatrix)(PushDataNow->dataID(),PushDataNow->axisID())
	= PushDataNow->data();
    }
  delete First;
  First = Last = NULL;

  if(Multivar!=NULL)
    delete Multivar;
  Multivar = new multivar(DataMatrix,FileName,NumAxis,FieldID,PFlag,OFlag);
  if(Multivar!=NULL)
    delete Multivar;
  Multivar = new multivar(DataMatrix,FileName,NumAxis,FieldID,PFlag,OFlag);

  calcRange();
}

void multiData::push(int I,int J,double Data)
{
  pushData* PushData = new pushData(I,J,Data);
  if(First == NULL)
    {
      First = Last = PushData;
    }
  else
    {
      Last->next(PushData);
      Last = PushData;
    }
}

void multiData::calcRange(void)
{
  double MaxDistance = 0.0;
  for(int I=0;I<NumData;++I)
    {
      double SumDistance2 = 0.0;
      for(int J=0;J<NumAxis;++J)
	{
	  double Distance   = (*DataMatrix)(I,J)-Multivar->statistic(J).mean();
	  SumDistance2 += Distance*Distance;
	}
      if(sqrt(SumDistance2) > MaxDistance)
	MaxDistance = sqrt(SumDistance2);
    }      
  Range   = (int)(MaxDistance/sqrt(Multivar->totalVar()))+1;
  NumHist = (NumData/100)*int(Range);
  if(NumHist%2==0)
    NumHist+=3;
  else
    NumHist+=2;
  Range   = Range+Range/(NumHist-1);
}

void multiData::printData(void)
{
  FILE* DataFile;
  if(NULL==(DataFile=fopen(Multivar->dataFileName(),"w")))
    {
      cerr << form("Can't open data file %s\n",Multivar->dataFileName());
      exit(1);
    }
  for(int I=0;I<NumData;++I)
    {
      int AxisID = 0;
      for(int J=0;J<=Multivar->statistic(NumAxis-1).id();++J)
	{
	  if(J==Multivar->statistic(AxisID).id())
	    {
	      fprintf(DataFile," %10.5lf",(*DataMatrix)(I,AxisID));
	      AxisID ++;
	    }
	  else
	    {
	      fprintf(DataFile," %10.5lf",0.0);
	    }
	}
      fprintf(DataFile,"\n");
    }
  fclose(DataFile);
}

void multiData::makeHistogram(void)
{
  Histogram = new matrix(NumHist,NumAxis);
  for(int I=0;I<Multivar->numData();++I)
    {
      matrix* PrincipalVal = makePrincipalVal(I);
      for(int Axis=0;Axis<NumAxis;++Axis)
	{
	  int IPrincipalVal = 
	    int(((*PrincipalVal)(0,Axis)+Range)/Range/2.0*NumHist);
	  (*Histogram)(IPrincipalVal,Axis) += 1.0;
	}
      delete PrincipalVal;
    }
}

void multiData::printHisto(matrix* Histo,char* Ext)
{
  char HistFileName[256];
  strcpy(HistFileName,Multivar->dataFileName());
  strcat(HistFileName,Ext);
  FILE* HistFile;
  if(NULL==(HistFile=fopen(HistFileName,"w")))
    {
      cerr << form("Can't open histogram file %s\n",HistFileName);
      exit(1);
    }
  for(int I=0;I<NumHist;++I)
    {
      fprintf(HistFile,"%10.5lf",Range*2/NumHist*(0.5+I)-Range);
      for(int J=0;J<NumAxis;++J)
	{
	  fprintf(HistFile," %10.5lf",(*Histo)(I,J));
	}
      fprintf(HistFile,"\n");
    }
  fclose(HistFile);
}

void multiData::printHistogram(void)
{
  if(Histogram==NULL)
    makeHistogram();
  printHisto(Histogram,".HST");
}

void multiData::printHistogramSmooth(void)
{
  if(Histogram==NULL)
    makeHistogram();
  if(HistogramSmooth==NULL)
    smoothHistogram();
  printHisto(HistogramSmooth,".HSTS");
}

void multiData::smoothHistogram(void)
{
  HistogramSmooth = new matrix(NumHist,NumAxis);

  if(Histogram==NULL)
    makeHistogram();

  double* TmpA = new double[NumHist*NumAxis];
  double* TmpB = new double[NumHist*NumAxis];

  int NumStandard = int(1.0/(Range*2.0/NumHist));
  for(int I=0;I<NumHist;++I)
    for(int J=0;J<NumAxis;++J)
      TmpA[NumAxis*I+J] = (*Histogram)(I,J);

  for(int K=0;K<NumStandard*NumStandard/100;++K)
    {
      for(int J=0;J<NumAxis;++J)
	{
	  TmpB[J] = TmpA[J]/2.0+TmpA[NumAxis+J]/4.0;
	  for(int I=1;I<NumHist-1;++I)
	    TmpB[NumAxis*I+J] = TmpA[NumAxis*I+J]/2.0
	      +TmpA[NumAxis*(I-1)+J]/4.0+TmpA[NumAxis*(I+1)+J]/4.0;
	  TmpB[NumAxis*(NumHist-1)+J] = TmpA[NumAxis*(NumHist-1)+J]/2.0
	    +TmpA[NumAxis*(NumHist-2)+J]/4.0;
	}
      double* TmpTmp = TmpA;
      TmpA = TmpB;
      TmpB = TmpTmp;
    }
  for(I=0;I<NumHist;++I)
    for(J=0;J<NumAxis;++J)
      (*HistogramSmooth)(I,J) = TmpA[NumAxis*I+J];
}

matrix* multiData::makePrincipalVal(matrix& OrigVal)
{
  matrix* PrincipalVal = new matrix(1,NumAxis);

  for(int J=0;J<NumAxis;++J)
    {
      OrigVal(0,J) -= Multivar->statistic(J).mean();
    }

  for(int Axis=0;Axis<NumAxis;++Axis)
    {
      (*PrincipalVal)(0,Axis) =
	(OrigVal*(Multivar->principal(Axis).axis()))(0,0)
	  /sqrt(Multivar->totalVar());
    }
  return PrincipalVal;
}

matrix* multiData::makePrincipalVal(int I)
{
  matrix  OrigVal(1,NumAxis);
  matrix* PrincipalVal = new matrix(1,NumAxis);

  for(int J=0;J<NumAxis;++J)
    {
      OrigVal(0,J) = (*DataMatrix)(I,J);
    }
  return makePrincipalVal(OrigVal);
}

void multiData::printPrincipalVal(void)
{
  char PValFileName[256];
  strcpy(PValFileName,Multivar->dataFileName());
  strcat(PValFileName,".PV");

  FILE* PVFile;
  if(NULL==(PVFile=fopen(PValFileName,"w")))
    {
      cerr << form("Can't open principal value file %s\n",PValFileName);
      exit(1);
    }
  for(int I=0;I<NumData;++I)
    {
      matrix* PrincipalVal = makePrincipalVal(I);
      for(int J=0;J<NumAxis;++J)
	{
	  fprintf(PVFile," %10.5lf",(*PrincipalVal)(0,J));
	}
      fprintf(PVFile,"\n");
      delete PrincipalVal;
    }
  fclose(PVFile);
}

void multiData::outPrincipalVal(char* FN)
{
  char FileName[256];
  strcpy(FileName,FN);

  FILE* InFile;
  if(NULL==(InFile=fopen(FileName,"r")))
    {
      cerr << form("Can't open original data file %s\n",FileName);
      exit(1);
    }

  FILE* OutFile;
  strcat(FileName,".PV");
  if(NULL==(OutFile=fopen(FileName,"w")))
    {
      cerr << form("Can't open out put file %s\n",FileName);
      exit(1);
    }
  char   Buff[1024];
  while(fgets(Buff,1023,InFile))
    {
      if(Buff[0]=='#')
	{
	  fprintf(OutFile,"%s",Buff);
	  continue;
	}
      String Field[32];
      int NW = getField(Buff,Field);
      matrix  OrigVal(1,NumAxis);
      for(int J=0;J<NumAxis;++J)
	{
	  OrigVal(0,J) =
	    atof(Field[Multivar->statistic(J).id()])
	      -Multivar->statistic(J).mean();
	}
      matrix* PrincipalVal = makePrincipalVal(OrigVal);
      for(J=0;J<NumAxis;++J)
	{
	  fprintf(OutFile," %10.5lf",(*PrincipalVal)(0,J));
	}
      fprintf(OutFile,"\n");
      delete PrincipalVal;
    }
  fclose(InFile);
  fclose(OutFile);
}


int multiData::div
(
 char* DataFileName,
 FILE* DivFile,
 int   Level,
 int   DivID,
 int   SmthFlag,
 int*  FieldID,
 int   PFlag,
 int   OFlag
)
{
  if(NumData<100) return 1;
  if(Histogram==NULL)
    {
      makeHistogram();
    }
  if(SmthFlag)
    {
      if(HistogramSmooth==NULL)
	{
	  smoothHistogram();
	}
    }
  matrix* Histo = (SmthFlag==0?Histogram:HistogramSmooth);

  double CntMin  [NumAxis];
  double CntHisto[NumAxis];
  int    DivPoint[NumAxis];
  for(int J=0;J<NumAxis;++J)
    {
      CntMin  [J] = double(NumData);
      CntHisto[J] = 0;
      DivPoint[J] = 0;
    }
  for(int I=0;I<NumHist;++I)
    {
      for(J=0;J<NumAxis;++J)
	{
	  if(
	     CntHisto[J]                              > double(NumData)/20.0 &&
	     double(NumData)-CntHisto[J]-(*Histo)(I,J)> double(NumData)/20.0
	     )
	    {
	      if(
		 (*Histo)(I,J)<(*Histo)(I-1,J) &&
		 (*Histo)(I,J)<(*Histo)(I+1,J)
		 )  
		{
		  if((*Histo)(I,J)<CntMin[J])
		    {
		      CntMin  [J] = (*Histo)(I,J);
		      DivPoint[J] = I;
		    }
		}
	    }
	  CntHisto[J] += (*Histo)(I,J);
	}
    }
  double AxisMin = double(NumData);
  int AxisID;
  for(J=0;J<NumAxis;++J)
    {
      if(CntMin[J] < AxisMin)
	{
	  AxisMin = CntMin[J];
	  AxisID = J;
	}
    }
  if(AxisMin == double(NumData))
    {
      return 1;
    }

  double Threshold = Range*2.0/NumHist*(DivPoint[AxisID]+0.5)-Range;
  fprintf(DivFile,"Division Level    : %10d %10d\n",Level,DivID);
	  
  fprintf(DivFile,"Axis for Division : %10d\n",AxisID);
  fprintf(DivFile,"Threshold         : %10.5lf\n",Threshold);
  Multivar->fprint(DivFile);

  multiData* MultiDataMinus = new multiData(NumAxis,FieldID);
  multiData* MultiDataPlus  = new multiData(NumAxis,FieldID);

  for(I=0;I<NumData;++I)
    {
      matrix* PrincipalVal = makePrincipalVal(I);
      int CntFlagMinus = 0;
      int CntFlagPlus  = 0;
      for(J=0;J<NumAxis;++J)
	{
	  if((*PrincipalVal)(0,AxisID)<Threshold)
	    {
	      MultiDataMinus->push(J,(*DataMatrix)(I,J));
	      CntFlagMinus = 1;
	    }
	  else
	    {
	      MultiDataPlus ->push(J,(*DataMatrix)(I,J));	      
	      CntFlagPlus = 1;
	    }
	}
      if(CntFlagMinus) MultiDataMinus->count();
      if(CntFlagPlus ) MultiDataPlus ->count();
    }
  char DataFileNameMinus[256];
  char DataFileNamePlus [256];
  strcpy(DataFileNameMinus,DataFileName);
  strcpy(DataFileNamePlus, DataFileName);
  sprintf(DataFileNameMinus+strlen(DataFileName),".%d",DivID*2+1);
  sprintf(DataFileNamePlus +strlen(DataFileName),".%d",DivID*2);
  MultiDataMinus->end(DataFileNameMinus,FieldID,PFlag,OFlag);
  MultiDataPlus ->end(DataFileNamePlus ,FieldID,PFlag,OFlag);
  if(MultiDataMinus->div(DataFileName,DivFile,Level+1,DivID*2+1,
			 SmthFlag,FieldID,PFlag,OFlag))
    {
      MultiDataMinus->multivar()->print();
      MultiDataMinus->printData();
      MultiDataList->push(MultiDataMinus);
    }
  if(MultiDataPlus ->div(DataFileName,DivFile,Level+1,DivID*2,
			 SmthFlag,FieldID,PFlag,OFlag))
    {
      MultiDataPlus ->multivar()->print();
      MultiDataPlus ->printData();
      MultiDataList->push(MultiDataPlus);
    }
  delete this;
  return 0;
}

multiDataList* MultiDataList;
