/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
#include <math.h>
#include <stdlib.h>
#include <stream.h>
#include "matrix.h"

     matrix::~matrix(void)            { if(Nsize >0) delete D;}
void matrix::zap    (void)            { delete D;}
void matrix::reset  (void)            { for(int I=0;I<Nsize;++I) D[I] = 0.0;}

void matrix::copy(matrix& A)
{
  int masize = A.sizeN();
       if(Nsize> 0) { if(Nsize != masize) { delete D; D=new double [masize];}}
  else if(Nsize==0) D = new double [masize];
  DimI=A.sizeI(); DimJ=A.sizeJ(); Nsize = masize;
  for(int I=0;I< DimI;I++) for (int J=0;J<DimJ;J++) D[I*DimJ+J] = A(I,J);
}

void matrix::print(void)
{
  printf("{\n");
  for(int I = 0; I < DimI; ++I)
    {
      printf("  {");
      for(int J=0;J<DimJ-1;++J)	printf("%12.4e,", D[I*DimJ + J]);
      printf("%12.4e", D[I*DimJ+DimJ-1]); printf("}\n");
    }
  printf("}\n");
}

double matrix::determinant(void)
{
  if(DimI!=DimJ){ cerr << "Can't calculate determinat!"; exit(1);}
  matrix Work   =(*this);
  int    Parity =      0;
  for(int K=0;K<DimI-1;++K)
    {
      int II   =K; int JJ =K;
      for(int I=K;I<DimI;++I)
	for(int J=K;J<DimI;++J)
	  if(fabs(Work(II,JJ))<=fabs(Work(I,J))){II=I;JJ=J; }
      if(JJ != K)
	{
	  Parity++;
	  for(int I=0;I<DimI;++I)
	    {
	      double TMP = Work(I, K);
	      Work(I, K) = Work(I,JJ);
	      Work(I,JJ) = TMP;
	    }
	}
      if(II != K)
	{
	  Parity++;
	  for(int J=0;J<DimI;++J)
	    {
	      double TMP = Work(K,J);
	      Work( K,J) = Work(II,J);
	      Work(II,J) = TMP;
	    }
	}
      for(   I=K+1;I<DimI;++I)
	{
	  double TMP = Work(I,K)/Work(K,K);
	  for(int J=K;J<DimI;++J) Work(I,J)=Work(I,J)-TMP*Work(K,J);
	}
    }
  double DD=(Parity%2==0?1.0:-1.0);
  for(    K=0;K<DimI;++K) DD *=  Work(K,K);
  return DD;
}

matrix matrix::tr(void)
{
  matrix C(DimJ,DimI);
  for(int I=0;I<DimI;++I) for(int J=0;J<DimJ;++J) C(J,I)=D[I*DimJ+J];
  return C;
}

matrix  matrix::inverse(void)
{
  if(DimI != DimJ) { cerr<<form("matrix sizes do not match \n"); exit(1);}
  matrix C(DimI,DimJ);
  for(int I=0;I<DimI;++I) for(int J=0;J<DimJ;++J) C(I,J)=D[I*DimJ+J];
  int P[DimI];int Q[DimI];int L = 0;
  for(    I=0;I<DimI;++I) P[I] = Q[I] = 0;
  for(int K=0;K<DimI;++K) 
    {
      L = C.pivot(Q,K); P[K]=L; Q[L]=K;
      for(int J=0;J<DimI;++J)
	if(J!=K)
	  {
	    C(L,J)/=C(L,K);
	    for(int I=0;I<DimI; ++I) if(I!=L) C(I,J) -= (C(L,J)*C(I,K));
	  }
      C(L,K) = (1.0/C(L,K));
      for(    J=0;J<DimI;++J)  if(J!=L) C(J,K) = -(C(J,K)*C(L,K));
    }
  for(    K=0;K<DimI;++K)
    {          
      L = P[K];
      if(K==L)      continue;
      for(int J=0;J<DimI;++J){ double sp=C(L,J);C(L,J)=C(K,   J);C(K,   J)=sp;}
      for(int I=0;I<DimI;++I){ double SP=C(I,K);C(I,K)=C(I,Q[K]);C(I,Q[K])=SP;}
      P[Q[K]]  =   L;P[K] =K;
      Q[   L]  =Q[K];Q[K] =K;
    }
  return C;
}

void   matrix::changeline  (int M1, int M2)
{
  double Temp;
  for(int I=0;I<DimJ;I++)
    { Temp = D[DimJ*M1+I]; D[DimJ*M1+I] = D[DimJ*M2+I]; D[DimJ*M2+I] = Temp;}
}

double matrix::innerproduct(int N, double* U, double* V)
{
  double  S  = 0.0;  int     N5 = N%5;
  double* WU =   U;  double* WV =   V;
  for(int I= 0;I<N5; ++I,++WU,++WV) S+=(*WU)*(*WV);
  for(    I=N5;I<N ;I+=5)
    {
      WU = (U+I); WV = (V+I);
      S += (*WU)*(*WV); WU++;WV++; S+=(*WU)*(*WV); WU++;WV++;
      S += (*WU)*(*WV); WU++;WV++; S+=(*WU)*(*WV); WU++;WV++;
      S += (*WU)*(*WV); 
    }
  return S;
}

double matrix::householder (int N,double* X)
{
  double S = sqrt(innerproduct(N,X,X));
  if(S != 0)
    {
      if(X[0] < 0) S= -S;
      X[0] += S;
      double  T= 1.0/sqrt(X[0]*S);
      for(int I=0;I<N;++I)X[I]*=T;
    }
  return -S;
}

void   matrix::operator=(matrix& B)   { copy(B);}
void   matrix::tridiagonalize(double* Dg, double* Eg)
{
  double V[DimJ],W[DimJ];
  for(int K=0;K<DimI-2;++K)
    {
      for(int L=0;L<DimJ;L++) V[L] = D[DimJ*K + L];
      Dg[K] = V[K]; Eg[K] = householder(DimI-K-1,&V[K+1]);
      if(Eg[K]==0) continue;
      for(int I=K+1   ;I<DimI;++I)
	{
	  double S  =0.0;
	  for (int J=K+1;J<   I;++J) S+=D[DimJ*J+I]*V[J];
	  for (    J=  I;J<DimI;++J) S+=D[DimJ*I+J]*V[J];
	  Dg[I]     =  S;
	}
      double  T=innerproduct(DimI-K-1,&V[K+1],&Dg[K+1])/2.0;
      for(    I=DimI-1;I>   K;--I)
	{
	  double P = V [I];
	  double Q = Dg[K]-T*P;
	  Dg[I]    = Q;
	  for(int J= I;J<DimI;J++) D[DimJ*I+J]-=P*Dg[J]+Q*V[J];
	}
      for(    L=     0;L<DimJ;L++) D[DimJ*K+L]=V[L];
    }
  if(DimI>=2)
    {
      Dg[DimI-2]=D[DimJ*(DimI-2)+(DimI-2)];
      Eg[DimI-2]=D[DimJ*(DimI-2)+(DimI-1)];
    }
  if(DimI>=1) Dg[DimI-1]=D[DimJ*(DimI-1)+(DimI-1)];
  for(    K=DimI-1;K>=0;--K)
    {
      for(int L=0;L<DimJ;++L)	V[L] = D[DimJ*K + L];
      if(K<DimI-2)
	for(int I=K+1; I<DimI;++I)
	  {
	    for(int L=  0;L<DimJ;++L) W[L]        = D[DimJ*I+L];
	    double  T=  innerproduct (DimI-K-1,&V[K+1],&W[K+1]);
	    for(int J=K+1;J<DimI;++J) W[J]       -=      T*V[J];
	    for(    L=  0;L<DimJ;++L) D[DimJ*I+L] =        W[L];
	  }
      for(int I=0;I<DimI;++I) V[I]       =   0;
      V[K] = 1;
      for(    L=0;L<DimJ;L++) D[DimJ*K+L]=V[L];
    }
}


void   operator *=(matrix& A,double B)
{ for(int I=0;I<A.DimI;++I) for(int J=0;J<A.DimJ;++J) A.D[I*A.DimJ+J] *= B;}

void operator /=(matrix& A,double B)
{ for(int I=0;I<A.DimI;++I) for(int J=0;J<A.DimJ;++J) A.D[I*A.DimJ+J] /= B;}

matrix operator-(matrix& A)
{
  int MaI = A.sizeI(); int Maj = A.sizeJ(); matrix C(MaI,Maj);
  for(int I=0;I<MaI;++I) for(int J=0;J<Maj;++J) C(I,J) = - A(I,J);
  return C;
}      

matrix operator+(matrix& A)           { return matrix(A);}      

matrix operator+(matrix& A,matrix& B)
{
  int MaI=A.sizeI(); int MaJ=A.sizeJ(); int mbi=B.sizeI(); int mbj=B.sizeJ(); 
  if(MaI!=mbi||MaJ!=mbj)
    { cerr << "matrix operator+ \nmatrix sizes do not match \n"; exit(1);}
  matrix C(MaI,MaJ);
  for(int I=0; I<MaI;I++) for(int J=0;J<MaJ;J++) C(I,J)=A(I,J)+B(I,J);
  return C;
}

matrix operator-(matrix& A,matrix& B)
{
  int MaI=A.sizeI(); int MaJ=A.sizeJ(); int mbi=B.sizeI(); int mbj=B.sizeJ(); 
  if (MaI != mbi || MaJ != mbj)
    { cerr << "matrix operator- \nmatrix sizes do not match \n"; exit(1);}
  matrix C(MaI,MaJ);
  for(int I=0;I<MaI;++I) for(int J=0;J<MaJ;++J) C(I,J)=A(I,J)-B(I,J);
  return C;
}

matrix operator*(matrix& A,matrix& B)
{
  int MaI=A.sizeI(); int MaJ=A.sizeJ();
  int MbI=B.sizeI(); int MbJ=B.sizeJ(); 
  if (MaJ != MbI)
    { cerr << "matrix operator* \nmatrix sizes do not match \n"; exit(1);}
  matrix C(MaI,MbJ);
  for(int I=0;I<MaI;I++)
    for(int J=0;J<MbJ;J++)
      {
	double Cij = 0.00;
	for(int K=0;K<MaJ;K++) Cij += (A(I,K) * B(K,J));
	C(I,J) = Cij;
      }
  return C;
}

matrix operator*(double  K,matrix& A)
{
  int MaI=A.sizeI(); int MaJ=A.sizeJ(); 
  matrix C(MaI,MaJ);
  for(int I=0;I<MaI;++I) for(int J=0;J<MaJ;++J) C(I,J) = K * A(I,J);
  return C;
}

matrix operator*(matrix& A,double  K) {                  return K *A;}
matrix operator/(matrix& A,double  K) { double AK=1.0/K; return AK*A;}


