/* Micro Quixote                 */
/* Copyright (C) 1993, 1994 ICOT */
/* Written by gniibe             */
/* $Id: subsump.c,v 1.2 1994/09/19 02:10:19 m-gniibe Exp $ */

/* subsumption operation handling routines */

#include <stdio.h>
#include "obstack.h"
#include "mq.h"
#include "internal.h"
#include "extern.h"

MQ_Atom mqA_Bottom, mqA_Top;
MQ_Obj  mqO_Bottom, mqO_Top;

/* function prototype for debugging */
static int hash _P((MQ_AtomList));
static MQ_AtomList sort_atom_list_sub _P((MQ_AtomList, MQ_AtomList));
static MQ_AtomList sort_atom_list _P((MQ_AtomList));
static MQ_AtomList make_atom_list _P((MQ_Atom, MQ_AtomList));
static int atom_list_cmp _P((MQ_AtomList, MQ_AtomList));
static int atom_set_exsits _P((MQ_AtomList));
static MQ_AtomSet intern_atom_set _P((MQ_AtomList));
static void delete_from_atom_set _P((MQ_AtomList));
static MQ_AtomList intersection _P((MQ_AtomList, MQ_AtomList));
static int includes _P((MQ_Atom, MQ_AtomList));
static void modify_l_bitmap _P((int, int));
static MQ_AtomList copy_al _P((MQ_AtomList));
static int subsume_subrel _P((MQ_Atom, MQ_Atom));
static MQ_Atom a_meet _P((MQ_Atom, MQ_Atom));
static MQ_Atom a_join _P((MQ_Atom, MQ_Atom));
static int a_subsume_p _P((MQ_Atom, MQ_Atom));
static MQ_AtomAtomSet make_atom_atomset _P((MQ_Atom, MQ_AtomList, MQ_AtomAtomSet));
static void variables_in_vterm _P((MQ_VTerm));


static struct obstack lattice_obstack;
static struct obstack *mm_lattice;
static unsigned char *lattice_first_obj;

#define SET_POOL_SIZE 511
static MQ_AtomSet atom_set_pool[SET_POOL_SIZE];
static MQ_Atom *lattice;

void
init_lattice ()
{
  int i;

  mm_lattice = &lattice_obstack;
  obstack_begin (mm_lattice, 0);
  mqA_Bottom = intern_atom ((unsigned char *)"&bottom");
  mqA_Top    = intern_atom ((unsigned char *)"&top");

  mm_current = mm_lattice;
  mqO_Bottom = make_object (mqA_Bottom, 0);
  mqO_Top    = make_object (mqA_Top, 0);;

  lattice_first_obj = (unsigned char *)obstack_alloc (mm_lattice, 0);

  for (i=0; i < SET_POOL_SIZE; i++)
    atom_set_pool[i] = NULL;

  mm_current = mm_rule;
}

void
free_lattice ()
{
  int i;

  obstack_free (mm_lattice, lattice_first_obj);
  lattice_first_obj = (unsigned char *)obstack_alloc (mm_lattice, 0);
  for (i=0; i < SET_POOL_SIZE; i++)
    atom_set_pool[i] = NULL;
}

static int
hash (atom_list)
     MQ_AtomList atom_list;
{
  MQ_AtomList al;
  unsigned int hash = 0;

  for (al=atom_list; al; al=al->next)
    hash += (unsigned int) al->atom;

  return hash % SET_POOL_SIZE;
}

static MQ_AtomList
make_atom_list (atom, next)
     MQ_Atom atom;
     MQ_AtomList next;
{
  MQ_AtomList new;

  new = (MQ_AtomList) obstack_alloc (mm_exec, sizeof (MQ_AtomList_Rec));
  new->atom = atom;
  new->next = next;
  return new;
}

static MQ_AtomAtomSet
make_atom_atomset (atom, atom_list, next)
     MQ_Atom atom;
     MQ_AtomList atom_list;
     MQ_AtomAtomSet next;
{
  MQ_AtomAtomSet new;

  new =(MQ_AtomAtomSet)obstack_alloc (mm_exec, sizeof (MQ_AtomAtomSet_Rec));
  new->atom = atom;
  new->atom_list = atom_list;
  new->next = next;
  return new;
}

static MQ_AtomList
sort_atom_list_sub (al_elm, al)
     MQ_AtomList al_elm, al;
{
  MQ_AtomList new, a, next, prev;

  prev = NULL;
  for (a=al; a; a=a->next)
    {
      if ((int)al_elm->atom < (int)a->atom)
	break;
      prev = a;
    }

  if (prev == NULL)
    {
      new = al_elm;
      al_elm->next = al;
    }
  else
    {
      new = al;
      next = prev->next;
      prev->next = al_elm;
      al_elm->next = next;
    }

  return new;
}

static MQ_AtomList
sort_atom_list (al)
     MQ_AtomList al;
{
  MQ_AtomList next;

  if (al == NULL)
    return NULL;

  next = sort_atom_list (al->next);
  return sort_atom_list_sub (al, next);
}

static int
atom_list_cmp (al1, al2)
     MQ_AtomList al1, al2;
{
  MQ_AtomList al, al_;

  for (al=al1, al_=al2; al; al=al->next, al_=al_->next)
    if (al_ == NULL)
      return FALSE;
    else if (al->atom != al_->atom)
      return FALSE;

  if (al_)
    return FALSE;
  return TRUE;
}

static int
atom_set_exsits (al)
     MQ_AtomList al;
{
  MQ_AtomList al1;
  MQ_AtomSet p;
  int val;

  al1 = copy_al (al);
  al1 = sort_atom_list (al1);
  val = hash (al1);
  for (p=atom_set_pool[val]; p; p=p->next_bucket)
    if (atom_list_cmp (al1, p->atom_list) == TRUE)
      return TRUE;
  return FALSE;
}

static MQ_AtomSet
intern_atom_set (al)
     MQ_AtomList al;
{
  MQ_AtomList al1;
  MQ_AtomSet p, new;
  int val;

  al1 = sort_atom_list (al);
  val = hash (al1);
  for (p=atom_set_pool[val]; p; p=p->next_bucket)
    if (atom_list_cmp (al1, p->atom_list) == TRUE)
      return p;

  new = (MQ_AtomSet) obstack_alloc (mm_exec, sizeof (MQ_AtomSet_Rec));
  new->next_bucket = atom_set_pool[val];
  atom_set_pool[val] = new;
  new->atom_list = al1;

  return new;
}

static void
delete_from_atom_set (al)
     MQ_AtomList al;
{
  MQ_AtomSet p, last;
  int val;

  last = NULL;
  val = hash (al);
  for (p=atom_set_pool[val]; p; p=p->next_bucket)
    {
      if (atom_list_cmp (al, p->atom_list) == TRUE)
	{
	  if (last)
	    last->next_bucket = p->next_bucket;
	  else
	    atom_set_pool[val] = p->next_bucket;
	  return;
	}
      last = p;
    }

  fatal ("delete_from_atom_set.\n");
}

static MQ_AtomList
intersection (al1, al2)
     MQ_AtomList al1, al2;
{
  MQ_AtomList al;

  al=NULL;
  while (al1 && al2)
    if ((int) al1->atom == (int) al2->atom)
      {
	al = make_atom_list (al1->atom, al);
	al1 = al1->next;
	al2 = al2->next;
      }
    else if ((int) al1->atom < (int) al2->atom)
      al1 = al1->next;
    else if ((int) al1->atom > (int) al2->atom)
      al2 = al2->next;

  return al;
}

static int
includes (a, al)
     MQ_Atom a;
     MQ_AtomList al;
{
  while (al)
    {
      if (al->atom == a)
	return TRUE;
      al = al->next;
    }
  return FALSE;
}

static char *l_bitmap;		/* well, it is not bitmap, actually... ;-) */
static int lno;

static void
modify_l_bitmap (i, j)
     int i, j;
{
  if ((i < 0) || (j < 0))
    fatal ("something wrong in lattice\n");

  l_bitmap[i*lno+j] = 1;
}

static MQ_AtomAtomSet aas_begin;

static MQ_AtomList
copy_al (al)
     MQ_AtomList al;
{
  MQ_AtomList next;

  if (al == NULL)
    return NULL;
  next = copy_al (al->next);
  return make_atom_list (al->atom, next);
}

static int
subsume_subrel (a1, a2)
     MQ_Atom a1, a2;
{
  MQ_SubRel s;

  if (a1 == a2)
    return TRUE;

  for (s = subrel_list; s; s=s->next)
    if ((s->a1 == a1))
      if (subsume_subrel (s->a2, a2))
	return TRUE;

  return FALSE;
}

void
generate_lattice ()
{
  MQ_SubRel s;
  MQ_AtomList al1, al2, al, all_a;
  MQ_AtomSet as;
  MQ_AtomAtomSet aas, aas1, aas2, aas3;
  int i,j;
  unsigned char *exec_obj;

  free_lattice ();
  exec_obj = (unsigned char *)obstack_alloc (mm_exec, 0);

  lno=0;
  aas = NULL;

  /* set for bottom */
  al = make_atom_list (mqA_Bottom, NULL);
  as = intern_atom_set (al);
  mqA_Bottom->lattice_index = lno++;
  aas = make_atom_atomset (mqA_Bottom, as->atom_list, aas);

  /* set for top */
  al = make_atom_list (mqA_Bottom, NULL); /* can't share */
  al = make_atom_list (mqA_Top, al);
  for (s=subrel_list; s; s=s->next)
    {
      int exists;

      exists = 0;
      for (al1=al; al1; al1=al1->next)
	if (al1->atom == s->a1)
	  {
	    exists = 1;
	    break;
	  }
      if (!exists)
	al = make_atom_list (s->a1, al);

      exists = 0;
      for (al1=al; al1; al1=al1->next)
	if (al1->atom == s->a2)
	  {
	    exists = 1;
	    break;
	  }
      if (!exists)
	al = make_atom_list (s->a2, al);
    }
  as = intern_atom_set (al);
  all_a = as->atom_list;
  mqA_Top->lattice_index = lno++;
  aas = make_atom_atomset (mqA_Top, as->atom_list, aas);

  for (al1 = all_a; al1; al1=al1->next)
    {
      if (al1->atom == mqA_Top || al1->atom == mqA_Bottom)
	continue;

      al = make_atom_list (mqA_Bottom, NULL);
      for (al2 = all_a; al2; al2=al2->next)
	if (subsume_subrel (al1->atom, al2->atom))
	  al = make_atom_list (al2->atom, al);

      al1->atom->lattice_index = lno++;
      as = intern_atom_set (al);
      aas = make_atom_atomset (al1->atom, as->atom_list, aas);
    }

  while (1)
    {
      MQ_AtomAtomSet last_aas;

      last_aas = aas;
      aas1=aas;
      while (aas1)
	{
	  aas1 = aas1->next;
	  for (aas2=aas1; aas2; aas2 = aas2->next)
	    {
	      al = intersection (aas1->atom_list, aas2->atom_list);
	      if (atom_set_exsits (al) == FALSE)
		{
		  char new_name[1024];
		  char *name;
		  int len;
		  MQ_Atom atom;

		  len = strlen (aas1->atom->name)
		    + strlen (aas2->atom->name) + 4;
		  if (len > 1023)
		    name = "too-complexed...";
		  else
		    {
		      name = new_name;
		      sprintf (new_name,
			       "&_%s_%s", aas1->atom->name, aas2->atom->name);
		    }

		  atom = make_atom (name, mm_lattice);
		  atom->lattice_index = lno++;

		  for (aas3=aas; aas3; aas3=aas3->next)
		    if (includes (aas1->atom, aas3->atom_list)
			|| includes (aas2->atom, aas3->atom_list))
		      {
			/* update atom_list of aas3 */
			delete_from_atom_set (aas3->atom_list);
			al1 = make_atom_list (atom, aas3->atom_list);
			as = intern_atom_set (al1);
			aas3->atom_list = as->atom_list;
		      }

		  al = make_atom_list (atom, al);
		  as = intern_atom_set (al);
		  aas = make_atom_atomset (atom, as->atom_list, aas);
		}
	    }
	}
      if (aas == last_aas)
	break;
    }
  aas_begin = aas;

  l_bitmap = (char *)obstack_alloc (mm_lattice, lno*lno);
  /* bottom */
  for (j=0; j<lno; j++)
    l_bitmap[0*lno+j]=0;
  l_bitmap[0*lno+0]=1;
  /* top */
  for (j=0; j<lno; j++)
    l_bitmap[1*lno+j]=1;
  for (i=2; i<lno; i++)
    {
      for (j=0; j<lno; j++)
	l_bitmap[i*lno+j]=0;
      /* self */
      l_bitmap[i*lno+i]=1;
      /* bottom */
      l_bitmap[i*lno+0]=1;
    }

  lattice = (MQ_Atom *)obstack_alloc (mm_lattice, sizeof (MQ_Atom)*lno);
  i = lno;
  for (aas=aas_begin; aas; aas=aas->next)
    {
      MQ_AtomList al;

      lattice[--i] = aas->atom;
      for (al=aas->atom_list; al; al=al->next)
	modify_l_bitmap (aas->atom->lattice_index, al->atom->lattice_index);
    }

  obstack_free (mm_exec, exec_obj);
}

static MQ_Atom
a_join (a1, a2)
     MQ_Atom a1, a2;
{
  int x;
  int min;
  int l1, l2;

  if (a1 == a2)
    return a1;
  if (a1 == mqA_Bottom)
    return a2;
  if (a2 == mqA_Bottom)
    return a1;

  l1 = a1->lattice_index; 
  l2 = a2->lattice_index; 

  if ((l1 < 0) || (l2 < 0))
    return mqA_Top;

  min = 1;
  for (x=0; x<lno; x++)
    if (l_bitmap[x*lno+l1]&l_bitmap[x*lno+l2])
      if (l_bitmap[min*lno+x])
	min = x;

  return lattice[min];
}

static MQ_Atom
a_meet (a1, a2)
     MQ_Atom a1, a2;
{
  int x;
  int max;
  int l1, l2;
  l1 = a1->lattice_index; 
  l2 = a2->lattice_index; 

  if (a1 == a2)
    return a1;
  if (a1 == mqA_Top)
    return a2;
  if (a2 == mqA_Top)
    return a1;

  if ((l1 < 0) || (l2 < 0))
    return mqA_Bottom;

  max = 0;
  for (x=0; x<lno; x++)
    if (l_bitmap[l1*lno+x]&l_bitmap[l2*lno+x])
      if (l_bitmap[x*lno+max])
	max = x;

  return lattice[max];
}

static int
a_subsume_p (a1, a2)
     MQ_Atom a1, a2;
{
  if (a1 == a2)
    return TRUE;

  if (a1 == mqA_Top)
    return TRUE;
  if (a2 == mqA_Bottom)
    return TRUE;
  if ((a1->lattice_index < 0) || (a2->lattice_index < 0))
    return FALSE;
  if (l_bitmap[a1->lattice_index*lno+a2->lattice_index])
    return TRUE;
  else
    return FALSE;
}

MQ_Obj
join (o1, o2)
     MQ_Obj o1, o2;
{
  MQ_Atom a;
  MQ_Obj o;
  int i,j;
  int new_arity;
  int found;
  IJ_List ij_list;
  struct obstack *mm_old;

  if (o1 == mqO_Bottom)
    return o2;

  if (o2 == mqO_Bottom)
    return o1;

  a = a_join (o1->atom, o2->atom);

  j = new_arity = 0;
  ij_list = NULL;

  /* Note: attr is sorted by label address */
  for (i=0; i < o1->arity; i++)
    {
      if (j == o2->arity)
	break;

      found = FALSE;
      while (j < o2->arity)
	if (o1->attr[i].label < o2->attr[j].label)
	  break;
	else if (o1->attr[i].label == o2->attr[j].label)
	  {
	    found = TRUE;
	    break;
	  }
	else
	  j++;

      if (found)
	{
	  ij_list = make_ij_list (i, j, ij_list);
	  new_arity++;
	}
    }

  mm_old = mm_current;
  mm_current = mm_exec;
  o = make_object (a, new_arity);
  mm_current = mm_old;
  while (ij_list)
    {
      --new_arity;
      o->attr[new_arity].label = o1->attr[ij_list->i].label;
      o->attr[new_arity].vterm
	= (MQ_VTerm) join ((MQ_Obj)o1->attr[ij_list->i].vterm,
			   (MQ_Obj)o2->attr[ij_list->j].vterm);
      ij_list = ij_list->next;
    }

  return o;
}

MQ_Obj
meet (o1, o2)
     MQ_Obj o1, o2;
{
  MQ_Atom a;
  MQ_Obj o;
  int i,j;
  int new_arity;
  int found;
  IJ_List ij_list;
  struct obstack *mm_old;

  if (o1 == mqO_Top)
    return o2;

  if (o2 == mqO_Top)
    return o1;

  a = a_meet (o1->atom, o2->atom);
  if (a == mqA_Bottom)
    return mqO_Bottom;

  j = new_arity = 0;
  ij_list = NULL;

  /* Note: attr is sorted by label address */
  for (i=0; i < o1->arity; i++)
    {
      found = FALSE;
      while (j < o2->arity)
	if (o1->attr[i].label < o2->attr[j].label)
	  break;
	else if (o1->attr[i].label == o2->attr[j].label)
	  {
	    found = TRUE;
	    break;
	  }
	else
	  {
	    ij_list = make_ij_list (-1, j, ij_list);
	    new_arity++;
	    j++;
	  }

      if (found)
	{
	  ij_list = make_ij_list (i, j, ij_list);
	  j++;
	}
      else
	ij_list = make_ij_list (i, -1, ij_list);
      new_arity++;
    }

  while (j < o2->arity)
    {
      ij_list = make_ij_list (-1, j, ij_list);
      new_arity++;
      j++;
    }
  
  mm_old = mm_current;
  mm_current = mm_exec;
  o = make_object (a, new_arity);
  mm_current = mm_old;
  while (ij_list)
    {
      --new_arity;
      if (ij_list->i == -1)
	{
	  o->attr[new_arity].label = o2->attr[ij_list->j].label;
	  o->attr[new_arity].vterm = o2->attr[ij_list->j].vterm;
	}
      else if (ij_list->j == -1)
	{
	  o->attr[new_arity].label = o1->attr[ij_list->i].label;
	  o->attr[new_arity].vterm = o1->attr[ij_list->i].vterm;
	}
      else
	{
	  o->attr[new_arity].label = o1->attr[ij_list->i].label;
	  o->attr[new_arity].vterm
	    = (MQ_VTerm) meet ((MQ_Obj)o1->attr[ij_list->i].vterm,
			       (MQ_Obj)o2->attr[ij_list->j].vterm);
	}
      ij_list = ij_list->next;
    }

  return o;
}

static MQ_VarList vl;

MQ_VarList
variables_in_obj (obj)
     MQ_Obj obj;
{
  vl = NULL;
  variables_in_vterm ((MQ_VTerm)obj);
  return vl;
}

static void
variables_in_vterm (vt)
     MQ_VTerm vt;
{
  MQ_Obj o;
  int i;

  switch (vt->type)
    {
    case TT_Var:
      vl = make_var_list ((MQ_Var)vt, vl);
      break;
    case TT_Obj:
      o = (MQ_Obj) vt;
      for (i=0; i < o->arity; i++)
	variables_in_vterm (o->attr[i].vterm);
      break;
    default:
      fatal ("something wrong in variables_in_vterm.\n");
      break;
    }
}

int
eval_subsumption (vt1, vt2, cnstrs_p)
     MQ_VTerm vt1, vt2;
     MQ_Constraints *cnstrs_p;
{
  MQ_Obj o1, o2;
  MQ_Constraint cnstr;
  int i, j;

  if (equal (vt1, vt2) == TRUE)
    return TRUE;

  switch (vt1->type)
    {
    case TT_Var:
      switch (vt2->type)
	{
	case TT_Var:
	  cnstr = make_cnstr (SubsumesVarVar, (MQ_Term)vt1, vt2);
	  *cnstrs_p = make_cnstrs (cnstr, *cnstrs_p);
	  (*cnstrs_p)->l_var_list = ((MQ_Var)vt1)->var_list;
	  (*cnstrs_p)->r_var_list = ((MQ_Var)vt2)->var_list;
	  return TRUE;
	  break;

	case TT_Obj:
	  cnstr = make_cnstr (SubsumesVarObj, (MQ_Term)vt1, vt2);
	  *cnstrs_p = make_cnstrs (cnstr, *cnstrs_p);
	  (*cnstrs_p)->l_var_list = ((MQ_Var)vt1)->var_list;
	  (*cnstrs_p)->r_var_list = variables_in_obj ((MQ_Obj)vt2);
	  return TRUE;
	  break;
	default:
	  fatal ("something wrong in eval_subsumption\n");
	  break;
	}
      break;

    case TT_Obj:
      switch (vt2->type)
	{
	case TT_Var:
	  cnstr = make_cnstr (SubsumesObjVar, (MQ_Term)vt1, vt2);
	  *cnstrs_p = make_cnstrs (cnstr, *cnstrs_p);
	  (*cnstrs_p)->l_var_list = variables_in_obj ((MQ_Obj)vt1);
	  (*cnstrs_p)->r_var_list = ((MQ_Var)vt2)->var_list;
	  return TRUE;
	  break;

	case TT_Obj:
	  o1 = (MQ_Obj)vt1;
	  o2 = (MQ_Obj)vt2;

	  if (o1->arity == 0)
	    return a_subsume_p (o1->atom, o2->atom);
	  else if (o2->arity == 0)
	    return FALSE;
	  else
	    {
	      if (a_subsume_p (o1->atom, o2->atom) == FALSE)
		return FALSE;

	      for (i=j=0; i < o1->arity; i++)
		{
		  while (j < o2->arity)
		    if (o1->attr[i].label == o2->attr[j].label)
		      {
			if (eval_subsumption (o1->attr[i].vterm,
					      o2->attr[j].vterm,
					      cnstrs_p) == FALSE)
			  return FALSE;
			else
			  break;
		      }
		    else
		      j++;

		  if (j == o2->arity)
		    return FALSE;
		}
	      return TRUE;
	    }
	  break;

	default:
	  fatal ("something wrong in eval_subsumption\n");
	  break;
	}
      break;
    default:
      fatal ("something wrong in eval_subsumption\n");
      break;
    }
}

static int x, y;

void
begin_get_atom_which_supersumes (a)
     MQ_Atom a;
{
  y = a->lattice_index;
  if (y < 0)
    x = lno;
  else
    x = 0;
}

MQ_Atom
get_next_atom_supersump ()
{
  while (x < lno)
    if (l_bitmap[y*lno+x] && x != y)
      return lattice[x++];
    else
      x++;
  return NULL;
}

void
begin_get_atom_which_subsumes (a)
     MQ_Atom a;
{
  y = a->lattice_index;
  if (y < 0)
    x = lno;
  else
    x = 0;
}

MQ_Atom
get_next_atom_subsump ()
{
  while (x < lno)
    if (l_bitmap[x*lno+y] && x != y)
      return lattice[x++];
    else
      x++;
  return NULL;
}
