/* Micro Quixote           */
/* Copyright (C) 1993 ICOT */
/* Written by gniibe       */
/* $Id: lookup.c,v 1.2 1994/09/19 02:07:53 m-gniibe Exp $ */

/* lookup routine */

#include <stdio.h>
#include "obstack.h"
#include "mq.h"
#include "internal.h"
#include "tree.h"
#include "extern.h"

static int var_number;
static MQ_VarList var_list;
static MQ_VTermList vterm_list_orig, vterm_list_tangled;

/* function prototype for debugging */
static MQ_Lookup make_lookup _P((void));
static MQ_NameVar lookup_tangle_var _P((MQ_Var));
static MQ_Obj lookup_tangle_obj _P((MQ_Obj));
static MQ_Dot lookup_tangle_dot _P((MQ_Dot));
static MQ_VTerm lookup_tangle_vterm _P((MQ_VTerm));
static MQ_Term lookup_tangle_term _P((MQ_Term));
static MQ_Constraint lookup_tangle_constraint _P((MQ_Constraint));
static MQ_Constraints lookup_tangle_constraints _P((MQ_Constraints));
static int compare_vterm _P((MQ_VTerm, MQ_VTerm));
static int compare_dot _P((MQ_Dot, MQ_Dot));
static int compare_term _P((MQ_Term, MQ_Term));
static int compare_cnstr _P((MQ_Constraint, MQ_Constraint));
static int compare_cnstrs _P((MQ_Constraints, MQ_Constraints));
static int compare_vterm_list _P((MQ_VTermList, MQ_VTermList));
static int compare_lookup _P((MQ_Lookup, MQ_Lookup));
static int hash _P((MQ_Lookup));
static unsigned int hash_term _P((MQ_Term));
static unsigned int hash_vterm _P((MQ_VTerm));
static unsigned int hash_vterm_list _P((MQ_VTermList));
static unsigned int hash_cnstr _P((MQ_Constraint));
static unsigned int hash_cnstrs _P((MQ_Constraints));

void init_lookup ()
{
  ;
}

void free_lookup ()
{
  ;
}

static
MQ_Lookup make_lookup ()
{
  MQ_Lookup new;

  new = (MQ_Lookup)obstack_alloc (mm_exec, sizeof (MQ_Lookup_Rec));
  new->entry = 0;
  new->head = NULL;
  new->head_cnstrs = NULL;
  new->body_cnstrs = NULL;
  new->body = NULL;
  return new;
}

int lookup_active_goals (g)
     MQ_Goal g;
{
  MQ_Lookup l_this;
  MQ_VTermList body;
  MQ_Goal g1;
  int entry;

  if (!mq_opt_lookup)
    return FALSE;

  var_list = NULL;
  var_number = 0;
  vterm_list_orig = vterm_list_tangled = NULL;
  l_this = make_lookup ();

  l_this->head = lookup_tangle_vterm (g->goal_vterm);
  l_this->head_cnstrs = lookup_tangle_constraints (g->head_cnstrs);
  body = NULL;
  for (g1=g->subgoal; g1; g1 = g1->next)
    {
      body = make_vterm_list (body);
      body->vterm = lookup_tangle_vterm (g1->goal_vterm);
    }
  l_this->body = body;
  l_this->body_cnstrs = lookup_tangle_constraints (g->body_cnstrs);
  while (var_list)
    {
      var_list->var->value = NULL;
      var_list = var_list->next;
    }

  entry = hash(l_this);
  l_this->entry = entry;
  g->lookup = l_this;

  for (g1=g->parent; g1; g1 = g1->parent)
    if (g1->lookup->entry == entry
	&& g->type == g1->type
	&& compare_lookup (l_this, g1->lookup) == TRUE)
      {
	g->lookup = NULL;
	return TRUE;
      }

  return FALSE;
}

static
MQ_NameVar lookup_tangle_var (var)
     MQ_Var var;
{
  MQ_NameVar nv;

  if (var->value)
    return (MQ_NameVar)var->value;

  nv = make_name_var (++var_number, NULL);
  var->value = (MQ_VTerm)nv;
  var_list = make_var_list (var, var_list);
  return nv;
}

static
MQ_Obj lookup_tangle_obj (obj)
     MQ_Obj obj;
{
  MQ_Obj new;
  int i, arity;
  MQ_VTermList vlo, vlt;
  MQ_VTerm vt;

  vt = (MQ_VTerm)obj;
  for (vlo=vterm_list_orig, vlt=vterm_list_tangled;
       vlo;
       vlo=vlo->next, vlt=vlt->next)
    if (vlo->vterm == vt)
      return (MQ_Obj)vlt->vterm;
  vterm_list_orig = make_vterm_list (vterm_list_orig);
  vterm_list_orig->vterm = vt;
  vterm_list_tangled = make_vterm_list (vterm_list_tangled);

  arity = obj->arity;
  new = make_object (obj->atom, arity);
  vterm_list_tangled->vterm = (MQ_VTerm)new;
  for (i=0; i < arity; i++)
    {
      new->attr[i].label = obj->attr[i].label;
      new->attr[i].vterm = lookup_tangle_vterm (obj->attr[i].vterm);
    }
  return new;
}

static
MQ_Dot lookup_tangle_dot (dot)
     MQ_Dot dot;
{
  MQ_Dot new;

  new = make_dot (dot->label);
  new->vterm = lookup_tangle_vterm (dot->vterm);
  return new;
}

static
MQ_VTerm lookup_tangle_vterm (vterm)
     MQ_VTerm vterm;
{
  MQ_VTerm new;

  switch (vterm->type)
    {
    case TT_NameVar:
      new = vterm;
      break;
    case TT_Var:
      new = (MQ_VTerm)lookup_tangle_var ((MQ_Var)vterm);
      break;
    case TT_Obj:
      new = (MQ_VTerm) lookup_tangle_obj ((MQ_Obj)vterm);
      break;
    default:
      fatal ("something wrong in lookup_tangle_vterm\n");
      break;
    }
  return new;
}

static
MQ_Term lookup_tangle_term (term)
     MQ_Term term;
{
  MQ_Term new;

  switch (term->type)
    {
    case TT_NameVar:
      new = term;
      break;
    case TT_Var:
      new = (MQ_Term) lookup_tangle_var ((MQ_Var)term);
      break;
    case TT_Obj:
      new = (MQ_Term) lookup_tangle_obj ((MQ_Obj)term);
      break;
    case TT_Dot:
      new = (MQ_Term) lookup_tangle_dot ((MQ_Dot)term);
      break;
    default:
      fatal ("something wrong in lookup_tangle_term\n");
      break;
    }
  return new;
}

static
MQ_Constraint lookup_tangle_constraint (cnstr)
     MQ_Constraint cnstr;
{
  MQ_Constraint new;

  new = make_cnstr (cnstr->rel, NULL, NULL);
  new->term = lookup_tangle_term (cnstr->term);
  new->vterm = lookup_tangle_vterm (cnstr->vterm);
  return new;
}

static
MQ_Constraints lookup_tangle_constraints (cnstrs)
     MQ_Constraints cnstrs;
{
  MQ_Constraints new, next;
  MQ_Constraint cnstr;

  if (cnstrs == NULL)
    fatal ("something wrong in lookup_tangle_constraints\n");

  if (cnstrs == mQ_void_cnstrs)
    return mQ_void_cnstrs;

  next = lookup_tangle_constraints (cnstrs->next);
  cnstr = lookup_tangle_constraint (cnstrs->cnstr);
  new = make_cnstrs (cnstr, next);
  return new;
}

static
int compare_vterm (vt1, vt2)
     MQ_VTerm vt1, vt2;
{
  MQ_Obj o1, o2;
  MQ_NameVar n1, n2;
  int i;

  if (vt1 == vt2)
    return TRUE;

  if ((vt1->type == TT_Var) || (vt2->type == TT_Var))
    fatal ("lookup: variable in the structure.\n");

  if ((vt1->type == TT_NameVar) && (vt2->type == TT_NameVar))
    {
      n1 = (MQ_NameVar) vt1;
      n2 = (MQ_NameVar) vt2;
      if (n1->number == n2->number)
	return TRUE;
      else
	return FALSE;
    }

  if ((vt1->type == TT_Obj) && (vt2->type == TT_Obj))
    {
      o1 = (MQ_Obj)vt1;
      o2 = (MQ_Obj)vt2;
      if ((o1->atom != o2->atom) || (o1->arity != o2->arity))
	return FALSE;

      for (i=0; i < o1->arity; i++)
	if (o1->attr[i].label != o2->attr[i].label)
	  return FALSE;

      for (i=0; i < o1->arity; i++)
	if (compare_vterm (o1->attr[i].vterm, o2->attr[i].vterm) == FALSE)
	  return FALSE;

      return TRUE;
    }
  return FALSE;
}

static
int compare_vterm_list (vl1, vl2)
     MQ_VTermList vl1, vl2;
{
  while (vl1)
    {
      if (vl2 == NULL)
	return FALSE;
      if (compare_vterm (vl1->vterm, vl2->vterm) == FALSE)
	return FALSE;
      vl1 = vl1->next;
      vl2 = vl2->next;
    }
  if (vl2 == NULL)
    return TRUE;
  else
    return FALSE;
}

static
int compare_dot (d1, d2)
     MQ_Dot d1, d2;
{
  if (d1->label != d2->label)
    return FALSE;
  return compare_vterm (d1->vterm, d2->vterm);
}

static
int compare_term (t1, t2)
     MQ_Term t1, t2;
{
  if (t1->type == TT_Dot)
    if (t2->type == TT_Dot)
      return compare_dot ((MQ_Dot)t1, (MQ_Dot)t2);
    else
      return FALSE;

  if (t2->type == TT_Dot)
      return FALSE;

  return compare_vterm ((MQ_VTerm)t1, (MQ_VTerm)t2);
}

static
int compare_cnstr (cnstr1, cnstr2)
     MQ_Constraint cnstr1, cnstr2;
{
  if (cnstr1->rel != cnstr2->rel)
    return FALSE;
  if (compare_term (cnstr1->term, cnstr2->term) == FALSE)
    return FALSE;
  if (compare_vterm (cnstr1->vterm, cnstr2->vterm) == FALSE)
    return FALSE;
  return TRUE;
}

static
int compare_cnstrs (cnstrs1, cnstrs2)
     MQ_Constraints cnstrs1, cnstrs2;
{
  while (cnstrs1 != mQ_void_cnstrs)
    {
      if (cnstrs2 == mQ_void_cnstrs)
	return FALSE;
      if (compare_cnstr (cnstrs1->cnstr, cnstrs2->cnstr) == FALSE)
	return FALSE;
      cnstrs1 = cnstrs1->next;
      cnstrs2 = cnstrs2->next;
    }
  if (cnstrs2 == mQ_void_cnstrs)
    return TRUE;
  else
    return FALSE;
}

static
int compare_lookup (l1, l2)
     MQ_Lookup l1, l2;
{
  if (compare_vterm (l1->head, l2->head)
      && compare_vterm_list (l1->body, l2->body)
      && compare_cnstrs (l1->head_cnstrs, l2->head_cnstrs)
      && compare_cnstrs (l1->body_cnstrs, l2->body_cnstrs))
    return TRUE;

  return FALSE;
}

static
int hash (l)
     MQ_Lookup l;
{
  unsigned int hash = 0;

  hash = hash_vterm (l->head) + hash_vterm_list (l->body)
    + hash_cnstrs (l->head_cnstrs) + hash_cnstrs (l->body_cnstrs);

  return hash;
}

static
unsigned int hash_term (t)
     MQ_Term t;
{
  unsigned int h;
  MQ_Dot dot;
  MQ_NameVar nv;
  MQ_Obj o;
  int i;

  switch (t->type)
    {
    case TT_NameVar:
      nv = (MQ_NameVar)t;
      h = nv->number;
      break;

    case TT_Obj:
      o = (MQ_Obj)t;
      h = (unsigned int)o->atom;
      for (i=0; i< o->arity; i++)
	h += (unsigned int)o->attr[i].label + hash_vterm (o->attr[i].vterm);
      break;

    case TT_Dot:
      dot = (MQ_Dot)t;
      h = (unsigned int)dot->label + hash_vterm (dot->vterm);
      break;

    default:
      fatal ("hash_vterm\n");
      break;
    }
  return h;
}

static
unsigned int hash_vterm (vt)
     MQ_VTerm vt;
{
  unsigned int h;
  MQ_NameVar nv;
  MQ_Obj o;
  int i;

  switch (vt->type)
    {
    case TT_NameVar:
      nv = (MQ_NameVar)vt;
      h = nv->number;
      break;

    case TT_Obj:
      o = (MQ_Obj)vt;
      h = (unsigned int)o->atom;
      for (i=0; i< o->arity; i++)
	h += (unsigned int)o->attr[i].label;
      break;

    default:
      fatal ("hash_vterm\n");
      break;
    }
  return h;
}

static
unsigned int hash_vterm_list (vl)
     MQ_VTermList vl;
{
  unsigned int h;

  for (h=0; vl; vl=vl->next)
    h += hash_vterm (vl->vterm);
  return h;
}

static
unsigned int hash_cnstr (cnstr)
     MQ_Constraint cnstr;
{
  unsigned int h;

  h = cnstr->rel;
  h += hash_term (cnstr->term);
  h += hash_vterm (cnstr->vterm);
  return h;
}

static
unsigned int hash_cnstrs (cnstrs)
     MQ_Constraints cnstrs;
{
  unsigned int h;

  for (h=0; cnstrs!=mQ_void_cnstrs; cnstrs=cnstrs->next)
    h += hash_cnstr (cnstrs->cnstr);
  return h;
}
