/* Micro Quixote           */
/* Copyright (C) 1993 ICOT */
/* Written by gniibe       */
/* $Id: lex.c,v 1.4 1994/11/14 08:37:40 m-gniibe Exp $ */

/* lexical analizer */

#include <stdio.h>
#include <string.h>
#include "obstack.h"
#include "mq.h"
#include "internal.h"
#include "tree.h"
#include "extern.h"
#include "parse_t.h"

/* function prototype for debugging */
static void help _P((void));
static int yygetc _P((void));
static void output_prompt _P((void));
static int mq_mode;

#define YYGETC()    yygetc ()
#define YYUNGETC()  ungetchar=c

struct keyword_table {
  char *keyword;
  int   token;
};

static struct keyword_table keyword_list[] = {
  {"bottom",      ATOM},
  {"clear",       CLEAR},
  {"end",         END},
  {"load",        LOAD},
  {"program",     PROGRAM},
  {"query",       QUERY},
  {"quit",        QUIT},
  {"rule",        RULE},
  {"set",         SET},
  {"show",        SHOW},
  {"subsumption", SUBSUMPTION},
  {"top",         ATOM},
  {"true",        LEX_TRUE},
};
#define NR_KEYWOARDS (sizeof keyword_list/sizeof(struct keyword_table))

typedef struct FileList_Rec {
  FILE *fp;
  struct FileList_Rec *next;
  char filename[1];
} FileList_Rec, *FileList;

static FileList file_list;

static struct obstack lex_obstack;
struct obstack *mm_lex;
unsigned char *lex_first_obj;

#define LEX_INITIAL_LEN 256

int last_char_is_newline;
unsigned char *yytext;
static int ungetchar;
static int allocated;

int lineno;
MQ_Atom external_cnstr_op;
static FILE *fp;
static unsigned char *string;

void
init_lex ()
{
  fp = NULL;
  string = NULL;
  file_list = NULL;
  mm_lex = &lex_obstack;
  obstack_begin (mm_lex, 0);
  lex_first_obj = (unsigned char *)obstack_alloc (mm_lex, 0);
  allocated = LEX_INITIAL_LEN;
  yytext = xmalloc (allocated);
  mq_mode = MODE_COMMAND;
  last_char_is_newline = 1;
  lineno = 1;
  reading_object = FALSE;
  ungetchar = 0;
  force_interactive = FALSE;
}

#define proceed_while(cond)                      \
  while (cond)                                   \
    {                                            \
      if (p >= yytext+allocated-1)               \
	{                                        \
	  allocated *= 2;                        \
	  yytext = xrealloc (yytext, allocated); \
	}                                        \
      *p++ = c;                                  \
      c = YYGETC();                              \
    }                                            \
  YYUNGETC ();                                   \
  *p = 0

#define DO_NEWLINE                               \
  if (error_recovering)                          \
    {						 \
      switch (mq_mode)				 \
	{					 \
	case MODE_COMMAND:			 \
	case MODE_PROGRAM:			 \
	  return TERMINATER;			 \
	  break;				 \
						 \
	case MODE_QUERY:			 \
	case MODE_QUERY_ONCE:			 \
	case MODE_EXPECT_PRD:			 \
	  return '.';				 \
	  break;				 \
						 \
	default:				 \
	  fatal ("wrong parser state\n");	 \
	  break;				 \
	}					 \
    }						 \
  last_char_is_newline++;			 \
  output_prompt (); 				 \
  lineno++; 					 \
  goto loop

int
yylex (lvalp)
     YYSTYPE *lvalp;
{
  int c;
  unsigned char *p;
  int i;

 loop:
  c = YYGETC ();
  if (c == EOF || c == PROGRAM_MODE || c == QUERY_MODE || c == QUERY_ONCE_MODE)
    return c;

  if (c == ' ' || c == '\t' || c == '\f')
    goto loop;

  if (c == '\n')
    {
      if (reading_object)
	{
	  YYUNGETC ();
	  return EOF;
	}
      DO_NEWLINE;
    }

  if (c == '%')
    while (1)
      {
	c = YYGETC ();
	if (c == EOF)
	  return EOF;

	if (c == '\n')
	  {
	    DO_NEWLINE;
	  }
      }

  if (c == '/')
    {
      c = YYGETC ();
      if (c == '*')
	{
	  while (1)
	    {
	      c = YYGETC ();
	      if (c == EOF)
		return EOF;
	      if (c != '*')
		continue;
	      c = YYGETC ();
	      if (c == '/')
		goto loop;
	    }
	}
      else
	{
	  YYUNGETC ();
	  return '/';
	}
    }

  last_char_is_newline = 0;
  if (c == '.')
    {
      c = YYGETC ();
      YYUNGETC ();
      if (c == EOF || c == '\n' || c == ' '  || c == '\t'  || c == '\f')
	return '.';
      return DOT;
    }

  if (c == '&') /* keyword */
    {
      p = yytext;
      *p++ = c;
      c = YYGETC ();
      proceed_while ((c != EOF) &&
		     (((c >= 'A') && (c <= 'Z'))
		      || ((c >= 'a') && (c <= 'z'))
		      || ((c >= '0') && (c <= '9'))
		      || (c == '_')));
      /* Well, just a linear search... */
      for (i = 0; i < NR_KEYWOARDS; i++)
	if (!strcmp (yytext+1, keyword_list[i].keyword))
	  {
	    if (keyword_list[i].token == ATOM)
	      lvalp->atom = intern_atom (yytext);

	    return keyword_list[i].token;
	  }

      return ERROR_TOKEN;
    }

  if (c == '#')
    {
      p = yytext;
      c = YYGETC ();
      proceed_while (c != EOF && c != '#');
      c = YYGETC ();
      if (yytext[0] == 0)
	lvalp->sr = ExternalExpr;
      else
	{
	  lvalp->sr = ExternalCnstr;
	  external_cnstr_op = intern_atom (yytext);
	}
      return SUBREL;
    }

  if (((c >= 'a') && (c <= 'z')) || ((c >= '0') && (c <= '9')) || (c >= 0x80))
    {
      p = yytext;
      *p++ = c;
      c = YYGETC ();
      proceed_while ((c != EOF) &&
		     (((c >= 'A') && (c <= 'Z'))
		      || ((c >= 'a') && (c <= 'z'))
		      || ((c >= '0') && (c <= '9'))
		      || (c == '_')
		      || (c >= 0x80)));
      lvalp->atom = intern_atom (yytext);
      return ATOM;
    }

  if (c == '"')
    {
      p = yytext;
      c = YYGETC ();
      proceed_while ((c != EOF) && (c != '"'));
      c = YYGETC ();
      lvalp->atom = intern_atom (yytext);
      return ATOM;
    }

  if (((c >= 'A') && (c <= 'Z')) || (c == '_'))
    {
      p = yytext;
      *p++ = c;
      c = YYGETC ();
      proceed_while ((c != EOF) &&
		     (((c >= 'A') && (c <= 'Z'))
		      || ((c >= 'a') && (c <= 'z'))
		      || ((c >= '0') && (c <= '9'))
		      || (c == '_')
		      || (c >= 0x80)));

      lvalp->variable = makeT_variable (yytext);
      return VARIABLE;
    }

  if (c == ';')
    {
      c = YYGETC ();
      if (c == ';')
	return TERMINATER;
      YYUNGETC ();
      return ';';
    }

  if (c == '-')
    {
      c = YYGETC ();
      if (c == '>')
	{
	  lvalp->ar = InstanceOf;
	  return ATTRREL;
	}
      YYUNGETC ();
      return '-';
    }

  if (c == '=')
    {
      c = YYGETC ();
      if (c == '=')
	{
	  lvalp->sr = Congruent;
	  return SUBREL;
	}
      if (c == '<')
	{
	  lvalp->sr = Supersumes;
	  return SUBREL;
	}
      YYUNGETC ();
      lvalp->ar = Equal;
      return ATTRREL;
    }

  if (c == '|')
    {
      c = YYGETC ();
      if (c == '|')
	return CONSTRAINED;
      YYUNGETC ();
      return '|';
    }

  if (c == '>')
    {
      c = YYGETC ();
      if (c == '=')
	{
	  lvalp->sr = Subsumes;
	  return SUBREL;
	}
      YYUNGETC ();
      return '>';
    }

  if (c == '<')
    {
      c = YYGETC ();
      if (c == '=')
	return COMMITS;
      if (c == '-')
	{
	  lvalp->ar = AbstractOf;
	  return ATTRREL;
	}
      YYUNGETC ();
      return '<';
    }

  if (c == '?')
    {
      c = YYGETC ();
      if (c == '-')
	return SOLVE;
      YYUNGETC ();
      return '?';
    }

  return c;
}

void
gobble_newline ()
{
  int c;

 loop:
  c = YYGETC ();
  if (c == EOF)
    return;

  if (c == ' ' || c == '\t' || c == '\f')
    goto loop;

  if (c == '\n')
    ;
  else
    YYUNGETC ();
}

void
read_line_to_yytext ()
{
  int c;
  unsigned char *p;

  yytext[0] = 0;
  p = yytext;
  gobble_newline ();
  c = YYGETC ();

  proceed_while ((c != EOF) && (c != '\n'));
  /* don't gobble the newline, this makes prompt. */
}

static void
help ()
{
  switch (mq_mode)
    {
    case MODE_COMMAND:
      printf ("\tThis is COMMAND mode.  Following commands are valid.\n");
      printf ("\n");
      printf ("\t&program;; (end with &end.)\n");
      printf ("\t\tInput program.\n");
      printf ("\t\t(If you input programs repeatedly,\n");
      printf ("\t\t they are inserted incrementally into the database.)\n");
      printf ("\t&quit;;\n");
      printf ("\t\tQuit Micro Quixote.\n");
      printf ("\t&query;;\n");
      printf ("\t\tGo to QUERY mode.\n");
      printf ("\t&set [ option [value] ];;\n");
      printf ("\t\tset option to the value.\n");
      printf ("\t&clear [ rule|subsumption ];;\n");
      printf ("\t\tclear rules/subsumptions.\n");
      printf ("\t&show [ rule|subsumption|program|version ];;\n");
      printf ("\t\tshow rule/subsumption/program.\n");
      printf ("\n");

      break;

    case MODE_PROGRAM:
      printf ("\tThis is PROGRAM mode.  Following commands are valid.\n");
      printf ("\n");
      printf ("\ta >= b;;\n");
      printf ("\t\tadd a subsumption relation `a subsumes b' into database.\n");
      printf ("\tapple/[color=green];;\n");
      printf ("\t\tadd a rule `apple with color=green' into database.\n");
      printf ("\ta <= b;;\n");
      printf ("\t\tadd a rule `if b then a' into database.\n");
      printf ("\t&end.\n");
      printf ("\t\tEnd program and go back to COMMAND mode.\n");
      break;

    case MODE_QUERY:
    case MODE_QUERY_ONCE:
      printf ("\tThis is QUERY mode.  Please input query.\n");
      printf ("\n");
      printf ("\tapple/[color=X].\n");
      printf ("\t\task Quixote for color of apple.\n");
      printf ("\t&end.\n");
      printf ("\t\tEnd query and go back to COMMAND mode.\n");
      break;

    case MODE_EXPECT_PRD:
      printf ("\t\n");
      break;

    default:
      fatal ("wrong parser state\n");
      break;
    }
  printf ("\n");
}

void
output_prompt_at_bol ()
{
  if (!interactive)
    return;

  switch (mq_mode)
    {
    case MODE_COMMAND:
      printf ("&> ");
      break;

    case MODE_PROGRAM:
      printf ("program> ");
      break;

    case MODE_QUERY_ONCE:
      break;

    case MODE_QUERY:
      printf ("?- ");
      break;

    case MODE_EXPECT_PRD:
      printf (".> ");
      break;

    default:
      fatal ("wrong parser state\n");
      break;
    }
  fflush (stdout);
}

static void
output_prompt ()
{
  if (!interactive)
    return;

  if (last_char_is_newline > 3)
    {
      help ();
      last_char_is_newline =1;
    }

  switch (mq_mode)
    {
    case MODE_COMMAND:
    case MODE_PROGRAM:
      printf (";;> ");
      break;

    case MODE_QUERY_ONCE:
    case MODE_QUERY:
    case MODE_EXPECT_PRD:
      printf (".> ");
      break;
    }
  fflush (stdout);
}

void
push_file (filename)
     char *filename;
{
  FileList new, fl;

  new = (FileList)obstack_alloc (mm_lex,
				 sizeof (FileList_Rec)+strlen (filename));
  file_list = new;
  strcpy (new->filename, filename);
  new->next = file_list;
  /* save the current file pointer */
  new->fp = fp;

  /* then, open the file */
  fp = fopen (file_list->filename, "r");
  if (fp == NULL)
    {
      fprintf (stderr, "Can't read file: %s\n", file_list->filename);
      fl = file_list;
      fp = file_list->fp;
      file_list = file_list->next;
      obstack_free (mm_lex, fl);
    }
  else
    {
      if (!mq_as_library)
	printf ("Loading \"%s\"...\n", file_list->filename);
      init_interactive (FALSE);
    }
}

int
pop_file ()
{
  FileList fl;

  if (file_list)
    {
      if (!mq_as_library)
	printf ("Loading \"%s\"...done\n", file_list->filename);
      fclose (fp);
      fl = file_list;
      fp = file_list->fp;	/* pop back to the file */
      file_list = file_list->next;
      obstack_free (mm_lex, fl);
      if (fp)
	return 1;
      else
	return 0;
    }
  else
    return 0;
}

void
set_mode_for_lex (mode)
     int mode;
{
  mq_mode = mode;
  switch (mq_mode)
    {
    case MODE_PROGRAM:
      ungetchar = PROGRAM_MODE;
      break;
    case MODE_QUERY_ONCE:
      ungetchar = QUERY_ONCE_MODE;
      break;
    case MODE_QUERY:
      ungetchar = QUERY_MODE;
      break;

    case MODE_EXPECT_PRD:
    case MODE_COMMAND:
      break;
    }
}

void
set_input_file (file_p)
     FILE *file_p;
{
  ungetchar = 0;
  fp = file_p;
}

void
set_input_string (s)
     char *s;
{
  ungetchar = 0;
  string = s;
}

static int
yygetc ()
{
  int c;

  if (ungetchar)
    {
      c = ungetchar;
      ungetchar = 0;
      return c;
    }

  if (fp)
    c = fgetc (fp);
  else if (string)
    {
      c = *string++;
      if (c == '\0')
	c = EOF;
    }
  else
    c = EOF;

  return c;
}

int
fetch_char ()
{
  int c;

  c = YYGETC ();
  YYUNGETC ();
  return c;
}
