/* Micro Quixote                 */
/* Copyright (C) 1993, 1994 ICOT */
/* Written by gniibe             */
/* $Id: cs.c,v 1.2 1994/09/19 02:05:58 m-gniibe Exp $ */

/* interface to constraint solvers */

#include <stdio.h>
#include "obstack.h"
#include "mq.h"
#include "internal.h"
#include "extern.h"

int constrain_failed;
MQ_Constraints dot_cnstrs, dot_asmpts, dot_hchcks, dot_bchcks;
MQ_Constraints sub_cnstrs, sub_asmpts, sub_hchcks, sub_bchcks;
MQ_Constraints ext_cnstrs, ext_asmpts, ext_hchcks, ext_bchcks;

MQ_CnstrsAsmpts cnstrs_asmpts;
MQ_Constraint mQ_void_cnstr;
MQ_Constraints mQ_void_cnstrs;
MQ_CnstrsAsmpts mQ_void_cnstrs_asmpts;

static struct obstack *mm_cnstrs;
static struct obstack cnstrs_obstack;
static unsigned char *cnstrs_first_obj;

typedef struct VariableProtect_Rec {
  struct VariableProtect_Rec *next;

  MQ_Var var;
  MQ_VTermAddrList vterm_addr_list;
} VariableProtect_Rec, *VariableProtect;

static VariableProtect vp;

/* function prototype for debugging */
static void constrain_with_head_and_body_front _P((MQ_Goal));
static void constrain_with_head_and_body_back _P((MQ_Goal));
static void constraint_solve_cnstrs _P((void));
static void constraint_solve_asmpts _P((void));
static void constraint_solve_hchcks _P((void));
static void constraint_solve_bchcks _P((void));
static void add_cnstr_to_cnstrs _P((MQ_Constraint));
static void add_cnstr_to_hchcks _P((MQ_Constraint));
static void add_cnstr_to_bchcks _P((MQ_Constraint));
static MQ_Constraint cs_tangle_cnstr _P((MQ_Constraint));
static MQ_Constraints cs_tangle_cnstrs _P((MQ_Constraints));
static MQ_Constraints copy_cnstrs _P((MQ_Constraints));
static void mark_variable _P((MQ_Var));
static void mark_variables_in_vterm _P((MQ_VTerm));
static void mark_variables_in_cnstrs _P((MQ_Constraints));
static MQ_CnstrsAsmpts make_cnstrs_asmpts _P((void));
static int cs_sub _P((void));
static void asmpts_generation _P((MQ_Constraint cnstr));

static int
cs_sub ()
{
  if (binding_changed == FALSE)
    return SUCCESS;

  while (1)
    {
      binding_changed = FALSE;
      constraint_solve_cnstrs ();
      if (constrain_failed)
	return FAILURE;

      constraint_solve_asmpts ();
      if (constrain_failed)
	return FAILURE;

      constraint_solve_hchcks ();
      if (constrain_failed)
	return FAILURE;

      constraint_solve_bchcks ();
      if (constrain_failed)
	return FAILURE;

      if (!binding_changed)
	break;
    }
  return SUCCESS;
}

/* 
  CONSTRAIN_FRONT:

  FRONT constraint solver.  This routine is called BEFORE the execution
  of body.

  Add head constraints and body constraints to the list `checks'.
  Then, test that `cnstrs' + `checks' doesn't deduce to contradiction.
  And also test that `asmpts' + `checks' doesn't deduce to contradiction.

  Doesn't change cnstrs, asmpt in the verification.

  Take G of goal, and returns FAILURE on failure, and SUCCESS on success.
 */

int
constrain_front (g)
     MQ_Goal g;
{
  MQ_CnstrsAsmpts new;
  VariableProtect vp1;

  if (mq_opt_constrain == 0)
    return SUCCESS;

  vp = NULL;
  constrain_failed = FALSE;

  /* Protect variables so that they don't refer the memory in mm_cnstrs after
     mm_cnstrs is garbage collected. */
  /* `mark_variables' marks var->value as PROTECTED,
     and make list of marked variables in VP. */
  mark_variables_in_cnstrs (g->head_cnstrs);
  mark_variables_in_cnstrs (g->body_cnstrs);
  mark_variables_in_cnstrs (cnstrs_asmpts->dot_cnstrs);
  mark_variables_in_cnstrs (cnstrs_asmpts->sub_cnstrs);
  mark_variables_in_cnstrs (cnstrs_asmpts->ext_cnstrs);
  mark_variables_in_cnstrs (cnstrs_asmpts->dot_asmpts);
  mark_variables_in_cnstrs (cnstrs_asmpts->sub_asmpts);
  mark_variables_in_cnstrs (cnstrs_asmpts->ext_asmpts);
  mark_variables_in_cnstrs (cnstrs_asmpts->dot_hchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->sub_hchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->ext_hchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->dot_bchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->sub_bchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->ext_bchcks);

  /* And we register these variable bindings so that backtracking works
     correctly */
  for (vp1=vp; vp1; vp1=vp1->next)
    {
      unwind_protect_variable_in_cnstrs (vp1->var, vp1->vterm_addr_list);
      vp1->var->value = NULL;	/* clear PROTECTED mark */
    }

  /* Copy constraints (in mm_exec) to mm_cnstrs.
     mm_cnstrs is temporary strage for constraints solving. */
  mm_current = mm_cnstrs;
  dot_cnstrs = cs_tangle_cnstrs (cnstrs_asmpts->dot_cnstrs);
  sub_cnstrs = cs_tangle_cnstrs (cnstrs_asmpts->sub_cnstrs);
  ext_cnstrs = cs_tangle_cnstrs (cnstrs_asmpts->ext_cnstrs);
  dot_asmpts = cs_tangle_cnstrs (cnstrs_asmpts->dot_asmpts);
  sub_asmpts = cs_tangle_cnstrs (cnstrs_asmpts->sub_asmpts);
  ext_asmpts = cs_tangle_cnstrs (cnstrs_asmpts->ext_asmpts);
  dot_hchcks = cs_tangle_cnstrs (cnstrs_asmpts->dot_hchcks);
  sub_hchcks = cs_tangle_cnstrs (cnstrs_asmpts->sub_hchcks);
  ext_hchcks = cs_tangle_cnstrs (cnstrs_asmpts->ext_hchcks);
  dot_bchcks = cs_tangle_cnstrs (cnstrs_asmpts->dot_bchcks);
  sub_bchcks = cs_tangle_cnstrs (cnstrs_asmpts->sub_bchcks);
  ext_bchcks = cs_tangle_cnstrs (cnstrs_asmpts->ext_bchcks);

  /* Because of unification, we need to invoke constraint solver */
  /* Force to invoke constraint solver */
  binding_changed = TRUE;
  if (cs_sub () == FAILURE)
    goto fail;

  constrain_with_head_and_body_front (g);
  if (constrain_failed)
    goto fail;

  /* SUCCESS */
  /* Restore variables. */
  for (vp1=vp; vp1; vp1=vp1->next)
    vp1->var->vterm_addr_list = vp1->vterm_addr_list;
  mm_current = mm_exec;
  new = make_cnstrs_asmpts ();
  /* Copy constraints (in mm_cnstrs) to mm_exec. Scavenging. */
  new->dot_cnstrs = cnstrs_asmpts->dot_cnstrs;
  new->sub_cnstrs = cnstrs_asmpts->sub_cnstrs;
  new->ext_cnstrs = cnstrs_asmpts->ext_cnstrs;
  new->dot_asmpts = cnstrs_asmpts->dot_asmpts;
  new->sub_asmpts = cnstrs_asmpts->sub_asmpts;
  new->ext_asmpts = cnstrs_asmpts->ext_asmpts;
  new->dot_hchcks = cs_tangle_cnstrs (dot_hchcks);
  new->sub_hchcks = cs_tangle_cnstrs (sub_hchcks);
  new->ext_hchcks = cs_tangle_cnstrs (ext_hchcks);
  new->dot_bchcks = cs_tangle_cnstrs (dot_bchcks);
  new->sub_bchcks = cs_tangle_cnstrs (sub_bchcks);
  new->ext_bchcks = cs_tangle_cnstrs (ext_bchcks);
  cnstrs_asmpts = new;
  /* free the memory used for constraints solving */
  obstack_free (mm_cnstrs, cnstrs_first_obj);
  cnstrs_first_obj = (unsigned char *)obstack_alloc (mm_cnstrs, 0);
  return SUCCESS;

 fail:
  /* restore variables */
  for (vp1=vp; vp1; vp1=vp1->next)
    vp1->var->vterm_addr_list = vp1->vterm_addr_list;
  mm_current = mm_exec;
  /* free the memory used for constraints solving */
  obstack_free (mm_cnstrs, cnstrs_first_obj);
  cnstrs_first_obj = (unsigned char *)obstack_alloc (mm_cnstrs, 0);
  return FAILURE;
}


static int
add_to_cnstrs (cnstrs)
     MQ_Constraints cnstrs;
{
  MQ_Constraints cs;

  for (cs = cnstrs; cs!=mQ_void_cnstrs; cs = cs->next)
    {
      binding_changed = FALSE;

      add_cnstr_to_cnstrs (cs->cnstr);
      if (constrain_failed)
	return FAILURE;

      if (cs_sub () == FAILURE)
	return FAILURE;
    }
  return SUCCESS;
}

/*
  Add head constraints and body constraints to constrain_checks.
  Take an argument G of goal.
 */
static void
constrain_with_head_and_body_front (g)
     MQ_Goal g;
{
  MQ_Constraints cs;

  /* Add head constraints to checks */
  for (cs = g->head_cnstrs; cs!=mQ_void_cnstrs; cs = cs->next)
    {
      binding_changed = FALSE;
      add_cnstr_to_hchcks (cs->cnstr);
      if (constrain_failed)
	return;

      if (cs_sub () == FAILURE)
	return;
    }

  /* Add body constraints to checks */
  for (cs = g->body_cnstrs; cs!=mQ_void_cnstrs; cs = cs->next)
    {
      binding_changed = FALSE;
      add_cnstr_to_bchcks (cs->cnstr);
      if (constrain_failed)
	return;

      if (cs_sub () == FAILURE)
	return;
    }

  /* add checks */
  if (add_to_cnstrs (dot_hchcks) == FAILURE
      || add_to_cnstrs (sub_hchcks) == FAILURE
      || add_to_cnstrs (ext_hchcks) == FAILURE
      || add_to_cnstrs (dot_bchcks) == FAILURE
      || add_to_cnstrs (ext_bchcks) == FAILURE
      || add_to_cnstrs (sub_bchcks) == FAILURE)
    return;
}

int
constrain_back (g)
     MQ_Goal g;
{
  MQ_CnstrsAsmpts new;
  VariableProtect vp1;

  if (mq_opt_constrain == 0)
    return SUCCESS;

  vp = NULL;
  constrain_failed = FALSE;

  /* protect variables against GC */
  mark_variables_in_cnstrs (g->head_cnstrs);
  mark_variables_in_cnstrs (g->body_cnstrs);
  mark_variables_in_cnstrs (cnstrs_asmpts->dot_cnstrs);
  mark_variables_in_cnstrs (cnstrs_asmpts->sub_cnstrs);
  mark_variables_in_cnstrs (cnstrs_asmpts->ext_cnstrs);
  mark_variables_in_cnstrs (cnstrs_asmpts->dot_asmpts);
  mark_variables_in_cnstrs (cnstrs_asmpts->sub_asmpts);
  mark_variables_in_cnstrs (cnstrs_asmpts->ext_asmpts);
  mark_variables_in_cnstrs (cnstrs_asmpts->dot_hchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->sub_hchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->ext_hchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->dot_bchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->sub_bchcks);
  mark_variables_in_cnstrs (cnstrs_asmpts->ext_bchcks);
  for (vp1=vp; vp1; vp1=vp1->next)
    {
      unwind_protect_variable_in_cnstrs (vp1->var, vp1->vterm_addr_list);
      vp1->var->value = NULL;	/* clear PROTECTED mark */
    }

  mm_current = mm_cnstrs;
  dot_cnstrs = cs_tangle_cnstrs (cnstrs_asmpts->dot_cnstrs);
  sub_cnstrs = cs_tangle_cnstrs (cnstrs_asmpts->sub_cnstrs);
  ext_cnstrs = cs_tangle_cnstrs (cnstrs_asmpts->ext_cnstrs);
  dot_asmpts = cs_tangle_cnstrs (cnstrs_asmpts->dot_asmpts);
  sub_asmpts = cs_tangle_cnstrs (cnstrs_asmpts->sub_asmpts);
  ext_asmpts = cs_tangle_cnstrs (cnstrs_asmpts->ext_asmpts);
  dot_hchcks = cs_tangle_cnstrs (cnstrs_asmpts->dot_hchcks);
  sub_hchcks = cs_tangle_cnstrs (cnstrs_asmpts->sub_hchcks);
  ext_hchcks = cs_tangle_cnstrs (cnstrs_asmpts->ext_hchcks);
  dot_bchcks = cs_tangle_cnstrs (cnstrs_asmpts->dot_bchcks);
  sub_bchcks = cs_tangle_cnstrs (cnstrs_asmpts->sub_bchcks);
  ext_bchcks = cs_tangle_cnstrs (cnstrs_asmpts->ext_bchcks);

  binding_changed = TRUE;
  if (cs_sub () == FAILURE)
    goto fail;

  constrain_with_head_and_body_back (g);
  if (constrain_failed)
    goto fail;

  /* SUCCESS */
  for (vp1=vp; vp1; vp1=vp1->next)
    vp1->var->vterm_addr_list = vp1->vterm_addr_list;
  mm_current = mm_exec;
  new = make_cnstrs_asmpts ();
  /* scavenging */
  new->dot_cnstrs = cs_tangle_cnstrs (dot_cnstrs);
  new->sub_cnstrs = cs_tangle_cnstrs (sub_cnstrs);
  new->ext_cnstrs = cs_tangle_cnstrs (ext_cnstrs);
  new->dot_asmpts = cs_tangle_cnstrs (dot_asmpts);
  new->sub_asmpts = cs_tangle_cnstrs (sub_asmpts);
  new->ext_asmpts = cs_tangle_cnstrs (ext_asmpts);
  new->dot_hchcks = cnstrs_asmpts->dot_hchcks;
  new->sub_hchcks = cnstrs_asmpts->sub_hchcks;
  new->ext_hchcks = cnstrs_asmpts->ext_hchcks;
  new->dot_bchcks = cnstrs_asmpts->dot_bchcks;
  new->sub_bchcks = cnstrs_asmpts->sub_bchcks;
  new->ext_bchcks = cnstrs_asmpts->ext_bchcks;
  cnstrs_asmpts = new;
  obstack_free (mm_cnstrs, cnstrs_first_obj); /* garbage collection */
  cnstrs_first_obj = (unsigned char *)obstack_alloc (mm_cnstrs, 0);
  return SUCCESS;

 fail:
  for (vp1=vp; vp1; vp1=vp1->next)
    vp1->var->vterm_addr_list = vp1->vterm_addr_list;
  mm_current = mm_exec;
  obstack_free (mm_cnstrs, cnstrs_first_obj); /* garbage collection */
  cnstrs_first_obj = (unsigned char *)obstack_alloc (mm_cnstrs, 0);
  return FAILURE;
}

static void
constrain_with_head_and_body_back (g)
     MQ_Goal g;
{
  MQ_Constraints cs;

  for (cs = g->body_cnstrs; cs!=mQ_void_cnstrs; cs = cs->next)
    asmpts_generation (cs->cnstr);

  for (cs = g->head_cnstrs; cs!=mQ_void_cnstrs; cs = cs->next)
    {
      binding_changed = FALSE;

      add_cnstr_to_cnstrs (cs->cnstr);
      if (constrain_failed)
	return;

      if (cs_sub () == FAILURE)
	return;
    }
}

static void
constraint_solve_cnstrs ()
{
  int binding_changed_saved = binding_changed;
  int binding_changed_1;

  binding_changed_1 = FALSE;
  binding_changed = FALSE;

  constraint_solve_dot_cnstrs ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 1: 0);

  binding_changed = FALSE;
  constraint_solve_sub_cnstrs ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 2: 0);

  binding_changed = FALSE;
  constraint_solve_ext_cnstrs ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 4: 0);

  while (binding_changed_1)
    {
      binding_changed_1 = FALSE;
      binding_changed_saved = TRUE;

      if (binding_changed_1 & 6)
	{
	  binding_changed = FALSE;
	  constraint_solve_dot_cnstrs ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 1: 0);
	}

      if (binding_changed_1 & 5)
	{
	  binding_changed = FALSE;
	  constraint_solve_sub_cnstrs ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 2: 0);
	}

      if (binding_changed_1 & 3)
	{
	  binding_changed = FALSE;
	  constraint_solve_ext_cnstrs ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 4: 0);
	}
    }
  binding_changed = binding_changed_saved;
  return;
}

static void
constraint_solve_asmpts ()
{
  int binding_changed_saved = binding_changed;
  int binding_changed_1;

  binding_changed_1 = FALSE;
  binding_changed = FALSE;

  constraint_solve_dot_asmpts ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 1: 0);

  binding_changed = FALSE;
  constraint_solve_sub_asmpts ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 2: 0);

  binding_changed = FALSE;
  constraint_solve_ext_asmpts ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 4: 0);

  while (binding_changed_1)
    {
      binding_changed_1 = FALSE;
      binding_changed_saved = TRUE;

      if (binding_changed_1 & 6)
	{
	  binding_changed = FALSE;
	  constraint_solve_dot_asmpts ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 1: 0);
	}

      if (binding_changed_1 & 5)
	{
	  binding_changed = FALSE;
	  constraint_solve_sub_asmpts ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 2: 0);
	}

      if (binding_changed_1 & 3)
	{
	  binding_changed = FALSE;
	  constraint_solve_ext_asmpts ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 4: 0);
	}
    }
  binding_changed = binding_changed_saved;
  return;
}

static void
constraint_solve_hchcks ()
{
  int binding_changed_saved = binding_changed;
  int binding_changed_1;

  binding_changed_1 = FALSE;
  binding_changed = FALSE;

  constraint_solve_dot_hchcks ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 1: 0);

  binding_changed = FALSE;
  constraint_solve_sub_hchcks ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 2: 0);

  binding_changed = FALSE;
  constraint_solve_ext_hchcks ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 4: 0);

  while (binding_changed_1)
    {
      binding_changed_1 = FALSE;
      binding_changed_saved = TRUE;

      if (binding_changed_1 & 6)
	{
	  binding_changed = FALSE;
	  constraint_solve_dot_hchcks ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 1: 0);
	}

      if (binding_changed_1 & 5)
	{
	  binding_changed = FALSE;
	  constraint_solve_sub_hchcks ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 2: 0);
	}

      if (binding_changed_1 & 3)
	{
	  binding_changed = FALSE;
	  constraint_solve_ext_hchcks ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 4: 0);
	}
    }
  binding_changed = binding_changed_saved;
  return;
}

static void
constraint_solve_bchcks ()
{
  int binding_changed_saved = binding_changed;
  int binding_changed_1;

  binding_changed_1 = FALSE;
  binding_changed = FALSE;

  constraint_solve_dot_bchcks ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 1: 0);

  binding_changed = FALSE;
  constraint_solve_sub_bchcks ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 2: 0);

  binding_changed = FALSE;
  constraint_solve_ext_bchcks ();
  if (constrain_failed)
    return;
  binding_changed_1 |= (binding_changed? 4: 0);

  while (binding_changed_1)
    {
      binding_changed_1 = FALSE;
      binding_changed_saved = TRUE;

      if (binding_changed_1 & 6)
	{
	  binding_changed = FALSE;
	  constraint_solve_dot_bchcks ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 1: 0);
	}

      if (binding_changed_1 & 5)
	{
	  binding_changed = FALSE;
	  constraint_solve_sub_bchcks ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 2: 0);
	}

      if (binding_changed_1 & 3)
	{
	  binding_changed = FALSE;
	  constraint_solve_ext_bchcks ();
	  if (constrain_failed)
	    return;
	  binding_changed_1 |= (binding_changed? 4: 0);
	}
    }
  binding_changed = binding_changed_saved;
  return;
}

static void
add_cnstr_to_cnstrs (cnstr)
     MQ_Constraint cnstr;
{
  int binding_changed_saved = binding_changed;

  binding_changed = FALSE;
  switch (cnstr->rel)
    {
    case DotCongruent:
    case DotCongruentObj:
    case DotCongruentVar:
      add_cnstr_to_dot_cnstrs (cnstr);
      break;

    case Subsumes:
    case SubsumesVarVar:
    case SubsumesVarObj:
    case SubsumesObjVar:
      add_cnstr_to_sub_cnstrs (cnstr);
      break;

    case Congruent:
      if (unify ((MQ_VTerm *)&cnstr->term, &cnstr->vterm) == FAILURE)
	constrain_failed = TRUE;
      break;

    case ExternalExpr:
    case ExternalCnstr:
      add_cnstr_to_ext_cnstrs (cnstr);
      break;

    default:
      fatal ("add_cnstr_to_csntrs\n");
      break;
    }
  if (constrain_failed || !binding_changed)
    {
      binding_changed = binding_changed_saved;
      return;
    }
  constraint_solve_cnstrs ();
  binding_changed = TRUE;
}

static void
add_cnstr_to_hchcks (cnstr)
     MQ_Constraint cnstr;
{
  int binding_changed_saved = binding_changed;

  binding_changed = FALSE;
  switch (cnstr->rel)
    {
    case DotCongruent:
    case DotCongruentObj:
    case DotCongruentVar:
      add_cnstr_to_dot_hchcks (cnstr);
      break;

    case Subsumes:
    case SubsumesVarVar:
    case SubsumesVarObj:
    case SubsumesObjVar:
      add_cnstr_to_sub_hchcks(cnstr);
      break;

    case Congruent:
      if (unify ((MQ_VTerm *)&cnstr->term, &cnstr->vterm) == FAILURE)
	constrain_failed = TRUE;
      break;

    case ExternalExpr:
    case ExternalCnstr:
      add_cnstr_to_ext_hchcks (cnstr);
      break;

    default:
      fatal ("add_cnstr_to_checks\n");
      break;
    }
  if (constrain_failed || !binding_changed)
    {
      binding_changed = binding_changed_saved;
      return;
    }
  constraint_solve_hchcks ();
  binding_changed = TRUE;
}

static void
add_cnstr_to_bchcks (cnstr)
     MQ_Constraint cnstr;
{
  int binding_changed_saved = binding_changed;

  binding_changed = FALSE;
  switch (cnstr->rel)
    {
    case DotCongruent:
    case DotCongruentObj:
    case DotCongruentVar:
      add_cnstr_to_dot_bchcks (cnstr);
      break;

    case Subsumes:
    case SubsumesVarVar:
    case SubsumesVarObj:
    case SubsumesObjVar:
      add_cnstr_to_sub_bchcks(cnstr);
      break;

    case Congruent:
      if (unify ((MQ_VTerm *)&cnstr->term, &cnstr->vterm) == FAILURE)
	constrain_failed = TRUE;
      break;

    case ExternalExpr:
    case ExternalCnstr:
      add_cnstr_to_ext_bchcks (cnstr);
      break;

    default:
      fatal ("add_cnstr_to_checks\n");
      break;
    }
  if (constrain_failed || !binding_changed)
    {
      binding_changed = binding_changed_saved;
      return;
    }
  constraint_solve_bchcks ();
  binding_changed = TRUE;
}

static void
asmpts_generation (cnstr)
     MQ_Constraint cnstr;
{
  switch (cnstr->rel)
    {
    case DotCongruent:
    case DotCongruentObj:
    case DotCongruentVar:
      if (!check_cnstr_in_dot_cnstrs (cnstr))
	add_cnstr_to_dot_asmpts (cnstr);
      break;

    case Subsumes:
    case SubsumesVarVar:
    case SubsumesVarObj:
    case SubsumesObjVar:
      if (!check_cnstr_in_sub_cnstrs (cnstr))
	add_cnstr_to_sub_asmpts (cnstr);
      break;

    case Congruent:
      break;

    case ExternalExpr:
    case ExternalCnstr:
      if (!check_cnstr_in_ext_cnstrs (cnstr))
	add_cnstr_to_ext_asmpts (cnstr);
      break;

    default:
      fatal ("asmpts_generation\n");
      break;
    }
}

static MQ_VarList
cs_tangle_var_list (vl)
     MQ_VarList vl;
{
  MQ_VarList next;

  if (vl == NULL)
    return NULL;
  next = cs_tangle_var_list (vl->next);
  return make_var_list (vl->var, next);
}

static MQ_Constraint
cs_tangle_cnstr (cnstr)
     MQ_Constraint cnstr;
{
  MQ_Constraint new;

  new = make_cnstr (cnstr->rel, cnstr->term, cnstr->vterm);
  return new;
}

static MQ_Constraints
cs_tangle_cnstrs (cnstrs)
     MQ_Constraints cnstrs;
{
  MQ_Constraints new, next;
  MQ_Constraint new_cnstr;

  if (cnstrs == mQ_void_cnstrs)
    return mQ_void_cnstrs;

  next = cs_tangle_cnstrs (cnstrs->next);

  if (cnstrs->cnstr->mark)
    new_cnstr = cs_tangle_cnstr (cnstrs->cnstr);
  else
    new_cnstr = cnstrs->cnstr;

  new = make_cnstrs (new_cnstr, next);
  new->l_var_list = cs_tangle_var_list (cnstrs->l_var_list);
  new->r_var_list = cs_tangle_var_list (cnstrs->r_var_list);
  return new;
}

static MQ_Constraints
copy_cnstrs (cnstrs)
     MQ_Constraints cnstrs;
{
  MQ_Constraints new, next;

  if (cnstrs == mQ_void_cnstrs)
    return mQ_void_cnstrs;

  next = copy_cnstrs (cnstrs->next);

  new = make_cnstrs (cnstrs->cnstr, next);
  new->l_var_list = cs_tangle_var_list (cnstrs->l_var_list);
  new->r_var_list = cs_tangle_var_list (cnstrs->r_var_list);
  return new;
}

static void
mark_variable (var)
     MQ_Var var;
{
  VariableProtect new;

  if (var->value == PROTECTED)
    return;
  if (var->value)
    fatal ("variable already has bound in mark_variables.\n");
  var->value = PROTECTED;
  new = (VariableProtect)
    obstack_alloc (mm_cnstrs, sizeof (VariableProtect_Rec));
  new->var = var;
  new->vterm_addr_list = var->vterm_addr_list;
  new->next = vp;
  vp = new;
}

static MQ_VTermList vterm_list_visited;

static void
mark_variables_in_vterm (vt)
     MQ_VTerm vt;
{
  MQ_Obj o;
  int i;
  MQ_VTermList vl;

  for (vl=vterm_list_visited; vl; vl=vl->next)
    if (vl->vterm == vt)
      return;
  vterm_list_visited = make_vterm_list (vterm_list_visited);
  vterm_list_visited->vterm = vt;

  switch (vt->type)
    {
    case TT_Var:
      mark_variable ((MQ_Var)vt);
      break;
    case TT_Obj:
      o = (MQ_Obj)vt;
      for (i=0; i< o->arity; i++)
	mark_variables_in_vterm (o->attr[i].vterm);
      break;
    default:
      fatal ("mark_variables_in_vterm.\n");
      break;
    }
}

static void 
mark_variables_in_cnstrs (cnstrs)
     MQ_Constraints cnstrs;
{
  MQ_Constraints cs;

  vterm_list_visited = NULL;
  for (cs=cnstrs; cs != mQ_void_cnstrs; cs= cs->next)
    {
      if (cs->cnstr->term->type == TT_Dot)
	;
      else
	mark_variables_in_vterm ((MQ_VTerm) cs->cnstr->term);
      mark_variables_in_vterm (cs->cnstr->vterm);
    }
}

MQ_Constraint
make_cnstr (rel, term, vterm)
     Rel rel;
     MQ_Term term;
     MQ_VTerm vterm;
{
  MQ_Constraint new;

  new = (MQ_Constraint) obstack_alloc (mm_current, sizeof (MQ_Constraint_Rec));
  new->rel = rel;
  new->term = term;
  new->vterm = vterm;
  if (term && executing && (term->type == TT_Var))
    ((MQ_Var)term)->vterm_addr_list
      = make_vterm_addr_list ((MQ_VTerm *)&new->term,
			      ((MQ_Var)term)->vterm_addr_list);
  if (vterm && executing && (vterm->type == TT_Var))
    ((MQ_Var)vterm)->vterm_addr_list
      = make_vterm_addr_list (&new->vterm, ((MQ_Var)vterm)->vterm_addr_list);
  if (mm_current == mm_cnstrs)
    new->mark = 1;
  else
    new->mark = 0;
  return new;
}

MQ_Constraints
make_cnstrs (cnstr, next)
     MQ_Constraint cnstr;
     MQ_Constraints next;
{
  MQ_Constraints new;

  new =(MQ_Constraints)obstack_alloc (mm_current, sizeof (MQ_Constraints_Rec));
  new->next = next;
  if (next && next != mQ_void_cnstrs)
    next->prev = new;
  new->cnstr = cnstr;
  new->prev = NULL;
  new->l_var_list = new->r_var_list = NULL;
  return new;
}

static MQ_CnstrsAsmpts
make_cnstrs_asmpts ()
{
  MQ_CnstrsAsmpts new;

  new = (MQ_CnstrsAsmpts)obstack_alloc (mm_exec,
					sizeof (MQ_CnstrsAsmpts_Rec));
  new->dot_cnstrs = new->dot_asmpts = new->dot_hchcks = new->dot_bchcks =
    new->sub_cnstrs = new->sub_asmpts = new->sub_hchcks = new->sub_bchcks =
      new->ext_cnstrs = new->ext_asmpts = new->ext_hchcks = new->ext_bchcks =
	NULL;
  return new;
}

void
init_constraints ()
{
  mm_cnstrs = &cnstrs_obstack;
  obstack_begin (mm_cnstrs, CONSTRAINTS_SIZE);

  mm_current = mm_cnstrs;
  mQ_void_cnstr = make_cnstr (Congruent, NULL, NULL);
  mQ_void_cnstrs = make_cnstrs (mQ_void_cnstr, NULL);
  mQ_void_cnstrs_asmpts = make_cnstrs_asmpts ();
  mQ_void_cnstrs_asmpts->dot_cnstrs = mQ_void_cnstrs_asmpts->dot_asmpts
    = mQ_void_cnstrs_asmpts->sub_cnstrs = mQ_void_cnstrs_asmpts->sub_asmpts
      = mQ_void_cnstrs_asmpts->ext_cnstrs = mQ_void_cnstrs_asmpts->ext_asmpts
	= mQ_void_cnstrs_asmpts->dot_hchcks
	  = mQ_void_cnstrs_asmpts->sub_hchcks
	    = mQ_void_cnstrs_asmpts->ext_hchcks
	      = mQ_void_cnstrs_asmpts->dot_bchcks
		= mQ_void_cnstrs_asmpts->sub_bchcks
		  = mQ_void_cnstrs_asmpts->ext_bchcks
	      = mQ_void_cnstrs;
  cnstrs_first_obj = (unsigned char *)obstack_alloc (mm_cnstrs, 0);
  cnstrs_asmpts = mQ_void_cnstrs_asmpts;
}
