/*
 *  Timer routines. (Used in the TFTP client program of Chapter12.)
 *
 *  These routines are structured so that there are only the following
 *  entry points:
 *
 *   void    timer_start()       start timer
 *   void    timer_stop()        stop  timer
 *   double  timer_getrtime()    return real(elapsed) time (second)
 *   double  timer_getutime()    return user time (second)
 *   double  timer_getstime()    return system time (second)
 *
 *  Purposely, there are no structures passed back and forth between
 *  the caller and these routines, and there are no include files
 *  required by the caller.
 */

#include  <stdio.h>

#ifdef BSD
#include  <sys/time.h>
#include  <sys/resource.h>
#endif

#ifdef SYS5
#include  <sys/types.h>
#include  <sys/times.h>
#include  <sys/param.h>     /* need the defintion of HZ */
#define   TICKS   HZ
#endif


#ifdef BSD
static struct timeval  time_start1, time_stop1;   /* for real time */
static struct rusage   ru_start, ru_stop;   /* for user & sys time */
#endif

#ifdef SYS5
static clock_t         time_start2, time_stop2;
static struct tms   tms_start, tms_stop;
clock_t                times();
#endif

/*
#ifdef SYS5
static long         time_start2, time_stop2;
static struct tms   tms_start, tms_stop; 
long                times();
#endif
*/

static double  start, stop, seconds;


/* Start the timer */
void  timer_start()
{
#ifdef BSD
    if(gettimeofday(&time_start1, (struct timezone *)0) < 0) 
	fprintf(stderr,"timer_start : gettimeofday() error");
    if(getrusage(RUSAGE_SELF, &ru_start) < 0) 
	fprintf(stderr,"timer_start : getrusage() error");
#endif /* BSD */

#ifdef SYS5
    if((time_start2=times(&tms_start)) == -1) 
	fprintf(stderr,"timer_start : times() error");
#endif /* SYS5 */
}


/* Stop the timer */
void  timer_stop()
{
#ifdef BSD
    if(getrusage(RUSAGE_SELF, &ru_stop) < 0) 
	fprintf(stderr,"timer_stop : getrusage() error");
    if(gettimeofday(&time_stop1, (struct timezone *)0) < 0) 
	fprintf(stderr,"timer_stop : gettimeofday() error");
#endif /* BSD */

#ifdef SYS5
    if((time_stop2=times(&tms_stop)) == -1) 
	fprintf(stderr,"timer_stop : times() error");
#endif /* SYS5 */
}


/* Return the user time in seconds. */
double  timer_getutime()
{
#ifdef BSD
    start = ((double)ru_start.ru_utime.tv_sec) * 1000000.0
	+ ru_start.ru_utime.tv_usec;
    stop = ((double)ru_stop.ru_utime.tv_sec) * 1000000.0
	+ ru_stop.ru_utime.tv_usec;
    seconds = (stop - start)/1000000.0;
#endif /* BSD */

#ifdef SYS5
    seconds = (double)(tms_stop.tms_utime - tms_start.tms_utime) /
	(double)TICKS;
#endif /* SYS5 */

    return seconds;
}



/* Return the system time in seconds. */
double  timer_getstime()
{
#ifdef BSD
    start = ((double)ru_start.ru_stime.tv_sec) * 1000000.0
	+ ru_start.ru_stime.tv_usec;
    stop = ((double)ru_stop.ru_stime.tv_sec) * 1000000.0
	+ ru_stop.ru_stime.tv_usec;
    seconds = (stop - start)/1000000.0;
#endif /* BSD */

#ifdef SYS5
    seconds = (double)(tms_stop.tms_stime - tms_start.tms_stime) /
	(double)TICKS;
#endif /* SYS5 */

    return seconds;
}


/* Return the real (elapsed) time in seconds. */
double  timer_getrtime()
{
#ifdef BSD
    start = ((double)time_start1.tv_sec) * 1000000.0
	+ time_start1.tv_usec;
    stop = ((double)time_stop1.tv_sec) * 1000000.0
	+ time_stop1.tv_usec;
    seconds = (stop - start)/1000000.0;
#endif /* BSD */

#ifdef SYS5
    seconds = (double)(time_stop2 - time_start2) / (double)TICKS;
#endif /* SYS5 */

    return seconds;
}

