%%==============================================================%%
%%                Window GUI System for Tcl/Tk                  %%
%%             Copyright (C) 1999 Tokuyasu KAKUTA               %%
%%                  1999.1.31 Version 0.184                     %%
%%                E-mail: kaku@juris.hokudai.ac.jp              %%
%%==============================================================%%

%:-module(window,
%    [init_window/0,create/3,createSubWin/3,kill/1,flush/0,get_event/2,
%     tk_set_var/3,tk_get_var/3,window_bell/1,iconify/1,deiconify/1,
%     hide/1,get_parts/3,on_SubWin/3,clear_canvas/1,get_canvas_size/3,
%     set_canvas_scroll_region/1,createBox/4,createOval/4,createText/4,
%     createLine/4,createPoly/4,createArc/4,set_fig_attr/2,
%     get_fig_attr/2,get_fig_info/2,get_owner_canvas/2,move/3,remove/1,
%     scale/5,lower/1,raise/1,visible/1,invisible/1,dragable/2,
%     click_notify/2,undragable/1,is_dragable/1,createGroup/1,regist/2,
%     release/1,release/2,get_text/2,putl_text/2,put_text/2,clear_text/1,
%     end_text/1,top_text/1,clear_entry/1,set_items/2,add_item/2,
%     insert_item/3,get_items/2,get_item/2,get_item/1,clear_items/1,
%     delete_item/2,get_items_size/2,set_parts_attr/2,get_parts_attr/3,
%     monoff/0,monon/0]
%     ).
%:-use_module(library(tcltk)).

:-dynamic window_trace/1, win_path_table/4, frame_count/1, window_id/2,
          visible_info/3, group_conter/1, group_register/2,
          group_index/2, entry_var_name/3, entry_var_count/1,'::='/2. 

:-op(980,xfx,::=).
:-op(900,xfy,(->)).
:-op(400,xfx,::).
:-op(300,xfx,:).
:-op(100,xfx,@).

%window_trace(on).
window_trace(off).

ok(X):-write(ok),tab(1),write(X),nl.

monon:-abolish(window_trace,1),assert(window_trace(on)).
monoff:-abolish(window_trace,1),assert(window_trace(off)).

%%--------------------------------------------------------------%%
%%                     Public Methods                           %%
%%--------------------------------------------------------------%%

init_window:-window_id(I,_),kill(I),fail.
init_window:-reset_pathname,abolish(frame_count,1),assert(frame_count(0)),
             abolish(visible_info,3),init_group,reset_entry_var_count.

tk_set_var(Win,Var,ValueString):-
   name(Var,TS),
   append_all(["set ",TS," ",ValueString],S),name(ComA,S),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

tk_get_var(Win,Var,ValueString):-
   name(Var,TS),
   append_all(["set ",TS],S),name(ComA,S),
   w_trace(ComA),
   tcl_eval(Win,ComA,ValueString).

create(Type,AttrL,Win):-
   (nonvar(AttrL),!;AttrL=[]),
   tk_new(AttrL,Win),
   assert(window_id(Win,Type)),
   layout(Win).

createSubWin(Win,SubName,Win/SubName):-
   set_pathname(Win,'.',SubName),
   name(SubName,St),
   append("toplevel .",St,ComS),
   name(ComA,ComS),
   w_trace(ComA),
   tcl_eval(Win,ComA,_),
   layout(Win,SubName).
   %hide(Win/SubName).

get_event(Win,Ev):-tk_next_event(Win,Ev).

flush:-tk_do_all_events.

set_tiltle(Win,String):-
    ComS1="wm title .",
   (Win=Root/Sub,!,
    name(Sub,NameS);
    Win=Root,
    NameS=[]),
   append_all([ComS1,NameS," ",String],ComS),
   name(Com,ComS),
   w_trace(Com),
   tcl_eval(Root,Com,_).

window_bell(Win):-
   w_trace(bell),
   tcl_eval(Win,bell,_).

kill(Win):-
   \+ Win=_/_,!,
   %reset_canvas(Win),
   Com='destroy .',
   w_trace(Com),
   tcl_eval(Win,Com,_),
   (retract(entry_var_name(Win,_,_)),fail;true),
   retract(window_id(Win,_)),!.
kill(Win/Sub):-
   ComS1="destroy .",
   name(Sub,NameS),
   append(ComS1,NameS,ComS),
   name(Com,ComS),
   w_trace(Com),
   tcl_eval(Win,Com,_).

iconify(Win):-
   \+ Win=_/_,!,
   ComA='wm iconify .',
   w_trace(ComA),
   tcl_eval(Win,ComA,_).
iconify(Win/Sub):-
   ComS1="wm iconify .",
   name(Sub,S2),
   append(ComS1,S2,ComS),
   name(ComA,ComS),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

deiconify(Win):-
   \+ Win=_/_,!,
   ComA='wm deiconify .',
   w_trace(ComA),
   tcl_eval(Win,ComA,_).
deiconify(Win/Sub):-
   ComS1="wm deiconify .",
   name(Sub,S2),
   append(ComS1,S2,ComS),
   name(ComA,ComS),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

hide(Win):-
   \+ Win=_/_,!,
   ComA='wm withdraw .',
   w_trace(ComA),
   tcl_eval(Win,ComA,_).
hide(Win/Sub):-
   ComS1="wm withdraw .",
   name(Sub,S2),
   append(ComS1,S2,ComS),
   name(ComA,ComS),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

%% + -
get_parts(V,Name,Win/Name):-nonvar(V),!,(V=Win/Sub,!;V=Win).
%% - +
get_parts(Win,Name,V):-nonvar(V),!,V=Win/Name.

on_SubWin(Win,P,Sub):-
   P=[F|P1],   %% F is the code of '.'
   get_first_parts(P1,Sub),
   append("winfo class ",[F|Sub],ComS),
   name(ComA,ComS),
   w_trace(ComA),
   tcl_eval(Win,ComA,AnsS),
   AnsS="Toplevel",!.

get_first_parts([F|_],[]):-name('.',[F]),!.
get_first_parts([F|L],[F|L1]):-get_first_parts(L,L1).

%redraw_parts(Win/Name,PredSymbol):-!,
%   get_pathname(Win/Name,P),
%   prepare_reshape_event(Win).
%   name(PredSymbol,PS),
%   %append_all(["bind . <Configure> {prolog ",PS,"}"],String),
%   append_all(["bind ",P," <Configure> {prolog ",PS,"}"],String),
%   name(ComA,String),
%   w_trace(ComA),
%   tcl_eval(Win,ComA,AnsS).
%ptest:-write('******** OK!! *********'),nl.

clear_canvas(Can):-
   Can=Win/_,
   get_pathname(Can,P),
   append_all([P," delete ",P," find all"],String),
   name(ComA,String),
   w_trace(ComA),
   tcl_eval(Win,ComA,AnsS).

get_canvas_size(Can,W,H):-
   get_parts_attr(Can,width,WS),
   name(W,WS),
   get_parts_attr(Can,height,HS),
   name(H,HS).

set_canvas_scroll_region(Can):-
   Can=Win/_,
   get_pathname(Can,P),
   append_all([P," bbox all"],String1),
   name(ComA1,String1),
   w_trace(ComA1),
   tcl_eval(Win,ComA1,AnsS1),
   append_all([P," configure -scrollregion {",AnsS1,"}"],String2),
   name(ComA2,String2),
   w_trace(ComA2),
   tcl_eval(Win,ComA2,_).

createBox(Can,CoordL,AttrL,Obj):-createFig(Can,rect,CoordL,AttrL,Obj).
createOval(Can,CoordL,AttrL,Obj):-createFig(Can,oval,CoordL,AttrL,Obj).
createText(Can,CoordL,AttrL,Obj):-createFig(Can,text,CoordL,AttrL,Obj).
createLine(Can,CoordL,AttrL,Obj):-createFig(Can,line,CoordL,AttrL,Obj).
createPoly(Can,CoordL,AttrL,Obj):-createFig(Can,poly,CoordL,AttrL,Obj).
createArc(Can,CoordL,AttrL,Obj):-createFig(Can,arc,CoordL,AttrL,Obj).
%createImage(Can,CoordL,AttrL,Obj):-createFig(Can,image,CoordL,AttrL,Obj).

get_owner_canvas(Win/Can-_,Win/Can):-!.

move(Obj,X,Y):-Obj='$group$'/_,!,group_method(move,Obj,[X,Y]).
move(Obj,X,Y):-move_fig(Obj,X,Y).
remove(Obj,X,Y):-Obj='$group$'/_,!,group_method(remove,Obj,[X,Y]).
remove(Obj,X,Y):-remove_fig(Obj,X,Y).
scale(Obj,X,Y,SX,SY):-Obj='$group$'/_,!,group_method(scale,Obj,[X,Y,SX,SY]).
scale(Obj,X,Y,SX,SY):-scale_fig(Obj,X,Y,SX,SY).
lower(Obj):-Obj='$group$'/_,!,group_method(lower,Obj,[X,Y]).
lower(Obj):-lower_fig(Obj).
raise(Obj):-Obj='$group$'/_,!,group_method(raise,Obj,[X,Y]).
raise(Obj):-raise_fig(Obj).
visible(Obj):-Obj='$group$'/_,!,group_method(visible,Obj,[X,Y]).
visible(Obj):-visible_fig(Obj).
invisible(Obj):-Obj='$group$'/_,!,group_method(invisible,Obj,[X,Y]).
invisible(Obj):-invisible_fig(Obj).
dragable(Obj,NCom):-Obj='$group$'/_,!,dragable_group(Obj,NCom).
dragable(Obj,NCom):-dragable_fig(Obj,NCom).
click_notify(Obj,Com):-Obj='$group$'/_,!,group_method(click_notify,Obj,[Com]).
click_notify(Obj,Com):-click_notify_fig(Obj,Com).
undragable(Obj):-Obj='$group$'/_,!,undragable_group(Obj).
undragable(Obj):-undragable_fig(Obj).

is_dragable(Obj):-
   (Obj='$group$'/N,
    name(N,NS),
    append("group_members",NS,TclVarS),
    get_group_member(Obj,FL),
    FL=[Win/Can-_|_],!;
    Obj=Win/Can-Item,
    name(Item,TclVarS)),
   get_pathname(Win/Can,P),
   append_all([P," bind ",TclVarS],String),
   name(Com,String),
   tcl_eval(Win,Com,Ans),
   w_trace(Com),
   is_substring(Ans,"<B1-Motion>").

createGroup(Group):-
   get_group_count(N),
   Group='$group$'/N,
   assert(group_register(Group,[])).

get_group_member(G,L):-group_register(G,L).

regist(_,[]):-!.
regist(G,[F|L]):-!,
   regist1(G,F),
   regist(G,L).
regist(G,Obj):-regist1(G,Obj).

release(G):-
   retract(group_register(G,L)),!,
   cancel_regist(L,G),
   assert(group_register(Group,[])).
release(G,L):-
   retract(group_register(G,L0)),!,
   cancel_regist(L,G),
   (is_dragable(G),!,dragable_group(G,_);ture),
   del_list_eq(L0,L,L1),
   assert(group_register(Group,L1)).

get_text(Win/Name,String):-
   entry_var_name(Win,Name,Var),
   name(Var,NS),
   append("set ",NS,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,String).

putl_text(Obj,String):-
   append(String,"\\n",StringNL),
   put_text(Obj,StringNL).

put_text(Obj,String):-
   to_tcl(Obj,insert,[end,String],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

clear_text(Obj):-
   to_tcl(Obj,delete,['0.0',end],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

end_text(Obj):-
   to_tcl(Obj,yview,[end],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

top_text(Obj):-
   to_tcl(Obj,yview,['0.0'],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

clear_entry(Obj):-
   to_tcl(Obj,delete,[0,end],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

set_items(Obj,L):-
   to_tcl(Obj,insert,[0|L],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

add_item(Obj,Item):-
   to_tcl(Obj,insert,[end,Item],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

insert_item(Obj,N,Item):-
   to_tcl(Obj,insert,[N,Item],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

get_items(Obj,L):-
   to_tcl(Obj,curselection,[],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,String),
   to_num_list(String,NL),
   get_n_items(NL,Obj,L).

get_item(Obj,X):-
   to_tcl(Obj,curselection,[],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,String),
   name(X,String).

get_item(Obj,N,X):-
   to_tcl(Obj,get,[N],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,String),
   name(X,String).

clear_items(Obj):-
   to_tcl(Obj,delete,[0,end],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

delete_item(Obj,N):-
   to_tcl(Obj,delete,[N],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,String),
   name(X,String).

get_items_size(Obj,N):-
   to_tcl(Obj,index,[end],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,String),
   name(N,String).

set_parts_attr(Obj,OL):-
   to_tcl(Obj,config,OL,Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

get_parts_attr(Obj,Opt,String):-
   to_tcl(Obj,cget,[-Opt],Win,ComL),
   name(ComA,ComL),
   w_trace(ComA),
   tcl_eval(Win,ComA,String).

%%--------------------------------------------------------------%%
%%                   local predicates                           %%
%%--------------------------------------------------------------%%

%% create Widget parts

create_parts(Type,InterP,Parent,Name,OptionList):-
   set_pathname(InterP,Parent,Name),
   (Type==entry,!,add_text_var(InterP,Name,OptionList,OptionList0);
    OptionList0=OptionList),
   to_tcl(Type,InterP/Name,OptionList0,_,TclComString),
   name(A1,TclComString),
   tcl_eval(InterP,A1,Path),
   append_all(["pack ",Path," -expand on"],PL),
   name(A2,PL),
   w_trace(A2),
   tcl_eval(InterP,A2,_).

%% to_tcl(+Arg1,+Arg2,+OptionsList,-InterP,-TclCommandString)
%%  !!Attention!! : before calling, need to set pathname. 

to_tcl(X,Y,L,Win,String):-
   (atom(X),!,name(X,X1);
    get_pathname(X,X1),get_parts(Win,_,X)),
   (atom(Y),!,name(Y,Y1);
    get_pathname(Y,Y1),get_parts(Win,_,Y)),
   get_options(L,SL),
   make_tcl_list([X1,Y1|SL],L1),
   append_all(L1,String),
   name(Atom,String),
   w_trace(Atom).

%% PATH Opreations :get_pathname, set_pathname, reset_pathname

get_pathname(I/N,Str):-
   win_path_table(I,N,Parent,Str).

set_pathname(I,Parent,N):-
   (win_path_table(I,N,Parent,_),!,warning('Already exists !'(N,Parent));
    (Parent=='.',!,PS=[];get_pathname(I/Parent,PS)),
    name(N,NS),
    append_all([PS,".",NS],Str),
    assert(win_path_table(I,N,Parent,Str)) ).

reset_pathname:-abolish(win_path_table,4).

get_options([],[]):-!.
get_options([F|L],H):-
   to_tcl_option(F,H,T),
   get_options(L,T).

to_tcl_option(A,[S|T],T):-atomic(A),!,name(A,S0),  %% for atom or number 
   tcl_esc_var(S0,S).
to_tcl_option([F|L],[[WQ|S]|T],T):-!,                   %% for string
   name('"',[WQ]),
   append([F|L],[WQ],S0),
   tcl_esc_var(S0,S).
to_tcl_option(W/C+Item,[S|T],T):-!,name(Item,S).   %% for figure object
to_tcl_option(-A,[[Min|S]|T],T):-atomic(A),!,      %% for sw-option cget
   name(A,S),"-"=[Min].
to_tcl_option(-Term,[[Min|S1]|H],T):-!,            %% for sw-option
   Term=..[F,A],"-"=[Min],name(F,S1),to_tcl_option(A,H,T).

make_tcl_list([F],[F]):-!.
make_tcl_list([F|L],[F," "|L0]):-make_tcl_list(L,L0).

tcl_esc_var([],[]):-!.
tcl_esc_var([D|St0],[C,D|St]):-"$"=[D],!,"\\"=[C],
   tcl_esc_var(St0,St).
tcl_esc_var([F|St0],[F|St]):-
   tcl_esc_var(St0,St).

append_all(L,A):-append_all(L,[],A).

append_all([],A,A):-!.
append_all([F|L],R,A):-
   append(R,F,R1),
   append_all(L,R1,A).

append([],L,L):-!.
append([F|L],B,[F|L1]):-append(L,B,L1).

puts([]):-!.
puts([X|L]):-put(X),puts(L).

is_substring([F|L],[F|L1]):-
    is_substring1(L,L1),!.
is_substring([_|L],L1):-
    is_substring1(L,L1).

is_substring1(_,[]):-!.
is_substring1([F|L],[F|L1]):-is_substring1(L,L1).

w_trace(CommandAtom):-window_trace(on),!,
   write('<TK>: '),
   write(CommandAtom),nl.
w_trace(_).

error(X):-write('ERROR(GUI):'),write(X),nl.
warning(X):-write('Warning(GUI):'),write(X),nl.

tk_do_all_events:-
   tk_do_one_event,!,
   tk_do_all_events.
tk_do_all_events.

pack_win(Win,C):-
   get_pathname(Win/C,PathS),
   append_all(["pack ",PathS," -expand on"],String),
   name(ComA,String),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

new_frame_id(New):-
   retract(frame_count(Num)),!,
   Num1 is Num+1,
   asserta(frame_count(Num1)),
   name(Num,NS),
   append("tkFrame",NS,String),
   name(New,String).

%%--------------------------------------------------------------------------
%%                                Layout
%%--------------------------------------------------------------------------

layout(Win):-
   window_id(Win,Type),!,
   window/Type ::= LayoutList,
   exec_layout(Win,LayoutList),
   default_attr(Win).

layout(Win,Sub):-   %% For sub_window
   window/Sub ::= LayoutList,
   exec_sub_layout(Win,Sub,LayoutList),
   default_attr(Win,Sub).

default_attr(Win):-
   window_id(Win,Type),
   get_all_deafult_attr(Type,AL),
   set_default_attrs(AL,Win).
default_attr(Win,Sub):-   %%%% No check !!!!
   get_all_deafult_attr(Sub,AL),
   set_default_attrs(AL,Win).

get_all_deafult_attr(Type,AL):-
   findall((X,L),attr/Type-X::=L,AL).

set_default_attrs([],_):-!.
set_default_attrs([(X,AL)|L],W):-
   set_default_attr(AL,W,X),
   set_default_attrs(L,W).

set_default_attr(AL,W,X):-
   conv_tcl_commands(AL,CL),
   to_tcl(W/X,conf,CL,_,String),
   name(ComA,String),
   tcl_eval(W,ComA,_).

conv_tcl_commands([],[]):-!.
conv_tcl_commands([Opt->Val|AL],Head):-!,
   conv_one_opt(Opt,Val,Head,Tail),
   conv_tcl_commands(AL,Tail).

conv_one_opt(size,(W,H),[-width(W),-height(H)|L],L):-!.
conv_one_opt(pos,(X,Y),[-posx(X),-posy(Y)|L],L):-!.
conv_one_opt(code,Val,[-command(Str)|L],L):-!,
   name(Val,VL),
   append("prolog_event ",VL,Str).
conv_one_opt(Opt,Val,[-Term|L],L):-!,Term=..[Opt,Val].  %%% others

exec_layout(Win,LL):-!,
   exec_layout(Win,.,LL,C),
   pack_win(Win,C).

exec_sub_layout(Win,Sub,LL):-!,
   exec_layout(Win,Sub,LL,C),
   pack_win(Win,C).

exec_layout(Win,Par,LL,Frame):- %% for frames
   (LL=[_|_],!,L=LL,For= +;
    functor(LL,For,1),arg(1,LL,L)),!,
   make_frame(Win,Par,Frame),
   exec_layout_children(L,Win,Frame,ChildL),
   pack_children(Win,ChildL,For).
exec_layout(Win,Par,LL,Frame):- %% for Scroll parts
   (LL=s+L,!,For= + ;LL=s-L,!,For= - ;LL=(s=L),!,For= +,Flag=both),
   (L=[X1,X2],!;L=[X1,X2,X3]),
   make_frame(Win,Par,Frame),
   conv_from_like_osl(X1,Type1,N1,Com1),
   (Type1==entry,!,add_text_var(Win,N1,Com1,Com10);Com10=Com1),
   (Flag==both,!,
      make_frame(Win,Frame,FrameC),
      set_pathname(Win,FrameC,N1);
    set_pathname(Win,Frame,N1)),
   get_pathname(Win/N1,Name1),
   to_tcl(Type1,Win/N1,Com10,_,String1),
   name(ComA1,String1),
   tcl_eval(Win,ComA1,_),
   (Flag==both,!,
      conv_from_like_osl(X3,Type3,N3,Com3),  % Type3 to be scrollbar
      set_pathname(Win,FrameC,N3),
      get_pathname(Win/N3,Name3), %Name3 is c-sc.
      append(Name1," yview",ScCom3),
      append(Com3,[-orient(vertical),-command(ScCom3)],Com31),
      to_tcl(Type3,Win/N3,Com31,_,String3),
      name(ComA3,String3),
      tcl_eval(Win,ComA3,_),
      append_all([Name1," conf -yscrollcommand \"",Name3," set\""],_,ConfStC),
      name(ComConfC,ConfStC),
      w_trace(ComConfC),
      tcl_eval(Win,ComConfC,_),
      conv_pathname_all([N1,N3],Win,CLc),
      make_tcl_list(CLc,CSc),     % [".b.c", ".b.d"] --> ".b.c .b.d"
      PackOpt0= " -side left -fill y",
      pack_scroll_frame(Win,CSc,PackOpt0);
   true),
   ((For== +,!,Fanc=xscrollcommand,Or= orient(horizontal),View=" xview";
     For== -,!,Fanc=yscrollcommand,Or= orient(vertical),  View=" yview")),
   conv_from_like_osl(X2,Type2,N2,Com2), % Type2 to be scrollbar
   set_pathname(Win,Frame,N2),
   get_pathname(Win/N2,Name2), %Name2 is p-sc.
   append(Name1,View,ScCom),
   append(Com2,[-Or,-command(ScCom)],Com21),
   to_tcl(Type2,Win/N2,Com21,_,String2),
   name(ComA2,String2),
   tcl_eval(Win,ComA2,_),
   name(Fanc,FancSt),
   append_all([Name1," conf -",FancSt," \"",Name2," set\""],_,ConfStP),
   name(ComConfP,ConfStP),
   w_trace(ComConfP),
   tcl_eval(Win,ComConfP,_),
   (Flag==both,!,Obj=FrameC;Obj=N1),
   conv_pathname_all([Obj,N2],Win,CL1),
   make_tcl_list(CL1,CS),     % [".b.c", ".b.d"] --> ".b.c .b.d"
   (For== -,!,OptP= " -side left -fill y";
    For== +,!,OptP= " -side top -fill x"),
   pack_scroll_frame(Win,CS,OptP).
exec_layout(Win,Par,X,N):- %%% for minimal parts
   conv_from_like_osl(X,Type,N,Com),
   (Type==entry,!,add_text_var(Win,N,Com,Com0);Com0=Com),
   set_pathname(Win,Par,N),
   to_tcl(Type,Win/N,Com0,_,String),
   name(ComA,String),
   tcl_eval(Win,ComA,_).

conv_from_like_osl(X:ST,Type,X,Com):-
   (atom(ST),!,Type=ST,Com=[];
    ST=..[Type|L],
    conv_tcl_commands(L,Com)).

make_frame(Win,Par,Name):-
   new_frame_id(Name),
   %set_pathname(Win,Par,Name),
   create_parts(frame,Win,Par,Name,[]).

exec_layout_children([],_,_,[]):-!.
exec_layout_children([F|L],Win,Par,[Name|L1]):-
   exec_layout(Win,Par,F,Name),
   exec_layout_children(L,Win,Par,L1).

pack_scroll_frame(Win,CS,Opt):-
   append(["pack "|CS],[Opt," -expand yes"],ComL),
   append_all(ComL,String),
   name(ComA,String),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

pack_children(Win,CL,For):-
   conv_pathname_all(CL,Win,CL1),
   make_tcl_list(CL1,CS),     % [".b.c", ".b.d"] --> ".b.c .b.d"
   (For== +,!,Opt= " -side top";
    For== -,!,Opt= " -side left"),
   append(["pack "|CS],[Opt," -expand yes"],ComL),
   append_all(ComL,String),
   name(ComA,String),
   w_trace(ComA),
   tcl_eval(Win,ComA,_).

conv_pathname_all([],_,[]):-!.
conv_pathname_all([P|L],W,[S|L1]):-
   get_pathname(W/P,S),
   conv_pathname_all(L,W,L1).

add_text_var(Win,Name,L,L1):-
   get_entry_var_count(N),
   name(entry_var,A),
   name(N,B),
   append(A,B,VarS),
   name(Var,VarS),
   assert(entry_var_name(Win,Name,Var)),
   append(L,[-textvariable(Var)],L1).

get_entry_var_count(N):-
   retract(entry_var_count(N)),!,
   N1 is N+1,
   assert(entry_var_count(N1)).

reset_entry_var_count:-
   abolish(entry_var_count),
   assert(entry_var_count(0)).

get_n_items([],_,[]):-!.
get_n_items([N|NL],Obj,[X|L]):-
   get_item(Obj,N,X),
   get_n_items(NL,Obj,L).

to_num_list([],[]):-!.
to_num_list(L,NL):-to_num_list(L,H,H,NL).

to_num_list([],H,[],[H]):-!.
to_num_list([C|L],H,[],[Num|NL]):-[C]=" ",!,
   name(Num,H),  %% or Num=H, also OK ?!
   to_num_list(L,H1,H1,NL).
to_num_list([C|L],H,[C|T],NL):-
   to_num_list(L,H,T,NL).

%%--------------------------------------------------------------------------
%%                              Figure
%%--------------------------------------------------------------------------

createFig(Can,Type,CoordL,AttrL,Win/C-Item):-
   Can=Win/C,
   conv_tcl_commands(AttrL,ComL),
   get_options(ComL,StL),
   conv_to_coord_list(CoordL,CoordStH,StL),
   make_tcl_list(CoordStH,StL1),
   name(Type,TypeS),
   get_pathname(Win/C,P),
   append_all([P," create ",TypeS," "|StL1],String),
   name(ComA,String),
   w_trace(ComA),
   tcl_eval(Win,ComA,ItemS),
   name(Item,ItemS).

conv_to_coord_list([],T,T):-!.
conv_to_coord_list([N|L],[NS|H],T):-
   name(N,NS),
   conv_to_coord_list(L,H,T).

conv_to_num_list(SL,NL):-conv_to_num_list(SL,H,H,NL).

conv_to_num_list([],H,[],[N]):-!,name(N,H).
conv_to_num_list([S|SL],H,[],[N|NL]):-" "=[S],!,
   name(N,H),
   conv_to_num_list(SL,NewH,NewH,NL).
conv_to_num_list([S|SL],H,[S|T],NL):-
   conv_to_num_list(SL,H,T,NL).

%reset_canvas(Win):-
%   (retract(canvas_register(Win,_));true),!.

%set_canvas(Win,Can):-
%   (canvas_register(Win,_),!,warning('Already assgined !! (CANVAS)');
%    assert(canvas_register(Win,Can))).

%get_canvas(Win,Can):-canvas_register(Win,Can).

get_fig_info([],_,_,_):-!.
get_fig_info([F->F1|L],Win,P,IS):-
   name(F,S),
   append_all([P," ",S," ",IS],String),
   name(ComA,String),
   tcl_eval(Win,ComA,F1),
   w_trace(ComA),
   get_fig_info(L,Win,P,IS).

get_fig_attr([],_,_,_,[]):-!.
get_fig_attr([OptName|L],Win,P,IS,[V|LV]):-
   name(OptName,OptS),
   append_all([P," itemcget ",IS," -",OptS],String),
   name(ComA,String),
   tcl_eval(Win,ComA,V),
   w_trace(ComA),
   get_fig_attr(L,Win,P,IS,LV).

divid_attr([],[],[]):-!.
divid_attr([A->V|L],[A|L1],[V|LV]):-
   divid_attr(L,L1,LV).

move_fig(Obj,X,Y):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,IS),
   name(X,XS),
   name(Y,YS),
   append_all([P," move ",IS," ",XS," ",YS],String),
   name(ComA,String),
   tcl_eval(Win,ComA,_),
   w_trace(ComA).

remove_fig(Obj):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,IS),
   append_all([P," delete ",IS],String),
   name(ComA,String),
   tcl_eval(Win,ComA,_),
   w_trace(ComA).

scale_fig(Obj,X,Y,SX,SY):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,IS),
   name(X,XS),
   name(Y,YS),
   name(SX,SXS),
   name(SY,SYS),
   append_all([P," scale ",IS," ",XS," ",YS," ",SXS," ",SYS],String),
   name(ComA,String),
   tcl_eval(Win,ComA,_),
   w_trace(ComA).

lower_fig(Obj):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,IS),
   append_all([P," lower ",IS],String),
   name(ComA,String),
   tcl_eval(Win,ComA,_),
   w_trace(ComA).

raise_fig(Obj):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,IS),
   append_all([P," raise ",IS],String),
   name(ComA,String),
   tcl_eval(Win,ComA,_),
   w_trace(ComA).

set_fig_attr(Obj,AttrL):-
   Obj=Win/Can-Item,
   conv_tcl_commands(AttrL,ComL),
   get_options(ComL,StL),
   make_tcl_list(StL,StL1),
   get_pathname(Win/Can,P),
   name(Item,ItemS),
   append_all([P," itemconf ",ItemS," "|StL1],String),
   name(ComA,String),
   tcl_eval(Win,ComA,_),
   w_trace(ComA).

get_fig_attr(Obj,AttrL):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,ItemS),
   divid_attr(AttrL,AttrL1,AttrLV),
   get_fig_attr(AttrL1,Win,P,ItemS,AttrLV).

get_fig_info(Obj,L):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,IS),
   get_fig_info(L,Win,P,IS).

visible_fig(Obj):-
   retract(visible_info(Obj,LC,FC)),!,
   set_fig_attr(Obj,[fill->FC,outline->LC]).

invisible_fig(Obj):-
   get_fig_attr(Obj,[fill->FC,outline->LC]),
   (retract(visible_info(Obj,_,_)),fail;true),!,
   asserta(visible_info(Obj,LC,FC)),
   set_fig_attr(Obj,[fill->[],outline->[]]).

dragable_fig(Obj,NCom):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,IS),
   append_all([P," coords ",IS],String1),
   name(ComA1,String1),
   tcl_eval(Win,ComA1,CoordS),
   w_trace(ComA1),
   conv_to_num_list(CoordS,[X1,Y1,X2,Y2]),
   DX is (X2-X1)/2,name(DX,DXS),
   DY is (Y2-Y1)/2,name(DY,DYS),
   append_all([P," bind ",IS," <B1-Motion> {",
              "set x1 [expr %x-",DXS,"];",
              "set y1 [expr %y-",DYS,"];",
              "set x2 [expr %x+",DXS,"];",
              "set y2 [expr %y+",DYS,"];",
              P," coords current $x1 $y1 $x2 $y2}"],String2),
   name(ComA2,String2),
   tcl_eval(Win,ComA2,_),
   w_trace(ComA2),
   (var(NCom),!;
    name(NCom,NComS),
    append_all([P," bind ",IS," <ButtonRelease> {prolog_event ",NComS,"}"],
               String3),
    name(ComA3,String3),
    tcl_eval(Win,ComA3,_),
    w_trace(ComA3)).

undragable_fig(Obj):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,IS),
   append_all([P," bind ",IS," <B1-Motion> {}"],String2),
   name(ComA2,String2),
   tcl_eval(Win,ComA2,_),
   w_trace(ComA2),
   append_all([P," bind ",IS," <ButtonRelease> {}"],String3),
   name(ComA3,String3),
   tcl_eval(Win,ComA3,_),
   w_trace(ComA3).

click_notify_fig(Obj,Com):-
   Obj=Win/Can-Item,
   get_pathname(Win/Can,P),
   name(Item,IS),
   name(Com,S),
   append_all([P," bind ",IS," <Double-Button> {prolog_event ",S,"}"],String),
   name(ComA,String),
   tcl_eval(Win,ComA,_),
   w_trace(ComA).

%%--------------------------------------------------------------------------
%%                              Group
%%--------------------------------------------------------------------------

init_group:-
   abolish(group_register,2),
   abolish(group_index,2),
   abolish(group_count,1),
   assert(group_count(0)).

get_group_count(N):-
   retract(group_count(N)),!,N1 is N+1,asserta(group_count(N1)).

regist1(G,Obj):-
   retract(group_register(G,L)),!,
   (retract(group_index(Obj,GL)),!;GL=[]),
   append(L,[Obj],NewL),
   append(GL,[G],NewGL),
   assert(group_register(G,NewL)),
   assert(group_index(Obj,GL)),
   regist_canvas_tag(G,Obj),
   (is_dragable(G),!,dragable_group(G,_);true).

cancel_regist([],_):-!.
cancel_regist([F|L],G):-
   retract(group_index(F,GL)),!,
   delete_canvas_tag(F,G),
   del_list_eq(GL,[G],NewGL),
   assert(group_index(Obj,GL)),
   cancel_regist(L,G).

del_list_eq([],_,[]):-!.
del_list_eq([F|L],L0,L1):-member(F,L0),!,del_list_eq(L,L0,L1).
del_list_eq([F|L],L0,[F|L1]):-del_list_eq(L,L0,L1).

member(F,[F1|_]):-F==F1,!.
member(F,[_|L]):-member(F,L).

get_member(F,[F|_]).
get_member(F,[_|L]):-get_member(F,L).

group_method(Pred,G,ArgL):-
   group_register(G,FigL),
   group_method_loop(FigL,Pred,ArgL).

group_method_loop([],_,_):-!.
group_method_loop([F|L],P,AL):-
   G=..[P,F|AL],
   call(G),
   group_method_loop(L,P,AL).

dragable_group(Group,NCom):-
   Group='$group$'/N,
   name(N,NS),
   append("group_members",NS,TclVarS),
   get_group_member(Group,FL),
   (FL=[Win/Can-_|_],!,
    get_pathname(Win/Can,P),
    append_all([P," coords ",TclVarS],String1),
    name(ComA1,String1),
    tcl_eval(Win,ComA1,CoordS),
    w_trace(ComA1),
    conv_to_num_list(CoordS,[X1,Y1,X2,Y2]),
    DX is (X2-X1)/2,name(DX,DXS),
    DY is (Y2-Y1)/2,name(DY,DYS),
    append_all([P," bind ",TclVarS," <B1-Motion> {",
              "set coordslist [",P," coords ",TclVarS,"];",
              "set x1 [lindex $coordslist 0];",
              "set y1 [lindex $coordslist 1];",
              "set x2 [expr %x-",DXS,"];",
              "set y2 [expr %y-",DYS,"];",
              "set dx [expr $x2-$x1];",
              "set dy [expr $y2-$y1];",
              P," move ",TclVarS," $dx $dy}"],String2),
    name(ComA2,String2),
    tcl_eval(Win,ComA2,_),
    w_trace(ComA2),
    (var(NCom),!;
     name(NCom,NComS),
     append_all([P," bind ",TclVarS,
                            " <ButtonRelease> {prolog_event ",NComS,"}"],
                String3),
     name(ComA3,String3),
     tcl_eval(Win,ComA3,_),
     w_trace(ComA3))
    ;
    true).

undragable_group(Group):-
   get_group_member(Group,FL),
   (FL=[Win/Can-_|_],!,
    Group='$group$'/N,
    name(N,NS),
    append("group_members",NS,TclVarS),
    get_pathname(Win/Can,P),
    append_all([P," bind ",TclVarS," <B1-Motion> {}"],String2),
    name(ComA2,String2),
    w_trace(ComA2),
    tcl_eval(Win,ComA2,_),
    append_all([P," bind ",TclVarS," <ButtonRelease> {}"],String3),
    name(ComA3,String3),
    w_trace(ComA3),
    tcl_eval(Win,ComA3,_);
    ture).

regist_canvas_tag(Group,Win/Can-Item):-
   Group='$group$'/N,
   name(N,NS),
   append("group_members",NS,TclVarS),
   get_pathname(Win/Can,P),
   name(Item,IS),
   append_all([P," addtag ",TclVarS," withtag ",IS],ComS),
   name(Com,ComS),
   tcl_eval(Win,Com,_),
   w_trace(Com).

delete_canvas_tag(Win/Can-Item,Group):-
   Group='$group$'/N,
   name(N,NS),
   append("group_members",NS,TclVarS),
   get_pathname(Win/Can,P),
   name(Item,IS),
   append_all([P," dtag ",IS," ",TclVarS],ComS),
   name(Com,ComS),
   tcl_eval(Win,Com,_),
   w_trace(Com).

