%=======================================================
%  COPLAS: A conditonal planner with sensing actions
%  by Jorge Lobo  (jorge@eecs.uic.edu)
%
%  Copyright: 1998, Jorge Lobo.
%

% To run the planner: consult your domain description and goal.
%                     ?- plan.


%==========HEADINGS=====================================


:- dynamic planstate/3.

:- multifile causes/3.
:- multifile sensing/1.
:- multifile initially/1.
:- multifile initially_unknown/1.
:- multifile axiom/2.
:- multifile goal/1.

:- dynamic causes/3.
:- dynamic sensing/1.
:- dynamic initially/1.
:- dynamic initially_unknown/1.
:- dynamic axiom/2.
:- dynamic goal/1.
%=======================================================


query(Goal):-
	clean_all,
        retractall(goal(_)),
	assert(goal(Goal)),
        plan.

plan :-
  statistics(runtime,[_,_]),
  goal( Goal),
  create_init_state( InitState),
  assert( planstate( [(Plan,InitState,root)],Plan, 1)),
  generate( Goal, Plan),
  statistics(runtime,[_,Time]),
  nl,write('Time needed: '), write(Time),nl,nl,
  write('Plan: '), nl, printplan(Plan,1),nl,!.

%=======================================================

create_init_state((Trues,Unknowns)) :-
	(setof(El,initially(El),Trues) -> true; Trues =[]),
        (setof(U,initially_unknown(U),Unknowns) -> true;Unknowns = []).

%=======================================================

foundgoal(_, []).
foundgoal( State, [X|Rest]) :- 
  true_in_state( X, State),
  foundgoal( State, Rest).

%=======================================================

closebranches([],_,[]).
closebranches([(B,State,OldState)|Branches],Goal,FinalBs) :-
	(foundgoal(State,Goal) -> (FinalBs = NewBs, B=close);
	                         FinalBs = [(B,State,OldState)|NewBs]),
        closebranches(Branches,Goal,NewBs).

%=======================================================

generate( Goal, close) :- 	
  planstate( [(close,InitState,_)],close, 1), assert(counter(1)),
  foundgoal( InitState, Goal).

generate(_, Plan) :-
	get_counter(N),
	generate_planstate( N),
	NNew is N + 1,
	planstate([],Plan,NNew).

%=======================================================

get_counter(N):-
   retract(counter(N)), 
% The following retraction is optional.  So far, for small
% examples does not make any difference.  It could if we get
% to generate very long plans.
   N0 is N-1, retractall(planstate(_,_,N0)),
   N1 is N+1,assert(counter(N1)).

% Notice that this below could lead to an infinite loop.
% We can used a constant to stop the loop.  The N reflects
% the length of the plan.

%  NOTE: if no planstate is generated we should stop 

get_counter(N) :- !, planstate(_,_,_), get_counter(N).  

%=======================================================

generate_planstate( N) :- 
  Next is N + 1,
  planstate(Branches,Plan, N),
  expandbranches(Branches,NewBs,Plan),
  goal(Goal),
  closebranches(NewBs, Goal, FinalBs),
  assert(planstate(FinalBs,Plan,Next)).

%=======================================================

expandbranches([],[],_).

expandbranches([(s(A,B),State,OldState)|Branches],
               [(B,NewState,State)|NewB],Plan) :-
  action(A),
% if_possible(  A, State),   %  independent exec. conditions
%                            uncomment the above line if you would like to
%                            work with unprocessed exec. conditions and
%                            comment next line.
  possible(  A, State),      %  dependent exec. conditions
  heuristic( A, State, Plan),
  phi(A,State,NewState),
  diff(NewState,OldState),
  expandbranches(Branches,NewB,Plan).

expandbranches([(b(A,Cond,B1,B2),State,_)|Branches],
	       [(B1,NewS1,State),(B2,NewS2,State)|NewB],Plan) :-
  sensing(A),
% sensing_if_possible(A,State),   %  independent exec. conditions
%                            uncomment the above line if you would like to
%                            work with unprocessed exec. conditions and
%                            comment next line.
  sensing_possible(  A, State),   %  dependent exec. conditions
% heuristic( A, State, Plan),
  phi(A,State,NewS1,NewS2,Cond),
  expandbranches(Branches,NewB,Plan).

%  At this moment there are no heuristics associated with
%  sensing actions.


%====================================
if_possible(A,State) :-
    possible(A,Conditions),
    check_if_possible(Conditions,State).

check_if_possible([],_).

check_if_possible([C|Conditions],State) :-
    ((C=diff(X,Y);C=(X=Y))-> call(C);
                            (fluent_literal(C),true_in_state(C,State))),
     check_if_possible(Conditions,State).
%=======================================================

sense_if_possible(A,State) :-
    causes_to_know(A,F,_),
    fluent(F),
    unknown_in_state(F,State),
    sensing_possible(A,Conditions),
    check_if_possible(Conditions,State).

%=======================================================

%   Phi for  non-sensing actions


phi(A,State,NewState) :-
	(causes(A,EffectProp) -> apply(EffectProp,State,State,InterState);
	                         State = InterState),
	(affects(A,AffectProp) ->
	     applynd(AffectProp,InterState,InterState,NewState);
	     InterState = NewState).


%=======================================================
%  Apply the effecst of deterministic actions

apply([],_,CurrentState,CurrentState).
apply([(Cond,Effect)|EffectProp],State,CurrentState,NewState) :-
	(setof(Effect,Cond^check(Cond,State),Effects) ->
                              apply(det,Effects,CurrentState,NState);
                              CurrentState = NState),
	(setof(Effect,Cond^check_u(Cond,State),Unknws) ->
                              apply_u(Unknws,NState,NUState);
                              NState = NUState),
	apply(EffectProp,State,NUState,NewState).

%=======================================================
%  Apply the effecst of non-deterministic actions

applynd([],_,CurrentState,CurrentState).
applynd([(Cond,Effect)|EffectProp],State,CurrentState,NewState) :-
	(setof(Effect,Cond^check(Cond,State),Effects) ->
                              apply(nondet,Effects,CurrentState,NState);
                              CurrentState = NState),
	applynd(EffectProp,State,NState,NewState).


%=======================================================

apply(_,[],State,State).
apply(Type,[E|Effects],State,NewState) :-
	apply_forall_effect(Type,E,State,NState),
	apply(Type,Effects,NState,NewState).

apply_forall_effect(Type,Forall_E,State,NState) :-
	(Forall_E = neg(E) -> setof(neg(E),fluent(E),Es);
	                      setof(Forall_E,fluent(Forall_E),Es)),
	apply_each_effect(Type,Es,State,NState).

apply_each_effect(_,[],State,State).
apply_each_effect(Type,[E|Es],State,NewState) :-
	apply_single_effect(Type,E,State,InterState),
        apply_each_effect(Type,Es,InterState,NewState).

apply_single_effect(det,E,State,NewState) :-
	(E = neg(F) -> falsefluent(State,F,InterState);
                       truefluent(State,E,InterState)),
	apply_axioms(E,InterState,NewState).

apply_single_effect(nondet,E,State,NewState) :-
	unkfluent(State,E,NewState).

%======================================================

causes(A,EffectProp) :-
	setof((Cond,Effect),causes(A,Effect,Cond),EffectProp).

affects(A,EffectProp) :-
	setof((Cond,Effect),affects(A,Effect,Cond),EffectProp).


%======================================================

check([],_).
check([C|Conditions],State) :-
	((C=diff(X,Y); C=(X=Y))-> call(C);
	                          true_in_state(C,State)),
	check(Conditions,State).

%====================================================== 

% check_u verifies if a conjunction of fluents is
% unknown -- E.i. at least one unknown and the rest
% unknown or true.

check_u([C|Conditions],State) :-
       ((C=diff(X,Y); C=(X=Y))-> (call(C),check_u(Conditions,State));
	 ((true_in_state(C,State), check_u(Conditions,State));
	  (unknown_in_state(C,State), check_rest_u(Conditions,State)))).

check_rest_u([],_).

check_rest_u([C|Conditions],State) :-
       ((C=diff(X,Y); C=(X=Y))-> (call(C),check_rest_u(Conditions,State));
	                         (true_in_state(C,State),
	                          check_rest_u(Conditions,State))).

check_rest_u([C|Conditions],State) :-
	unknown_in_state(C,State),
	check_rest_u(Conditions,State).

%====================================================== 

apply_u([],State,State).
apply_u([E|Effects],State,NewState) :-
	apply_forall_effect_u(E,State,NState),
	apply_u(Effects,NState,NewState).

apply_forall_effect_u(Forall_E,State,NState) :-
	(Forall_E = neg(E) -> setof(neg(E),fluent(E),Es);
	                      setof(Forall_E,fluent(Forall_E),Es)),
	apply_each_effect_u(Es,State,NState).

apply_each_effect_u([],State,State).
apply_each_effect_u([E|Es],State,NewState) :-
	apply_single_effect_u(E,State,InterState),
        apply_each_effect_u(Es,InterState,NewState).


apply_single_effect_u(neg(E),State,NewState) :-
	true_in_state(E,State),
	unkfluent(State,E,NewState).

apply_single_effect_u(E,(T,U),(T,NewU)) :-
	\+ E = neg(_),
	false_in_state(E,(T,U)),
        insert(U,E,NewU).



%====================================================== 
%  Phi for sensing actions
%  In this definiton it is assummed that the
%  sensing actions do not have pre-conditions and there
%  is only one effect per action.
%  It is also assumed that Fluent is unknown.

phi(A,(T,U),(NewT1,NewU1),(NewT2,NewU2),Fluent) :-
	causes_to_know(A,Fluent,_),
	delete(U,Fluent,InterU),
	insert(T,Fluent,InterT),
	apply_axioms(Fluent,(InterT,InterU), (NewT1,NewU1)),
	apply_axioms(neg(Fluent),(T,InterU), (NewT2,NewU2)).

%======================================================


apply_axioms(E,State,NewState) :-
	setof(Effect,axiom(E,Effect),Effects)
	   -> apply_axiom_effects(Effects,State,NewState);
	      State = NewState.

apply_axiom_effects([],State,State).

apply_axiom_effects([E|Effects],State,NewState) :-
	(E = neg(F) -> falsefluent(State,F,InterState);
                       truefluent(State,E,InterState)),
	apply_axiom_effects(Effects,InterState,NewState).


%============================================================
%  The following set of predicate definitons are tools used
%  by the planner.
%=======================================================

eq( X, X).
diff( X, Y) :- \+ eq( X, Y).

%=======================================================

append([],Y,Y).
append([X|L],Y,[X|Z]):- append(L,Y,Z).

%======================================================

/*  This is set membership. Element is assumed to be ground and
    the list representing the set ordered
*/

memberL( Element, [Element|_]) :-  !.
memberL( Element, [XX|Tail]) :- 
	XX @< Element,
	memberL( Element, Tail).

%=======================================================

% Standard list memebership.  It can be used only to check
% not to generate.

member( Element, [Element|_]) :-  !.
member( Element, [_|Tail]) :- 
	member( Element, Tail).

%=======================================================

fluent_literal(F) :-
       (F=neg(G) -> fluent(G);fluent(F)).			     

%=======================================================

/*
If we have large states this operations can be optimized
by representing states using trees or hash tables. At this
moment the planner uses sorted lists.

true_in_state, unknow_in_state and false_in_state are similar to lookups.
falsefluent, truefluent and unkfluent use both  inserts and deletes.
*/

%=======================================================
/*   ***NOTE***

  This true_in_state check if there is AN instance of the
  fluent literal that is true in the state.
*/

true_in_state( El, (T,U)) :- 
	El = neg(F) -> (fluent(F),\+ memberL(F,T), \+ memberL(F,U)); 
                       (fluent(El), memberL( El,T)).


%=======================================================

unknown_in_state(Fluent,(_,U)) :-
	fluent(Fluent),
	memberL(Fluent,U).

%=======================================================

false_in_state(Fluent,State) :-
	\+ true_in_state(Fluent,State),
	\+ unknown_in_state(Fluent,State).

%=======================================================

falsefluent(([],U),El,([],NewU)) :- delete(U,El,NewU).
falsefluent(([El|Set],U),El,(Set,U)) :- !.  
falsefluent(([XX|Set],U),El,([XX|Newset],NewU)) :-
        XX @< El,
	falsefluent((Set,U),El,(Newset,NewU)).
falsefluent(([XX|Set],U),El,([XX|Set],NewU)) :-
        El @< XX,
	delete(U,El,NewU).

truefluent((Set,U),El,(NewSet,NewU)) :-
	insert(Set,El,NewSet), 
        delete(U,El,NewU).

unkfluent((Set,U),El,(NewSet,NewU)) :-
	insert(U,El,NewU), 
	delete(Set,El,NewSet).

%=======================================================

delete([],_,[]).
delete([El|Us],El,Us) :- !.
delete([XX|Us],El,[XX|NewUs]) :-
	XX @< El,
	delete(Us,El,NewUs).
delete([XX|Us],El,[XX|Us]) :-
	El @< XX.

insert([],El,[El]).
insert([XX|Set],El,[El,XX|Set]) :-
	El @< XX.
insert([El|Set],El,[El|Set]) :-  !.
insert([XX|Set],El,[XX|NewSet]) :-
	XX @< El,
	insert(Set,El,NewSet).


%======================================================

clean_all :- 
	retractall(counter(_)), 
	retractall(planstate(_,_,_)).

%=======================================================
 
printplan(close,_).

printplan(s(A,Plan),Tab) :-
	tab(Tab), 
	print(A),write(';'),nl,
	printplan(Plan,Tab).

printplan(b(A,C,P1,P2),Tab) :-
	 tab(Tab),
	 print(A),
	 write(';'), nl,
	 tab(Tab), write('if '),
	 print(C), write( ' then '),nl,
	 NewTab is Tab + 3,
	 printplan(P1,NewTab),
	 \+ P2 = close -> (tab(Tab),write('else '),nl,printplan(P2,NewTab)).

%=======================================================
/*
   The following predicates are used to implement the entailment relation.
   Given a possible incomplete initial states, it can be used to verify if
   a conjunction of fluent literals is true after the execution of a plan.
   For example, f1, f2, f3 and f4 are fluent literals, and a1, a2, a4, a4, a5
   and a6 are actions, we can ask:

     ?- entails ([f1,f2],
                 [a1,
                  if(f3,[a2]),
                  if(f4,[a2,a3],[a2,a4]),
                  while(f3,[a4,a5]),
                  a1
                 ]
                ).


   The answer is yes if after the execution of the plan f1 and f2
   are true in every resulting state.  There is an error if the condition
   of an if or a while is unknown at the moment of the execution of the
   if or while.
*/
%=======================================================

entails(Goal,Plan) :- 
	create_init_state(State),
	exec([State],Plan,NewStates),
	check_goal_in_states(Goal,NewStates).

check_goal_in_states([],_).

check_goal_in_states([Literal|Goal],States):-
	check_literal_in_states(Literal,States),
	check_goal_in_states(Goal,States).

check_literal_in_states(_,[]).

check_literal_in_states(Literal,[S|States]):-
	true_in_state(Literal,S),
	check_literal_in_states(Literal,States).

%========================================================

exec(States,[],States).

exec(States,[if(C,P1,P2)|Plan],NewStates):-
	execIfTS(States,C,P1,P2,IfStates),
	exec(IfStates,Plan,NewStates).

exec(States,[if(C,P)|Plan],NewStates):-
	execIfT(States,C,P,IfStates),
	exec(IfStates,Plan,NewStates).

exec(States,[while(C,P)|Plan],NewStates):-
	execWhile(States,C,P,WhileStates),
	exec(WhileStates,Plan,NewStates).

exec(States,[Action|Plan],NewStates):-
	action(Action),
	execAction(Action,States,PartialStates),
	exec(PartialStates,Plan,NewStates).

exec(States,[Sensing|Plan],NewStates):-
	sensing(Sensing),
	execSensingAction(Sensing,States,PartialStates),
	exec(PartialStates,Plan,NewStates).

execIfTS([],_,_,_,[]).

execIfTS([S|States],C,ThenPlan,ElsePlan,NewStates):-
	true_in_state(C,S), !,
	exec([S],ThenPlan,ThenStates),
	execIfTS(States,C,ThenPlan,ElsePlan,PartialStates),
	append(ThenStates,PartialStates,NewStates).

execIfTS([S|States],C,ThenPlan,ElsePlan,NewStates):-
	false_in_state(C,S) ->
	(exec([S],ElsePlan,ThenStates),
	 execIfTS(States,C,ThenPlan,ElsePlan,PartialStates),
	 append(ThenStates,PartialStates,NewStates));(write('error in if'), nl, abort).

%execIfTS(_,_,_,_,_):-
%	 write('error in if'), nl, abort.

execIfT([],_,_,[]).

execIfT([S|States],C,ThenPlan,NewStates):-
	true_in_state(C,S), !,
	exec([S],ThenPlan,ThenStates),
	execIfT(States,C,ThenPlan,PartialStates),
	append(ThenStates,PartialStates,NewStates).

execIfT([S|States],C,ThenPlan,[S|NewStates]):-
	false_in_state(C,S) ->
	execIfT(States,C,ThenPlan,NewStates);(write('error in if'), nl, abort).

%execIfT(_,_,_,_):-
%	write('error in if'), nl, abort.

execWhile([],_,_,[]).

execWhile([S|States],C,Plan,[S|WhileStates]):-
	false_in_state(C,S), !,
	execWhile(States,C,Plan,WhileStates).

execWhile([S|States],C,Plan,WhileStates):-
	true_in_state(C,S), !,
	exec([S],Plan,NewStates),
	append(NewStates,States,NewWhileStates),
	execWhile(NewWhileStates,C,Plan,WhileStates).

execWhile(_,_,_,_,_) :-  
	write('error in while'), nl, abort.

execAction(Action,States,NewStates):-
	possible_in_each_state(Action,States),
	exec_action(Action,States,NewStates).

possible_in_each_state(_,[]).

possible_in_each_state(Action,[S|States]):-
	possible(Action,S),
	possible_in_each_state(Action,States).

exec_action(_,[],[]).

exec_action(A,[S|States],[NewS|NewStates]):-
	phi(A,S,NewS),
	exec_action(A,States,NewStates).

%=========================================================
execSensingAction(Action,States,NewStates):-
	sensing_possible_in_each_state(Action,States),
	exec_sensing_action(Action,States,NewStates).

sensing_possible_in_each_state(_,[]).

sensing_possible_in_each_state(Action,[S|States]):-
	sensing_possible(Action,S),
	sensing_possible_in_each_state(Action,States).

exec_sensing_action(_,[],[]).

exec_sensing_action(A,[S|States],[NewS1,NewS2|NewStates]):-
	phi(A,S,NewS1,NewS2,_),
	exec_sensing_action(A,States,NewStates).

%======================================================

