/***************************************************************************
* Functions
*
* FunTree.c (Management of binary trees representing functions)
*
* Date:    1 feb 98
* Author:  L. Granvilliers - LIFO Orleans
****************************************************************************/

#include "FunTree.h"
#include <string.h>
#include <ctype.h>
#include <bssolve.h>


/* Global variables needed for the parsing of wrapped strings */
char *FunParseWstring;
int FunParseWstringPos;
int FunParseUnilex;

FunTree *FunFreeTree(FunTree *f)
/***************************************************************************
*  To desallocate a tree
*/
{
  if( f==NULL ) return( NULL );

  switch( FunTypeTree(f) )
  {
    case FunNodeOp:
         FunFreeTree(FunLeftTree(f));
         FunFreeTree(FunRightTree(f));
         break;

    case FunNodeVar:
         free(FunVarTree(f));
         break;

    case FunNodeItv:
         /*  free(FunItvTree(f));  */
         break;
  }
/*  free(FunItvBwd(f));
  free(FunItvFwd(f));
*/
  free(f);
  return( NULL );
}


FunTree *FunCopyTree(FunTree *f)
/***************************************************************************
*  To make a copy of f
*/
{
  if( f==NULL ) return( NULL );

  switch( FunTypeTree(f) )
  {
    case FunNodeOp:
         return( FunNewOpTree(FunOpTree(f),
                              FunDleftTree(f),
                              FunDrightTree(f),
                              FunDallTree(f),
                              FunCopyTree(FunLeftTree(f)),
                              FunCopyTree(FunRightTree(f))) );

    case FunNodeVar:
         return( FunNewVarTree(FunGlobVarTree(f),FunLocVarTree(f)) );

    case FunNodeItv:
         return( FunNewItvTree(FiaNewCopyI(FunItvTree(f))) );
  }
}


FunTree *FunNewOpTree(FiaEvalOpI evalII,
                      FunEvalBwdPath evalDleft,
                      FunEvalBwdPath evalDright,
                      FunEvalBwdPath evalDall,
                      FunTree *l,
                      FunTree *r)
/***************************************************************************
*  To create a tree'snode for the operator evalII with subtrees l and r
*/
{
  FunTree *f = (FunTree *)malloc(sizeof(FunTree));
  FunTypeTree(f) = FunNodeOp;

  FunStructOpTree(f) = (struct FunTvalFunc *)malloc(sizeof(struct FunTvalFunc));
  FunOpTree(f) = evalII;
  FunDleftTree(f) = evalDleft;
  FunDrightTree(f) = evalDright;
  FunDallTree(f) = evalDall;

  FunLeftTree(f) = l;
  FunRightTree(f) = r;

  return( f );
}

FunTree *FunNewExpTree(int exp)
/***************************************************************************
*  To create a tree'snode for the interval [exp,exp]
*/
{
/*  FiaBounds *i = FiaNewRRtoI(exp,exp);*/


  FunTree *f = (FunTree *)malloc(sizeof(FunTree));
  FiaSetRR(FunItvTree(f),exp,exp);
  FiaCopyI(FunItvFwd(f),FunItvTree(f));   /* The value during the forward evaluation
                                 which will never be changed ! */

  FunLeftTree(f) = FunRightTree(f) = NULL;
  FunTypeTree(f) = FunNodeItv;

  return( f );
}

FunTree *FunNewItvTree(FiaItv i)
/***************************************************************************
*  To create a tree'snode for the interval i
*/
{
  FunTree *f = (FunTree *)malloc(sizeof(FunTree));

  FiaCopyI(FunItvFwd(f),i);   /* The value during the forward evaluation
                                 which will never be changed ! */
  FiaCopyI(FunItvTree(f),i);
  free(i);

  FunLeftTree(f) = FunRightTree(f) = NULL;
  FunTypeTree(f) = FunNodeItv;

  return( f );
}


FunTree *FunNewEmptyItvTree()
/***************************************************************************
*  To create a tree'snode for the interval i
*/
{
  FunTree *f = (FunTree *)malloc(sizeof(FunTree));

  FunLeftTree(f) = FunRightTree(f) = NULL;
  FunTypeTree(f) = FunNodeItv;

  return( f );
}


FunTree *FunNewUselessTree()
/***************************************************************************
*  To create a tree's node needed for the forward evaluation
*  in the call of FunEvalFwdTree
*/
{
  FunTree *f = (FunTree *)malloc(sizeof(FunTree));
  FunTypeTree(f) = FunNodeUseless;
  return( f );
}

FunTree *FunNewVarTree(long globvar, long locvar)
/***************************************************************************
*  To create a tree's node for the variable globvar
*/
{
  FunTree *f = (FunTree *)malloc(sizeof(FunTree));
  FunLeftTree(f) = FunRightTree(f) = NULL;
  FunTypeTree(f) = FunNodeVar;
  FunVarTree(f) = (int *)malloc(2*sizeof(int));
  FunGlobVarTree(f) = globvar;
  FunLocVarTree(f) = locvar;
  return( f );
}


int FunTreeZero(FunTree *f)
/***************************************************************************
*  To test if f is equal to 0
*/
{
  if( FunTypeTree(f)==FunNodeItv )
  {
    if( FiaItvZero(FunItvTree(f)) ) return( 1 );
    else return( 0 );
  }
  else return( 0 );
}

int FunExistVarInTree(FunTree *f)
/***************************************************************************
*  To test if f contains one variable at least
*/
{
  if( f==NULL ) return( 0 );
  if( FunTypeTree(f)==FunNodeVar ) return( 1 );
  else if( FunTypeTree(f)==FunNodeOp )
  {
    if( FunExistVarInTree(FunLeftTree(f)) ) return( 1 );
    else return( FunExistVarInTree(FunRightTree(f)) );
  }
  else return( 0 );
}


void FunEvalFwdConstantTree(FunTree *f)
/***************************************************************************
*  To evaluate the real function represented by f which contains no variable
*/
{
  if( f==NULL ) return;

  if( FunTypeTree(f)==FunNodeOp )
  {
    FunEvalFwdConstantTree(FunLeftTree(f));
    FunEvalFwdConstantTree(FunRightTree(f));
    (* FunOpTree(f))(FunItvFwd(f),
                     FunItvFwd(FunLeftTree(f)),
                     FunItvFwd(FunRightTree(f)));
  }
}


FunTree *FunRemoveConstantTrees(FunTree *f)
/***************************************************************************
*  To evaluate and remove from f the sub-trees with no variables
   and containing one operator at least
*/
{
  int l,r;
  FunTree *newf;
  FiaBounds *newi;

  if( FunTypeTree(f)==FunNodeOp )
  {
    l = FunExistVarInTree((FunTree *)FunLeftTree(f));
    r = FunExistVarInTree((FunTree *)FunRightTree(f));
    if( !(l || r) )   /* Sub-tree which has to be evaluated */
    {
      FunEvalFwdConstantTree(f);
      /* The interval constant is now in FunItvFwd(f); */
      newi = FiaNewCopyI(FunItvFwd(f));
      newf = FunNewItvTree(newi);
      FunFreeTree(f);
      return( newf );
    }
    else
    {
      FunLeftTree(f) = FunRemoveConstantTrees(FunLeftTree(f));
      FunRightTree(f) = FunRemoveConstantTrees(FunRightTree(f));
      return( f );
    }
  }
  else return( f );
}


int FunUnitTree(FunTree *f)
/***************************************************************************
*  Auxiliary function used to write a function's tree 
*  in order to write the minimum of brackets for the infix operators
*  considering their usual priorities
*/
{
  if( FunTypeTree(f) != FunNodeOp )         return( 1 );
  else
  if( FunOpTree(f)==FunOpSqrtI )            return( 1 );
  else
  if( (    (FunOpTree(f)==FunOpSqrI) 
            || (FunOpTree(f)==FunOpPowI))
        && (FunUnitTree(FunLeftTree(f))) )  return( 1 );
  else                                      return( 0 );
}


void FunWriteTree(FILE *out, FunTree *f, int digits)
/***************************************************************************
*  To write f on out
*/
{
  if( f==NULL ) return;

  switch( FunTypeTree(f) )
  {
    case FunNodeVar :
         fprintf(out,"v(%ld)",FunVarTree(f));
         return;

    case FunNodeItv :
         FiaWriteI(out,FunItvTree(f),digits);
         return;

    case FunNodeOp :
         if( FunOpTree(f)==FunOpAddII )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"+");
           FunWriteTree(out,FunRightTree(f),digits);
	 }
         else if( FunOpTree(f)==FunOpAddRI )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"+");
           FunWriteTree(out,FunRightTree(f),digits);
	 }
         else if( FunOpTree(f)==FunOpSubII )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"-");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpSubRI )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"-");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpSubIR )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"-");
           FunWriteTree(out,FunRightTree(f),digits);
	 }
         else if( FunOpTree(f)==FunOpNegI )
         {
           fprintf(out,"-");
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunLeftTree(f),digits);
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpMulII )
         {
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunLeftTree(f),digits);
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,")");
           fprintf(out,"*");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpMulRI )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"*");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpMulRnegI )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"*");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpMulRposI )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"*");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpDivII )
         {
          if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunLeftTree(f),digits);
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,")");
           fprintf(out,"/");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpDivIR )
         {
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunLeftTree(f),digits);
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,")");
           fprintf(out,"/");
           FunWriteTree(out,FunRightTree(f),digits);
	 }
         else if( FunOpTree(f)==FunOpDivRI )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"/");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpDivIRneg )
         {
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunLeftTree(f),digits);
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,")");
           fprintf(out,"/");
           FunWriteTree(out,FunRightTree(f),digits);
	 }
         else if( FunOpTree(f)==FunOpDivIRpos )
         {
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunLeftTree(f),digits);
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,")");
           fprintf(out,"/");
           FunWriteTree(out,FunRightTree(f),digits);
	 }
         else if( FunOpTree(f)==FunOpDivRnegI )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"/");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpDivRposI )
         {
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,"/");
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunRightTree(f),digits);
           if( !FunUnitTree(FunRightTree(f)) ) fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpSqrI )
         {
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunLeftTree(f),digits);
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,")");
           fprintf(out,"^2");
	 }
         else if( FunOpTree(f)==FunOpSqrtI )
         {
           fprintf(out,"sqrt(");
           FunWriteTree(out,FunLeftTree(f),digits);
           fprintf(out,")");
	 }
         else if( FunOpTree(f)==FunOpPowI )
         {
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,"(");
           FunWriteTree(out,FunLeftTree(f),digits);
           if( !FunUnitTree(FunLeftTree(f)) ) fprintf(out,")");
           fprintf(out,"^%d",(int)FiaMinI(FunItvTree(FunRightTree(f))));
	 }
       }
}



void FunEvalFwdTreeBis(FunFunction *FunF)
/***************************************************************************
*  To evaluate the real function represented by FunF->f
*    - d contains the domains of the other variables
*/
{
  struct FunLtreeIn *l = FunF->EvalFwd->first;

  while( l!=NULL )
  {
    switch( FunTypeTree(l->node) )
    {
      case FunNodeOp:
           (* FunOpTree(l->node))(FunItvFwd(l->node),
                                  FunItvFwd(FunLeftTree(l->node)),
                                  FunItvFwd(FunRightTree(l->node)));
           break;

      case FunNodeVar:
           FiaCopyI(FunItvFwd(l->node),FunGetExternDom(FunGlobVarTree(l->node)));
           break;
    }
    l = l->next;
  }
}



 


void FunEvalFwdTree(FunFunction *FunF, FiaItv ivar, long globvar)
/***************************************************************************
*  To evaluate the real function represented by FunF->f
*    - ivar is the domain of the variable of global index globvar
*    - d contains the domains of the other variables
*/
{
  struct FunLtreeIn *l = FunF->EvalFwd->first;

  while( l!=NULL )
  {
    switch( FunTypeTree(l->node) )
    {
      case FunNodeOp:
           (* FunOpTree(l->node))(FunItvFwd(l->node),
                                  FunItvFwd(FunLeftTree(l->node)),
                                  FunItvFwd(FunRightTree(l->node)));
           break;

      case FunNodeVar:
           if( FunGlobVarTree(l->node)==globvar )
                { FiaCopyI(FunItvFwd(l->node),ivar); }
           else { FiaCopyI(FunItvFwd(l->node),
                           FunGetExternDom(FunGlobVarTree(l->node))); }
           break;
    }
    l = l->next;
  }
}


void FunEvalCenterFwdTree(FunFunction *FunF)
/***************************************************************************
*  To evaluate the real function represented by FunF->f
*    - ivar is the domain of the variable of global index globvar
*    - d contains the domains of the other variables
*/
{
  struct FunLtreeIn *l = FunF->EvalFwd->first;
  FiaBounds *i;

  while( l!=NULL )
  {
    switch( FunTypeTree(l->node) )
    {

      case FunNodeOp:
           (* FunOpTree(l->node))(FunItvFwd(l->node),
                                  FunItvFwd(FunLeftTree(l->node)),
                                  FunItvFwd(FunRightTree(l->node)));
           break;

      case FunNodeVar:
               i = FunGetExternDom(FunGlobVarTree(l->node));
               FiaRoundDown;
               FiaMinI(FunItvFwd(l->node)) = FiaMidI(i);
               FiaRoundUp;
               FiaMaxI(FunItvFwd(l->node)) = FiaMidI(i);
           break;
    }
    l = l->next;
  }
}


void FunReEvalFwdTree(FunFunction *FunF, FiaItv ivar, long globvar, long locvar)
/***************************************************************************
*  
*/
{
  struct FunLtreeIn *l = FunF->vars[locvar].ReEvalFwd->first;

  while( l!=NULL )
  {
    switch( FunTypeTree(l->node) )
    {
      case FunNodeOp:
           (* FunOpTree(l->node))(FunItvFwd(l->node),
                                  FunItvFwd(FunLeftTree(l->node)),
                                  FunItvFwd(FunRightTree(l->node)));
           break;

      case FunNodeVar:
           if( FunGlobVarTree(l->node)==globvar )
                { FiaCopyI(FunItvFwd(l->node),ivar); }
           else { FiaCopyI(FunItvFwd(l->node),
                           FunGetExternDom(FunGlobVarTree(l->node))); }
           break;
    }
    l = l->next;
  }
}


/***************************************************************************
* PARTIAL DERIVATIVES
****************************************************************************/

void FunEvalBwdAddII(FunTree *f)
{
  FiaCopyI(FunItvBwd(FunLeftTree(f)),FunItvBwd(f));
  FiaCopyI(FunItvBwd(FunRightTree(f)),FunItvBwd(f));
}

void FunEvalBwdLeftAddII(FunTree *f)  /* d(u+v)/d(u) = 1 */
{
  FiaCopyI(FunItvBwd(FunLeftTree(f)),FunItvBwd(f));
}

void FunEvalBwdRightAddII(FunTree *f)  /* d(u+v)/d(v) = 1 */
{
  FiaCopyI(FunItvBwd(FunRightTree(f)),FunItvBwd(f));
}

void FunEvalBwdSubII(FunTree *f)
{
  FiaItv useless;
  FiaCopyI(FunItvBwd(FunLeftTree(f)),FunItvBwd(f));
  FiaNegI(FunItvBwd(FunRightTree(f)),FunItvBwd(f),useless);
}

void FunEvalBwdLeftSubII(FunTree *f)  /* d(u-v)/d(u) = 1 */
{
  FiaCopyI(FunItvBwd(FunLeftTree(f)),FunItvBwd(f));
}

void FunEvalBwdRightSubII(FunTree *f)  /* d(u*v)/d(v) = -1 */
{
  FiaItv useless;
  FiaNegI(FunItvBwd(FunRightTree(f)),FunItvBwd(f),useless);
}

void FunEvalBwdNegI(FunTree *f)  /* d(-u)/d(u) = -1 */
{
  FiaItv useless;
  FiaNegI(FunItvBwd(FunLeftTree(f)),FunItvBwd(f),useless);
}

void FunEvalBwdMulII(FunTree *f)
{
  FiaMulII(FunItvBwd(FunLeftTree(f)),
           FunItvBwd(f),
           FunItvFwd(FunRightTree(f)));

  FiaMulII(FunItvBwd(FunRightTree(f)),
           FunItvBwd(f),
           FunItvFwd(FunLeftTree(f)));
}

void FunEvalBwdLeftMulII(FunTree *f)  /* d(u*v)/d(u) = v */
{
  FiaMulII(FunItvBwd(FunLeftTree(f)),
           FunItvBwd(f),
           FunItvFwd(FunRightTree(f)));
}

void FunEvalBwdRightMulII(FunTree *f)  /* d(u*v)/d(v) = u */
{
  FiaMulII(FunItvBwd(FunRightTree(f)),
           FunItvBwd(f),
           FunItvFwd(FunLeftTree(f)));
}


void FunEvalBwdDivII(FunTree *f)
{
  FiaItv i1, i2, i3;

  FiaNegI(i1,FunItvFwd(FunLeftTree(f)),i1);  /* i1 = -u */
  FiaSqrI(i2,FunItvFwd(FunRightTree(f)),i2);    /* i2 = v^2 */
  FiaDivII(i3,i1,i2);   /* i3 = -u/v^2 */

  FiaMulII(FunItvBwd(FunRightTree(f)),
           FunItvBwd(f),
           i3);

  FiaDivRposIinternal(i1,1.0,FunItvFwd(FunRightTree(f)));  /* i1 = 1/v */
  FiaMulII(FunItvBwd(FunLeftTree(f)),
           FunItvBwd(f),
           i1);
}

void FunEvalBwdLeftDivII(FunTree *f)  /* d(u/v)/d(u) = 1/v */
{
  FiaItv i1;

  FiaDivRposIinternal(i1,1.0,FunItvFwd(FunRightTree(f)));  /* i1 = 1/v */
  FiaMulII(FunItvBwd(FunLeftTree(f)),
           FunItvBwd(f),
           i1);
}

void FunEvalBwdRightDivII(FunTree *f)  /* d(u/v)/d(v) = -u/v^2 */
{
  FiaItv i1, i2, i3;

  FiaNegI(i1,FunItvFwd(FunLeftTree(f)),i1);  /* i1 = -u */
  FiaSqrI(i2,FunItvFwd(FunRightTree(f)),i2);    /* i2 = v^2 */
  FiaDivII(i3,i1,i2);   /* i3 = -u/v^2 */

  FiaMulII(FunItvBwd(FunRightTree(f)),
           FunItvBwd(f),
           i3);
}

void FunEvalBwdSqrI(FunTree *f)  /* d(u^2) = 2*u */
{
  FiaItv i1;
  FiaMulRI(i1,
           FunItvTree(FunRightTree(f)),
           FunItvFwd(FunLeftTree(f)));  /* i1 = 2*u */

  FiaMulII(FunItvBwd(FunLeftTree(f)),
           FunItvBwd(f),
           i1);
}

void FunEvalBwdPowI(FunTree *f)  /* d(u^n) = n*u^(n-1) */
{
  FiaItv i1, i2;
  int n = FiaMinI(FunItvTree(FunRightTree(f)));

  FiaPowIinternal(i1,FunItvFwd(FunLeftTree(f)),n-1);  /* i1 = u^(n-1) */

  FiaMulRposIinternal(i2,n,i1);  /* i2 = n*u^(n-1) */

  FiaMulII(FunItvBwd(FunLeftTree(f)),
           FunItvBwd(f),
           i2);
}

void FunEvalBwdSqrtI(FunTree *f)  /* d(sqrt(u)) = 0.5/sqrt(u) */
{
  FiaItv i1;
  /* sqrt(u) = FunItvFwd(f) */

  FiaDivRposIinternal(i1,0.5,FunItvFwd(f));  /* i1 = 0.5/sqrt(u) */

  FiaMulII(FunItvBwd(FunLeftTree(f)),
           FunItvBwd(f),
           i1);
}



void FunEvalBwdTreeOne(FunFunction *func, long locvar)
/***************************************************************************
*  To evaluate the partial derivative D[func->f]/D[locvar]
*/
{
  FiaBounds *sum = FunLocVarDeriv(func,locvar);
  struct FunLevalTreeIn *l = func->vars[locvar].EvalPath->first;
  struct FunLtreeIn *lt = func->vars[locvar].VarNodes->first;

  /* All the backward intervals on the paths associated to the variable
     must be evaluated. Finally, the partial derivative will be contained
     in FunLocVarDeriv(func,locvar) */

  while( l!=NULL )
  {
    (*(l->eval))(l->node);    /* evaluation of one backward interval */
    l = l->next;
  }
  
  /* This interval will contain the partial derivative computed as a sum
     of all the backward intervals for the nodes containing an occurrence
     of this variable -> it is initialized to [0,0] */
  FiaSetRR(sum,0.0,0.0);

  while( lt!=NULL )
  {
    FiaAddII(sum,sum,FunItvBwd(lt->node));
    lt = lt->next;
  }
}


void FunEvalBwdTreeAllRec(FunTree *f)
/***************************************************************************
*  To evaluate all the backward intervals if f
*/
{
  if( f==NULL ) return;

  if( FunTypeTree(f)==FunNodeOp )
  {
    (* FunDallTree(f))(f);
    FunEvalBwdTreeAllRec(FunLeftTree(f));
    FunEvalBwdTreeAllRec(FunRightTree(f));
  }
}


void FunEvalBwdTreeAll(FunFunction *func)
/***************************************************************************
*  To evaluate all the partial derivatives of func->f
*/
{
  int i;
  FiaBounds *sum;
  struct FunLtreeIn *lt;
  struct FunLevalTreeIn *l = func->EvalBwd->first;

  /* All the backward intervals must be evaluated */
  while( l!=NULL )
  {
    (*(l->eval))(l->node);  /* evaluation of one backward interval */
    l = l->next;
  }


  /* All the backward intervals for variables must be added */
  for( i=0; i<func->Nvar; i++ )
  {
    sum = FunLocVarDeriv(func,i);
    FiaSetRR(sum,0.0,0.0);
    lt = func->vars[i].VarNodes->first;

    while( lt!=NULL )
    {
      FiaAddII(sum,sum,FunItvBwd(lt->node));
      lt = lt->next;
    }
  }
}



/***************************************************************************
* LIST OF TREE NODES needed for partial derivative computations
****************************************************************************/

FunLevalTree *FunLevalTreeNew()
/***************************************************************************
*  Creation
*/
{
  FunLevalTree *l = (FunLevalTree *)malloc(sizeof(FunLevalTree));
  l->first = l->end = NULL;
  return( l );
}

void FunLevalTreeFree(FunLevalTree *l)
/***************************************************************************
*  Deallocation
*/
{
  struct FunLevalTreeIn *l0, *l1 = l->first;
  while( l1!=NULL )
  {
    l0 = l1;
    l1 = l1->next;
    free(l0);
  }
  free( l );
}

void FunLevalTreeAdd(FunLevalTree *l, FunTree *f, FunEvalBwdPath e)
/***************************************************************************
*  Addition in l of the node f associated to the backward evaluation
*  function e which is applied during the computation of the partial
*  derivative with respect to the corresponding variable
*/
{
  struct FunLevalTreeIn *l1 = (struct FunLevalTreeIn *)
                               malloc(sizeof(struct FunLevalTreeIn));
  l1->node = f;
  l1->eval = e;
  l1->next = NULL;
  if( l->first==NULL )  /* add first */
  {
    l->first = l->end = l1;
  }
  else  /* add end */
  {
    l->end->next = l1;
    l->end = l1;
  }
}



FunLtree *FunLtreeNew()
/***************************************************************************
*  Creation
*/
{
  FunLtree *l = (FunLtree *)malloc(sizeof(FunLtree));
  l->first = l->end = NULL;
  return( l );
}

void FunLtreeFree(FunLtree *l)
/***************************************************************************
*  Dellocation
*/
{
  struct FunLtreeIn *l0, *l1 = l->first;
  while( l1!=NULL )
  {
    l0 = l1;
    l1 = l1->next;
    free(l0);
  }
  free( l );
}

void FunLtreeAddEnd(FunLtree *l, FunTree *f)
/***************************************************************************
*  To add a node f at the end of l
*/
{
  struct FunLtreeIn *l1 = (struct FunLtreeIn *)malloc(sizeof(struct FunLtreeIn));

  l1->node = f;
  l1->next = NULL;
  if( l->first==NULL )  /* add first */
  {
    l->first = l->end = l1;
  }
  else  /* add end */
  {
    l->end->next = l1;
    l->end = l1;
  }
}

void FunLtreeAddFirst(FunLtree *l, FunTree *f)
/***************************************************************************
*  To add a node f at the beginning of l
*/
{
  struct FunLtreeIn *l1 = (struct FunLtreeIn *)malloc(sizeof(struct FunLtreeIn));
  l1->node = f;
  l1->next = l->first;
  l->first = l1;
  if( l->end==NULL ) l->end = l->first;
}


void FunLtreeDelEnd(FunLtree *l)
/***************************************************************************
*  To add a node f at the beginning of l
*/
{
  struct FunLtreeIn *l1 = l->first;

  if( l->first==l->end )
  { 
    free( l->first);
    l->first = l->end = NULL;
    return;
  }

  while( l1->next != l->end ) l1 = l1->next;

  free(l1->next);
  l1->next = NULL;
  l->end = l1;
}


int FunLocVarInTree(FunTree *f, long locvar)
/***************************************************************************
*  To test if the variable of local index locvar is in f
*/
{
  if( f==NULL ) return( 0 );
  if( FunTypeTree(f)==FunNodeVar )
  {
    if( FunLocVarTree(f)==locvar ) return( 1 );
    else return( 0 );
  }
  else if( FunTypeTree(f)==FunNodeOp )
  {
    if( FunLocVarInTree(FunLeftTree(f),locvar) ) return( 1 );
    else return( FunLocVarInTree(FunRightTree(f),locvar) );
  }
  else return( 0 );
}


void FunCreateOneEvalPath(FunTree *f, long locvar,
                          FunLevalTree *leval, FunLtree *reeval)
/***************************************************************************
*  For the local variable locvar, creation in leval of the list of nodes which
*  must be backward evaluated to compute the partial derivative d(f)/d(locvar)
*  In reeval, creation of the list of nodes which must be forward evaluated
*  in order to compute f'(X) after the forward evaluation of f(center(X))
*  in the Newton narrowing operator
*/
{
  int l,r;

  if( f==NULL ) return;

  if( FunTypeTree(f)==FunNodeVar )  /* it is necessary locvar */
  {
    FunLtreeAddFirst(reeval,f);
  }

  l = FunLocVarInTree(FunLeftTree(f),locvar);
  r = FunLocVarInTree(FunRightTree(f),locvar);
  if( l )
  {
    if( r )
    {
      FunLevalTreeAdd(leval,f,FunDallTree(f));
      FunLtreeAddFirst(reeval,f);
      FunCreateOneEvalPath(FunLeftTree(f),locvar,leval,reeval);
      FunCreateOneEvalPath(FunRightTree(f),locvar,leval,reeval);
    }
    else
    {
      FunLevalTreeAdd(leval,f,FunDleftTree(f));
      FunLtreeAddFirst(reeval,f);
      FunCreateOneEvalPath(FunLeftTree(f),locvar,leval,reeval);
    }
  }
  else if( r )
  {
    FunLevalTreeAdd(leval,f,FunDrightTree(f));
    FunLtreeAddFirst(reeval,f);
    FunCreateOneEvalPath(FunRightTree(f),locvar,leval,reeval);
  }
}


void FunCreateEvalPaths(FunFunction *func)
/***************************************************************************
*  For each local variable, creation of the list of nodes which must be
*  backward evaluated to compute the partial derivative for this variable
*/
{
  int i;
  for( i=0; i<func->Nvar; i++ )
  {
    FunCreateOneEvalPath(func->f,i,func->vars[i].EvalPath,func->vars[i].ReEvalFwd);

    /* Removal in ReEvalFwd of the root of func->f which is the final element */
    FunLtreeDelEnd(func->vars[i].ReEvalFwd);
  }
}




/***************************************************************************
* CREATION OF FUNCTIONS AFTER THE PARSING OF THEIR EXPRESSIONS
****************************************************************************/


int FunInLocVar(FunLocVar *vars, int Nvar, long globvar)
/***************************************************************************
*  To test if the variable of global index globvar was added
*  in the array vars of local variables
*/
{
  int i;
  for( i=0; i<Nvar; i++ )
  {
    if( vars[i].globindex==globvar ) return( i );
  }
  return( -1 );
}


int FunLocVarAdd(FunFunction *func, long globvar)
/***************************************************************************
*  To add the variable of global index globvar in the array func->vars
*  of the local variables appearing in func->f
*/
{
  int i = FunInLocVar(func->vars,func->Nvar,globvar);
  if( i<0 )
  {
    if( func->Nvar == 0 )        /* then func->vars was not created */
    {
      func->vars = (FunLocVar *)malloc(FunLocVarSizeUnit*sizeof(FunLocVar));
      func->Nfree = FunLocVarSizeUnit;
    }

    if( func->Nfree == 0 )       /* func->vars is full */
    {
      func->vars = (FunLocVar *)
                    realloc(func->vars,(func->Nvar + FunLocVarSizeUnit) *
                                        sizeof(FunLocVar));
      func->Nfree = FunLocVarSizeUnit;
    }

    i = func->Nvar;  /* local index of the variable */

    /* Initializations */
    func->vars[i].globindex = globvar;
    func->vars[i].Nocc = 0;
    func->vars[i].VarNodes = FunLtreeNew();
    func->vars[i].EvalPath = FunLevalTreeNew();
    func->vars[i].ReEvalFwd = FunLtreeNew();

    func->Nvar ++;
    func->Nfree --;
   }

  /* One more occurrence */
  FunLocVarNocc(func,i) ++;

  return( i );
}


void FunCreateLocVars(FunFunction *func, FunTree *f)
/***************************************************************************
*  To create the array for the local variables appearing in f
*/
{
  int var;
  if( FunTypeTree(f)==FunNodeVar )
  {
    /* addition of the variable in the local array */
    var = FunLocVarAdd(func,FunGlobVarTree(f));

    /* set the local index in the tree */
    FunLocVarTree(f) = var;

    /* add the node in func->vars[var].VarNodes */
    FunLtreeAddEnd(func->vars[var].VarNodes,f);
  }
  else if( FunTypeTree(f)==FunNodeOp )
  {
    if( (FunOpTree(f)==FunOpNegI) || (FunOpTree(f)==FunOpSqrtI) ||
        (FunOpTree(f)==FunOpSqrI) || (FunOpTree(f)==FunOpPowI) )
    {
      /* Only the left sub-tree of f has to be considered */
      FunCreateLocVars(func,FunLeftTree(f));
    }
    else
    {
      FunCreateLocVars(func,FunLeftTree(f));
      FunCreateLocVars(func,FunRightTree(f));
    }
  }
}


void FunCreateEvalPathFwdBwd(FunTree *f, FunLtree *fwd, FunLevalTree *bwd)
/***************************************************************************
*  
*/
{
  int l,r;

  if( FunTypeTree(f)==FunNodeVar )
  {
    FunLtreeAddFirst(fwd,f);
  }
  else if( FunTypeTree(f)==FunNodeOp )
  {
    FunLtreeAddFirst(fwd,f);

    if( FunTypeTree(FunLeftTree(f))==FunNodeItv )
    {
      FunLevalTreeAdd(bwd,f,FunDrightTree(f));
      FunCreateEvalPathFwdBwd(FunRightTree(f),fwd,bwd);
    }
    else if( FunTypeTree(FunRightTree(f))==FunNodeItv )
    {
      FunLevalTreeAdd(bwd,f,FunDleftTree(f));
      FunCreateEvalPathFwdBwd(FunLeftTree(f),fwd,bwd);
    }
    else
    {
      FunLevalTreeAdd(bwd,f,FunDallTree(f));
      FunCreateEvalPathFwdBwd(FunLeftTree(f),fwd,bwd);
      FunCreateEvalPathFwdBwd(FunRightTree(f),fwd,bwd);
    }
  }
}


FunFunction *FunCreateF(FunTree *f, int pred)
/***************************************************************************
*  To initialize a function from the tree f representing its expression
*/
{
  FunFunction *func = (FunFunction *)malloc(sizeof(FunFunction));

  func->Nfree = 0;  /* no place in func->vars */
  func->Nvar = 0;   /* no variables in func->vars */
  func->f = f;
  func->pred = pred;

  /* To evaluate and remove from f the sub-trees with no variables
     and containing one operator at least */
  FunRemoveConstantTrees(f);

  /* Creation of the informations local to func associated to the variables
     in func->f */
  FunCreateLocVars(func,f);

  /* For each variable x in f, creation of the list of the nodes of f which
     have to be (backward) evaluated for the computation of Df/Dx */
  FunCreateEvalPaths(func);

  func->EvalFwd = FunLtreeNew();
  func->EvalBwd = FunLevalTreeNew();
  FunCreateEvalPathFwdBwd(func->f,func->EvalFwd,func->EvalBwd);

  /* The backward interval of the tree's root must be initialized to [1,1]
     for every backward evaluation -> this will never change */
  FiaSetRR(FunItvBwd(func->f),1.0,1.0);

  func->vars = (FunLocVar *)realloc(func->vars,func->Nvar*sizeof(FunLocVar));

  return( func );
}


void FunFreeF(FunFunction *FunF)
/***************************************************************************
*  Deallocation of FunF
*/
{
  int i;
  for( i=0; i<FunF->Nvar; i++ )
  {
    /* FiaFreeI(FunF->vars[i].deriv);*/
    FunLtreeFree(FunF->vars[i].VarNodes);
    FunLevalTreeFree(FunF->vars[i].EvalPath);
    FunLtreeFree(FunF->vars[i].ReEvalFwd);
  }
  free(FunF->vars); 
  FunFreeTree(FunF->f);
  FunLtreeFree(FunF->EvalFwd);
  FunLevalTreeFree(FunF->EvalBwd);
  free(FunF);
}






/***************************************************************************
* INTERFACE WITH BSSSOLVE
****************************************************************************/


void FunNewtonInter(int uselesstag, FiaBounds **res, int *size,
                    FiaBounds *itv1, int size1, FiaBounds *itv2, int size2)
/***************************************************************************
*  To intersect two domains -> to be used in multifold
*  The result is the interval *res allocated in this function
*  which has the size sizeof(FiaItv)
*/
{
  FiaBounds *itv = FiaNewI();
  /*  size = (int *)malloc(sizeof(int)); */
  *size = sizeof(FiaItv);

  FiaInterII(itv,itv1,itv2);

  *res = itv;
}


FiaBounds *FunGetExternDom(long globvar)
/***************************************************************************
*  To get the external domain of a variable  -> addition of the internal
*  representation in BSSolve structures if it does not exist yet
*/
{
  FiaBounds *itv;
  BssVar *bsrep;
  char *bsstring;
  void *locrep;

  locrep = LocVarGetRep(globvar,NEWTON_REP);

  if( locrep==NULL )
  {
    /* get the bssolve rep. of the variable */
    bsrep = LocVarGetVar(globvar);

    /* get the wrapped string representing the domain of the variable */

    bsstring = Bss2wstring(VarValue(bsrep));
    /*
    printf( "VARIABLE %d -> %s -> ", globvar, bsstring);
    */
    /* create the interval in my representation */
    itv = wstring2FiaItv(bsstring);
    /*
    FiaWriteI( stdout, itv, 25);
      putchar('\n');
    */
    /* add the representation in the structure of the variable */
    VarAddRep(bsrep,NEWTON_REP,(void *)itv);

    return( itv );    
  }
  else
    {
      itv = ((FiaBounds **)locrep)[0];
      /*
      printf( "VARIABLE %d DEJA VUE -> ", globvar);
      FiaWriteI( stdout, itv, 25);
      putchar('\n');
      */
      return( itv );
    }
}



FunFunction *ParseFunTree(char * s)
/***************************************************************************
*  Parsing of the constraint represented by the wrapped string in s
*/
{
  FunTree *fun1, *fun2, *fun3;
  int pred, ipred;

/*
printf("\nWSTRING:!%s!\n\n",s);
*/


  FunParseWstring = s; FunParseWstringPos = 0;

  FunParseWstringPos+=2;    /* o( */
  
  pred = Pred();
  ipred = InvPred(pred);

        FunParseWstringPos++;    /* ( */
  fun1 = Expr();
        FunParseWstringPos++;    /* , */  
  fun2 = Expr();
        FunParseWstringPos++;    /* ) */

  if( FunTreeZero(fun1) )
  {
    if( FunTreeZero(fun2) )  /* 0=0, 0>=0, 0<=0 */
    {
       /* then do nothing */
       FunFreeTree(fun1);
       FunFreeTree(fun2);
    }
    else  /* 0 pred f */
    {
      FunFreeTree(fun1);
      return( FunCreateF(fun2,ipred) );
    }
  }
  else if( FunTreeZero(fun2) )  /* f pred 0 */
  {
    FunFreeTree(fun2);
    return( FunCreateF(fun1,pred) );
  }
  else  /* f pred g */
  {
    fun3 = FunNewOpTree(FunOpSubII,
                        (FunEvalBwdPath)FunEvalBwdLeftSubII,
                        (FunEvalBwdPath)FunEvalBwdRightSubII,
                        (FunEvalBwdPath)FunEvalBwdSubII,
                        fun1,
                        fun2);
/*
printf("\nPARSE TREE:!");
FunWriteTree(stdout,fun3,16);
printf("!\n\n");
*/

    return( FunCreateF(fun3,pred) );
  }
}


int InvPred(int pred)
{
  if( pred==FunCstrEq ) return( FunCstrEq );
  else if( pred==FunCstrSup ) return( FunCstrInf );
  else if( pred==FunCstrInf ) return( FunCstrSup );
}


int Pred()
{
  if( FunParseWstring[FunParseWstringPos]=='=' )
  {
    FunParseWstringPos++;
    return( FunCstrEq );
  }
  else if( FunParseWstring[FunParseWstringPos]=='>' )
  {
    FunParseWstringPos+=2;
    return( FunCstrSup );
  }
  else
  {
    FunParseWstringPos+=2;
    return( FunCstrInf );
  }
}


void GetIdent( char *s)
{
  int i=0;
  FunParseWstringPos++;
  while( (isalpha(FunParseWstring[FunParseWstringPos])) && (FunParseWstring[FunParseWstringPos]!='\0') )
  {
    s[i] = FunParseWstring[FunParseWstringPos];
    FunParseWstringPos++;
    i++;
  }
  s[i] = '\0';
}


void GetTok()
{
  char c;
  char s[40];

  c = FunParseWstring[FunParseWstringPos];
  FunParseWstringPos++;

  switch(c) {
  case 'o':
    FunParseWstringPos++; /* ( */
    c = FunParseWstring[FunParseWstringPos];
    FunParseWstringPos++; /* .( */
    switch(c){
    case '+': FunParseUnilex = ADD; return;
    case '-': FunParseUnilex = SUB; return;
    case '*': FunParseUnilex = MUL; return;
    case '/': FunParseUnilex = DIV; return;
    default:
      FunParseWstringPos-=2;
      GetIdent(s);
      if( strcmp(s,"pow")==0 )
	{
	  FunParseUnilex = POW; return;
	}
      else if( strcmp(s,"neg")==0 )
	{
	  FunParseUnilex = MINUS; return;
	}
      else if( strcmp(s,"sqr")==0 )
	{
	  FunParseUnilex = SQR; return;
	}
      else if( strcmp(s,"sqrt")==0 )
	{
	  FunParseUnilex = SQRT; return;
	}
    }
  case '(': FunParseUnilex = LBR; return;
  case ')': FunParseUnilex = RBR; return;
  case ',': FunParseUnilex = SEP; return;
  case '\0': FunParseUnilex = END; return;
  case 'i': FunParseUnilex = TOKINT; return;
  case 'f': FunParseUnilex = TOKRE; return;
  case 'q': FunParseUnilex = TOKRAT; return;
  case 'r': FunParseUnilex = TOKITV; return;
  case 'v': FunParseUnilex = VAR; return;
  }
}


FunTree * Expr()
{
  FunTree *fun1, *fun2;
  FiaBounds *r1;
  int n;

  GetToken
  switch( FunParseUnilex )
  {
    case ADD:
       FunParseWstringPos++; /* ( */
       fun1 = Expr();
       FunParseWstringPos++; /* , */
       fun2 = Expr();
       FunParseWstringPos+=2; /* )) */
       return( FunNewOpTree(FunOpAddII,
                           (FunEvalBwdPath)FunEvalBwdLeftAddII,
                           (FunEvalBwdPath)FunEvalBwdRightAddII,
                           (FunEvalBwdPath)FunEvalBwdAddII,
                           fun1,
                           fun2) );

    case SUB:
       FunParseWstringPos++; /* ( */
       fun1 = Expr();
       FunParseWstringPos++; /* , */
       fun2 = Expr();
       FunParseWstringPos+=2; /* )) */
       return( FunNewOpTree(FunOpSubII,
                           (FunEvalBwdPath)FunEvalBwdLeftSubII,
                           (FunEvalBwdPath)FunEvalBwdRightSubII,
                           (FunEvalBwdPath)FunEvalBwdSubII,
                           fun1,
                           fun2) );

    case MUL:
       FunParseWstringPos++; /* ( */
       fun1 = Expr();
       FunParseWstringPos++; /* , */
       fun2 = Expr();
       FunParseWstringPos+=2; /* )) */
       return( FunNewOpTree(FunOpMulII,
                           (FunEvalBwdPath)FunEvalBwdLeftMulII,
                           (FunEvalBwdPath)FunEvalBwdRightMulII,
                           (FunEvalBwdPath)FunEvalBwdMulII,
                           fun1,
                           fun2) );

    case DIV:
       FunParseWstringPos++; /* ( */
       fun1 = Expr();
       FunParseWstringPos++; /* , */
       fun2 = Expr();
       FunParseWstringPos+=2; /* )) */
       return( FunNewOpTree(FunOpDivII,
                           (FunEvalBwdPath)FunEvalBwdLeftDivII,
                           (FunEvalBwdPath)FunEvalBwdRightDivII,
                           (FunEvalBwdPath)FunEvalBwdDivII,
                           fun1,
                           fun2) );

   case POW:
       FunParseWstringPos++; /* ( */
       fun1 = Expr();
       FunParseWstringPos+=2; /* ,i */
       n = ExprInt();
       FunParseWstringPos+=2; /* ) */
       return( FunNewOpTree(FunOpPowI,
                           (FunEvalBwdPath)FunEvalBwdPowI,
                           (FunEvalBwdPath)FunEvalBwdPowI,
                           (FunEvalBwdPath)FunEvalBwdPowI,
                           fun1,
                           FunNewExpTree(n)) );

   case MINUS:
       FunParseWstringPos++; /* ( */
       fun1 = Expr();
       FunParseWstringPos+=2; /* )) */
       return( FunNewOpTree(FunOpNegI,
                           (FunEvalBwdPath)FunEvalBwdNegI,
                           (FunEvalBwdPath)FunEvalBwdNegI,
                           (FunEvalBwdPath)FunEvalBwdNegI,
                           fun1,
                           FunNewUselessTree()) );

   case SQRT:
       FunParseWstringPos++; /* ( */
       fun1 = Expr();
       FunParseWstringPos+=2; /* )) */
       return( FunNewOpTree(FunOpSqrtI,
                           (FunEvalBwdPath)FunEvalBwdSqrtI,
                           (FunEvalBwdPath)FunEvalBwdSqrtI,
                           (FunEvalBwdPath)FunEvalBwdSqrtI,
                           fun1,
                           FunNewUselessTree()) );

   case SQR:
       FunParseWstringPos++; /* ( */
       fun1 = Expr();
       FunParseWstringPos+=2; /* )) */
       return( FunNewOpTree(FunOpSqrI,
                           (FunEvalBwdPath)FunEvalBwdSqrI,
                           (FunEvalBwdPath)FunEvalBwdSqrI,
                           (FunEvalBwdPath)FunEvalBwdSqrI,
                           fun1,
                           FunNewExpTree(2)) );

   case TOKINT:
       n = ExprInt();
       return( FunNewItvTree(FiaNewRRtoI(n,n)) );

   case TOKRE:
       return( FunNewItvTree(ExprFloat()) );

   case TOKRAT:
       return( FunNewItvTree(ExprRat()) );

   case TOKITV:
       return( FunNewItvTree(ExprItv()) );

   case VAR:
       return( FunNewVarTree(ExprLong(),-1) );
  }
}


int ExprInt()
{
  int n, i, j;
  char strnum[MAXCARFLOAT];

  FunParseWstringPos++;   /* ( */
  j = 0;
  i = FunParseWstringPos;
  while( FunParseWstring[i]!=')' )
  {
    strnum[j] = FunParseWstring[i];
    j++; i++;
  }
  strnum[j] = '\0';
  FunParseWstringPos = i;
  FunParseWstringPos++;   /* ) */
  return( FiaStrToInt(strnum) );
}


FiaBounds *ExprFloat()
{
  double x1,x2;
  char strnum[MAXCARFLOAT];
  int i,j;

  FunParseWstringPos++;   /* ( */
  j = 0;
  i = FunParseWstringPos;
  while( FunParseWstring[i]!=')' )
  {
    strnum[j] = FunParseWstring[i];
    j++; i++;
  }
  strnum[j] = '\0';
  FunParseWstringPos = i;
  FunParseWstringPos++;   /* ) */
  FiaRoundDown;
  x1 = FiaStrToR(strnum);
  FiaRoundUp;
  x2 = FiaStrToR(strnum);

  return( FunNewItvTree(FiaNewRRtoI(x1,x2)) );
}

long ExprLong()
{
  int i, j;
  long n;
  char strnum[MAXCARFLOAT];

  FunParseWstringPos++;   /* ( */
  j = 0;
  i = FunParseWstringPos;
  while( FunParseWstring[i]!=')' )
  {
    strnum[j] = FunParseWstring[i];
    j++; i++;
  }
  strnum[j] = '\0';
  FunParseWstringPos = i;
  FunParseWstringPos++;   /* ) */
  return( FiaStrToLong(strnum) );
}

FiaBounds *ExprItv()
{
  double x1,x2;
  char strnum1[MAXCARFLOAT], strnum2[MAXCARFLOAT];
  int i,j;

  FunParseWstringPos+=4;   /* ([i( */

  /* LEFT BOUND */
  j = 0;
  i = FunParseWstringPos;
  while( FunParseWstring[i]!=')' )
  {
    strnum1[j] = FunParseWstring[i];
    j++; i++;
  }
  strnum1[j] = '\0';
  FunParseWstringPos = i;


  FunParseWstringPos+=5;   /* )..f( */


  /* RIGHT BOUND */
  j = 0;
  i = FunParseWstringPos;
  while( FunParseWstring[i]!=')' )
  {
    strnum2[j] = FunParseWstring[i];
    j++; i++;
  }
  strnum2[j] = '\0';
  FunParseWstringPos = i;


  FunParseWstringPos+=3;   /* )]) */

  FiaRoundDown;
  x1 = FiaStrToR(strnum1);
  FiaRoundUp;
  x2 = FiaStrToR(strnum2);

  return FiaNewRRtoI(x1,x2);
}


FiaBounds *ExprRat()
{
  double x1,x2;
  char strnum1[MAXCARFLOAT], strnum2[MAXCARFLOAT], s1[50], s2[50],
       exp1[10], exp2[10], sf1[50], sf2[50];
  int i, j, sgn, len1, len2;
  mpq_t q; mpz_t num; mpz_t den;
  mpf_t f1, f2;
  long exponent1, exponent2;

  FunParseWstringPos++;   /* ( */

  /* LEFT BOUND */
  j = 0;

  i = FunParseWstringPos;
  if( FunParseWstring[i]=='+' ) i++;

  while( FunParseWstring[i]!=':' )
  {
    strnum1[j] = FunParseWstring[i];
    j++; i++;
  }
  strnum1[j] = '\0';
  FunParseWstringPos = i;


  FunParseWstringPos++;   /* : */


  /* RIGHT BOUND */
  j = 0;
  i = FunParseWstringPos;
  while( FunParseWstring[i]!=')' )
  {
    strnum2[j] = FunParseWstring[i];
    j++; i++;
  }
  strnum2[j] = '\0';
  FunParseWstringPos = i;

  FunParseWstringPos++;   /* ) */

  mpz_init(den);
  mpz_init(num);
  mpz_set_str(num,strnum1,10);
  mpz_set_str(den,strnum2,10);
  mpq_init(q);
  mpq_set_num(q,num);
  mpq_set_den(q,den);
  mpq_canonicalize(q);


  sgn = mpq_sgn(q);
  mpf_init2(f1,51); mpf_init2(f2,51);
  FiaRoundDown;
  mpf_set_q(f1,q);
  mpf_get_str(s1,&exponent1,10,50,f1);
  FiaRoundUp;
  mpf_set_q(f2,q);
  mpf_get_str(s2,&exponent2,10,50,f2);

  len1 = strlen(s1);
  len2 = strlen(s2);
  if( exponent1>len1 ) for( i=0; i<exponent1-len1; i++ ) strcat(s1,"0");
  if( exponent2>len2 ) for( i=0; i<exponent2-len2; i++ ) strcat(s2,"0");

  exponent1 = strlen(s1) - exponent1;
  exponent2 = strlen(s2) - exponent2;
  sprintf(exp1,"%ld",exponent1);
  sprintf(exp2,"%ld",exponent2);

  if( sgn==-1 ) { strcpy(sf1,"-"); strcpy(sf2,"-"); }
  else { strcpy(sf1,""); strcpy(sf2,""); }
  strcat(sf1,s1); strcat(sf2, s2);
  strcat(sf1,"e-");  strcat(sf1,exp1);
  strcat(sf2,"e-");  strcat(sf2,exp2);
  FiaRoundDown;
  x1 = FiaStrToR(sf1);
  FiaRoundUp;
  x2 = FiaStrToR(sf2);

  mpf_clear(f1); mpf_clear(f2);      
  mpz_clear(den); mpz_clear(num);
  mpq_clear(q);

  return( FiaNewRRtoI(x1,x2) );
}



FiaBounds *wstring2FiaItv(char *s)
{
  double x1,x2;
  char strnum1[MAXCARFLOAT], strnum2[MAXCARFLOAT];
  int i,j, bp;

  bp = 5; /* r([i( */

  /* LEFT BOUND */
  j = 0;
  i = bp;
  while( s[i]!=')' )
  {
    strnum1[j] = s[i];
    j++; i++;
  }
  strnum1[j] = '\0';
  bp = i;


  bp+=5;   /* )..f( */


  /* RIGHT BOUND */
  j = 0;
  i = bp;
  while( s[i]!=')' )
  {
    strnum2[j] = s[i];
    j++; i++;
  }
  strnum2[j] = '\0';
  bp = i;

  if( strcmp(strnum1,"-Inf")==0 ) x1 = FiaNegInf;
  else if( strcmp(strnum1,"+Inf")==0 ) x1 = FiaPlusInf;
  else
  {
    FiaRoundDown;
    x1 = FiaStrToR(strnum1);
  }

  if( strcmp(strnum2,"-Inf")==0 ) x2 = FiaNegInf;
  else if( strcmp(strnum2,"+Inf")==0 ) x2 = FiaPlusInf;
  else
  {
    FiaRoundUp;
    x2 = FiaStrToR(strnum2);
  }

  return( FiaNewRRtoI(x1,x2) );
}

