/***************************************************************************
* Funtional Interval Arithmetic
*
* FiaInterval.c (basic types and functions)
*
* Date:    1 feb 98
* Author:  L. Granvilliers - LIFO Orleans
****************************************************************************/
/*************************************************************************/
/*                                                                       */
/*       Copyright (C) 1998 Universite de Nantes                         */
/*                                                                       */
/*************************************************************************/

#include "FiaInterval.h"
#include <stdlib.h>
#include <string.h>
#include <bssolve.h>


FiaBounds *FiaNewI()
/***************************************************************************
*  Allocation of an interval
*/
{
  return( (FiaBounds *)malloc(sizeof(FiaBounds)) );
}

FiaBounds *FiaNewLargestI()
/***************************************************************************
*  Allocation of [-oo,+oo]
*/
{
  FiaBounds *i;
  i = (FiaBounds *)malloc(sizeof(FiaBounds));
  FiaMinI(i) = FiaNegInf;
  FiaMaxI(i) = FiaPlusInf; 
  return( i );
}


void FiaToLargestI(FiaItv i)
/***************************************************************************
*  i <- [-oo,+oo]
*/
{
  FiaMinI(i) = FiaNegInf;
  FiaMaxI(i) = FiaPlusInf; 
}


FiaBounds *FiaNewCopyI(FiaItv i)
/***************************************************************************
*  Allocation of an interval intialized to i
*/
{
  FiaBounds *i1;
  i1 = (FiaBounds *)malloc(sizeof(FiaBounds));
  FiaMinI(i1) = FiaMinI(i);
  FiaMaxI(i1) = FiaMaxI(i); 
  return( i1 );
}


FiaBounds *FiaNewRRtoI(FiaR x1, FiaR x2)
/***************************************************************************
*  Allocation of an interval initialized to [x1,x2]
*/
{
  FiaBounds *i;
  i = (FiaBounds *)malloc(sizeof(FiaBounds));
  FiaMinI(i) = x1;
  FiaMaxI(i) = x2; 
  return( i );
}


void FiaToIntegerI(FiaItv i)
/***************************************************************************
*  i=[a,b] <- [ceil(a),floor(b)]
*  (i where the bounds are rounded to integers)
*/
{
  FiaMinI(i) = ceil(FiaMinI(i));
  FiaMaxI(i) = floor(FiaMaxI(i)); 
}


void FiaAbsI(FiaItv Result, FiaItv i)
/***************************************************************************
*  Result = |i|
*/
{
  if( FiaMinI(i)>=0.0 )
  {
    FiaMinI(Result) = FiaMinI(i);
    FiaMaxI(Result) = FiaMaxI(i);
    return;
  }

  if( FiaMaxI(i)<=0.0 )
  {
    FiaMinI(Result) = -(FiaMaxI(i));
    FiaMaxI(Result) = -(FiaMinI(i));
    return;
  }

  FiaMinI(Result) = 0.0;
  FiaMaxI(Result) = FiaMaxRR(-(FiaMinI(i)),FiaMaxI(i));
}


void FiaAddII(FiaItv Result, FiaItv i1, FiaItv i2)
/***************************************************************************
*  Result = i1 + i2
*/
{
  FiaRoundDown; FiaMinI(Result) = FiaMinI(i1) + FiaMinI(i2);
  FiaRoundUp;   FiaMaxI(Result) = FiaMaxI(i1) + FiaMaxI(i2);
}


void FiaAddRI(FiaItv Result, FiaItv x, FiaItv i)
/***************************************************************************
*  Result = (x=[r,r]) + i
*/
{
  FiaRoundDown; FiaMinI(Result) = FiaMinI(x) + FiaMinI(i);
  FiaRoundUp;   FiaMaxI(Result) = FiaMinI(x) + FiaMaxI(i);
}


void FiaSubII(FiaItv Result, FiaItv i1, FiaItv i2)
/***************************************************************************
*  Result = i1 - i2
*/
{
  FiaRoundDown; FiaMinI(Result) = FiaMinI(i1) - FiaMaxI(i2);
  FiaRoundUp;   FiaMaxI(Result) = FiaMaxI(i1) - FiaMinI(i2);
}


void FiaSubRI(FiaItv Result, FiaItv x, FiaItv i)
/***************************************************************************
*  Result = (x=[r,r]) - i
*/
{
  FiaRoundDown; FiaMinI(Result) = FiaMinI(x) - FiaMaxI(i);
  FiaRoundUp;   FiaMaxI(Result) = FiaMinI(x) - FiaMinI(i);
}


void FiaSubIR(FiaItv Result, FiaItv i, FiaItv x)
/***************************************************************************
*  Result = i - (x=[r,r])
*/
{
  FiaRoundDown; FiaMinI(Result) = FiaMinI(i) - FiaMinI(x);
  FiaRoundUp;   FiaMaxI(Result) = FiaMaxI(i) - FiaMinI(x);
}


void FiaNegI(FiaItv Result, FiaItv i, FiaItv useless)
/***************************************************************************
*  Result = -i
*/
{
  FiaMinI(Result) = -(FiaMaxI(i));
  FiaMaxI(Result) = -(FiaMinI(i));
}


void FiaMulII(FiaItv Result, FiaItv i1, FiaItv i2)
/***************************************************************************
*  Result = i1*i2
*/
{
  FiaR acDn, adDn, bcDn, bdDn, acUp, adUp, bcUp, bdUp;

  FiaRoundDown;
  acDn = FiaMulRR(FiaMinI(i1),FiaMinI(i2));
  adDn = FiaMulRR(FiaMinI(i1),FiaMaxI(i2));
  bcDn = FiaMulRR(FiaMaxI(i1),FiaMinI(i2));
  bdDn = FiaMulRR(FiaMaxI(i1),FiaMaxI(i2));

  if( acDn<adDn )
  {
    if( bcDn<bdDn )
    {
      FiaMinI(Result) = FiaMinRR(acDn,bcDn);
      FiaRoundUp;
      adUp = FiaMulRR(FiaMinI(i1),FiaMaxI(i2));
      bdUp = FiaMulRR(FiaMaxI(i1),FiaMaxI(i2));
      FiaMaxI(Result) = FiaMaxRR(adUp,bdUp);
    }
    else
    {
      FiaMinI(Result) = FiaMinRR(acDn,bdDn);
      FiaRoundUp;
      adUp = FiaMulRR(FiaMinI(i1),FiaMaxI(i2));
      bcUp = FiaMulRR(FiaMaxI(i1),FiaMinI(i2));
      FiaMaxI(Result) = FiaMaxRR(adUp,bcUp);
    }
  }
  else
  {
    if( bcDn<bdDn )
    {
      FiaMinI(Result) = FiaMinRR(adDn,bcDn);
      FiaRoundUp;
      acUp = FiaMulRR(FiaMinI(i1),FiaMinI(i2));
      bdUp = FiaMulRR(FiaMaxI(i1),FiaMaxI(i2));
      FiaMaxI(Result) = FiaMaxRR(acUp,bdUp);
    }
    else
    {
      FiaMinI(Result) = FiaMinRR(adDn,bdDn);
      FiaRoundUp;
      acUp = FiaMulRR(FiaMinI(i1),FiaMinI(i2));
      bcUp = FiaMulRR(FiaMaxI(i1),FiaMinI(i2));
      FiaMaxI(Result) = FiaMaxRR(acUp,bcUp);
    }
  }
}


void FiaMulRI(FiaItv Result, FiaItv x, FiaItv i)
/***************************************************************************
*  Result = (x=[r,r])*i
*/
{
  FiaRoundDown;
  if( FiaMinI(x)<0 )
  {
    FiaMinI(Result) = FiaMulRR(FiaMinI(x),FiaMaxI(i));
    FiaRoundUp;
    FiaMaxI(Result) = FiaMulRR(FiaMinI(x),FiaMinI(i));
  }
  else
  {
    FiaMinI(Result) = FiaMulRR(FiaMinI(x),FiaMinI(i));
    FiaRoundUp;
    FiaMaxI(Result) = FiaMulRR(FiaMinI(x),FiaMaxI(i));
  }
}


void FiaMulRnegI(FiaItv Result, FiaItv x, FiaItv i)
/***************************************************************************
*  Result = (x=[r,r])*i with r<0
*/
{
  FiaRoundDown;
  FiaMinI(Result) = FiaMulRR(FiaMinI(x),FiaMaxI(i));
  FiaRoundUp;
  FiaMaxI(Result) = FiaMulRR(FiaMinI(x),FiaMinI(i));
}


void FiaMulRposI(FiaItv Result, FiaItv x, FiaItv i)
/***************************************************************************
*  Result = (x=[r,r])*i with r>=0
*/
{
  FiaRoundDown;
  FiaMinI(Result) = FiaMulRR(FiaMinI(x),FiaMinI(i));
  FiaRoundUp;
  FiaMaxI(Result) = FiaMulRR(FiaMinI(x),FiaMaxI(i));
}


void FiaMulRposIinternal(FiaItv Result, FiaR x, FiaItv i)
/***************************************************************************
*  Result = x*i with x>=0
*/
{
  FiaRoundDown;
  FiaMinI(Result) = FiaMulRR(x,FiaMinI(i));
  FiaRoundUp;
  FiaMaxI(Result) = FiaMulRR(x,FiaMaxI(i));
}


void FiaDivII(FiaItv Result, FiaItv i1, FiaItv i2)
/***************************************************************************
*  Result = i1 / i2
*/
{
  FiaR acDn, adDn, bcDn, bdDn, acUp, adUp, bcUp, bdUp;

  if( FiaRinI(i2,0.0) )
  {
    FiaMinI(Result) = FiaNegInf;
    FiaMaxI(Result) = FiaPlusInf;
    return;
  }
  FiaRoundDown;
  acDn = FiaDivRR(FiaMinI(i1),FiaMinI(i2));
  adDn = FiaDivRR(FiaMinI(i1),FiaMaxI(i2));
  bcDn = FiaDivRR(FiaMaxI(i1),FiaMinI(i2));
  bdDn = FiaDivRR(FiaMaxI(i1),FiaMaxI(i2));

  if( acDn<adDn )
  {
    if( bcDn<bdDn )
    {
      FiaMinI(Result) = FiaMinRR(acDn,bcDn);
      FiaRoundUp;
      adUp = FiaDivRR(FiaMinI(i1),FiaMaxI(i2));
      bdUp = FiaDivRR(FiaMaxI(i1),FiaMaxI(i2));
      FiaMaxI(Result) = FiaMaxRR(adUp,bdUp);
    }
    else
    {
      FiaMinI(Result) = FiaMinRR(acDn,bdDn);
      FiaRoundUp;
      adUp = FiaDivRR(FiaMinI(i1),FiaMaxI(i2));
      bcUp = FiaDivRR(FiaMaxI(i1),FiaMinI(i2));
      FiaMaxI(Result) = FiaMaxRR(adUp,bcUp);
    }
  }
  else
  {
    if( bcDn<bdDn )
    {
      FiaMinI(Result) = FiaMinRR(adDn,bcDn);
      FiaRoundUp;
      acUp = FiaDivRR(FiaMinI(i1),FiaMinI(i2));
      bdUp = FiaDivRR(FiaMaxI(i1),FiaMaxI(i2));
      FiaMaxI(Result) = FiaMaxRR(acUp,bdUp);
    }
    else
    {
      FiaMinI(Result) = FiaMinRR(adDn,bdDn);
      FiaRoundUp;
      acUp = FiaDivRR(FiaMinI(i1),FiaMinI(i2));
      bcUp = FiaDivRR(FiaMaxI(i1),FiaMinI(i2));
      FiaMaxI(Result) = FiaMaxRR(acUp,bcUp);
    }
  }
}


void FiaDivIR(FiaItv Result, FiaItv i, FiaItv x)
/***************************************************************************
*  Result = i / (x=[r,r])
*/
{
  if( FiaMinI(x)==0.0 )
  {
    FiaMinI(Result) = FiaNegInf;
    FiaMaxI(Result) = FiaPlusInf;
    return;
  }

  FiaRoundDown;
  if( FiaMinI(x)<0 )
  {
    FiaMinI(Result) = FiaDivRR(FiaMaxI(i),FiaMinI(x));
    FiaRoundUp;
    FiaMaxI(Result) = FiaDivRR(FiaMinI(i),FiaMinI(x));
  }
  else
  {
    FiaMinI(Result) = FiaDivRR(FiaMinI(i),FiaMinI(x));
    FiaRoundUp;
    FiaMaxI(Result) = FiaDivRR(FiaMaxI(i),FiaMinI(x));
  }
}


void FiaDivIRneg(FiaItv Result, FiaItv i, FiaItv x)
/***************************************************************************
*  Result = i / (x=[r,r]) with r<0
*/
{
  FiaRoundDown;
  FiaMinI(Result) = FiaDivRR(FiaMaxI(i),FiaMinI(x));
  FiaRoundUp;
  FiaMaxI(Result) = FiaDivRR(FiaMinI(i),FiaMinI(x));
}


void FiaDivIRpos(FiaItv Result, FiaItv i, FiaItv x)
/***************************************************************************
*  Result = i / (x=[r,r]) with r>0
*/
{
  FiaRoundDown;
  FiaMinI(Result) = FiaDivRR(FiaMinI(i),FiaMinI(x));
  FiaRoundUp;
  FiaMaxI(Result) = FiaDivRR(FiaMaxI(i),FiaMinI(x));
}


void FiaDivRI(FiaItv Result, FiaItv x, FiaItv i)
/***************************************************************************
*  Result = (x=[r,r]) / i
*/
{
  if( FiaRinI(i,0.0) )
  {
    FiaMinI(Result) = FiaNegInf;
    FiaMaxI(Result) = FiaPlusInf;
    return;
  }

  FiaRoundDown;
  if( FiaMinI(x)<0 )
  {
    FiaMinI(Result) = FiaDivRR(FiaMinI(x),FiaMinI(i));
    FiaRoundUp;
    FiaMaxI(Result) = FiaDivRR(FiaMinI(x),FiaMaxI(i));
  }
  else
  {
    FiaMinI(Result) = FiaDivRR(FiaMinI(x),FiaMaxI(i));
    FiaRoundUp;
    FiaMaxI(Result) = FiaDivRR(FiaMinI(x),FiaMinI(i));
  }
}

void FiaDivRnegI(FiaItv Result, FiaItv x, FiaItv i)
/***************************************************************************
*  Result = [*x,*x] / i with *x<0
*/
{
  if( FiaRinI(i,0.0) )
  {
    FiaMinI(Result) = FiaNegInf;
    FiaMaxI(Result) = FiaPlusInf;
    return;
  }

  FiaRoundDown;
  FiaMinI(Result) = FiaDivRR(FiaMinI(x),FiaMinI(i));
  FiaRoundUp;
  FiaMaxI(Result) = FiaDivRR(FiaMinI(x),FiaMaxI(i));
}


void FiaDivRposI(FiaItv Result, FiaItv x, FiaItv i)
/***************************************************************************
*  Result = [*x,*x] / i with *x >=0
*/
{
  if( FiaRinI(i,0.0) )
  {
    FiaMinI(Result) = FiaNegInf;
    FiaMaxI(Result) = FiaPlusInf;
    return;
  }

  FiaRoundDown;
  FiaMinI(Result) = FiaDivRR(FiaMinI(x),FiaMaxI(i));
  FiaRoundUp;
  FiaMaxI(Result) = FiaDivRR(FiaMinI(x),FiaMinI(i));
}


void FiaDivRposIinternal(FiaItv Result, FiaR x, FiaItv i)
/***************************************************************************
*  Result = [x,x] / i with x >=0
*/
{
  if( FiaRinI(i,0.0) )
  {
    FiaMinI(Result) = FiaNegInf;
    FiaMaxI(Result) = FiaPlusInf;
    return;
  }

  FiaRoundDown;
  FiaMinI(Result) = FiaDivRR(x,FiaMaxI(i));
  FiaRoundUp;
  FiaMaxI(Result) = FiaDivRR(x,FiaMinI(i));
}


void FiaSqrI(FiaItv Result, FiaItv i, FiaItv useless)
/***************************************************************************
*  Result = i^2
*/
{
  FiaR x;

  if( FiaMinI(i)>=0.0 )
  {
    FiaRoundDown;
    FiaMinI(Result) = FiaMulRR(FiaMinI(i),FiaMinI(i));
    FiaRoundUp;
    FiaMaxI(Result) = FiaMulRR(FiaMaxI(i),FiaMaxI(i));
    return;
  }

  if( FiaMaxI(i)<=0.0 )
  {
    FiaRoundDown;
    FiaMinI(Result) = FiaMulRR(FiaMaxI(i),FiaMaxI(i));
    FiaRoundUp;
    FiaMaxI(Result) = FiaMulRR(FiaMinI(i),FiaMinI(i));
    return;
  }
  FiaMinI(Result) = 0.0;
  x = FiaMaxRR(-(FiaMinI(i)),FiaMaxI(i));
  FiaMaxI(Result) = FiaMulRR(x,x);
}


void FiaSqrtI(FiaItv Result, FiaItv i, FiaItv useless)
/***************************************************************************
*  Result = sqrt(i)
*/
{
  if( FiaMinI(i)>=0.0 )
  {
    FiaRoundDown;
    FiaMinI(Result) = sqrt(FiaMinI(i));
    FiaRoundUp;
    FiaMaxI(Result) = sqrt(FiaMaxI(i));
    return;
  }

  if( FiaMaxI(i)<=0.0 )
  {
    FiaRoundDown;
    FiaMinI(Result) = sqrt(-(FiaMaxI(i)));
    FiaRoundUp;
    FiaMaxI(Result) = sqrt(-(FiaMinI(i)));
    return;
  }
  FiaMinI(Result) = 0.0;
  FiaMaxI(Result) = sqrt(FiaMaxRR(-(FiaMinI(i)),FiaMaxI(i)));
}


FiaR FiaPowerRInt(FiaR x, int n)
/***************************************************************************
*  Return x^n
*/
{
  FiaR y = x, z = 1.0;

  for( ;; )
  {
    if( FiaOddInt(n) )
    {
      z *= y;
      n >>= 1;
      if( n==0 ) return( z );
    }
    else n >>= 1;
    y *= y;
  }
}


void FiaPowI(FiaItv Result, FiaItv i, FiaItv n)
/***************************************************************************
*  Result =i^j where n=[j,j] 
*/
{
  FiaItv i1;

  if( FiaEvenInt((int)FiaMinI(n)) )
  {
    FiaAbsI(i1,i);
    FiaRoundDown;
    FiaMinI(Result) = FiaPowerRInt(FiaMinI(i1),FiaMinI(n));
    FiaRoundUp;
    FiaMaxI(Result) = FiaPowerRInt(FiaMaxI(i1),FiaMinI(n));
    return;
  }

  FiaRoundDown;
  FiaMinI(Result) = FiaPowerRInt(FiaMinI(i),FiaMinI(n));
  FiaRoundUp;
  FiaMaxI(Result) = FiaPowerRInt(FiaMaxI(i),FiaMinI(n));
}


void FiaPowIinternal(FiaItv Result, FiaItv i, int n)
/***************************************************************************
*  Result =i^n
*/
{
  FiaItv i1;

  if( FiaEvenInt(n) )
  {
    FiaAbsI(i1,i);
    FiaRoundDown;
    FiaMinI(Result) = FiaPowerRInt(FiaMinI(i1),n);
    FiaRoundUp;
    FiaMaxI(Result) = FiaPowerRInt(FiaMaxI(i1),n);
    return;
  }

  FiaRoundDown;
  FiaMinI(Result) = FiaPowerRInt(FiaMinI(i),n);
  FiaRoundUp;
  FiaMaxI(Result) = FiaPowerRInt(FiaMaxI(i),n);
}


/***************************************************************************
*  Result =exp(i)
void FiaExpI(FiaI *Result, FiaI *i, FiaI *useless)

{
  FiaMinI(Result) = exp(FiaMinI(i))-FiaEpsilon;
  FiaMaxI(Result) = exp(FiaMaxI(i))+FiaEpsilon;
  if( FiaMinI(Result)<0.0 ) FiaMinI(Result) = 0.0;
}
void FiaLogI(FiaI *Result, FiaI *i, FiaI *useless)

{
  if( FiaMaxI(i)<=0.0 )
  {
    FiaMaxI(Result) = FiaNegInf;
    return;
  }
  if( FiaMinI(i)<=0.0 )
  {
    FiaMinI(Result) = FiaNegInf;
    FiaMaxI(Result) = log(FiaMaxI(i))+FiaEpsilon;
    return;
  }
  FiaMinI(Result) = log(FiaMinI(i))-FiaEpsilon;
  FiaMaxI(Result) = log(FiaMaxI(i))+FiaEpsilon;
}
***************************************************************************/


void FiaInterII(FiaItv Result, FiaItv i1, FiaItv i2)
/***************************************************************************
*  Result = i1 inter i2
*/
{
  FiaMinI(Result) = FiaMaxRR(FiaMinI(i1),FiaMinI(i2));
  FiaMaxI(Result) = FiaMinRR(FiaMaxI(i1),FiaMaxI(i2));
}


int FiaIsEqualInterII(FiaItv Result, FiaItv i1, FiaItv i2)
/***************************************************************************
*  Result = i1 inter i2
*  Returns 1 if Result = i1
*/
{
  if( FiaMinI(i2)>FiaMinI(i1) )
  {
    FiaMinI(Result) = FiaMinI(i2);
    if( FiaMaxI(i2)<FiaMaxI(i1) )
    {
      FiaMaxI(Result) = FiaMaxI(i2);
    }
    else
    {
      FiaMaxI(Result) = FiaMaxI(i1);
    }
    return( 0 );
  }
  else
  {
    FiaMinI(Result) = FiaMinI(i1);
    if( FiaMaxI(i2)<FiaMaxI(i1) )
    {
      FiaMaxI(Result) = FiaMaxI(i2);
      return( 0 );
    }
    else
    {
      FiaMaxI(Result) = FiaMaxI(i1);
      return( 1 );
    }
  }
}


void FiaWriteI(FILE *out, FiaItv i, int digits)
/***************************************************************************
*  Write i on out with the digits precision
*/
{
  if( FiaEmptyI(i) )
  {
    fprintf(out,"[empty]");
    return;
  }
  if( FiaFloatI(i) )
  {
    if( FiaMinI(i)>=0 ) fprintf(out,"%.*g",digits,FiaMinI(i));
    else fprintf(out,"%+.*g",digits,FiaMinI(i));
  }
  else
  {
    if( FiaMinI(i)>=0 )
    {
      FiaRoundDown;
      fprintf(out,"[%.*g , ",digits,FiaMinI(i));
      FiaRoundUp;
      if( FiaMaxI(i)==FiaPlusInf ) fprintf(out,"+oo]");
      else fprintf(out,"%.*g]",digits,FiaMaxI(i));
    }
    else
    {
      FiaRoundDown;
      if( FiaMinI(i)==FiaNegInf ) fprintf(out,"[-oo , ");
      else fprintf(out,"[%+.*g , ",digits,FiaMinI(i));
      FiaRoundUp;
      if( FiaMaxI(i)==FiaPlusInf ) fprintf(out,"+oo]");
      else fprintf(out,"%+.*g]",digits,FiaMaxI(i));
    }
  }
}

BssValue *Fia2Bss( FiaItv i)
{
  BssValueP v, l, r;

  FltRoundDownWard;
  l = BssAllocInit( Float, FiaMinI(i));
  FltRoundUpWard;
  r = BssAllocInit( Float, FiaMaxI( i));
  
  v = BssAllocInit( Interval, l,r, 1, 1);

  return v;
}
