/***************************************************************************/
/* interval.h                                                              */
/*                                                                         */
/* Description file for operations on intervals                            */
/*                                                                         */
/* Interval Arithmetic Library                                             */
/*									   */
/* Drawn from Laurent Granvilliers and Fre'de'ric Goualard solvers	   */
/***************************************************************************/

/*************************************************************************/
/*                                                                       */
/*       Copyright (C) 1998 Universite de Nantes                         */
/*                                                                       */
/*************************************************************************/
 
/************* 
 Edit history: 
       27/2/98 - Nicolas Romero : Creation
*************/

#ifndef INTERVAL_H
#define INTERVAL_H

#include <stdio.h>
#include "value.h"
#include "infinity.h"

#define ItvMin MinusInf
#define ItvMax PlusInf

/*-------------------------------------------------------------------------*/
/* BSS INTERVAL                                                            */
/*-------------------------------------------------------------------------*/
struct bssitv {
  char brackets;  /* ----**** */
                  /*     ^^^^_ 1 => right closed */
                  /*     |||__ 1 => right open   */
                  /*     ||___ 1 => left closed  */
                  /*     |____ 1 => left open    */
  BssValueP left;
  BssValueP right;
};

typedef struct bssitv  BssItvType;
typedef  BssItvType BssItv[1];
typedef  BssItvType *BssItvP;

#define LEFT  12  /* 00001100 */
#define RIGHT  3  /* 00000011 */

#define MASK_LEFT  243  /* 11110011 */
#define MASK_RIGHT 252  /* 11111100 */
#define MASK       240  /* 11110000 */

#define LEFT_OPEN    8  /* 00001000 */
#define LEFT_CLOSED  4  /* 00000100 */
#define RIGHT_OPEN   2  /* 00000010 */
#define RIGHT_CLOSED 1  /* 00000001 */

#define ItvBracketsR(r)  (r).brackets
#define ItvMinR(r)     (r).left
#define ItvMaxR(r)     (r).right
#define ItvBracketsRP(r) (r)[0].brackets
#define ItvMinRP(r)    (r)[0].left
#define ItvMaxRP(r)    (r)[0].right

#define ItvLeftBracket(r)   (ItvBracketsRP(r) & LEFT)
#define ItvRightBracket(r)  (ItvBracketsRP(r) & RIGHT)
#define ItvIsLeftOpen(r)    (ItvBracketsRP(r) & LEFT_OPEN)
#define ItvIsRightOpen(r)   (ItvBracketsRP(r) & RIGHT_OPEN) 
#define ItvIsLeftClosed(r)  (ItvBracketsRP(r) & LEFT_CLOSED)  
#define ItvIsRightClosed(r) (ItvBracketsRP(r) & RIGHT_CLOSED) 
#define ItvIsOpen(r)        (ItvIsLeftOpen( r) && ItvIsRightOpen( r))
#define ItvIsClosed(r)      (ItvIsLeftClosed( r) && ItvIsRightClosed( r))

#define ItvGetBrackets(r) ItvBracketsRP(r)
#define ItvSetBrackets(r, b)\
    (ItvBracketsRP( r) = b)
#define ItvSetBracketsOpen(r)\
    (ItvBracketsRP( r) = LEFT_OPEN | RIGHT_OPEN)
#define ItvSetBracketsClosed(r)\
     (ItvBracketsRP( r) = LEFT_CLOSED | RIGHT_CLOSED)
/* #define ItvSetLeftBracket(r, b) \
 *     (ItvBracketsRP( r) = (ItvBracketsRP( r) & MASK_LEFT) | (b))
 * #define ItvSetRightBracket(r, b)\
 *       ((ItvBracketsRP( r)) = (((ItvBracketsRP( r)) & MASK_RIGHT) | (b)))
 */

void ItvSetLeftBracket(BssItv r, char b);
void ItvSetRightBracket(BssItv r, char b);

#define ItvSetLeftBracketOpen(r) \
    (ItvBracketsRP( r) = (ItvBracketsRP( r) & MASK_LEFT) | LEFT_OPEN)
#define ItvSetRightBracketOpen(r)\
     (ItvBracketsRP( r) = (ItvBracketsRP( r) & MASK_RIGHT) | RIGHT_OPEN)
#define ItvSetLeftBracketClosed(r) \
     (ItvBracketsRP( r) = (ItvBracketsRP( r) & MASK_LEFT) | LEFT_CLOSED)
#define ItvSetRightBracketClosed(r) \
     (ItvBracketsRP( r) = (ItvBracketsRP( r) & MASK_RIGHT) | RIGHT_CLOSED)

#define ItvCopyBrackets( r, r1)\
     (ItvBracketsRP( r) = ItvBracketsRP( r1))
#define ItvCopyLeftBracket( r, r1)\
     (ItvBracketsRP( r) = (ItvBracketsRP( r) & MASK_LEFT)\
                          | ItvLeftBracket( r1))
#define ItvCopyRightBracket( r, r1)\
     (ItvBracketsRP( r) = (ItvBracketsRP( r) & MASK_RIGHT)\
                          | ItvRightBracket( r1))
#define ItvExchangeBrackets( r, r1)\
     (ItvBracketsRP( r) = ((ItvBracketsRP( r) & MASK)\
                          | (((ItvRightBracket( r1) << 2) & MASK_RIGHT)\
			     | ItvLeftBracket( r1) >> 2)))
#define ItvInvertBrackets( r, r1)\
     (ItvBracketsRP( r) = ((ItvBracketsRP( r) & MASK)\
                          | ((~ItvRightBracket( r1) & MASK_LEFT)\
			     | (~ItvLeftBracket( r1) & MASK_RIGHT))))
#define ItvInvertLeftBrackets( r, r1)\
     (ItvBracketsRP( r) = ((ItvBracketsRP( r) & MASK)\
                          | ((~ItvRightBracket( r1) & MASK_LEFT)\
			     | (~ItvLeftBracket( r1) & MASK_RIGHT))))
     
#define ItvIsEmptyRange(r) (ItvIsLeftOpen(r) || ItvIsRightOpen( r) \
			         ? (BssCmp(ItvMinRP(r),ItvMaxRP(r),&_itv_cmp)>=0) \
			         : (BssCmp(ItvMinRP(r),ItvMaxRP(r),&_itv_cmp)>0))

int ItvIsZeroRange(BssItv r);

/* #define ItvIsEmptyRange(r)     (ItvIsOpen(r) ? \
			        (ItvMinRP(r)>=ItvMaxRP(r)): \
			        (ItvMinRP(r)>ItvMaxRP(r)))
#define ItvIsNotEmptyRange(r)  (ItvIsOpen(r) ? \
			        (ItvMinRP(r)<ItvMaxRP(r)): \
			        (ItvMinRP(r)<=ItvMaxRP(r)))
#define ItvIsNumInRange(r,x)   (ItvIsOpen(r)? \
			        ((x>ItvMinRP(r)) || (x<ItvMaxRP(r))): \
			        ((x>=ItvMinRP(r)) || (x<=ItvMaxRP(r))))
#define ItvRangeIsEqualRange(r1,r2) \
     ((ItvGetBrackets(r1)==ItvGetBrackets(r2)) \
     && (ItvMinRP(r1)==ItvMinRP(r2)) && (ItvMaxRP(r1)==ItvMaxRP(r2)))
#define ItvSetEmptyRange(r)    (ItvMaxRP(r) = ItvMin)
#define ItvIsCanonicalRange(r) (ItvMaxRP(r)<= FltNextDouble(ItvMinRP(r)))


#define ItvRangeInitR(r,x) ItvMinRP(r)=ItvMaxRP(r)=x

#define ItvWidthRange(r)   (ItvMaxRP(r)-ItvMinRP(r))


#define ItvNarIsCanonicalRange(min,max,prec) (max<=(FltNextDouble(min+prec)))



#define ItvDomIsImproved(r1,r2) (ItvWidthRange(r2)<(ItvWidthRange(r1)*ItvMulImprove))

#define RangeIsReducible(r) (ItvWidthRange(r)>ItvPrecision)
 */

extern double ItvImprove;
extern double ItvPrecision;

void ItvSetMaxRange(BssItv r);
void ItvSetZeroRange(BssItv r);
#define ItvSetEmptyRange(r)    (BssSet( ItvMaxRP(r), Minus_infinity ),\
				BssSet( ItvMinRP(r), Plus_infinity ))

/*-------------------------------------------------------------------------*/
/* BssItv functions                                                         */
/*-------------------------------------------------------------------------*/
BssItvP ItvAlloc();

BssItvP ItvAllocCopy(const BssItv r);
BssItvP ItvAllocInit(const BssValue min, const BssValue max,
		     const char leftclosed, const char rightclosed);
                     /*1 if closed*/

void ItvInit(BssItv r);

void ItvSet(BssItv r, const BssValue min, const BssValue max,
	    const char leftclosed, const char rightclosed); /*1 if closed*/
void ItvCopy(BssItv r, const BssItv r1);
void ItvFree(BssItvP r);

void ItvAdd( BssItv r, const BssItv r1, const BssItv r2);
void ItvSub( BssItv r, const BssItv r1, const BssItv r2);
void ItvMinus( BssItv r, const BssItv r1);
void ItvMul( BssItv r, const BssItv r1, const BssItv r2);
void ItvDiv( BssItv r, const BssItv r1, const BssItv r2);
void ItvSqrt( BssItv r, const BssItv r1);
void ItvAbs( BssItv r, const BssItv r1);
void ItvSqr( BssItv r, const BssItv r1);
void ItvPow( BssItv r, const BssItv r1, const int n);

int ItvInter( BssItv r, const BssItv r1, const BssItv r2);

void ItvWrite(FILE *out, const BssItv r, int digits);

char* Itv2bstring( BssItv v, unsigned int *size);
BssItvP bstring2Itv( char *s);
     
extern BssItv *klic2Itv( void *);
extern void *Itv2klic( const BssItv );

char *Itv2wstring( const BssItv v);
BssItvP wstring2Itv( char *s);

int ItvCmp( const BssItv r1, const BssItv r2, int *cmp);
#endif
